// -*-c++-*-

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "bhv_basic_tackle.h"

#include <rcsc/action/body_tackle_to_point.h>
#include <rcsc/action/neck_turn_to_ball_or_scan.h>

#include <rcsc/player/player_agent.h>
#include <rcsc/player/debug_client.h>
#include <rcsc/player/intercept_table.h>

#include <rcsc/common/logger.h>
#include <rcsc/common/server_param.h>

/*-------------------------------------------------------------------*/
/*!
  execute action
*/
bool
Bhv_BasicTackle::execute( rcsc::PlayerAgent * agent )
{
    const rcsc::WorldModel & wm = agent->world();
    const rcsc::ServerParam & sp = rcsc::ServerParam::i();

    int self_min = wm.interceptTable()->selfReachCycle();
    int mate_min = wm.interceptTable()->teammateReachCycle();
    int opp_min = wm.interceptTable()->opponentReachCycle();

    if ( ! wm.existKickableOpponent()
         || ( opp_min > 3
              && opp_min > self_min
              && opp_min > mate_min )
         || wm.self().tackleProbability() < M_min_prob )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": Bhv_BasicTackle. no kickableopp or low tackle_prob=%.2f",
                            wm.self().tackleProbability() );
        return false;
    }

    //
    // v11 or older
    //

    if ( agent->config().version() < 12.0 )
    {
        double tackle_power = sp.maxTacklePower();
        if ( wm.self().body().abs() < M_body_thr )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                __FILE__": Bhv_BasicTackle. (old) to body dir" );

            agent->debugClient().addMessage( "Tackle+" );
            agent->doTackle( tackle_power );
            agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
            return true;
        }

        tackle_power = - sp.maxBackTacklePower();
        if ( tackle_power < 0.0
             && wm.self().body().abs() > 180.0 - M_body_thr )
        {
            rcsc::dlog.addText( rcsc::Logger::TEAM,
                                __FILE__": Bhv_BasicTackle. (old) to body reverse dir" );

            agent->debugClient().addMessage( "Tackle-" );
            agent->doTackle( tackle_power );
            agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
            return true;
        }

        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": Bhv_BasicTackle. v12- failed." );
        return false;
    }

    //
    // v12+
    //

    const double min_speed = 2.0; // XXX: need to be parameterized.

    const rcsc::AngleDeg ball_rel_angle = wm.ball().rpos().th() - wm.self().body();
    const double tackle_rate
        = sp.tacklePowerRate()
        * ( 1.0 - 0.5 * ball_rel_angle.abs() / 180.0 );

    rcsc::AngleDeg best_angle = 180.0;
    double max_speed = 0.0;

    const double angle_step = M_body_thr * 2.0 / 10;
    rcsc::AngleDeg angle = -M_body_thr;
    for ( int i = 0; i <= 10; ++i, angle += angle_step )
    {
        rcsc::AngleDeg target_rel_angle = angle - wm.self().body();
        double eff_power
            = sp.maxBackTacklePower()
            + ( ( sp.maxTacklePower() - sp.maxBackTacklePower() )
                * ( 1.0 - target_rel_angle.abs() / 180.0 ) );
        eff_power *= tackle_rate;

        rcsc::Vector2D vel = wm.ball().vel()
            + rcsc::Vector2D::polar2vector( eff_power, angle );
        double speed = vel.r();

        if ( vel.th().abs() > M_body_thr
             || speed < min_speed )
        {
            continue;
        }

        if ( speed > max_speed )
        {
            max_speed = speed;
            best_angle = angle;
        }
    }

    if ( max_speed > 0.0 )
    {
        rcsc::dlog.addText( rcsc::Logger::TEAM,
                            __FILE__": Bhv_BasicTackle. angle=%.1f  speed=%.2f",
                            best_angle.degree(), max_speed );
        agent->debugClient().addMessage( "BasicTackle" );

        agent->doTackle( ( best_angle - wm.self().body() ).degree() );
        agent->setNeckAction( new rcsc::Neck_TurnToBallOrScan() );
        return true;
    }

    rcsc::dlog.addText( rcsc::Logger::TEAM,
                        __FILE__": Bhv_BasicTackle. v12+ failed." );
    return false;
}
