// -*-c++-*-

/*!
  \file view_config_dialog.cpp
  \brief View Config Dialog class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <QtGui>

#include "view_config_dialog.h"

#include "main_data.h"
#include "view_config.h"

#include <iostream>
#include <cassert>

/*-------------------------------------------------------------------*/
/*!

*/
ViewConfigDialog::ViewConfigDialog( QWidget * parent,
                                    const MainData & main_data,
                                    ViewConfig & view_config )


    : QDialog( parent )
    , M_main_data( main_data )
    , M_view_config( view_config )
{
    this->setWindowTitle( tr( "View Preference" ) );

    //QFont font = this->font();
    //font.setPointSize( 8 );
    //this->setFont( font );

    createWidgets();
}

/*-------------------------------------------------------------------*/
/*!

*/
ViewConfigDialog::~ViewConfigDialog()
{
    //std::cerr << "delete ViewConfigDialog" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::createWidgets()
{
    M_tab_widget = new QTabWidget();
    QVBoxLayout * top_layout = new QVBoxLayout();

    // canvas
    {
        QFrame * frame = new QFrame();
        QVBoxLayout * layout = new QVBoxLayout();
        layout->setSizeConstraint( QLayout::SetFixedSize );
        layout->setMargin( 4 );
        layout->setSpacing( 4 );

        layout->addWidget( createZoomControls(),
                           0, Qt::AlignLeft );
        layout->addWidget( createObjectSizeControls(),
                           0, Qt::AlignLeft );
        layout->addWidget( createCanvasSizeControls(),
                           0, Qt::AlignLeft );
        layout->addWidget( createFieldStyleControls(),
                           0, Qt::AlignLeft );
        layout->addWidget( createMiscControls(),
                           0, Qt::AlignLeft );

        frame->setLayout( layout );
        M_tab_widget->addTab( frame, tr( "Canvas" ) );
    }
    // show
    {
        QFrame * frame = new QFrame();
        QVBoxLayout * layout = new QVBoxLayout();
        layout->setSizeConstraint( QLayout::SetFixedSize );
        layout->setMargin( 4 );
        layout->setSpacing( 4 );

        layout->addWidget( createShowControls(),
                           0, Qt::AlignLeft );
        layout->addWidget( createPlayersDetailControls(),
                           0, Qt::AlignLeft );
        layout->addWidget( createReverseControls(),
                           0, Qt::AlignLeft );
        layout->addWidget( createGridStepControls(),
                           0, Qt::AlignLeft );
        layout->addWidget( createCompGeomControls(),
                           0, Qt::AlignLeft );

        frame->setLayout( layout );
        M_tab_widget->addTab( frame, tr( "Show" ) );
    }
    // trace / future
    {
        QFrame * frame = new QFrame();
        QVBoxLayout * layout = new QVBoxLayout();
        layout->setSizeConstraint( QLayout::SetFixedSize );
        layout->setMargin( 4 );
        layout->setSpacing( 4 );

        layout->addWidget( createFocusControls(),
                           0, Qt::AlignLeft );
        layout->addWidget( createPlayerSelectionControls(),
                           0, Qt::AlignLeft );
        layout->addWidget( createTraceControls(),
                           0, Qt::AlignLeft );
        layout->addWidget( createInertiaMoveControls(),
                           0, Qt::AlignLeft );

        frame->setLayout( layout );
        M_tab_widget->addTab( frame, tr( "Trace" ) );
    }

    top_layout->addWidget( M_tab_widget );
    this->setLayout( top_layout );
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createZoomControls()
{
    QGroupBox * group_box = new QGroupBox( tr( "Zoom" ) );

    QVBoxLayout * top_layout = new QVBoxLayout();
    top_layout->setMargin( 1 );
    top_layout->setSpacing( 0 );

    const QFontMetrics metrics = this->fontMetrics();
    const int button_height = metrics.height() + 12;

    {
        QHBoxLayout * layout = new QHBoxLayout();
        layout->setSpacing( 0 );

        QPushButton * zoomin_button = new QPushButton( tr( "+" ) );
        zoomin_button->setMaximumSize( 48, button_height );
        connect( zoomin_button, SIGNAL( clicked() ),
                 this, SLOT( zoomIn() ) );
        layout->addWidget( zoomin_button );
        //
        QPushButton * zoomout_button = new QPushButton( tr( "-" ) );
        zoomout_button->setMaximumSize( 48, button_height );
        connect( zoomout_button, SIGNAL( clicked() ),
                 this, SLOT( zoomOut() ) );
        layout->addWidget( zoomout_button );
        //
        layout->addSpacing( 1 );
        layout->addStretch( 1 );
        //
        QPushButton * fit_button = new QPushButton( tr( "Fit" ) );
        fit_button->setMaximumSize( 60, button_height );
        connect( fit_button, SIGNAL( clicked() ),
                 this, SLOT( unzoom() ) );
        layout->addWidget( fit_button );

        top_layout->addLayout( layout );
    }

    {
        QHBoxLayout * layout = new QHBoxLayout();
        layout->setSpacing( 0 );

        layout->addWidget( new QLabel( tr( "Scale:" ) ) );

        layout->addSpacing( 2 );

        M_scale_slider = new QSlider( Qt::Horizontal );
        M_scale_slider->setRange( static_cast< int >( ViewConfig::MIN_FIELD_SCALE
                                                      * 10.0 ),
                                  static_cast< int >( ViewConfig::MAX_FIELD_SCALE
                                                      * 10.0 ) );
        M_scale_slider->setValue( 80 );
        M_scale_slider->setSingleStep( 1 );
        M_scale_slider->setPageStep( 1 ); // set the minimal increment/decrement step
        M_scale_slider->setMinimumSize( 260, 24 );
        connect( M_scale_slider, SIGNAL( sliderMoved( int ) ),
                 this, SLOT( slideFieldScale( int ) ) );
        layout->addWidget( M_scale_slider );

        layout->addSpacing( 2 );

        M_scale_text = new QLineEdit( tr( "0.0" ) );
        M_scale_text->setValidator( new QDoubleValidator( 0.0, 400.0, 3, M_scale_text ) );
        M_scale_text->setMaximumSize( 48, 24 );
        connect( M_scale_text, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( editFieldScale( const QString & ) ) );
        layout->addWidget( M_scale_text );

        top_layout->addLayout( layout );
    }


    group_box->setLayout( top_layout );
    return group_box;;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createObjectSizeControls()
{
    QGroupBox * group_box = new QGroupBox( tr( "Object Size" ) );

    QVBoxLayout * top_layout = new QVBoxLayout();
    top_layout->setMargin( 1 );
    top_layout->setSpacing( 0 );

    M_enlarge_cb = new QCheckBox( tr( "Enlarge Objects" ) );
    M_enlarge_cb->setChecked( M_view_config.isEnlarged() );
    connect( M_enlarge_cb, SIGNAL( clicked( bool ) ),
             this, SLOT( clickEnlarge( bool ) ) );
    top_layout->addWidget( M_enlarge_cb );


    QHBoxLayout * box = new QHBoxLayout();
    //
    box->addWidget( new QLabel( tr( "Ball Size:" ) ) );
    //
    M_ball_size_text = new QLineEdit( tr( "0.35" ) );
    M_ball_size_text->setValidator( new QDoubleValidator( 0.01, 100.0, 3, M_ball_size_text ) );
    M_ball_size_text->setMaximumSize( 48, 24 );
    connect( M_ball_size_text, SIGNAL( textChanged( const QString & ) ),
             this, SLOT( editBallSize( const QString & ) ) );
    box->addWidget( M_ball_size_text );
    //
    box->addWidget( new QLabel( tr( " Player Size:" ) ) );
    //
    M_player_size_text = new QLineEdit( tr( "0.0" ) );
    M_player_size_text->setValidator( new QDoubleValidator( 0.0, 100.0, 3, M_ball_size_text ) );
    M_player_size_text->setMaximumSize( 48, 24 );
    connect( M_player_size_text, SIGNAL( textChanged( const QString & ) ),
             this, SLOT( editPlayerSize( const QString & ) ) );
    box->addWidget( M_player_size_text );
    //
    top_layout->addLayout( box );

    group_box->setLayout( top_layout );
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createCanvasSizeControls()
{
    QGroupBox * group_box = new QGroupBox( tr( "Canvas Size" ) );

    QHBoxLayout * layout = new QHBoxLayout();
    layout->setMargin( 1 );
    layout->setSpacing( 0 );

    layout->addWidget( new QLabel( tr( " Width:" ) ) );

    M_canvas_width_text = new QLineEdit( tr( "640" ) );
    M_canvas_width_text->setValidator( new QIntValidator( 100, 3000, M_canvas_width_text ) );
    M_canvas_width_text->setMaximumSize( 48, 24 );
    layout->addWidget( M_canvas_width_text );

    layout->addWidget( new QLabel( tr( " Height:" ) ) );

    M_canvas_height_text = new QLineEdit( tr( "480" ) );
    M_canvas_height_text->setValidator( new QIntValidator( 100, 3000, M_canvas_height_text ) );
    M_canvas_height_text->setMaximumSize( 48, 24 );
    layout->addWidget( M_canvas_height_text );

    layout->addSpacing( 12 );

    QPushButton * apply_canvas_size_btn = new QPushButton( tr( "apply" ) );
    apply_canvas_size_btn->setMaximumSize( 60, this->fontMetrics().height() + 12 );
    connect( apply_canvas_size_btn, SIGNAL( clicked() ),
             this, SLOT( applyCanvasSize() ) );
    layout->addWidget( apply_canvas_size_btn );

    group_box->setLayout( layout );
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createReverseControls()
{
    QGroupBox * group_box = new QGroupBox( tr( "Reverse" ) );

    QHBoxLayout * layout = new QHBoxLayout();
    layout->setMargin( 1 );
    layout->setSpacing( 0 );

    M_reverse_side_cb = new QCheckBox( tr( "Reverse Side" ));
    M_reverse_side_cb->setChecked( M_view_config.reverseSide() );
    connect( M_reverse_side_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickReverseSide( bool ) ) );
    layout->addWidget( M_reverse_side_cb );

    M_player_reverse_draw_cb = new QCheckBox( tr( "Reverse Player Draw Order" ) );
    M_player_reverse_draw_cb->setChecked( M_view_config.playerReverseDraw() );
    connect( M_player_reverse_draw_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickPlayerReverseDraw( bool ) ) );
    layout->addWidget( M_player_reverse_draw_cb );

    group_box->setLayout( layout );
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createPlayersDetailControls()
{
    QGroupBox * group_box = new QGroupBox( tr( "Players\' Detail" ) );

    QVBoxLayout * top_layout = new QVBoxLayout();
    top_layout->setMargin( 1 );
    top_layout->setSpacing( 0 );

    {
        QHBoxLayout * layout = new QHBoxLayout();
        layout->setMargin( 0 );
        layout->setSpacing( 0 );

        M_player_number_cb = new QCheckBox( tr( "Unum" ) );
        M_player_number_cb->setChecked( M_view_config.isShownPlayerNumber() );
        connect( M_player_number_cb, SIGNAL( clicked( bool ) ),
                 this, SLOT( clickShowPlayerNumber( bool ) ) );
        layout->addWidget( M_player_number_cb );
        //
        M_hetero_number_cb = new QCheckBox( tr( "Type" ) );
        M_hetero_number_cb->setChecked( M_view_config.isShownHeteroNumber() );
        connect( M_hetero_number_cb, SIGNAL( clicked( bool ) ),
                 this, SLOT( clickShowHeteroNumber( bool ) ) );
        layout->addWidget( M_hetero_number_cb );
        //
        M_stamina_cb = new QCheckBox( tr( "Stamina" ) );
        M_stamina_cb->setChecked( M_view_config.isShownStamina() );
        connect( M_stamina_cb, SIGNAL( clicked( bool ) ),
                 this, SLOT( clickShowStamina( bool ) ) );
        layout->addWidget( M_stamina_cb );

        top_layout->addLayout( layout );
    }
    {
        QHBoxLayout * layout = new QHBoxLayout();
        layout->setMargin( 0 );
        layout->setSpacing( 0 );

        M_view_cone_cb = new QCheckBox( tr( "View Cone" ) );
        M_view_cone_cb->setChecked( M_view_config.isShownViewCone() );
        connect( M_view_cone_cb, SIGNAL( clicked( bool ) ),
                 this, SLOT( clickShowViewCone( bool ) ) );
        layout->addWidget( M_view_cone_cb );
        //
        M_body_shadow_cb = new QCheckBox( tr( "Body Shadow" ) );
        M_body_shadow_cb->setChecked( M_view_config.isShownBodyShadow() );
        connect( M_body_shadow_cb, SIGNAL( clicked( bool ) ),
                 this, SLOT( clickShowBodyShadow( bool ) ) );
        layout->addWidget( M_body_shadow_cb );
        //
        M_control_area_cb = new QCheckBox( tr( "Control Area" ) );
        M_control_area_cb->setChecked( M_view_config.isShownControlArea() );
        connect( M_control_area_cb, SIGNAL( clicked( bool ) ),
                 this, SLOT( clickShowControlArea( bool ) ) );
        layout->addWidget( M_control_area_cb );

        top_layout->addLayout( layout );
    }

    group_box->setLayout( top_layout );
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createShowControls()
{
    QGroupBox * group_box = new QGroupBox( tr( "Show" ) );

    QVBoxLayout * top_layout = new QVBoxLayout();
    top_layout->setMargin( 1 );
    top_layout->setSpacing( 0 );

    {
        QHBoxLayout * layout = new QHBoxLayout();
        layout->setMargin( 0 );
        layout->setSpacing( 0 );

        M_anonymous_mode_cb = new QCheckBox( tr( "Anonymous Mode" ) );
        M_anonymous_mode_cb->setChecked( M_view_config.anonymousMode() );
        connect( M_anonymous_mode_cb, SIGNAL( clicked( bool ) ),
                 this, SLOT( clickAnonymousMode( bool ) ) );
        layout->addWidget( M_anonymous_mode_cb );

        top_layout->addLayout( layout );
    }
    {
        QHBoxLayout * layout = new QHBoxLayout();
        layout->setMargin( 0 );
        layout->setSpacing( 0 );

        M_show_score_board_cb = new QCheckBox( tr( "Score Board" ) );
        M_show_score_board_cb->setChecked( M_view_config.isShownScoreBoard() );
        connect( M_show_score_board_cb, SIGNAL( clicked( bool ) ),
                 this, SLOT( clickShowScoreBoard( bool ) ) );
        layout->addWidget( M_show_score_board_cb );
        //
        M_show_team_logo_cb = new QCheckBox( tr( "Team Logo" ) );
        M_show_team_logo_cb->setChecked( M_view_config.isShownTeamLogo() );
        connect( M_show_team_logo_cb, SIGNAL( clicked( bool ) ),
                 this, SLOT( clickShowTeamLogo( bool ) ) );
        layout->addWidget( M_show_team_logo_cb );
        //
        M_show_flags_cb = new QCheckBox( tr( "Flag" ) );
        M_show_flags_cb->setChecked( M_view_config.isShownFlags() );
        connect( M_show_flags_cb, SIGNAL( clicked( bool ) ),
                 this, SLOT( clickShowFlags( bool ) ) );
        layout->addWidget( M_show_flags_cb );

        top_layout->addLayout( layout );
    }
    {
        QHBoxLayout * layout = new QHBoxLayout();
        layout->setMargin( 0 );
        layout->setSpacing( 0 );

        M_show_ball_cb = new QCheckBox( tr( "Ball" ) );
        M_show_ball_cb->setChecked( M_view_config.isShownBall() );
        connect( M_show_ball_cb, SIGNAL( clicked( bool ) ),
                 this, SLOT( clickShowBall( bool ) ) );
        layout->addWidget( M_show_ball_cb );
        //
        M_show_players_cb = new QCheckBox( tr( "Player" ) );
        M_show_players_cb->setChecked( M_view_config.isShownPlayers() );
        connect( M_show_players_cb, SIGNAL( clicked( bool ) ),
                 this, SLOT( clickShowPlayers( bool ) ) );
        layout->addWidget( M_show_players_cb );
        //
        M_show_offside_line_cb = new QCheckBox( tr( "Offside Line" ) );
        M_show_offside_line_cb->setChecked( M_view_config.isShownOffsideLine() );
        connect( M_show_offside_line_cb, SIGNAL( clicked( bool ) ),
                 this, SLOT( clickShowOffsideLine( bool ) ) );
        layout->addWidget( M_show_offside_line_cb );

        top_layout->addLayout( layout );
    }

    group_box->setLayout( top_layout );
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createCompGeomControls()
{
    QGroupBox * group_box = new QGroupBox( tr( "Compututational Geometry" ) );
    QVBoxLayout * top_layout = new QVBoxLayout();
    top_layout->setMargin( 1 );
    top_layout->setSpacing( 0 );

    QHBoxLayout * layout = new QHBoxLayout();
    layout->setMargin( 1 );
    layout->setSpacing( 0 );

    {
        M_show_voronoi_cb = new QCheckBox( tr( "VoronoiDiagram" ) );
        M_show_voronoi_cb->setChecked( M_view_config.isShownVoronoiDiagram() );
        connect( M_show_voronoi_cb, SIGNAL( clicked( bool ) ),
                 this, SLOT( clickShowVoronoiDiagram( bool ) ) );
        layout->addWidget( M_show_voronoi_cb );
    }
    {
        M_show_delaunay_cb = new QCheckBox( tr( "DelaunayTrianglation" ) );
        M_show_delaunay_cb->setChecked( M_view_config.isShownDelaunayTrianglation() );
        connect( M_show_delaunay_cb, SIGNAL( clicked( bool ) ),
                 this, SLOT( clickShowDelaunayTrianglation( bool ) ) );
        layout->addWidget( M_show_delaunay_cb );
    }
    top_layout->addLayout( layout );

    {
        M_voronoi_choice = new QComboBox();
        M_voronoi_choice->addItem( tr( "All" ) );
        M_voronoi_choice->addItem( tr( "Left Team" ) );
        M_voronoi_choice->addItem( tr( "Right Team" ) );
        M_voronoi_choice->setMaximumSize( this->fontMetrics().width( tr( "Right Team" ) ) + 32,
                                          this->fontMetrics().height() + 12 );
        M_voronoi_choice->setCurrentIndex( 0 );
        connect( M_voronoi_choice, SIGNAL( currentIndexChanged( int ) ),
                 this, SLOT( selectCompGeomSide( int ) ) );
        top_layout->addWidget( M_voronoi_choice );
    }

    group_box->setLayout( top_layout );
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createFocusControls()
{
    QGroupBox * group_box = new QGroupBox( tr( "Focus" ) );

    QHBoxLayout * layout = new QHBoxLayout();
    layout->setMargin( 1 );
    layout->setSpacing( 0 );

    {
        M_focus_ball_rb = new QRadioButton( tr( "Ball" ) );
        connect( M_focus_ball_rb, SIGNAL( clicked() ),
                 this, SLOT( clickFocusBall() ) );
        layout->addWidget( M_focus_ball_rb );
    }
    {
        M_focus_player_rb = new QRadioButton( tr( "Selected Player" ) );
        connect( M_focus_player_rb, SIGNAL( clicked() ),
                 this, SLOT( clickFocusPlayer() ) );
        layout->addWidget( M_focus_player_rb );
    }
    {
        M_focus_fix_rb = new QRadioButton( tr( "Fix" ) );
        connect( M_focus_fix_rb, SIGNAL( clicked() ),
                 this, SLOT( setFocusFix() ) );
        layout->addWidget( M_focus_fix_rb );
    }
    M_focus_fix_rb->setChecked( true );

    group_box->setLayout( layout );
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createPlayerSelectionControls()
{
    QGroupBox * group_box = new QGroupBox( tr( "Player Selection" ) );

    QVBoxLayout * top_layout = new QVBoxLayout();
    top_layout->setMargin( 1 );
    top_layout->setSpacing( 0 );

    // selection type
    {
        QHBoxLayout * layout = new QHBoxLayout();
        layout->setMargin( 0 );
        layout->setSpacing( 0 );
        {
            M_select_all_rb = new QRadioButton( tr( "Auto" ) );
            connect( M_select_all_rb, SIGNAL( clicked() ),
                     this, SLOT( clickSelectAutoAll() ) );
            layout->addWidget( M_select_all_rb );
        }
        {
            M_select_left_rb = new QRadioButton( tr( "Auto left" ) );
            connect( M_select_left_rb, SIGNAL( clicked() ),
                     this, SLOT( clickSelectAutoLeft() ) );
            layout->addWidget( M_select_left_rb );
        }
        {
            M_select_right_rb = new QRadioButton( tr( "Auto right" ) );
            connect( M_select_right_rb, SIGNAL( clicked() ),
                     this, SLOT( clickSelectAutoRight() ) );
            layout->addWidget( M_select_right_rb );
        }
        {
            M_select_fix_rb = new QRadioButton( tr( "Fix" ) );
            connect( M_select_all_rb, SIGNAL( clicked() ),
                     this, SLOT( clickSelectFix() ) );
            layout->addWidget( M_select_fix_rb );
        }
        {
            M_unselect_rb = new QRadioButton( tr( "Unselect" ) );
            connect( M_unselect_rb, SIGNAL( clicked() ),
                     this, SLOT( setUnselect() ) );
            layout->addWidget( M_unselect_rb );
        }
        M_unselect_rb->setChecked( true );

        top_layout->addLayout( layout );
    }

    {
        QHBoxLayout * layout = new QHBoxLayout();
        layout->setMargin( 0 );
        layout->setSpacing( 0 );
        //layout->addSpacing( 16 );

        M_player_choice = new QComboBox();
        M_player_choice->addItem( tr ( "No Selection" ) );
        for ( int i = 1; i <= 11; ++i )
        {
            M_player_choice->addItem( QString( "Left number: %1" ).arg( i ) );
        }
        for ( int i = 1; i <= 11; ++i )
        {
            M_player_choice->addItem( QString( "Right number: %1" ).arg( i ) );
        }
        M_player_choice->setCurrentIndex( 0 );
        int id_width = this->fontMetrics().width( tr( "Right Number: 11" ) );
        M_player_choice->setMaximumWidth( id_width + 40 );
        connect( M_player_choice, SIGNAL( currentIndexChanged( int ) ),
                 this, SLOT( choicePlayer( int ) ) );
        layout->addWidget( M_player_choice, 0, Qt::AlignLeft );

        layout->addSpacing( 16 );

        top_layout->addLayout( layout );
    }

    group_box->setLayout( top_layout );
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createFieldStyleControls()
{
    QGroupBox * group_box = new QGroupBox( tr( "Field Style" ) );

    QVBoxLayout * top_layout = new QVBoxLayout();
    top_layout->setMargin( 1 );
    top_layout->setSpacing( 0 );

    QHBoxLayout * layout = new QHBoxLayout();
    layout->setMargin( 1 );
    layout->setSpacing( 0 );

    {
        M_keepaway_mode_cb = new QCheckBox( tr( "Keepaway" ) );
        M_keepaway_mode_cb->setChecked( M_view_config.keepawayMode() );
        connect( M_keepaway_mode_cb, SIGNAL( clicked( bool ) ),
                 this, SLOT( clickKeepawayMode( bool ) ) );
        layout->addWidget( M_keepaway_mode_cb );
    }
    {
        M_grass_normal_rb = new QRadioButton( tr( "mono" ) );
        connect( M_grass_normal_rb, SIGNAL( clicked() ),
                 this, SLOT( clickGrassNormal() ) );
        layout->addWidget( M_grass_normal_rb );
    }
    {
        M_grass_line_rb = new QRadioButton( tr( "lines" ) );
        connect( M_grass_line_rb, SIGNAL( clicked() ),
                 this, SLOT( clickGrassLine() ) );
        layout->addWidget( M_grass_line_rb );
    }
    {
        M_grass_checker_rb = new QRadioButton( tr( "checker" ) );
        connect( M_grass_checker_rb, SIGNAL( clicked() ),
                 this, SLOT( clickGrassChecker() ) );
        layout->addWidget( M_grass_checker_rb );
    }
    top_layout->addLayout( layout );

    {
        M_draw_type_choice = new QComboBox();
        M_draw_type_choice->addItem( tr( "default" ) );
        M_draw_type_choice->addItem( tr( "rcssmonitor" ) );
        connect( M_draw_type_choice, SIGNAL( currentIndexChanged( int ) ),
                 this, SLOT( selectDrawType( int ) ) );
        top_layout->addWidget( M_draw_type_choice );
    }

    switch ( M_view_config.grassType() ) {
    case ViewConfig::GRASS_NORMAL:
        M_grass_normal_rb->setChecked( true );
        break;
    case ViewConfig::GRASS_LINES:
        M_grass_line_rb->setChecked( true );
        break;
    case ViewConfig::GRASS_CHECKER:
        M_grass_checker_rb->setChecked( true );
        break;
    default:
        M_grass_normal_rb->setChecked( true );
        break;
    }

    group_box->setLayout( top_layout );
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createMiscControls()
{
    QGroupBox * group_box = new QGroupBox( tr( "Misc" ) );
    QVBoxLayout * top_layout = new QVBoxLayout();
    top_layout->setMargin( 1 );
    top_layout->setSpacing( 0 );

    M_cursor_hide_cb = new QCheckBox( tr( "Audomatic Cursor Hide" ) );
    M_cursor_hide_cb->setChecked( M_view_config.cursorHide() );
    connect( M_cursor_hide_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickCursorHide( bool ) ) );
    top_layout->addWidget( M_cursor_hide_cb );
    //
    M_anti_aliasing_cb = new QCheckBox( tr( "Anti Aliasing" ) );
    M_anti_aliasing_cb->setChecked( M_view_config.antiAliasing() );
    connect( M_anti_aliasing_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickAntiAliasing( bool ) ) );
    top_layout->addWidget( M_anti_aliasing_cb );
    //
    M_gradient_cb = new QCheckBox( tr( "Gradient" ) );
    M_gradient_cb->setChecked( M_view_config.gradient() );
    connect( M_gradient_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickGradient( bool ) ) );
    top_layout->addWidget( M_gradient_cb );

    group_box->setLayout( top_layout );
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createGridStepControls()
{
    QGroupBox * group_box = new QGroupBox( tr( "Grid Lines" ) );
    QVBoxLayout * top_layout = new QVBoxLayout();
    top_layout->setMargin( 1 );
    top_layout->setSpacing( 0 );

    M_show_grid_coord_cb = new QCheckBox( tr( "Grid Coordinate" ) );
    M_show_grid_coord_cb->setChecked( M_view_config.isShownGridCoord() );
    connect( M_show_grid_coord_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickShowGridCoord( bool ) ) );
    top_layout->addWidget( M_show_grid_coord_cb );

    //
    QHBoxLayout * layout = new QHBoxLayout();
    //layout->setMargin( 0 );
    //layout->setSpacing( 0 );
    //
    layout->addWidget( new QLabel( tr( "Step:" ) ) );
    //
    M_grid_step_slider = new QSlider( Qt::Horizontal );
    M_grid_step_slider->setRange( 0, 1000 );
    M_grid_step_slider->setValue( 0 );
    M_grid_step_slider->setSingleStep( 1 );
    M_grid_step_slider->setPageStep( 1 );
    M_grid_step_slider->setMinimumSize( 300, 24 );
    connect( M_grid_step_slider, SIGNAL( sliderMoved( int ) ),
             this, SLOT( slideGridStep( int ) ) );
    layout->addWidget( M_grid_step_slider );

    //
    M_grid_step_text = new QLineEdit( tr( " 000.0" )  );
    M_grid_step_text->setValidator( new QDoubleValidator( 0.0, 200.0, 2, M_grid_step_text ) );
    M_grid_step_text->setMaximumSize( 48, 24 );
    connect( M_grid_step_text, SIGNAL( textChanged( const QString & ) ),
             this, SLOT( editGridStep( const QString & ) ) );
    layout->addWidget( M_grid_step_text );

    top_layout->addLayout( layout );
    group_box->setLayout( top_layout );
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createTraceControls()
{
    QGroupBox * group_box = new QGroupBox( tr( "Trace" ) );

    QVBoxLayout * top_layout = new QVBoxLayout();
    top_layout->setMargin( 0 );
    top_layout->setSpacing( 0 );

//     QHBoxLayout * first_layout = new QHBoxLayout();
//     first_layout->setMargin( 0 );
//     first_layout->setSpacing( 0 );
    {
        QGroupBox * ball_box = new QGroupBox( tr( "Ball" ) );

        QHBoxLayout * layout = new QHBoxLayout();
        layout->setSizeConstraint( QLayout::SetFixedSize );
        layout->setMargin( 0 );
        layout->setSpacing( 0 );

        M_ball_trace_start = new QLineEdit( tr( "0" ) );
        M_ball_trace_start->setValidator( new QIntValidator( 0, 20000, M_ball_trace_start ) );
        M_ball_trace_start->setMaximumSize( 36, 24 );
        connect( M_ball_trace_start, SIGNAL( textEdited( const QString & ) ),
                 this, SLOT( editBallTraceStart( const QString & ) ) );
        layout->addWidget( M_ball_trace_start, 0, Qt::AlignVCenter );

        layout->addWidget( new QLabel( tr( "-" ) ), 0, Qt::AlignVCenter );

        M_ball_trace_end = new QLineEdit( tr( "0" ) );
        M_ball_trace_end->setValidator( new QIntValidator( 0, 20000, M_ball_trace_end ) );
        M_ball_trace_end->setMaximumSize( 36, 24 );
        connect( M_ball_trace_end, SIGNAL( textEdited( const QString & ) ),
                 this, SLOT( editBallTraceEnd( const QString & ) ) );
        layout->addWidget( M_ball_trace_end, 0, Qt::AlignVCenter );

        QPushButton * all_btn = new QPushButton( tr( "All" ) );
        all_btn->setMaximumSize( 48, this->fontMetrics().height() + 12 );
        all_btn->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
        connect( all_btn, SIGNAL( clicked() ),
                 this, SLOT( clickBallTraceAll() ) );
        layout->addWidget( all_btn, 0, Qt::AlignVCenter );

        M_ball_trace_cb = new QCheckBox( tr( "Auto" ) );
        M_ball_trace_cb->setChecked( false );
        connect( M_ball_trace_cb, SIGNAL( clicked( bool ) ),
                 this, SLOT( clickAutoBallTrace( bool ) ) );
        layout->addWidget( M_ball_trace_cb, 0, Qt::AlignVCenter );

        ball_box->setLayout( layout );
        //first_layout->addWidget( ball_box );
        top_layout->addWidget( ball_box );
    }
    //first_layout->addSpacing( 2 );
    top_layout->addSpacing( 2 );
    {
        QGroupBox * player_box = new QGroupBox( tr( "Player" ) );

        QHBoxLayout * layout= new QHBoxLayout();
        layout->setSizeConstraint( QLayout::SetFixedSize );
        layout->setMargin( 0 );
        layout->setSpacing( 0 );

        M_player_trace_start = new QLineEdit( tr( "0" ) );
        M_player_trace_start->setValidator( new QIntValidator( 0, 20000, M_player_trace_start ) );
        M_player_trace_start->setMaximumSize( 36, 24 );
        connect( M_player_trace_start, SIGNAL( textEdited( const QString & ) ),
                 this, SLOT( editPlayerTraceStart( const QString & ) ) );
        layout->addWidget( M_player_trace_start, 0, Qt::AlignVCenter );

        layout->addWidget( new QLabel( tr( "-" ) ), 0, Qt::AlignVCenter );

        M_player_trace_end = new QLineEdit( tr( "0" ) );
        M_player_trace_end->setValidator( new QIntValidator( 0, 20000, M_player_trace_end ) );
        M_player_trace_end->setMaximumSize( 36, 24 );
        connect( M_ball_trace_end, SIGNAL( textEdited( const QString & ) ),
                 this, SLOT( editBallTraceEnd( const QString & ) ) );
        layout->addWidget( M_player_trace_end, 0, Qt::AlignVCenter );

        QPushButton * all_btn = new QPushButton( tr( "All" ) );
        all_btn->setMaximumSize( 48, this->fontMetrics().height() + 12 );
        all_btn->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
        connect( all_btn, SIGNAL( clicked() ),
                 this, SLOT( clickPlayerTraceAll() ) );
        layout->addWidget( all_btn, 0, Qt::AlignVCenter );

        M_player_trace_cb =  new QCheckBox( tr( "Auto" ) );
        M_player_trace_cb->setChecked( false );
        connect( M_player_trace_cb, SIGNAL( clicked( bool ) ),
                 this, SLOT( clickAutoPlayerTrace( bool ) ) );
        layout->addWidget( M_player_trace_cb, 0, Qt::AlignVCenter );

        player_box->setLayout( layout );
        //first_layout->addWidget( player_box );
        top_layout->addWidget( player_box );
    }

    //top_layout->addLayout( first_layout );
    top_layout->addSpacing( 1 );

    QHBoxLayout * second_layout = new QHBoxLayout();
    second_layout->setMargin( 0 );
    second_layout->setSpacing( 0 );
    {
        QGroupBox * trace_box = new QGroupBox( tr( "Auto Trace Period" ) );

        QHBoxLayout * layout = new QHBoxLayout();
        layout->setMargin( 0 );
        layout->setSpacing( 0 );

        layout->addWidget( new QLabel( tr( "Start:" ) ), 0, Qt::AlignVCenter );

        layout->addSpacing( 2 );

        M_auto_trace_start = new QLineEdit( tr( "-10" ) );
        M_auto_trace_start->setValidator( new QIntValidator( -20000, 20000, M_auto_trace_start ) );
        M_auto_trace_start->setMinimumSize( 36, this->fontMetrics().height() + 4 );
        M_auto_trace_start->setMaximumSize( 48, this->fontMetrics().height() + 4 );
        connect( M_auto_trace_start, SIGNAL( textEdited( const QString & ) ),
                 this, SLOT( editAutoTraceStart( const QString & ) ) );
        layout->addWidget( M_auto_trace_start, 0, Qt::AlignVCenter );

        layout->addSpacing( 6 );

        layout->addWidget( new QLabel( tr( "Period:" ) ), 0, Qt::AlignVCenter );

        layout->addSpacing( 2 );

        M_auto_trace_period = new QSpinBox();
        M_auto_trace_period->setValue( 10 );
        M_auto_trace_period->setRange( 1, 12000 );
        //M_auto_trace_period->setMaximumSize( 36, 24 );
        connect( M_auto_trace_period, SIGNAL( valueChanged( int ) ),
                 this, SLOT( changeAutoTracePeriod( int ) ) );
        layout->addWidget( M_auto_trace_period, 0, Qt::AlignVCenter );

        trace_box->setLayout( layout );

        second_layout->addWidget( trace_box, 1 );
    }

    second_layout->addSpacing( 12 );
    {
//         QHBoxLayout * layout = new QHBoxLayout();
//         layout->setMargin( 0 );
//         layout->setSpacing( 0 );

//         layout->addStretch( 1 );

        QPushButton * line_point_btn = new QPushButton( tr( "Line/Point" ) );
        line_point_btn->setMaximumSize( 80, this->fontMetrics().height() + 12 );
        connect( line_point_btn, SIGNAL( clicked() ),
                 this, SLOT( clickLinePointButton() ) );

        second_layout->addWidget( line_point_btn, 0, Qt::AlignLeft );
//         layout->addWidget( line_point_btn, 2 );

//         layout->addStretch( 1 );

//         second_layout->addLayout( layout, 1 );
    }
    second_layout->addSpacing( 6 );

    top_layout->addLayout( second_layout );

    group_box->setLayout( top_layout );
    return group_box;;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createInertiaMoveControls()
{
    QGroupBox * group_box = new QGroupBox( tr( "Inertia Move" ) );

    QHBoxLayout * top_layout = new QHBoxLayout();
    top_layout->setMargin( 0 );
    top_layout->setSpacing( 0 );

    top_layout->addWidget( new QLabel( tr( "Ball:" ) ) );

    top_layout->addSpacing( 2 );

    M_ball_future = new QSpinBox();
    //M_ball_future->setMinimumSize( 60, 24 );
    M_ball_future->setValue( 0 );
    M_ball_future->setRange( 0, 50 );
    connect( M_ball_future, SIGNAL( valueChanged( int ) ),
             this, SLOT( changeBallFutureCycle( int ) ) );
    top_layout->addWidget( M_ball_future );

    top_layout->addSpacing( 6 );

    top_layout->addWidget( new QLabel( tr( "Player:" ) ) );

    top_layout->addSpacing( 2 );

    M_player_future = new QSpinBox();
    //M_player_future->setMaximumSize( 60, 24 );
    M_player_future->setValue( 0 );
    M_player_future->setRange( 0, 10 );
    connect( M_player_future, SIGNAL( valueChanged( int ) ),
             this, SLOT( changePlayerFutureCycle( int ) ) );
    top_layout->addWidget( M_player_future );

    group_box->setLayout( top_layout );
    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::showEvent( QShowEvent * event )
{
    updateAll();
    event->accept();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::updateAll()
{
    M_enlarge_cb->setChecked( M_view_config.isEnlarged() );
    M_ball_size_text->setText( QString::number( M_view_config.ballSize() ) );
    M_player_size_text->setText( QString::number( M_view_config.playerSize() ) );

    updateFieldScale();

    M_canvas_width_text
        ->setText( QString::number( M_view_config.canvasWidth() ) );

    M_canvas_height_text
        ->setText( QString::number( M_view_config.canvasHeight() ) );

    M_player_number_cb->setChecked( M_view_config.isShownPlayerNumber() );
    M_hetero_number_cb->setChecked( M_view_config.isShownHeteroNumber() );
    M_stamina_cb->setChecked( M_view_config.isShownStamina() );
    M_view_cone_cb->setChecked( M_view_config.isShownViewCone() );
    M_body_shadow_cb->setChecked( M_view_config.isShownBodyShadow() );
    M_control_area_cb->setChecked( M_view_config.isShownControlArea() );

    M_anonymous_mode_cb->setChecked( M_view_config.anonymousMode() );
    M_show_score_board_cb->setChecked( M_view_config.isShownScoreBoard() );
    M_show_team_logo_cb->setChecked( M_view_config.isShownTeamLogo() );
    M_show_ball_cb->setChecked( M_view_config.isShownBall() );
    M_show_players_cb->setChecked( M_view_config.isShownPlayers() );
    M_show_flags_cb->setChecked( M_view_config.isShownFlags() );
    M_show_offside_line_cb->setChecked( M_view_config.isShownOffsideLine() );

    switch ( M_view_config.grassType() ) {
    case ViewConfig::GRASS_NORMAL:
        M_grass_normal_rb->setChecked( true );
        break;
    case ViewConfig::GRASS_LINES:
        M_grass_line_rb->setChecked( true );
        break;
    case ViewConfig::GRASS_CHECKER:
        M_grass_checker_rb->setChecked( true );
        break;
    default:
        M_grass_normal_rb->setChecked( true );
        break;
    }

    M_keepaway_mode_cb->setChecked( M_view_config.keepawayMode() );

    M_draw_type_choice->setCurrentIndex( M_view_config.drawType() );

    M_cursor_hide_cb->setChecked( M_view_config.cursorHide() );
    M_anti_aliasing_cb->setChecked( M_view_config.antiAliasing() );
    M_gradient_cb->setChecked( M_view_config.gradient() );

    M_show_voronoi_cb->setChecked( M_view_config.isShownVoronoiDiagram() );
    M_show_delaunay_cb->setChecked( M_view_config.isShownDelaunayTrianglation() );

    switch ( M_view_config.voronoiTarget() ) {
    case rcsc::NEUTRAL:
        M_voronoi_choice->setCurrentIndex( 0 );
        break;
    case rcsc::LEFT:
        M_voronoi_choice->setCurrentIndex( 1 );
        break;
    case rcsc::RIGHT:
        M_voronoi_choice->setCurrentIndex( 2 );
        break;
    default:
        M_voronoi_choice->setCurrentIndex( 0 );
        break;
    }

    switch( M_view_config.focusType() ) {
    case ViewConfig::FOCUS_BALL:
        M_focus_ball_rb->setChecked( true );
        break;
    case ViewConfig::FOCUS_PLAYER:
        M_focus_player_rb->setChecked( true );
        break;
    case ViewConfig::FOCUS_POINT:
        M_focus_fix_rb->setChecked( true );
        break;
    default:
        M_focus_fix_rb->setChecked( true );
        break;
    }

    switch ( M_view_config.playerSelectType() ) {
    case ViewConfig::SELECT_AUTO_ALL:
        M_select_all_rb->setChecked( true );
        break;
    case ViewConfig::SELECT_AUTO_LEFT:
        M_select_left_rb->setChecked( true );
        break;
    case ViewConfig::SELECT_AUTO_RIGHT:
        M_select_right_rb->setChecked( true );
        break;
    case ViewConfig::SELECT_UNSELECT:
        M_unselect_rb->setChecked( true );
        break;
    default:
        M_unselect_rb->setChecked( true );
        break;
    }

    M_ball_trace_start->setText( QString::number( M_view_config.ballTraceStart() ) );
    M_ball_trace_end->setText( QString::number( M_view_config.ballTraceEnd() ) );
    M_ball_trace_cb->setChecked( M_view_config.isBallAutoTrace() );

    M_player_trace_start->setText( QString::number( M_view_config.playerTraceStart() ) );
    M_player_trace_end->setText( QString::number( M_view_config.playerTraceEnd() ) );
    M_player_trace_cb->setChecked( M_view_config.isPlayerAutoTrace() );

    M_auto_trace_start->setText( QString::number( M_view_config.autoTraceStart() ) );
    M_auto_trace_period->setValue( M_view_config.autoTracePeriod() );

    M_ball_future->setValue( M_view_config.ballFutureCycle() );
    M_player_future->setValue( M_view_config.playerFutureCycle() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::updateFieldScale()
{
    int iscale = static_cast< int >( rint( M_view_config.fieldScale() * 10.0 ) );
    if ( M_scale_slider->value() != iscale )
    {
        M_scale_slider->setValue( iscale );
    }

    char buf[16];
    std::snprintf( buf, 16, "%.2f", M_view_config.fieldScale() );
    M_scale_text->setText( QString::fromAscii( buf ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::zoomIn()
{
    M_view_config.zoomIn();

    updateFieldScale();

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::zoomOut()
{
    M_view_config.zoomOut();

    updateFieldScale();

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::unzoom()
{
    M_view_config.unzoom();
    updateFieldScale();
    M_focus_fix_rb->setChecked( true );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickEnlarge( bool checked )
{
    if ( M_view_config.isEnlarged() != checked )
    {
        M_view_config.toggleEnlarge();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleEnlarge()
{
    M_view_config.toggleEnlarge();
    M_enlarge_cb->setChecked( M_view_config.isEnlarged() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editBallSize( const QString & text )
{
    bool ok = true;
    double value = text.toDouble( &ok );

    if ( ok && value > 0.0 )
    {
        M_view_config.setBallSize( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editPlayerSize( const QString & text )
{
    bool ok = true;
    double value = text.toDouble( &ok );

    if ( ok )
    {
        M_view_config.setPlayerSize( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::slideFieldScale( int value )
{
    double scale = value * 0.1;

    if ( std::fabs( scale - M_view_config.fieldScale() ) >= 0.01 )
    {
        M_scale_text->setText( QString::number( scale ) );

        M_view_config.setFieldScale( scale );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editFieldScale( const QString & text )
{
    bool ok = true;
    double value = text.toDouble( &ok );

    if ( ok
         && std::fabs( value - M_view_config.fieldScale() ) >= 0.01 )
    {
        int ivalue = static_cast< int >( rint( value * 10.0 ) );
        M_scale_slider->setValue( ivalue );

        M_view_config.setFieldScale( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::applyCanvasSize()
{
    bool ok_w = true;
    bool ok_h = true;
    int width = M_canvas_width_text->text().toInt( & ok_w );
    int height = M_canvas_height_text->text().toInt( & ok_h );

    if ( ok_w
         && ok_h
         && width > 0
         && height > 0 )
    {
        emit canvasResized( QSize( width, height ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickReverseSide( bool checked )
{
    if ( M_view_config.reverseSide() != checked )
    {
        M_view_config.toggleReverseSide();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleReverseSide()
{
    M_view_config.toggleReverseSide();
    M_reverse_side_cb->setChecked( M_view_config.reverseSide() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickPlayerReverseDraw( bool checked )
{
    if ( M_view_config.playerReverseDraw() != checked )
    {
        M_view_config.togglePlayerReverseDraw();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::togglePlayerReverseDraw()
{
    M_view_config.togglePlayerReverseDraw();
    M_player_reverse_draw_cb->setChecked( M_view_config.playerReverseDraw() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowPlayerNumber( bool checked )
{
    if ( M_view_config.isShownPlayerNumber() != checked )
    {
        M_view_config.toggleShowPlayerNumber();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowPlayerNumber()
{
    M_view_config.toggleShowPlayerNumber();
    M_player_number_cb->setChecked( M_view_config.isShownPlayerNumber() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowHeteroNumber( bool checked )
{
    if ( M_view_config.isShownHeteroNumber() != checked )
    {
        M_view_config.toggleShowHeteroNumber();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowHeteroNumber()
{
    M_view_config.toggleShowHeteroNumber();
    M_hetero_number_cb->setChecked( M_view_config.isShownHeteroNumber() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowStamina( bool checked )
{
    if ( M_view_config.isShownStamina() != checked )
    {
        M_view_config.toggleShowStamina();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowStamina()
{
    M_view_config.toggleShowStamina();
    M_stamina_cb->setChecked( M_view_config.isShownStamina() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowViewCone( bool checked )
{
    if ( M_view_config.isShownViewCone() != checked )
    {
        M_view_config.toggleShowViewCone();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowViewCone()
{
    M_view_config.toggleShowViewCone();
    M_view_cone_cb->setChecked( M_view_config.isShownViewCone() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowBodyShadow( bool checked )
{
    if ( M_view_config.isShownBodyShadow() != checked )
    {
        M_view_config.toggleShowBodyShadow();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowBodyShadow()
{
    M_view_config.toggleShowBodyShadow();
    M_body_shadow_cb->setChecked( M_view_config.isShownBodyShadow() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowControlArea( bool checked )
{
    if ( M_view_config.isShownControlArea() != checked )
    {
        M_view_config.toggleShowControlArea();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowControlArea()
{
    M_view_config.toggleShowControlArea();
    M_control_area_cb->setChecked( M_view_config.isShownControlArea() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickAnonymousMode( bool checked )
{
    if ( M_view_config.anonymousMode() != checked )
    {
        M_view_config.toggleAnonymousMode();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleAnonymousMode()
{
    M_view_config.toggleAnonymousMode();
    M_show_score_board_cb->setChecked( M_view_config.anonymousMode() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowScoreBoard( bool checked )
{
    if ( M_view_config.isShownScoreBoard() != checked )
    {
        M_view_config.toggleShowScoreBoard();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowScoreBoard()
{
    M_view_config.toggleShowScoreBoard();
    M_show_score_board_cb->setChecked( M_view_config.isShownScoreBoard() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowTeamLogo( bool checked )
{
    if ( M_view_config.isShownTeamLogo() != checked )
    {
        M_view_config.toggleShowTeamLogo();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowTeamLogo()
{
    M_view_config.toggleShowTeamLogo();
    M_show_score_board_cb->setChecked( M_view_config.isShownTeamLogo() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowBall( bool checked )
{
    if ( M_view_config.isShownBall() != checked )
    {
        M_view_config.toggleShowBall();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowBall()
{
    M_view_config.toggleShowBall();
    M_show_ball_cb->setChecked( M_view_config.isShownBall() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowPlayers( bool checked )
{
    if ( M_view_config.isShownPlayers() != checked )
    {
        M_view_config.toggleShowPlayers();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowPlayers()
{
    M_view_config.toggleShowPlayers();
    M_show_players_cb->setChecked( M_view_config.isShownPlayers() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowFlags( bool checked )
{
    if ( M_view_config.isShownFlags() != checked )
    {
        M_view_config.toggleShowFlags();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowFlags()
{
    M_view_config.toggleShowFlags();
    M_show_flags_cb->setChecked( M_view_config.isShownFlags() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowOffsideLine( bool checked )
{
    if ( M_view_config.isShownOffsideLine() != checked )
    {
        M_view_config.toggleShowOffsideLine();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowOffsideLine()
{
    M_view_config.toggleShowOffsideLine();
    M_show_offside_line_cb->setChecked( M_view_config.isShownOffsideLine() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickGrassNormal()
{
    if ( M_view_config.grassType() != ViewConfig::GRASS_NORMAL )
    {
        M_view_config.setGrassType( ViewConfig::GRASS_NORMAL );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickGrassLine()
{
    if ( M_view_config.grassType() != ViewConfig::GRASS_LINES )
    {
        M_view_config.setGrassType( ViewConfig::GRASS_LINES );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickGrassChecker()
{
    if ( M_view_config.grassType() != ViewConfig::GRASS_CHECKER )
    {
        M_view_config.setGrassType( ViewConfig::GRASS_CHECKER );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickKeepawayMode( bool checked )
{
    if ( M_view_config.keepawayMode() != checked )
    {
        M_view_config.toggleKeepawayMode();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowGridCoord( bool checked )
{
    if ( M_view_config.isShownGridCoord() != checked )
    {
        M_view_config.toggleShowGridCoord();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleKeepawayMode()
{
    M_view_config.toggleKeepawayMode();
    M_keepaway_mode_cb->setChecked( M_view_config.keepawayMode() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickCursorHide( bool checked )
{
    if ( M_view_config.cursorHide() != checked )
    {
        M_view_config.toggleCursorHide();

        //emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickAntiAliasing( bool checked )
{
    if ( M_view_config.antiAliasing() != checked )
    {
        M_view_config.toggleAntiAliasing();

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickGradient( bool checked )
{
    if ( M_view_config.gradient() != checked )
    {
        M_view_config.toggleGradient();

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::slideGridStep( int value )
{
    M_grid_step_text->setText( QString::number( value * 0.1 ) );

    M_view_config.setGridStep( value );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editGridStep( const QString & text )
{
    bool ok = true;
    double value = text.toDouble( & ok );

    if ( ok )
    {
        M_view_config.setGridStep( static_cast< int >( rint( value * 10.0 ) ) );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::selectDrawType( int index )
{
    if ( static_cast< int >( M_view_config.drawType() ) != index )
    {
        M_view_config.setDrawType( static_cast< ViewConfig::DrawType >( index ) );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowVoronoiDiagram( bool checked )
{
    if ( M_view_config.isShownVoronoiDiagram() != checked )
    {
        M_view_config.toggleShowVoronoiDiagram();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowDelaunayTrianglation( bool checked )
{
    if ( M_view_config.isShownDelaunayTrianglation() != checked )
    {
        M_view_config.toggleShowDelaunayTrianglation();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::selectCompGeomSide( int index )
{
    switch ( index ) {
    case 0:
        if ( M_view_config.voronoiTarget() != rcsc::NEUTRAL )
        {
            M_view_config.setVoronoiTarget( rcsc::NEUTRAL );
            emit configured();
        }
        break;
    case 1:
        if ( M_view_config.voronoiTarget() != rcsc::LEFT )
        {
            M_view_config.setVoronoiTarget( rcsc::LEFT );
            emit configured();
        }
        break;
    case 2:
        if ( M_view_config.voronoiTarget() != rcsc::RIGHT )
        {
            M_view_config.setVoronoiTarget( rcsc::RIGHT );
            emit configured();
        }
        break;
    default:
        std::cerr << "ViewConfigDialog::selectCompGeomSide() unsupported index "
                  << index
                  << std::endl;
        break;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickFocusBall()
{
    if ( M_view_config.focusType() != ViewConfig::FOCUS_BALL )
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_BALL );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleFocusBall()
{
    if ( M_view_config.focusType() == ViewConfig::FOCUS_BALL )
    {
        setFocusFix();
    }
    else
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_BALL );
        M_focus_ball_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickFocusPlayer()
{
    if ( M_view_config.focusType() != ViewConfig::FOCUS_PLAYER )
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_PLAYER );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleFocusPlayer()
{
    if ( M_view_config.focusType() == ViewConfig::FOCUS_PLAYER )
    {
        setFocusFix();
    }
    else
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_PLAYER );
        M_focus_player_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setFocusFix()
{
    if ( M_view_config.focusType() != ViewConfig::FOCUS_POINT )
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_POINT );
        M_focus_fix_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setFocusPoint( const QPoint & point )
{
    M_view_config.setFocusPoint( point.x(), point.y() );
    M_focus_fix_rb->setChecked( true );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickSelectAutoAll()
{
    if ( M_view_config.playerSelectType() != ViewConfig::SELECT_AUTO_ALL )
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_ALL );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleSelectAutoAll()
{
    if ( M_view_config.playerSelectType() == ViewConfig::SELECT_AUTO_ALL )
    {
        setUnselect();
    }
    else
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_ALL );
        M_select_all_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickSelectAutoLeft()
{
    ViewConfig::PlayerSelectType type = ( M_view_config.reverseSide()
                                          ? ViewConfig::SELECT_AUTO_RIGHT
                                          : ViewConfig::SELECT_AUTO_LEFT );

    if ( M_view_config.playerSelectType() != type )
    {
        M_view_config.setPlayerSelectType( type );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleSelectAutoLeft()
{
    ViewConfig::PlayerSelectType type = ( M_view_config.reverseSide()
                                          ? ViewConfig::SELECT_AUTO_RIGHT
                                          : ViewConfig::SELECT_AUTO_LEFT );

    if ( M_view_config.playerSelectType() == type )
    {
        setUnselect();
    }
    else
    {
        M_view_config.setPlayerSelectType( type );
        M_select_left_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickSelectAutoRight()
{
    ViewConfig::PlayerSelectType type = ( M_view_config.reverseSide()
                                          ? ViewConfig::SELECT_AUTO_LEFT
                                          : ViewConfig::SELECT_AUTO_RIGHT );

    if ( M_view_config.playerSelectType() != type )
    {
        M_view_config.setPlayerSelectType( type );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleSelectAutoRight()
{
    ViewConfig::PlayerSelectType type = ( M_view_config.reverseSide()
                                          ? ViewConfig::SELECT_AUTO_LEFT
                                          : ViewConfig::SELECT_AUTO_RIGHT );

    if ( M_view_config.playerSelectType() == type )
    {
        setUnselect();
    }
    else
    {
        M_view_config.setPlayerSelectType( type );
        M_select_right_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickSelectFix()
{
    if ( M_view_config.playerSelectType() != ViewConfig::SELECT_FIX )
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_RIGHT );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleSelectFix()
{
    if ( M_view_config.playerSelectType() == ViewConfig::SELECT_FIX )
    {
        setUnselect();
    }
    else
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_FIX );
        M_select_fix_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setUnselect()
{
    M_view_config.unselectPlayer();
    M_unselect_rb->setChecked( true );
    M_player_choice->setCurrentIndex( 0 );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::selectPlayer( int number )
{
    // left side:  number = [1,11]
    // right side:  number = [12,22]

    if ( number <= 0
         || 22 < number )
    {
        setUnselect();
        return;
    }

    int orig_number = number;

    if ( M_view_config.reverseSide() )
    {
        if ( number <= 11 )
        {
            number += 11;
        }
        else
        {
            number -= 11;
        }
    }

    M_view_config.setPlayerSelectType( ViewConfig::SELECT_FIX );
    if ( number <= 11 )
    {
        M_view_config.setSelectedNumber( rcsc::LEFT, number );
    }
    else
    {
        M_view_config.setSelectedNumber( rcsc::RIGHT, number - 11 );
    }

    M_player_choice->setCurrentIndex( orig_number );
    M_select_fix_rb->setChecked( true );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::choicePlayer( int number )
{
    int orig_number = number;
    if ( M_view_config.reverseSide() )
    {
        if ( number <= 11 )
        {
            number += 11;
        }
        else
        {
            number -= 11;
        }
    }

    if ( number <= 11
         && M_view_config.isSelectedPlayer( rcsc::LEFT, number ) )
    {
        return;
    }

    if ( number > 11
         && M_view_config.isSelectedPlayer( rcsc::RIGHT, number - 11 ) )
    {
        return;
    }

    selectPlayer( orig_number );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::selectPlayerWithKey()
{
    QAction * action = qobject_cast< QAction * >( sender() );

    if ( ! action )
    {
        return;
    }

    int key = action->shortcut()[0];

    for ( int i = 1; i < 10; ++i )
    {
        if ( key == Qt::Key_0 + i )
        {
            selectPlayer( i );
            return;
        }
        else if (
#ifdef Q_WS_MAC
                 key == int( Qt::META + Qt::Key_0 + i )
#else
                 key == int( Qt::CTRL + Qt::Key_0 + i )
#endif
                 )
        {
            selectPlayer( i + 11 );
            return;
        }
    }

    if ( key == Qt::Key_0 )
    {
        selectPlayer( 10 );
        return;
    }

    if (
#ifdef Q_WS_MAC
         key == Qt::META + Qt::Key_0
#else
         key == Qt::CTRL + Qt::Key_0
#endif
         )
    {
        selectPlayer( 21 );
        return;
    }

    if ( key == Qt::Key_Minus )
    {
        selectPlayer( 11 );
        return;
    }

    if (
#ifdef Q_WS_MAC
        key == Qt::META + Qt::Key_Minus
#else
        key == Qt::CTRL + Qt::Key_Minus
#endif
        )
    {
        selectPlayer( 22 );
        return;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editBallTraceStart( const QString & text )
{
    bool ok = true;
    long value = text.toLong( & ok );

    if ( ok )
    {
        M_view_config.setBallTraceStart( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editBallTraceEnd( const QString & text )
{
    bool ok = true;
    long value = text.toLong( & ok );

    if ( ok )
    {
        M_view_config.setBallTraceEnd( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickBallTraceAll()
{
    const std::vector< MonitorViewPtr > & view
        = M_main_data.viewHolder().monitorViewCont();

    if ( ! view.empty() )
    {
        M_ball_trace_cb->setChecked( false );

        if ( M_view_config.ballTraceStart() == view.front()->cycle()
             && M_view_config.ballTraceEnd() == view.back()->cycle() )
        {
            M_ball_trace_start->setText( QString::number( 0 ) );
            M_ball_trace_end->setText( QString::number( 0 ) );

            M_view_config.setBallTraceStart( 0 );
            M_view_config.setBallTraceEnd( 0 );
        }
        else
        {
            M_ball_trace_start->setText( QString::number( view.front()->cycle() ) );
            M_ball_trace_end->setText( QString::number( view.back()->cycle() ) );

            M_view_config.setBallTraceStart( view.front()->cycle() );
            M_view_config.setBallTraceEnd( view.back()->cycle() );
        }

        if ( M_view_config.isBallAutoTrace() )
        {
            M_view_config.toggleBallAutoTrace();
        }

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickAutoBallTrace( bool checked )
{
    if ( M_view_config.isBallAutoTrace() != checked )
    {
        M_view_config.toggleBallAutoTrace();

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editPlayerTraceStart( const QString & text )
{
    bool ok = true;
    long value = text.toLong( & ok );

    if ( ok )
    {
        M_view_config.setPlayerTraceStart( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editPlayerTraceEnd( const QString & text )
{
    bool ok = true;
    long value = text.toLong( & ok );

    if ( ok )
    {
        M_view_config.setPlayerTraceEnd( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickPlayerTraceAll()
{
    const std::vector< MonitorViewPtr > & view
        = M_main_data.viewHolder().monitorViewCont();
    if ( ! view.empty() )
    {
        M_player_trace_cb->setChecked( false );


        if ( M_view_config.playerTraceStart() == view.front()->cycle()
             && M_view_config.playerTraceEnd() == view.back()->cycle() )
        {
            M_player_trace_start->setText( QString::number( 0 ) );
            M_player_trace_end->setText( QString::number( 0 ) );

            M_view_config.setPlayerTraceStart( 0 );
            M_view_config.setPlayerTraceEnd( 0 );
        }
        else
        {
            M_player_trace_start->setText( QString::number( view.front()->cycle() ) );
            M_player_trace_end->setText( QString::number( view.back()->cycle() ) );

            M_view_config.setPlayerTraceStart( view.front()->cycle() );
            M_view_config.setPlayerTraceEnd( view.back()->cycle() );
        }

        if ( M_view_config.isPlayerAutoTrace() )
        {
            M_view_config.togglePlayerAutoTrace();
        }

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickAutoPlayerTrace( bool checked )
{
    if ( M_view_config.isPlayerAutoTrace() != checked )
    {
        M_view_config.togglePlayerAutoTrace();

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editAutoTraceStart( const QString & text )
{
    bool ok = true;
    long value = text.toLong( & ok );

    if ( ok )
    {
        M_view_config.setAutoTraceStart( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::changeAutoTracePeriod( int value )
{
    if ( value >= 1 )
    {
        M_view_config.setAutoTracePeriod( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickLinePointButton()
{
    M_view_config.toggleLineTrace();

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::changeBallFutureCycle( int value )
{
    M_view_config.setBallFutureCycle( value );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::changePlayerFutureCycle( int value )
{
    M_view_config.setPlayerFutureCycle( value );

    emit configured();
}
