// -*-c++-*-

/*!
  \file plaeyr_painter.cpp
  \brief player painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <QtGui>

#include "player_painter.h"

#include "draw_config.h"
// model
#include "main_data.h"

#include <rcsc/common/player_type.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/angle_deg.h>

#include <cstring>
#include <cstdio>
#include <cmath>

/*-------------------------------------------------------------------*/
/*

*/
inline
PlayerPainter::Param::Param( const Player & player,
                             const Ball & ball,
                             const ViewConfig & view_conf,
                             const rcsc::PlayerType & ptype )
    : x_( view_conf.screenX( player.x() ) )
    , y_( view_conf.screenY( player.y() ) )
    , body_( player.body() + ( view_conf.reverseSide() ? 180.0 : 0.0 ) )
    , head_( player.head() + ( view_conf.reverseSide() ? 180.0 : 0.0 ) )
    , body_radius_( view_conf.scale( ptype.playerSize() ) )
    , kick_radius_( view_conf.scale( ptype.kickableArea() ) )
    , have_full_effort_( player.hasFullEffort( ptype.effortMax() ) )
    , player_( player )
    , ball_( ball )
    , player_type_( ptype )
{

    if ( body_radius_ < 1 ) body_radius_ = 1;
    if ( kick_radius_ < 5 ) kick_radius_ = 5;

    if ( view_conf.isEnlarged() )
    {
        if ( view_conf.playerSize() > 0.0 )
        {
            draw_radius_ = view_conf.scale( view_conf.playerSize() );
            if ( draw_radius_ < 1 ) draw_radius_ = 1;
        }
        else
        {
            draw_radius_ = kick_radius_;
        }
    }
    else
    {
        draw_radius_ = body_radius_;
    }
}

/*-------------------------------------------------------------------*/
/*

*/
PlayerPainter::PlayerPainter( const MainData & main_data )
    : M_main_data( main_data )
{

}

/*-------------------------------------------------------------------*/
/*

*/
void
PlayerPainter::draw( QPainter & painter )
{
    if ( ! M_main_data.viewConfig().isShownPlayers() )
    {
        return;
    }

    MonitorViewConstPtr view = M_main_data.getViewData( M_main_data.viewIndex() );

    if ( ! view )
    {
        return;
    }

    const Ball & ball = view->ball();

    if ( M_main_data.viewConfig().playerReverseDraw() )
    {
        const std::vector< Player >::const_reverse_iterator end = view->players().rend();
        for ( std::vector< Player >::const_reverse_iterator it = view->players().rbegin();
              it != end;
              ++it )
        {
            drawAll( painter, *it, ball );
        }
    }
    else
    {
        const std::vector< Player >::const_iterator end = view->players().end();
        for ( std::vector< Player >::const_iterator it = view->players().begin();
              it != end;
              ++it )
        {
            drawAll( painter, *it, ball );
        }
    }
}

/*-------------------------------------------------------------------*/
/*

*/
void
PlayerPainter::drawAll( QPainter & painter,
                        const Player & player,
                        const Ball & ball ) const
{
    const Param param( player,
                       ball,
                       M_main_data.viewConfig(),
                       M_main_data.viewHolder().playerType( player.type() ) );

    drawBody( painter, param );
    if ( M_main_data.viewConfig().isShownBodyShadow() )
    {
        drawShadow( painter, param );
    }
    drawEdge( painter, param );

    if ( M_main_data.viewConfig().isSelectedPlayer( player.side(), player.unum() )
         && M_main_data.viewConfig().playerFutureCycle() > 0
         && player.hasDelta() )
    {
        drawFuture( painter, param );
    }

    if ( player.hasView()
         && M_main_data.viewConfig().isShownViewCone() )
    {
        drawViewCone( painter, param );
    }

    if ( M_main_data.viewConfig().isShownControlArea() )
    {
        drawControlArea( painter, param );
        drawPointTo( painter, param );
    }

    drawText( painter, param );
}

/*-------------------------------------------------------------------*/
/*

*/
void
PlayerPainter::drawBody( QPainter & painter,
                         const PlayerPainter::Param & param ) const
{
    const DrawConfig & dconf = DrawConfig::instance();

    // decide base color
    painter.setPen( dconf.playerPen() );

    rcsc::SideID side = param.player_.side();
    if ( M_main_data.viewConfig().reverseSide() )
    {
        side = static_cast< rcsc::SideID >( -1 * side );
    }

    switch ( side ) {
    case rcsc::LEFT:
        if ( param.player_.isGoalie() )
        {
            painter.setBrush( dconf.leftGoalieBrush() );
        }
        else
        {
            painter.setBrush( dconf.leftTeamBrush() );
        }
        break;
    case rcsc::RIGHT:
        if ( param.player_.isGoalie() )
        {
            painter.setBrush( dconf.rightGoalieBrush() );
        }
        else
        {
            painter.setBrush( dconf.rightTeamBrush() );
        }
        break;
    case rcsc::NEUTRAL:
        painter.setBrush( dconf.shadowBrush() );
        break;
    default:
        painter.setBrush( dconf.shadowBrush() );
        break;
    }


    // decide status color
    if ( ! param.player_.isAlive() )
    {
        painter.setBrush( dconf.shadowBrush() );
    }
    if ( param.player_.isKicking() )
    {
        painter.setPen( dconf.kickPen() );
    }
    if ( param.player_.isKickingFault() )
    {
        painter.setPen( dconf.kickFaultPen() );
        painter.setBrush( dconf.kickFaultBrush() );
    }
    if ( param.player_.isCatching() )
    {
        painter.setBrush( dconf.catchBrush() );
    }
    if ( param.player_.isCatchingFault() )
    {
        painter.setBrush( dconf.catchFaultBrush() );
    }
    if ( param.player_.isTackling() )
    {
        painter.setPen( dconf.tacklePen() );
        painter.setBrush( dconf.tackleBrush() );
    }
    if ( param.player_.isTacklingFault() )
    {
        painter.setPen( dconf.tacklePen() );
        painter.setBrush( dconf.tackleFaultBrush() );
    }
    if ( param.player_.isCollidedBall() )
    {
        painter.setBrush( dconf.collideBallBrush() );
    }
    if ( param.player_.isCollidedPlayer() )
    {
        painter.setBrush( dconf.collidePlayerBrush() );
    }

    if ( M_main_data.viewConfig().gradient() )
    {
        QRadialGradient gradient( param.x_,
                                  param.y_,
                                  param.draw_radius_,
                                  param.x_ - param.draw_radius_,
                                  param.y_ - param.draw_radius_ );
                                  //param.x_ - param.draw_radius_*0.8,
                                  //param.y_ - param.draw_radius_*0.8 );
        //gradient.setColorAt( 0.0, painter.brush().color().lighter( 170 ) );
        gradient.setColorAt( 0.0, painter.brush().color().lighter( 160 ) );
        gradient.setColorAt( 1.0, painter.brush().color() );
        painter.setBrush( gradient );
    }

    painter.drawEllipse( param.x_ - param.draw_radius_ ,
                         param.y_ - param.draw_radius_ ,
                         param.draw_radius_ * 2,
                         param.draw_radius_ * 2 );
}

/*-------------------------------------------------------------------*/
/*

*/
void
PlayerPainter::drawShadow( QPainter & painter,
                           const PlayerPainter::Param & param ) const
{
    const DrawConfig & dconf = DrawConfig::instance();

    // set size
    int shadow_radius = static_cast< int >( param.draw_radius_ * 0.9 );

    // decide shadow color
#if 0
    double stamina_rate = ( param.player_.hasStamina()
                            ? ( param.player_.stamina()
                                / param.server_param_.staminaMax() )
                            : 1.0 );
    const QColour & base_col = dconf.playerShadowBrush().color();
    int r = 255 - static_cast< int >( (255 - base_col.red()) * stamina_rate );
    int g = 255 - static_cast< int >( (255 - base_col.green()) * stamina_rate );
    int b = 255 - static_cast< int >( (255 - base_col.blue()) * stamina_rate );
    QBrush tmp_brush( QColour( r, g, b ), Qt::SolidPattern );
    painter.setPen( dconf.transparentPen() );
    painter.setBrush( tmp_brush );
#elif 0
    // create temporary brush
    int col = 255
        - static_cast< int >( 255
                              * ( param.player_.hasStamina()
                                  ? ( param.player_.stamina()
                                      / rcsc::ServerParam::i().staminaMax() )
                                  : 1.0 ) );
    //QBrush tmp_brush( QColor( col, col, col ), Qt::SolidPattern );
    painter.setPen( dconf.transparentPen() );
    //painter.setBrush( tmp_brush );
    painter.setBrush( dconf.shadowBrush( col ) );
#else
    painter.setPen( dconf.transparentPen() );
    double stamina_rate = ( param.player_.hasStamina()
                            ? ( param.player_.stamina()
                                / rcsc::ServerParam::i().staminaMax() )
                            : 1.0 );
    //int level = 255 - (int)rint( 255 * rint( stamina_rate * 16.0 ) / 16.0 );
    int level = 255 - (int)rint( 255 * rint( stamina_rate * 8.0 ) / 8.0 );
    painter.setBrush( dconf.shadowBrush( level ) );
#endif

    if ( M_main_data.viewConfig().gradient() )
    {
        QRadialGradient gradient( param.x_,
                                  param.y_,
                                  param.draw_radius_,
                                  param.x_ - param.draw_radius_,
                                  param.y_ - param.draw_radius_ );
        gradient.setColorAt( 0.0, Qt::lightGray );
        gradient.setColorAt( 1.0, painter.brush().color() );
        painter.setBrush( gradient );
    }

    // draw half circle of shadow
#if 0
    QPainterPath shadow_path;
    shadow_path.moveTo( param.x_, param.y_ );
    shadow_path.arcTo( param.x_ - shadow_radius,
                       param.y_ - shadow_radius,
                       shadow_radius * 2,
                       shadow_radius * 2,
                       ( - param.body_ + 90.0 ),
                       180.0 );

    painter.drawPath( shadow_path );
#else
    // set angle
    int shadow_start_dir
            = static_cast< int >( rint( ( - param.body_ + 90.0 ) * 16 ) );
    //double shadow_end_dir = shadow_start_dir + 180.0;
    painter.drawPie( param.x_ - shadow_radius, // left x
                     param.y_ - shadow_radius, // top y
                     shadow_radius * 2, // width
                     shadow_radius * 2, // height
                     shadow_start_dir,
                     180 * 16 ); // span angle
#endif
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainter::drawEdge( QPainter & painter,
                         const PlayerPainter::Param & param ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    int edge_radius = 1;
    if ( vconf.isEnlarged() )
    {
        if ( vconf.isShownBodyShadow() )
        {
            painter.setPen( dconf.realBodyPen() );
        }
        else
        {
            painter.setPen( dconf.playerPen() );
        }

        edge_radius = param.body_radius_;
    }
    else
    {
        // draw kickable area edge
        rcsc::SideID side = param.player_.side();
        if ( M_main_data.viewConfig().reverseSide() )
        {
            side = static_cast< rcsc::SideID >( -1 * side );
        }

        if ( side == rcsc::LEFT )
        {
            painter.setPen( dconf.leftTeamPen() );
        }
        else
        {
            painter.setPen( dconf.rightTeamPen() );
        }

        edge_radius = param.kick_radius_;
    }

    painter.setBrush( dconf.transparentBrush() );
    painter.drawEllipse( param.x_ - edge_radius,
                         param.y_ - edge_radius,
                         edge_radius * 2,
                         edge_radius * 2 );

    // body direction line
    if ( ! vconf.isShownBodyShadow() )
    {
        int end_x = vconf.absScreenX( param.player_.x() * vconf.reverseValue()
                                      + param.player_type_.kickableArea()
                                      * std::cos( param.body_ * rcsc::AngleDeg::DEG2RAD ) );
        int end_y = vconf.absScreenY( param.player_.y() * vconf.reverseValue()
                                      + param.player_type_.kickableArea()
                                      * std::sin( param.body_ * rcsc::AngleDeg::DEG2RAD ) );
        // same pen & brush
        painter.drawLine( param.x_, param.y_, end_x, end_y );
    }

    // draw stamina status if effort or recovery is decayed.
    if ( param.player_.hasStamina() )
    {
        if ( ! param.have_full_effort_ )
        {
            painter.setPen( dconf.effortDecayedPen() );
            painter.setBrush( dconf.transparentBrush() );
            int radius = param.draw_radius_ + 2;
            painter.drawEllipse( param.x_ - radius,
                                 param.y_ - radius,
                                 radius * 2,
                                 radius * 2 );
        }
        else if ( ! param.player_.hasFullRecovery() )
        {
            painter.setPen( dconf.recoveryDecayedPen() );
            painter.setBrush( dconf.transparentBrush() );
            int radius = param.draw_radius_ + 2;
            painter.drawEllipse( param.x_ - radius,
                                 param.y_ - radius,
                                 radius * 2,
                                 radius * 2 );
        }
    }
}


/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainter::drawFuture( QPainter & painter,
                           const PlayerPainter::Param & param ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    if ( vconf.antiAliasing() )
    {
        painter.setRenderHint( QPainter::Antialiasing, false );
    }

    // draw future state
    rcsc::Vector2D ppos( param.player_.x(), param.player_.y() );
    rcsc::Vector2D pvel( param.player_.deltaX(), param.player_.deltaY() );
    if ( vconf.reverseSide() )
    {
        ppos *= -1.0;
        pvel *= -1.0;
    }

    const int last = vconf.playerFutureCycle();

    QPoint first_point( vconf.absScreenX( ppos.x ),
                        vconf.absScreenY( ppos.y ) ) ;
    QPoint last_point = first_point;

    // draw kickable area edge
    rcsc::SideID side = param.player_.side();
    if ( vconf.reverseSide() )
    {
        side = static_cast< rcsc::SideID >( -1 * side );
    }

    if ( side == rcsc::LEFT )
    {
        painter.setPen( dconf.rightTeamPen() );
    }
    else
    {
        painter.setPen( dconf.leftTeamPen() );
    }

    painter.setBrush( dconf.transparentBrush() );

    QPainterPath path;
    for ( int i = 0; i < last; ++i )
    {
        ppos += pvel;
        pvel *= param.player_type_.playerDecay();

        QPoint pt( vconf.absScreenX( ppos.x ),
                   vconf.absScreenY( ppos.y ) );
        if ( std::abs( last_point.x() - pt.x() ) < 1
             && std::abs( last_point.y() - pt.y() ) < 1 )
        {
            break;
        }

        path.addEllipse( pt.x() - 1,
                         pt.y() - 1,
                         2,
                         2 );
        path.addEllipse( pt.x() - param.kick_radius_,
                         pt.y() - param.kick_radius_,
                         param.kick_radius_ * 2,
                         param.kick_radius_ * 2 );

        last_point = pt;
    }

    painter.drawPath( path );

    // draw move line
    painter.setPen( dconf.debugTargetPen() );
    painter.drawLine( first_point, last_point );

    if ( vconf.antiAliasing() )
    {
        painter.setRenderHint( QPainter::Antialiasing );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainter::drawViewCone( QPainter & painter,
                             const PlayerPainter::Param & param ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double view_width = param.player_.viewWidth();

    const double visible_dist = rcsc::ServerParam::i().visibleDistance();
    const int visible_radius = vconf.scale( visible_dist );

    const double view_start_angle = -param.head_ - view_width * 0.5;
    const double view_start_angle_real = ( param.head_ - view_width * 0.5 ) * rcsc::AngleDeg::DEG2RAD;
    const double view_end_angle_real = ( param.head_ + view_width * 0.5 ) * rcsc::AngleDeg::DEG2RAD;

    // draw enlarged view cone and feel area
    if ( vconf.isSelectedPlayer( param.player_.side(),
                                 param.player_.unum() ) )
    {
        const int UNUM_FAR = vconf.scale( 20.0 );
        const int TEAM_FAR = vconf.scale( 40.0 );
        const int TEAM_TOOFAR = vconf.scale( 60.0 );

        painter.setPen( dconf.linePen() );
        //painter.setBrush( dconf.transparentBrush() );

        int view_start_angle_int
            = static_cast< int >( rint( view_start_angle * 16 ) );
        int span_angle = static_cast< int >( rint( view_width * 16 ) );

        painter.drawArc( param.x_ - UNUM_FAR, // left x
                         param.y_ - UNUM_FAR, // toop y
                         UNUM_FAR * 2, // width
                         UNUM_FAR * 2, // height
                         view_start_angle_int,
                         span_angle );
        painter.drawArc( param.x_ - TEAM_FAR, // left x
                         param.y_ - TEAM_FAR, // toop y
                         TEAM_FAR * 2, // width
                         TEAM_FAR * 2, // height
                         view_start_angle_int,
                         span_angle );
        // pie, no an arc
        painter.drawArc( param.x_ - TEAM_TOOFAR, // left x
                         param.y_ - TEAM_TOOFAR, // toop y
                         TEAM_TOOFAR * 2, // width
                         TEAM_TOOFAR * 2, // height
                         view_start_angle_int,
                         span_angle );

        // draw feeling area circle
        painter.drawArc( param.x_ - visible_radius,
                         param.y_ - visible_radius,
                         visible_radius * 2,
                         visible_radius * 2,
                         0,
                         360 * 16 );
        // left side view cone end point x
        int lx = param.x_
            + vconf.scale( 60.0 * std::cos( view_start_angle_real ) );
        // left side view cone end point y
        int ly = param.y_
            + vconf.scale( 60.0  * std::sin( view_start_angle_real ) );
        // right side view cone end point x
        int rx = param.x_
            + vconf.scale( 60.0 * std::cos( view_end_angle_real ) );
        // right side view cone end point y
        int ry = param.y_
            + vconf.scale( 60.0 * std::sin( view_end_angle_real ) );
        painter.drawLine( lx, ly, param.x_, param.y_ );
        painter.drawLine( rx, ry, param.x_, param.y_ );
    }
    else
    {
        // draw normal view cone
        painter.setPen( dconf.viewConePen() );
        painter.setBrush( dconf.transparentBrush() );

        QPainterPath view_cone_path;
        view_cone_path.moveTo( param.x_, param.y_ );
        view_cone_path.arcTo( param.x_ - visible_radius,
                              param.y_ - visible_radius,
                              visible_radius * 2,
                              visible_radius * 2,
                              view_start_angle,
                              view_width );
        view_cone_path.closeSubpath();

        painter.drawPath( view_cone_path );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainter::drawControlArea( QPainter & painter,
                                const PlayerPainter::Param & param ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    if ( param.player_.isGoalie() )
    {
        rcsc::SideID side = param.player_.side();
        if ( M_main_data.viewConfig().reverseSide() )
        {
            side = static_cast< rcsc::SideID >( -1 * side );
        }

        int catchable = vconf.scale( rcsc::ServerParam::i().catchableArea() );
        painter.setPen( ( side == rcsc::LEFT )
                        ? dconf.leftGoaliePen()
                        : dconf.rightGoaliePen() );
        painter.setBrush( dconf.transparentBrush() );;

        painter.drawEllipse( param.x_ - catchable,
                             param.y_ - catchable,
                             catchable * 2,
                             catchable * 2 );
    }

    rcsc::Vector2D ppos( param.player_.x(),
                         param.player_.y() );
    rcsc::Vector2D bpos( param.ball_.x(),
                         param.ball_.y() );
    if ( vconf.reverseSide() )
    {
        ppos *= -1.0;
        bpos *= -1.0;
    }

    rcsc::Vector2D player_to_ball = bpos - ppos;
    player_to_ball.rotate( - param.body_ );

    // draw tackle area & probability
    double tackle_dist = ( player_to_ball.x > 0.0
                           ? rcsc::ServerParam::i().tackleDist()
                           : rcsc::ServerParam::i().tackleBackDist() );
    double tackle_prob = ( std::pow( player_to_ball.absX() / tackle_dist,
                                     rcsc::ServerParam::i().tackleExponent() )
                           + std::pow( player_to_ball.absY() / rcsc::ServerParam::i().tackleWidth(),
                                       rcsc::ServerParam::i().tackleExponent() ) );
    if ( tackle_prob < 1.0 )
    {
        rcsc::AngleDeg body_angle = param.body_;
        rcsc::AngleDeg body_angle_side = body_angle + 90.0;
        double body_x = body_angle.cos();
        double body_y = body_angle.sin();
        double forward_x = rcsc::ServerParam::i().tackleDist() * body_x;
        double forward_y = rcsc::ServerParam::i().tackleDist() * body_y;
        double back_x = rcsc::ServerParam::i().tackleBackDist() * -body_x;
        double back_y = rcsc::ServerParam::i().tackleBackDist() * -body_y;
        double right_x = rcsc::ServerParam::i().tackleWidth() * body_angle_side.cos();
        double right_y = rcsc::ServerParam::i().tackleWidth() * body_angle_side.sin();

        QPoint pts[5];
        pts[0].setX( vconf.absScreenX( ppos.x + forward_x + right_x ) );
        pts[0].setY( vconf.absScreenY( ppos.y + forward_y + right_y ) );
        pts[1].setX( vconf.absScreenX( ppos.x + forward_x - right_x ) );
        pts[1].setY( vconf.absScreenY( ppos.y + forward_y - right_y ) );
        pts[2].setX( vconf.absScreenX( ppos.x + back_x - right_x ) );
        pts[2].setY( vconf.absScreenY( ppos.y + back_y - right_y ) );
        pts[3].setX( vconf.absScreenX( ppos.x + back_x + right_x ) );
        pts[3].setY( vconf.absScreenY( ppos.y + back_y + right_y ) );
        pts[4] = pts[0];

        painter.setPen( dconf.tackleAreaPen() );
        painter.setBrush( dconf.transparentBrush() );

        //painter.drawConvexPolygon( pts, 4 );
        painter.drawPolyline( pts, 5 );

        int text_radius = param.draw_radius_;
        if ( text_radius > 40 )
        {
            text_radius = 40;
        }

        painter.setFont( dconf.playerFont() );
        painter.setPen( dconf.tacklePen() );

        char msg[32];
        std::snprintf( msg, 32, "TackleProb=%.3f", 1.0 - tackle_prob );
        painter.drawText( param.x_ + text_radius,
                          param.y_ - text_radius + painter.fontMetrics().height(),
                          QString::fromAscii( msg ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainter::drawPointTo( QPainter & painter,
                            const PlayerPainter::Param & param ) const
{
    if ( ! param.player_.isPointing() )
    {
        return;
    }

    int px = M_main_data.viewConfig().screenX( param.player_.data().point_x_ );
    int py = M_main_data.viewConfig().screenX( param.player_.data().point_y_ );

    painter.setPen( DrawConfig::instance().pointtoPen() );
    painter.setBrush( DrawConfig::instance().transparentBrush() );

    painter.drawLine( param.x_, param.y_, px, py );
    painter.drawLine( px - 2, py - 2, px + 2, py + 2 );
    painter.drawLine( px - 2, py + 2, px + 2, py - 2 );

}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainter::drawText( QPainter & painter,
                         const PlayerPainter::Param & param ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const int text_radius = std::min( 40, param.draw_radius_ );

    if ( vconf.isShownPlayerNumber()
         && vconf.isShownHeteroNumber() )
    {
        char buf[32];
        std::snprintf( buf, 32,
                       "%d,t%d",
                       param.player_.unum(), param.player_.type() );

        painter.setPen( dconf.playerNumberFontPen() );
        painter.setFont( dconf.playerFont() );
        painter.drawText( param.x_ + text_radius,
                          param.y_ + 4,
                          QString::fromAscii( buf ) );
    }
    else if ( vconf.isShownPlayerNumber() )
    {
        painter.setPen( dconf.playerNumberFontPen() );
        painter.setFont( dconf.playerFont() );
        painter.drawText( param.x_ + text_radius,
                          param.y_ + 4,
                          QString::number( param.player_.unum() ) );
    }
    else if ( vconf.isShownHeteroNumber() )
    {
        char buf[32];
        std::snprintf( buf, 32,
                       "t%d",
                       param.player_.type() );
        painter.setPen( dconf.playerNumberFontPen() );
        painter.drawText( param.x_ + text_radius,
                          param.y_ + 4,
                          QString::fromAscii( buf ) );
    }

    if ( param.player_.hasStamina()
         && vconf.isShownStamina() )
    {
        // this player is selected
        if ( vconf.isSelectedPlayer( param.player_.side(),
                                     param.player_.unum() ) )
        {
            char buf[32];
            std::snprintf( buf, 32,
                           "%4.0f,e%.3f,r%.3f",
                           param.player_.stamina(),
                           param.player_.effort(),
                           param.player_.recovery() );
            painter.setPen( dconf.playerNumberFontPen() );
            painter.drawText( param.x_ - text_radius,
                              param.y_ - text_radius - 3,
                              QString::fromAscii( buf ) );
        }
        else
        {
            char buf[16];
            std::snprintf( buf, 16, "%4d",
                           static_cast< int >( rint( param.player_.stamina() ) ) );

            painter.setPen( DrawConfig::instance().playerStaminaFontPen() );
            painter.setFont( DrawConfig::instance().playerFont() );
            painter.drawText( param.x_ - text_radius,
                              param.y_ - text_radius - 3,
                              QString::fromAscii( buf ) );
        }
    }
}
