// -*-c++-*-

/*!
  \file ball_trace_painter.cpp
  \brief ball trace painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <QtGui>

#include "ball_trace_painter.h"

#include "draw_config.h"

#include "main_data.h"
#include "monitor_view_data.h"


/*-------------------------------------------------------------------*/
/*!

*/
BallTracePainter::BallTracePainter( const MainData & main_data )
    : M_main_data( main_data )
    , M_line_trace( false )
    , M_center_x( 0 )
    , M_center_y( 0 )
    , M_field_scale( 0.0 )
{
    M_point_pixmap
        = boost::shared_ptr< QPixmap >( new QPixmap( 5, 5 ) );

    M_point_pixmap->fill( Qt::black );

    QPainter painter( M_point_pixmap.get() );
    painter.setPen( DrawConfig::instance().ballPen() );
    painter.setBrush( Qt::gray );
    painter.drawEllipse( 0, 0, 5, 5 );
    painter.end();

    M_point_pixmap->setMask( M_point_pixmap->createMaskFromColor( Qt::black ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
BallTracePainter::draw( QPainter & painter )
{
    const ViewConfig & vconf = M_main_data.viewConfig();

    if ( ! vconf.isShownBallTrace() )
    {
        return;
    }

#if 1
    if ( vconf.antiAliasing() )
    {
        painter.setRenderHint( QPainter::Antialiasing, false );
    }

    drawTrace( painter );

    if ( vconf.antiAliasing() )
    {
        painter.setRenderHint( QPainter::Antialiasing );
    }
#else
    if ( vconf.isBallAutoTrace()
         || vconf.focusType() != ViewConfig::FOCUS_POINT )
    {
        drawTrace( painter );
        return;
    }
    bool redraw = vconf.isBallAutoTrace();
    if ( ! M_all_pixmap
         || M_all_pixmap->size() != painter.window().size() )
    {
        M_all_pixmap
            = boost::shared_ptr< QPixmap >
            ( new QPixmap( painter.window().size() ) );
        redraw = true;
    }

    if ( M_center_x != vconf.fieldCenter().x
         || M_center_y != vconf.fieldCenter().y
         || M_field_scale != vconf.fieldScale()
         || M_line_trace != vconf.isLineTrace() )
    {
        M_center_x = vconf.fieldCenter().x;
        M_center_y = vconf.fieldCenter().y;
        M_field_scale = vconf.fieldScale();
        M_line_trace = vconf.isLineTrace();
        redraw = true;
    }

    if ( redraw )
    {
        M_all_pixmap->fill( Qt::yellow );

        QPainter pm_painter( M_all_pixmap.get() );
        if ( ! drawTrace( pm_painter ) )
        {
            return;
        }
        pm_painter.end();

        M_all_pixmap->setMask( M_all_pixmap->createMaskFromColor( Qt::yellow ) );
    }

    painter.drawPixmap( 0, 0, *M_all_pixmap );
#endif
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
BallTracePainter::drawTrace( QPainter & painter )
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    const ViewHolder & holder = M_main_data.viewHolder();

    long start_cycle = vconf.ballTraceStart();
    long end_cycle = vconf.ballTraceEnd();

    const MonitorViewCont & vc = holder.monitorViewCont();

    if ( vc.empty() )
    {
        return false;
    }

    if ( vconf.isBallAutoTrace() )
    {
        MonitorViewConstPtr view = M_main_data.getViewData( M_main_data.viewIndex() );
        if ( ! view )
        {
            return false;
        }
        start_cycle = std::max( 1L, view->cycle() + vconf.autoTraceStart() );
    }
    else if ( start_cycle >= end_cycle )
    {
        return false;
    }

    const std::size_t first = holder.getIndexOf( start_cycle );
    const std::size_t last = ( vconf.isBallAutoTrace()
                               ? std::min( first + vconf.autoTracePeriod(),
                                           vc.size() - 1 )
                               : holder.getIndexOf( end_cycle ) );
    if ( first >= last )
    {
        return false;
    }

    const bool line_trace = vconf.isLineTrace();

    painter.setBrush( DrawConfig::instance().transparentBrush() );

    const QPen & bpen = DrawConfig::instance().ballPen();
    QPen black_dot_pen( Qt::black );
    black_dot_pen.setStyle( Qt::DotLine );

    std::size_t i = first;
    int prev_x = vconf.screenX( vc[i]->ball().x() );
    int prev_y = vconf.screenY( vc[i]->ball().y() );
    ++i;
    for ( ; i <= last; ++i )
    {
        switch ( vc[i]->playmode().mode() ) {
        case rcsc::PM_BeforeKickOff:
        case rcsc::PM_TimeOver:
        case rcsc::PM_KickOff_Left:
        case rcsc::PM_KickOff_Right:
            prev_x = vconf.screenX( 0.0 );
            prev_y = vconf.screenY( 0.0 );
            continue;
        case rcsc::PM_PlayOn:
        case rcsc::PM_AfterGoal_Left:
        case rcsc::PM_AfterGoal_Right:
            painter.setPen( bpen );
            break;
        default:
            painter.setPen( black_dot_pen );
            break;
        }

        const Ball & b = vc[i]->ball();
        int ix = vconf.screenX( b.x() );
        int iy = vconf.screenY( b.y() );

        painter.drawLine( prev_x, prev_y, ix, iy );
        if ( ! line_trace )
        {
            painter.drawPixmap( ix - 2, iy - 2, *M_point_pixmap );
        }
        prev_x = ix;
        prev_y = iy;
    }

    return true;
}
