// -*-c++-*-

/*!
  \file ball_painter_rcss.cpp
  \brief rcssmonitor style: ball painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa Akiyama

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <QtGui>

#include "ball_painter_rcss.h"

#include "draw_config.h"

#include "main_data.h"
#include "monitor_view_data.h"
#include "app_config.h"

#include <rcsc/common/server_param.h>
#include <rcsc/geom/vector_2d.h>

#include <vector>

/*-------------------------------------------------------------------*/
/*!

*/
void
BallPainterRCSS::draw( QPainter & painter )
{
    const ViewConfig & vconf = M_main_data.viewConfig();

    if ( ! vconf.isShownBall() )
    {
        return;
    }

    MonitorViewConstPtr view
        = M_main_data.getViewData( M_main_data.viewIndex() );
    if ( ! view )
    {
        return;
    }

    const DrawConfig & dconf = DrawConfig::instance();

    // decide radius
    const int ball_radius
        = std::max( 1,
                    ( vconf.isEnlarged()
                      ? vconf.scale( 0.3 )
                      : vconf.scale( rcsc::ServerParam::i().ballSize() ) )
                    );
    const int kickable_radius
        = std::max( 1,
                    vconf.scale( rcsc::ServerParam::i().defaultKickableArea() ) );

    const int ix = vconf.screenX( view->ball().x() );
    const int iy = vconf.screenY( view->ball().y() );

    // draw ball body
    painter.setPen( dconf.transparentPen() );
    painter.setBrush( dconf.ballBrush() );
    painter.drawEllipse( ix - ball_radius,
                         iy - ball_radius,
                         ball_radius * 2,
                         ball_radius * 2 );

    // draw kickable margin
    painter.setPen( dconf.ballPen() );
    painter.setBrush( dconf.transparentBrush() );
    painter.drawEllipse( ix - kickable_radius,
                         iy - kickable_radius,
                         kickable_radius * 2,
                         kickable_radius * 2 );

    // draw future status
    if ( vconf.ballFutureCycle() > 0
         && view->ball().hasDelta() )
    {
        drawFutureState( painter );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
BallPainterRCSS::drawFutureState( QPainter & painter ) const
{
    static QPen pen( QColor( 255, 0, 0 ), Qt::SolidLine );

    MonitorViewConstPtr view
        = M_main_data.getViewData( M_main_data.viewIndex() );

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();
    const rcsc::ServerParam & sparam = rcsc::ServerParam::i();

    QPainter::RenderHints hints = painter.renderHints();
    if ( vconf.antiAliasing() )
    {
        painter.setRenderHint( QPainter::Antialiasing, false );
    }

    rcsc::Vector2D bpos( view->ball().x(),
                         view->ball().y() );
    rcsc::Vector2D bvel( view->ball().deltaX(),
                         view->ball().deltaY() );
    const double ball_speed = bvel.r();

    bpos *= vconf.reverseValue();
    bvel *= vconf.reverseValue();

    QPoint first_point( vconf.absScreenX( bpos.x ),
                        vconf.absScreenY( bpos.y ) );

    painter.save();
    painter.translate( first_point.x(), first_point.y() );
    painter.rotate( bvel.th().degree() );

    painter.setPen( pen );
    painter.setBrush( dconf.transparentBrush() );

    double travel = 0.0;
    double speed = ball_speed;
    int last_x = 0;
    const int max_loop = vconf.ballFutureCycle();
    for ( int i = 0; i < max_loop; ++i )
    {
        travel += speed;
        speed *= sparam.ballDecay();

        if ( speed < 0.5 )
        {
            break;
        }

        int x = vconf.scale( travel );
        if ( std::abs( last_x - x ) < 1 )
        {
            break;
        }
        last_x = x;
        int ispeed_y = vconf.scale( speed * 0.5 );
        painter.drawLine( x, -ispeed_y, x, ispeed_y );
    }

    last_x = vconf.scale( ball_speed / ( 1.0 - sparam.ballDecay() ) );
    painter.drawLine( 0, 0, last_x, 0 );

    painter.restore();

    if ( vconf.antiAliasing() )
    {
        painter.setRenderHint( QPainter::Antialiasing );
    }
}
