// -*-c++-*-

/*!
  \file view_config_dialog.cpp
  \brief View Config Dialog class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "view_config_dialog.h"

#include "main_data.h"
#include "view_config.h"

#include <iostream>
#include <cassert>

/*-------------------------------------------------------------------*/
/*!

*/
ViewConfigDialog::ViewConfigDialog( QWidget * parent,
                                    const MainData & main_data,
                                    ViewConfig & view_config )


    //    : QDialog( parent )
    : QTabDialog( parent )
    , M_main_data( main_data )
    , M_view_config( view_config )
{
    this->setCaption( tr( "View Preference" ) );

    //QFont font = this->font();
    //font.setPointSize( 8 );
    //this->setFont( font );

    createWidgets();

    this->setOkButton( QString::null ); // no ok button
}

/*-------------------------------------------------------------------*/
/*!

*/
ViewConfigDialog::~ViewConfigDialog()
{
    //std::cerr << "delete ViewConfigDialog" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::createWidgets()
{
    // canvas
    {
        QVBox * box = new QVBox( this );

        createZoomControls( box );
        createObjectSizeControls( box );
        createCanvasSizeControls( box );
        createFieldStyleControls( box );
        createMiscControls( box );

        box->layout()->setResizeMode( QLayout::Fixed );
        this->addTab( box, tr( "Canvas" ) );
    }
    // show
    {
        QVBox * box = new QVBox( this );

        createShowControls( box );
        createPlayersDetailControls( box );
        createReverseControls( box );
        createGridStepControls( box );
        createCompGeomControls( box );

        box->layout()->setResizeMode( QLayout::Fixed );
        this->addTab( box, tr( "Show" ) );
    }
    // trace / future
    {
        QVBox * box = new QVBox( this );

        createFocusControls( box );
        createPlayerSelectionControls( box );
        createTraceControls( box );
        createInertiaMoveControls( box );

        box->layout()->setResizeMode( QLayout::Fixed );
        this->addTab( box, tr( "Trace" ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createZoomControls( QWidget * parent )
{
    QVGroupBox * group_box = new QVGroupBox( tr( "Zoom" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    const QFontMetrics metrics = parent->fontMetrics();
    const int button_height = metrics.height() + 12;

    {
        QFrame * box = new QFrame( group_box );
        QHBoxLayout * layout = new QHBoxLayout( box );

        QPushButton * zoomin_button = new QPushButton( tr( "+" ), box );
        zoomin_button->setMaximumSize( 48, button_height );
        connect( zoomin_button, SIGNAL( clicked() ),
                 this, SLOT( zoomIn() ) );
        layout->addWidget( zoomin_button, 0, Qt::AlignLeft );
        //
        QPushButton * zoomout_button = new QPushButton( tr( "-" ), box );
        zoomout_button->setMaximumSize( 48, button_height );
        connect( zoomout_button, SIGNAL( clicked() ),
                 this, SLOT( zoomOut() ) );
        layout->addWidget( zoomout_button, 0, Qt::AlignLeft );
        //
        layout->addStretch( 1 );
        //
        QPushButton * fit_button = new QPushButton( tr( "Fit" ), box );
        fit_button->setMaximumSize( 60, button_height );
        connect( fit_button, SIGNAL( clicked() ),
                 this, SLOT( unzoom() ) );
        layout->addWidget( fit_button, 0, Qt::AlignRight );
    }
//     {
//         M_score_board_auto_fit = new QCheckBox( tr( "Score Board Auto Fit" ), group_box );
//         M_score_board_auto_fit->setChecked( M_view_config.scoreBoardAutoFit() );
//         connect( M_score_board_auto_fit_cb, SIGNAL( toggled( bool ) ),
//                  this, SLOT( clickScoreBoardAutoFit( bool ) );
//     }
    {
        QHBox * box = new QHBox( group_box );

        new QLabel( tr( "Scale: " ), box );
        //
        M_scale_slider = new QSlider( Qt::Horizontal, box );
        M_scale_slider->setRange( static_cast< int >( ViewConfig::MIN_FIELD_SCALE
                                                      * 10.0 ),
                                  static_cast< int >( ViewConfig::MAX_FIELD_SCALE
                                                      * 10.0 ) );
        M_scale_slider->setValue( 80 );
        M_scale_slider->setSteps( 1, 1 );
        M_scale_slider->setMinimumSize( 300, 16 );
        connect( M_scale_slider, SIGNAL( sliderMoved( int ) ),
                 this, SLOT( slideFieldScale( int ) ) );
        //
        M_scale_text = new QLineEdit( tr( "0.0" ), box );
        M_scale_text->setValidator( new QDoubleValidator( 0.0, 400.0, 3, M_scale_text ) );
        M_scale_text->setMaximumSize( 48, 32 );
        M_scale_text->setMinimumSize( 40, 24 );
        connect( M_scale_text, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( editFieldScale( const QString & ) ) );
    }

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createObjectSizeControls( QWidget * parent )
{
    QVGroupBox * group_box = new QVGroupBox( tr( "Object Size" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    M_enlarge_cb = new QCheckBox( tr( "Enlarge Objects" ), group_box );
    M_enlarge_cb->setChecked( M_view_config.isEnlarged() );
    connect( M_enlarge_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickEnlarge( bool ) ) );

    QHBox * box = new QHBox( group_box );
    //
    new QLabel( tr( " Ball Size:" ), box );
    M_ball_size_text = new QLineEdit( tr( "0.35" ), box );
    M_ball_size_text->setValidator( new QDoubleValidator( 0.01, 100.0, 3, M_ball_size_text ) );
    M_ball_size_text->setMaximumSize( 48, 24 );
    connect( M_ball_size_text, SIGNAL( textChanged( const QString & ) ),
             this, SLOT( editBallSize( const QString & ) ) );
    //
    new QLabel( tr( " Player Size:" ), box );
    M_player_size_text = new QLineEdit( tr( "0.0" ), box );
    M_player_size_text->setValidator( new QDoubleValidator( 0.0, 100.0, 3, M_ball_size_text ) );
    M_player_size_text->setMaximumSize( 48, 24 );
    connect( M_player_size_text, SIGNAL( textChanged( const QString & ) ),
             this, SLOT( editPlayerSize( const QString & ) ) );

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createCanvasSizeControls( QWidget * parent )
{
    QHGroupBox * group_box = new QHGroupBox( tr( "Canvas Size" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    new QLabel( tr( " Width:" ), group_box );

    M_canvas_width_text = new QLineEdit( tr( "640" ), group_box );
    M_canvas_width_text->setValidator( new QIntValidator( 100, 3000, M_canvas_width_text ) );
    M_canvas_width_text->setMaximumSize( 48, 24 );

    new QLabel( tr( " Height:" ), group_box );

    M_canvas_height_text = new QLineEdit( tr( "480" ), group_box );
    M_canvas_height_text->setValidator( new QIntValidator( 100, 3000, M_canvas_height_text ) );
    M_canvas_height_text->setMaximumSize( 48, 24 );

    QPushButton * apply_canvas_size_btn = new QPushButton( tr( "apply" ), group_box );
    apply_canvas_size_btn->setMaximumSize( 60, parent->fontMetrics().height() + 12 );
    connect( apply_canvas_size_btn, SIGNAL( clicked() ),
             this, SLOT( applyCanvasSize() ) );

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createReverseControls( QWidget * parent )
{
    QHGroupBox * group_box = new QHGroupBox( tr( "Reverse" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    M_reverse_side_cb = new QCheckBox( tr( "Reverse Side" ), group_box );
    M_reverse_side_cb->setChecked( M_view_config.reverseSide() );
    connect( M_reverse_side_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickReverseSide( bool ) ) );

    M_player_reverse_draw_cb = new QCheckBox( tr( "Reverse Player Draw Order" ),
                                              group_box );
    M_player_reverse_draw_cb->setChecked( M_view_config.playerReverseDraw() );
    connect( M_player_reverse_draw_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickPlayerReverseDraw( bool ) ) );

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createPlayersDetailControls( QWidget * parent )
{
    QVGroupBox * group_box = new QVGroupBox( tr( "Players\' Detail" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 0 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    QHBox * top_box = new QHBox( group_box );
    top_box->setSpacing( 2 );

    //
    M_player_number_cb = new QCheckBox( tr( "Unum" ), top_box );
    M_player_number_cb->setChecked( M_view_config.isShownPlayerNumber() );
    connect( M_player_number_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickShowPlayerNumber( bool ) ) );
    //
    M_hetero_number_cb = new QCheckBox( tr( "Type" ), top_box );
    M_hetero_number_cb->setChecked( M_view_config.isShownHeteroNumber() );
    connect( M_hetero_number_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickShowHeteroNumber( bool ) ) );
    //
    M_stamina_cb = new QCheckBox( tr( "Stamina" ), top_box );
    M_stamina_cb->setChecked( M_view_config.isShownStamina() );
    connect( M_stamina_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickShowStamina( bool ) ) );

    //
    QHBox * bottom_box = new QHBox( group_box );
    bottom_box->setSpacing( 2 );

    //
    M_view_cone_cb = new QCheckBox( tr( "View Cone" ), bottom_box );
    M_view_cone_cb->setChecked( M_view_config.isShownViewCone() );
    connect( M_view_cone_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickShowViewCone( bool ) ) );
    //
    M_body_shadow_cb = new QCheckBox( tr( "Body Shadow" ), bottom_box );
    M_body_shadow_cb->setChecked( M_view_config.isShownBodyShadow() );
    connect( M_body_shadow_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickShowBodyShadow( bool ) ) );
    //
    M_control_area_cb = new QCheckBox( tr( "Control Area" ), bottom_box );
    M_control_area_cb->setChecked( M_view_config.isShownControlArea() );
    connect( M_control_area_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickShowControlArea( bool ) ) );

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createShowControls( QWidget * parent )
{
    QVGroupBox * group_box = new QVGroupBox( tr( "Show" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    {
        QHBox * box = new QHBox( group_box );
        box->setSpacing( 2 );
        //
        M_anonymous_mode_cb = new QCheckBox( tr( "Anonymous Mode" ), box );
        M_anonymous_mode_cb->setChecked( M_view_config.anonymousMode() );
        connect( M_anonymous_mode_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickAnonymousMode( bool ) ) );
    }
    {
        QHBox * top_box = new QHBox( group_box );
        top_box->setSpacing( 2 );

        //
        M_show_score_board_cb = new QCheckBox( tr( "Score Board" ), top_box );
        M_show_score_board_cb->setChecked( M_view_config.isShownScoreBoard() );
        connect( M_show_score_board_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowScoreBoard( bool ) ) );
        //
        M_show_team_logo_cb = new QCheckBox( tr( "Team Logo" ), top_box );
        M_show_team_logo_cb->setChecked( M_view_config.isShownTeamLogo() );
        connect( M_show_team_logo_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowTeamLogo( bool ) ) );
        //
        M_show_flags_cb = new QCheckBox( tr( "Flag" ), top_box );
        M_show_flags_cb->setChecked( M_view_config.isShownFlags() );
        connect( M_show_flags_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowFlags( bool ) ) );
    }
    {
        QHBox * bottom_box = new QHBox( group_box );
        //
        M_show_ball_cb = new QCheckBox( tr( "Ball" ), bottom_box );
        M_show_ball_cb->setChecked( M_view_config.isShownBall() );
        connect( M_show_ball_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowBall( bool ) ) );
        //
        M_show_players_cb = new QCheckBox( tr( "Player" ), bottom_box );
        M_show_players_cb->setChecked( M_view_config.isShownPlayers() );
        connect( M_show_players_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowPlayers( bool ) ) );
        //
        M_show_offside_line_cb = new QCheckBox( tr( "Offside Line" ), bottom_box );
        M_show_offside_line_cb->setChecked( M_view_config.isShownOffsideLine() );
        connect( M_show_offside_line_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowOffsideLine( bool ) ) );
    }

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createCompGeomControls( QWidget * parent )
{
    QVGroupBox * group_box = new QVGroupBox( tr( "Compututational Geometry" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    {
        QHBox * above_box = new QHBox( group_box );
        //
        M_show_voronoi_cb = new QCheckBox( tr( "VoronoiDiagram" ), above_box );
        M_show_voronoi_cb->setChecked( M_view_config.isShownVoronoiDiagram() );
        connect( M_show_voronoi_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowVoronoiDiagram( bool ) ) );
        //
        M_show_delaunay_cb = new QCheckBox( tr( "DelaunayTrianglation" ),
                                            above_box );
        M_show_delaunay_cb->setChecked( M_view_config.isShownDelaunayTrianglation() );
        connect( M_show_delaunay_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickShowDelaunayTrianglation( bool ) ) );
    }

    M_voronoi_choice = new QComboBox( group_box );
    M_voronoi_choice->insertItem( tr( "All" ) );
    M_voronoi_choice->insertItem( tr( "Left Team" ) );
    M_voronoi_choice->insertItem( tr( "Right Team" ) );
    M_voronoi_choice->setMaximumSize( this->fontMetrics().width( tr( "Right Team" ) ) + 32,
                                      this->fontMetrics().height() + 12 );
    M_voronoi_choice->setCurrentItem( 0 );
    connect( M_voronoi_choice, SIGNAL( activated( int ) ),
             this, SLOT( selectCompGeomSide( int ) ) );

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createFocusControls( QWidget * parent )
{
    QGroupBox * group_box = new QHButtonGroup( tr( "Focus" ),
                                               parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    //
    M_focus_ball_rb = new QRadioButton( tr( "Ball" ), group_box );
    connect( M_focus_ball_rb, SIGNAL( clicked() ),
             this, SLOT( clickFocusBall() ) );
    //
    M_focus_player_rb = new QRadioButton( tr( "Selected Player" ),
                                          group_box );
    connect( M_focus_player_rb, SIGNAL( clicked() ),
             this, SLOT( clickFocusPlayer() ) );
    //
    M_focus_fix_rb = new QRadioButton( tr( "Fix" ), group_box );
    connect( M_focus_fix_rb, SIGNAL( clicked() ),
             this, SLOT( setFocusFix() ) );
    //
    M_focus_fix_rb->setChecked( true );

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createPlayerSelectionControls( QWidget * parent )
{
    QVGroupBox * group_box = new QVGroupBox( tr( "Player Selection" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    // selection type
    {
        QGroupBox * box = new QHButtonGroup( group_box );
        box->setFlat( true );
        box->setInsideMargin( 0 );
        box->setInsideSpacing( 0 );
        //
        M_select_all_rb = new QRadioButton( tr( "Auto" ), box );
        connect( M_select_all_rb, SIGNAL( clicked() ),
                 this, SLOT( clickSelectAutoAll() ) );
        //
        M_select_left_rb = new QRadioButton( tr( "Auto left" ), box );
        connect( M_select_left_rb, SIGNAL( clicked() ),
                 this, SLOT( clickSelectAutoLeft() ) );
        //
        M_select_right_rb = new QRadioButton( tr( "Auto right" ), box );
        connect( M_select_right_rb, SIGNAL( clicked() ),
                 this, SLOT( clickSelectAutoRight() ) );
        //
        M_select_fix_rb = new QRadioButton( tr( "Fix" ), box );
        connect( M_select_all_rb, SIGNAL( clicked() ),
                 this, SLOT( clickSelectFix() ) );
        //
        M_unselect_rb = new QRadioButton( tr( "Unselect" ), box );
        connect( M_unselect_rb, SIGNAL( clicked() ),
                 this, SLOT( setUnselect() ) );
        //
        M_unselect_rb->setChecked( true );
    }

    {
        QFrame * box = new QFrame( group_box );
        QHBoxLayout * layout = new QHBoxLayout( box );
        //layout->setMargin( 0 );
        //layout->setSpacing( 0 );

        //layout->addSpacing( 16 ); // need to add the space before next choice

        M_player_choice = new QComboBox( box );
        M_player_choice->insertItem( tr ( "No Selection" ) );
        for ( int i = 1; i <= 11; ++i )
        {
            M_player_choice->insertItem( QString( "Left number: %1" ).arg( i ) );
        }
        for ( int i = 1; i <= 11; ++i )
        {
            M_player_choice->insertItem( QString( "Right number: %1" ).arg( i ) );
        }
        M_player_choice->setCurrentItem( 0 );
        int id_width = this->fontMetrics().width( tr( "Right Number: 11" ) );
        M_player_choice->setMaximumWidth( id_width + 40 );
        connect( M_player_choice, SIGNAL( activated( int ) ),
                 this, SLOT( choicePlayer( int ) ) );
        layout->addWidget( M_player_choice, 0, Qt::AlignLeft );
    }

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createFieldStyleControls( QWidget * parent )
{
    QGroupBox * group_box = new QVGroupBox( tr( "Field Style" ),
                                                   parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    {
        QHButtonGroup * hbox = new QHButtonGroup( group_box );
        hbox->setFlat( true );
        hbox->setInsideMargin( 0 );
        hbox->setInsideSpacing( 0 );
        //
        M_keepaway_mode_cb = new QCheckBox( tr( "Keepaway" ), hbox );
        M_keepaway_mode_cb->setChecked( M_view_config.keepawayMode() );
        connect( M_keepaway_mode_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickKeepawayMode( bool ) ) );
        //
        M_grass_normal_rb = new QRadioButton( tr( "mono" ), hbox );
        connect( M_grass_normal_rb, SIGNAL( clicked() ),
                 this, SLOT( clickGrassNormal() ) );
        //
        M_grass_line_rb = new QRadioButton( tr( "lines" ), hbox );
        connect( M_grass_line_rb, SIGNAL( clicked() ),
                 this, SLOT( clickGrassLine() ) );
        //
        M_grass_checker_rb = new QRadioButton( tr( "checker" ), hbox );
        connect( M_grass_checker_rb, SIGNAL( clicked() ),
                 this, SLOT( clickGrassChecker() ) );
    }
    {
        M_draw_type_choice = new QComboBox( group_box );
        M_draw_type_choice->insertItem( tr( "default" ) );
        M_draw_type_choice->insertItem( tr( "rcssmonitor" ) );
        connect( M_draw_type_choice, SIGNAL( activated( int ) ),
                 this, SLOT( selectDrawType( int ) ) );
    }

    switch ( M_view_config.grassType() ) {
    case ViewConfig::GRASS_NORMAL:
        M_grass_normal_rb->setChecked( true );
        break;
    case ViewConfig::GRASS_LINES:
        M_grass_line_rb->setChecked( true );
        break;
    case ViewConfig::GRASS_CHECKER:
        M_grass_checker_rb->setChecked( true );
        break;
    default:
        M_grass_normal_rb->setChecked( true );
        break;
    }

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createMiscControls( QWidget * parent )
{
    QGroupBox * group_box = new QVGroupBox( tr( "Misc" ),
                                               parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );

    M_cursor_hide_cb = new QCheckBox( tr( "Audomatic Cursor Hide" ), group_box );
    M_cursor_hide_cb->setChecked( M_view_config.cursorHide() );
    connect( M_cursor_hide_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickCursorHide( bool ) ) );

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createGridStepControls( QWidget * parent )
{
    QGroupBox * group_box = new QVGroupBox( tr( "Grid Lines" ),
                                               parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
//     group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
//                                            QSizePolicy::Fixed ) );


    M_show_grid_coord_cb = new QCheckBox( tr( "Grid Coordinate" ), group_box );
    M_show_grid_coord_cb->setChecked( M_view_config.isShownGridCoord() );
    connect( M_show_grid_coord_cb, SIGNAL( toggled( bool ) ),
             this, SLOT( clickShowGridCoord( bool ) ) );
    //
    //
    QHBox * hbox = new QHBox( group_box );
    //
    new QLabel( tr( "Step:" ), hbox );
    //
    M_grid_step_slider = new QSlider( Qt::Horizontal, hbox );
    M_grid_step_slider->setRange( 0, 1000 );
    M_grid_step_slider->setValue( 0 );
    M_grid_step_slider->setSteps( 1, 1 );
    //        M_grid_step_slider->setMinimumSize( 300, -1 );
    connect( M_grid_step_slider, SIGNAL( sliderMoved( int ) ),
             this, SLOT( slideGridStep( int ) ) );
    //
    M_grid_step_text = new QLineEdit( tr( "0.0" ), hbox );
    M_grid_step_text->setValidator( new QDoubleValidator( 0.0, 200.0, 2, M_grid_step_text ) );
    M_grid_step_text->setMaximumSize( 48, 24 );
    connect( M_grid_step_text, SIGNAL( textChanged( const QString & ) ),
             this, SLOT( editGridStep( const QString & ) ) );

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createTraceControls( QWidget * parent )
{
    QVGroupBox * group_box = new QVGroupBox( tr( "Trace" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    QVBox * top_box = new QVBox( group_box );
    {
        QHGroupBox * ball_box = new QHGroupBox( tr( "Ball" ),
                                                top_box );
        ball_box->setInsideMargin( 5 );
        ball_box->setInsideSpacing( 2 );
        ball_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                              QSizePolicy::Fixed ) );
        //
        M_ball_trace_start = new QLineEdit( tr( "0" ), ball_box );
        M_ball_trace_start->setValidator( new QIntValidator( 0, 20000, M_ball_trace_start ) );
        M_ball_trace_start->setMaximumSize( 36, 24 );
        connect( M_ball_trace_start, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( editBallTraceStart( const QString & ) ) );
        //
        new QLabel( tr( "-" ), ball_box );
        //
        M_ball_trace_end = new QLineEdit( tr( "0" ), ball_box );
        M_ball_trace_end->setValidator( new QIntValidator( 0, 20000, M_ball_trace_end ) );
        M_ball_trace_end->setMaximumSize( 36, 24 );
        connect( M_ball_trace_end, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( editBallTraceEnd( const QString & ) ) );
        //
        QPushButton * all_btn = new QPushButton( tr( "All" ), ball_box );
        all_btn->setMaximumSize( 48, this->fontMetrics().height() + 12 );
        all_btn->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
        connect( all_btn, SIGNAL( clicked() ),
                 this, SLOT( clickBallTraceAll() ) );
        //
        M_ball_trace_cb = new QCheckBox( tr( "Auto" ), ball_box );
        M_ball_trace_cb->setChecked( false );
        connect( M_ball_trace_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickAutoBallTrace( bool ) ) );
    }
    {
        QHGroupBox * player_box = new QHGroupBox( tr( "Player" ),
                                                  top_box );
        player_box->setInsideMargin( 5 );
        player_box->setInsideSpacing( 2 );
        player_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                                QSizePolicy::Fixed ) );
        //
        M_player_trace_start = new QLineEdit( tr( "0" ), player_box );
        M_player_trace_start->setValidator( new QIntValidator( 0, 20000, M_player_trace_start ) );
        M_player_trace_start->setMaximumSize( 36, 24 );
        connect( M_player_trace_start, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( editPlayerTraceStart( const QString & ) ) );
        //
        new QLabel( tr( "-" ), player_box );
        //
        M_player_trace_end = new QLineEdit( tr( "0" ), player_box );
        M_player_trace_end->setValidator( new QIntValidator( 0, 20000, M_player_trace_end ) );
        M_player_trace_end->setMaximumSize( 36, 24 );
        connect( M_ball_trace_end, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( editBallTraceEnd( const QString & ) ) );
        //
        QPushButton * all_btn = new QPushButton( tr( "All" ), player_box );
        all_btn->setMaximumSize( 48, this->fontMetrics().height() + 12 );
        all_btn->setSizePolicy( QSizePolicy::Fixed, QSizePolicy::Fixed );
        connect( all_btn, SIGNAL( clicked() ),
                 this, SLOT( clickPlayerTraceAll() ) );
        //
        M_player_trace_cb =  new QCheckBox( tr( "Auto" ), player_box );
        M_player_trace_cb->setChecked( false );
        connect( M_player_trace_cb, SIGNAL( toggled( bool ) ),
                 this, SLOT( clickAutoPlayerTrace( bool ) ) );
    }

    QHBox * bottom_box = new QHBox( top_box );
    {
        QHGroupBox * trace_box = new QHGroupBox( tr( "Auto Trace Period" ),
                                                 bottom_box );
        trace_box->setInsideMargin( 5 );
        trace_box->setInsideSpacing( 2 );
        //
        new QLabel( tr( "Start:" ), trace_box );
        //
        M_auto_trace_start = new QLineEdit( tr( "-10" ), trace_box );
        M_auto_trace_start->setValidator( new QIntValidator( -20000, 20000, M_auto_trace_start ) );
        M_auto_trace_start->setMaximumSize( 36, 24 );
        connect( M_auto_trace_start, SIGNAL( textChanged( const QString & ) ),
                 this, SLOT( editAutoTraceStart( const QString & ) ) );
        //
        new QLabel( tr( "Period:" ), trace_box );
        //
        M_auto_trace_period = new QSpinBox( trace_box );
        M_auto_trace_period->setValue( 10 );
        M_auto_trace_period->setRange( 1, 12000 );
        //M_auto_trace_period->setMaximumSize( 36, 24 );
        connect( M_auto_trace_period, SIGNAL( valueChanged( int ) ),
                 this, SLOT( changeAutoTracePeriod( int ) ) );
    }
    {
        QPushButton * line_point_btn = new QPushButton( tr( "Line/Point" ),
                                                        bottom_box );
        line_point_btn->setMaximumSize( 80, this->fontMetrics().height() + 12 );
        connect( line_point_btn, SIGNAL( clicked() ),
                 this, SLOT( clickLinePointButton() ) );
    }

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
QWidget *
ViewConfigDialog::createInertiaMoveControls( QWidget * parent )
{
    QHGroupBox * group_box = new QHGroupBox( tr( "Inertia Move" ),
                                             parent );
    group_box->setInsideMargin( 5 );
    group_box->setInsideSpacing( 2 );
    group_box->setSizePolicy( QSizePolicy( QSizePolicy::Fixed,
                                           QSizePolicy::Fixed ) );

    //
    new QLabel( tr( "Ball:" ), group_box );
    //
    M_ball_future = new QSpinBox( group_box );
    //M_ball_future->setMinimumSize( 60, 24 );
    M_ball_future->setValue( 0 );
    M_ball_future->setRange( 0, 50 );
    connect( M_ball_future, SIGNAL( valueChanged( int ) ),
             this, SLOT( changeBallFutureCycle( int ) ) );
    //
    new QLabel( tr( "Player:" ), group_box );
    //
    M_player_future = new QSpinBox( group_box );
    //M_player_future->setMaximumSize( 60, 24 );
    M_player_future->setValue( 0 );
    M_player_future->setRange( 0, 10 );
    connect( M_player_future, SIGNAL( valueChanged( int ) ),
             this, SLOT( changePlayerFutureCycle( int ) ) );

    return group_box;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::showEvent( QShowEvent * event )
{
    updateAll();
    //event->accept();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::updateAll()
{
    M_enlarge_cb->setChecked( M_view_config.isEnlarged() );
    M_ball_size_text->setText( QString::number( M_view_config.ballSize() ) );
    M_player_size_text->setText( QString::number( M_view_config.playerSize() ) );

    updateFieldScale();

    M_canvas_width_text
        ->setText( QString::number( M_view_config.canvasWidth() ) );

    M_canvas_height_text
        ->setText( QString::number( M_view_config.canvasHeight() ) );

    M_player_number_cb->setChecked( M_view_config.isShownPlayerNumber() );
    M_hetero_number_cb->setChecked( M_view_config.isShownHeteroNumber() );
    M_stamina_cb->setChecked( M_view_config.isShownStamina() );
    M_view_cone_cb->setChecked( M_view_config.isShownViewCone() );
    M_body_shadow_cb->setChecked( M_view_config.isShownBodyShadow() );
    M_control_area_cb->setChecked( M_view_config.isShownControlArea() );

    M_anonymous_mode_cb->setChecked( M_view_config.anonymousMode() );
    M_show_score_board_cb->setChecked( M_view_config.isShownScoreBoard() );
    M_show_ball_cb->setChecked( M_view_config.isShownBall() );
    M_show_players_cb->setChecked( M_view_config.isShownPlayers() );
    M_show_flags_cb->setChecked( M_view_config.isShownFlags() );
    M_show_offside_line_cb->setChecked( M_view_config.isShownOffsideLine() );

    switch ( M_view_config.grassType() ) {
    case ViewConfig::GRASS_NORMAL:
        M_grass_normal_rb->setChecked( true );
        break;
    case ViewConfig::GRASS_LINES:
        M_grass_line_rb->setChecked( true );
        break;
    case ViewConfig::GRASS_CHECKER:
        M_grass_checker_rb->setChecked( true );
        break;
    default:
        M_grass_normal_rb->setChecked( true );
        break;
    }

    M_keepaway_mode_cb->setChecked( M_view_config.keepawayMode() );

    M_draw_type_choice->setCurrentItem( M_view_config.drawType() );

    M_cursor_hide_cb->setChecked( M_view_config.cursorHide() );

    M_show_voronoi_cb->setChecked( M_view_config.isShownVoronoiDiagram() );
    M_show_delaunay_cb->setChecked( M_view_config.isShownDelaunayTrianglation() );

    switch ( M_view_config.voronoiTarget() ) {
    case rcsc::NEUTRAL:
        M_voronoi_choice->setCurrentItem( 0 );
        break;
    case rcsc::LEFT:
        M_voronoi_choice->setCurrentItem( 1 );
        break;
    case rcsc::RIGHT:
        M_voronoi_choice->setCurrentItem( 2 );
        break;
    default:
        M_voronoi_choice->setCurrentItem( 0 );
        break;
    }

    switch( M_view_config.focusType() ) {
    case ViewConfig::FOCUS_BALL:
        M_focus_ball_rb->setChecked( true );
        break;
    case ViewConfig::FOCUS_PLAYER:
        M_focus_player_rb->setChecked( true );
        break;
    case ViewConfig::FOCUS_POINT:
        M_focus_fix_rb->setChecked( true );
        break;
    default:
        M_focus_fix_rb->setChecked( true );
        break;
    }

    switch ( M_view_config.playerSelectType() ) {
    case ViewConfig::SELECT_AUTO_ALL:
        M_select_all_rb->setChecked( true );
        break;
    case ViewConfig::SELECT_AUTO_LEFT:
        M_select_left_rb->setChecked( true );
        break;
    case ViewConfig::SELECT_AUTO_RIGHT:
        M_select_right_rb->setChecked( true );
        break;
    case ViewConfig::SELECT_UNSELECT:
        M_unselect_rb->setChecked( true );
        break;
    default:
        M_unselect_rb->setChecked( true );
        break;
    }

    M_ball_trace_start->setText( QString::number( M_view_config.ballTraceStart() ) );
    M_ball_trace_end->setText( QString::number( M_view_config.ballTraceEnd() ) );
    M_ball_trace_cb->setChecked( M_view_config.isBallAutoTrace() );

    M_player_trace_start->setText( QString::number( M_view_config.playerTraceStart() ) );
    M_player_trace_end->setText( QString::number( M_view_config.playerTraceEnd() ) );
    M_player_trace_cb->setChecked( M_view_config.isPlayerAutoTrace() );

    M_auto_trace_start->setText( QString::number( M_view_config.autoTraceStart() ) );
    M_auto_trace_period->setValue( M_view_config.autoTracePeriod() );

    M_ball_future->setValue( M_view_config.ballFutureCycle() );
    M_player_future->setValue( M_view_config.playerFutureCycle() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::updateFieldScale()
{
    int iscale = static_cast< int >( rint( M_view_config.fieldScale() * 10.0 ) );
    if ( M_scale_slider->value() != iscale )
    {
        M_scale_slider->setValue( iscale );
    }

    char buf[16];
    std::snprintf( buf, 16, "%.2f", M_view_config.fieldScale() );
    M_scale_text->setText( QString::fromAscii( buf ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::zoomIn()
{
    M_view_config.zoomIn();

    updateFieldScale();

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::zoomOut()
{
    M_view_config.zoomOut();

    updateFieldScale();

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::unzoom()
{
    M_view_config.unzoom();
    updateFieldScale();
    M_focus_fix_rb->setChecked( true );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::slideFieldScale( int value )
{
    double scale = value * 0.1;

    if ( std::fabs( scale - M_view_config.fieldScale() ) >= 0.01 )
    {
        M_scale_text->setText( QString::number( scale ) );

        M_view_config.setFieldScale( scale );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editFieldScale( const QString & text )
{
    bool ok = true;
    double value = text.toDouble( &ok );

    if ( ok
         && std::fabs( value - M_view_config.fieldScale() ) >= 0.01 )
    {
        int ivalue = static_cast< int >( rint( value * 10.0 ) );
        M_scale_slider->setValue( ivalue );

        M_view_config.setFieldScale( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
// void
// ViewConfigDialog::clickScoreBoardAutoFit( bool checked )
// {
//     if ( M_view_config.scoreBoardAutoFit() != checked )
//     {
//         M_view_config.toggleScoreBoardAutoFit();
//         emit configured();
//     }
//}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickEnlarge( bool checked )
{
    if ( M_view_config.isEnlarged() != checked )
    {
        M_view_config.toggleEnlarge();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleEnlarge()
{
    M_view_config.toggleEnlarge();
    M_enlarge_cb->setChecked( M_view_config.isEnlarged() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editBallSize( const QString & text )
{
    bool ok = true;
    double value = text.toDouble( &ok );

    if ( ok && value > 0.0 )
    {
        M_view_config.setBallSize( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editPlayerSize( const QString & text )
{
    bool ok = true;
    double value = text.toDouble( &ok );

    if ( ok )
    {
        M_view_config.setPlayerSize( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::applyCanvasSize()
{
    bool ok_w = true;
    bool ok_h = true;
    int width = M_canvas_width_text->text().toInt( & ok_w );
    int height = M_canvas_height_text->text().toInt( & ok_h );

    if ( ok_w
         && ok_h
         && width > 0
         && height > 0 )
    {
        emit canvasResized( QSize( width, height ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickReverseSide( bool checked )
{
    if ( M_view_config.reverseSide() != checked )
    {
        M_view_config.toggleReverseSide();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleReverseSide()
{
    M_view_config.toggleReverseSide();
    M_reverse_side_cb->setChecked( M_view_config.reverseSide() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickPlayerReverseDraw( bool checked )
{
    if ( M_view_config.playerReverseDraw() != checked )
    {
        M_view_config.togglePlayerReverseDraw();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::togglePlayerReverseDraw()
{
    M_view_config.togglePlayerReverseDraw();
    M_player_reverse_draw_cb->setChecked( M_view_config.playerReverseDraw() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowPlayerNumber( bool checked )
{
    if ( M_view_config.isShownPlayerNumber() != checked )
    {
        M_view_config.toggleShowPlayerNumber();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowPlayerNumber()
{
    M_view_config.toggleShowPlayerNumber();
    M_player_number_cb->setChecked( M_view_config.isShownPlayerNumber() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowHeteroNumber( bool checked )
{
    if ( M_view_config.isShownHeteroNumber() != checked )
    {
        M_view_config.toggleShowHeteroNumber();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowHeteroNumber()
{
    M_view_config.toggleShowHeteroNumber();
    M_hetero_number_cb->setChecked( M_view_config.isShownHeteroNumber() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowStamina( bool checked )
{
    if ( M_view_config.isShownStamina() != checked )
    {
        M_view_config.toggleShowStamina();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowStamina()
{
    M_view_config.toggleShowStamina();
    M_stamina_cb->setChecked( M_view_config.isShownStamina() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowViewCone( bool checked )
{
    if ( M_view_config.isShownViewCone() != checked )
    {
        M_view_config.toggleShowViewCone();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowViewCone()
{
    M_view_config.toggleShowViewCone();
    M_view_cone_cb->setChecked( M_view_config.isShownViewCone() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowBodyShadow( bool checked )
{
    if ( M_view_config.isShownBodyShadow() != checked )
    {
        M_view_config.toggleShowBodyShadow();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowBodyShadow()
{
    M_view_config.toggleShowBodyShadow();
    M_body_shadow_cb->setChecked( M_view_config.isShownBodyShadow() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowControlArea( bool checked )
{
    if ( M_view_config.isShownControlArea() != checked )
    {
        M_view_config.toggleShowControlArea();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowControlArea()
{
    M_view_config.toggleShowControlArea();
    M_control_area_cb->setChecked( M_view_config.isShownControlArea() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickAnonymousMode( bool checked )
{
    if ( M_view_config.anonymousMode() != checked )
    {
        M_view_config.toggleAnonymousMode();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleAnonymousMode()
{
    M_view_config.toggleAnonymousMode();
    M_anonymous_mode_cb->setChecked( M_view_config.anonymousMode() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowScoreBoard( bool checked )
{
    if ( M_view_config.isShownScoreBoard() != checked )
    {
        M_view_config.toggleShowScoreBoard();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowScoreBoard()
{
    M_view_config.toggleShowScoreBoard();
    M_show_score_board_cb->setChecked( M_view_config.isShownScoreBoard() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowTeamLogo( bool checked )
{
    if ( M_view_config.isShownTeamLogo() != checked )
    {
        M_view_config.toggleShowTeamLogo();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowTeamLogo()
{
    M_view_config.toggleShowTeamLogo();
    M_show_team_logo_cb->setChecked( M_view_config.isShownTeamLogo() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowBall( bool checked )
{
    if ( M_view_config.isShownBall() != checked )
    {
        M_view_config.toggleShowBall();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowBall()
{
    M_view_config.toggleShowBall();
    M_show_ball_cb->setChecked( M_view_config.isShownBall() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowPlayers( bool checked )
{
    if ( M_view_config.isShownPlayers() != checked )
    {
        M_view_config.toggleShowPlayers();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowPlayers()
{
    M_view_config.toggleShowPlayers();
    M_show_players_cb->setChecked( M_view_config.isShownPlayers() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowFlags( bool checked )
{
    if ( M_view_config.isShownFlags() != checked )
    {
        M_view_config.toggleShowFlags();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowFlags()
{
    M_view_config.toggleShowFlags();
    M_show_flags_cb->setChecked( M_view_config.isShownFlags() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowOffsideLine( bool checked )
{
    if ( M_view_config.isShownOffsideLine() != checked )
    {
        M_view_config.toggleShowOffsideLine();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleShowOffsideLine()
{
    M_view_config.toggleShowOffsideLine();
    M_show_offside_line_cb->setChecked( M_view_config.isShownOffsideLine() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickGrassNormal()
{
    if ( M_view_config.grassType() != ViewConfig::GRASS_NORMAL )
    {
        M_view_config.setGrassType( ViewConfig::GRASS_NORMAL );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickGrassLine()
{
    if ( M_view_config.grassType() != ViewConfig::GRASS_LINES )
    {
        M_view_config.setGrassType( ViewConfig::GRASS_LINES );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickGrassChecker()
{
    if ( M_view_config.grassType() != ViewConfig::GRASS_CHECKER )
    {
        M_view_config.setGrassType( ViewConfig::GRASS_CHECKER );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickKeepawayMode( bool checked )
{
    if ( M_view_config.keepawayMode() != checked )
    {
        M_view_config.toggleKeepawayMode();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowGridCoord( bool checked )
{
    if ( M_view_config.isShownGridCoord() != checked )
    {
        M_view_config.toggleShowGridCoord();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::slideGridStep( int value )
{
    if ( value != M_view_config.gridStep() )
    {
        M_grid_step_text->setText( QString::number( value * 0.1 ) );

        M_view_config.setGridStep( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editGridStep( const QString & text )
{
    bool ok = true;
    double value = text.toDouble( &ok );

    if ( ok )
    {
        int ivalue = static_cast< int >( rint( value * 10.0 ) );
        if ( ivalue != M_view_config.gridStep() )
        {
            M_grid_step_slider->setValue( ivalue );
            M_grid_step_text->setText( QString::number( ivalue * 0.1 ) );

            M_view_config.setGridStep( ivalue );

            emit configured();
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::selectDrawType( int index )
{
    if ( static_cast< int >( M_view_config.drawType() ) != index )
    {
        M_view_config.setDrawType( static_cast< ViewConfig::DrawType >( index ) );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleKeepawayMode()
{
    M_view_config.toggleKeepawayMode();
    M_keepaway_mode_cb->setChecked( M_view_config.keepawayMode() );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickCursorHide( bool checked )
{
    if ( M_view_config.cursorHide() != checked )
    {
        M_view_config.toggleCursorHide();

        //emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowVoronoiDiagram( bool checked )
{
    if ( M_view_config.isShownVoronoiDiagram() != checked )
    {
        M_view_config.toggleShowVoronoiDiagram();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickShowDelaunayTrianglation( bool checked )
{
    if ( M_view_config.isShownDelaunayTrianglation() != checked )
    {
        M_view_config.toggleShowDelaunayTrianglation();
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::selectCompGeomSide( int index )
{
    switch ( index ) {
    case 0:
        if ( M_view_config.voronoiTarget() != rcsc::NEUTRAL )
        {
            M_view_config.setVoronoiTarget( rcsc::NEUTRAL );
            emit configured();
        }
        break;
    case 1:
        if ( M_view_config.voronoiTarget() != rcsc::LEFT )
        {
            M_view_config.setVoronoiTarget( rcsc::LEFT );
            emit configured();
        }
        break;
    case 2:
        if ( M_view_config.voronoiTarget() != rcsc::RIGHT )
        {
            M_view_config.setVoronoiTarget( rcsc::RIGHT );
            emit configured();
        }
        break;
    default:
        std::cerr << "ViewConfigDialog::selectCompGeomSide() unsupported index "
                  << index
                  << std::endl;
        break;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickFocusBall()
{
    if ( M_view_config.focusType() != ViewConfig::FOCUS_BALL )
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_BALL );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleFocusBall()
{
    if ( M_view_config.focusType() == ViewConfig::FOCUS_BALL )
    {
        setFocusFix();
    }
    else
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_BALL );
        M_focus_ball_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickFocusPlayer()
{
    if ( M_view_config.focusType() != ViewConfig::FOCUS_PLAYER )
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_PLAYER );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleFocusPlayer()
{
    if ( M_view_config.focusType() == ViewConfig::FOCUS_PLAYER )
    {
        setFocusFix();
    }
    else
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_PLAYER );
        M_focus_player_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setFocusFix()
{
    if ( M_view_config.focusType() != ViewConfig::FOCUS_POINT )
    {
        M_view_config.setFocusType( ViewConfig::FOCUS_POINT );
        M_focus_fix_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setFocusPoint( const QPoint & point )
{
    M_view_config.setFocusPoint( point.x(), point.y() );
    M_focus_fix_rb->setChecked( true );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickSelectAutoAll()
{
    if ( M_view_config.playerSelectType() != ViewConfig::SELECT_AUTO_ALL )
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_ALL );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleSelectAutoAll()
{
    if ( M_view_config.playerSelectType() == ViewConfig::SELECT_AUTO_ALL )
    {
        setUnselect();
    }
    else
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_ALL );
        M_select_all_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickSelectAutoLeft()
{
    ViewConfig::PlayerSelectType type = ( M_view_config.reverseSide()
                                          ? ViewConfig::SELECT_AUTO_RIGHT
                                          : ViewConfig::SELECT_AUTO_LEFT );

    if ( M_view_config.playerSelectType() != type )
    {
        M_view_config.setPlayerSelectType( type );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleSelectAutoLeft()
{
    ViewConfig::PlayerSelectType type = ( M_view_config.reverseSide()
                                          ? ViewConfig::SELECT_AUTO_RIGHT
                                          : ViewConfig::SELECT_AUTO_LEFT );

    if ( M_view_config.playerSelectType() == type )
    {
        setUnselect();
    }
    else
    {
        M_view_config.setPlayerSelectType( type );
        M_select_left_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickSelectAutoRight()
{
    ViewConfig::PlayerSelectType type = ( M_view_config.reverseSide()
                                          ? ViewConfig::SELECT_AUTO_LEFT
                                          : ViewConfig::SELECT_AUTO_RIGHT );

    if ( M_view_config.playerSelectType() != type )
    {
        M_view_config.setPlayerSelectType( type );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleSelectAutoRight()
{
    ViewConfig::PlayerSelectType type = ( M_view_config.reverseSide()
                                          ? ViewConfig::SELECT_AUTO_LEFT
                                          : ViewConfig::SELECT_AUTO_RIGHT );

    if ( M_view_config.playerSelectType() == type )
    {
        setUnselect();
    }
    else
    {
        M_view_config.setPlayerSelectType( type );
        M_select_right_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickSelectFix()
{
    if ( M_view_config.playerSelectType() != ViewConfig::SELECT_FIX )
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_AUTO_RIGHT );
        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::toggleSelectFix()
{
    if ( M_view_config.playerSelectType() == ViewConfig::SELECT_FIX )
    {
        setUnselect();
    }
    else
    {
        M_view_config.setPlayerSelectType( ViewConfig::SELECT_FIX );
        M_select_fix_rb->setChecked( true );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::setUnselect()
{
    M_view_config.unselectPlayer();
    M_unselect_rb->setChecked( true );
    M_player_choice->setCurrentItem( 0 );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::selectPlayer( int number )
{
    // left side:  number = [1,11]
    // right side:  number = [12,22]
    if ( number <= 0
         || 22 < number )
    {
        setUnselect();
        return;
    }

    int orig_number = number;
    if ( M_view_config.reverseSide() )
    {
        if ( number <= 11 )
        {
            number += 11;
        }
        else
        {
            number -= 11;
        }
    }

    M_view_config.setPlayerSelectType( ViewConfig::SELECT_FIX );
    if ( number <= 11 )
    {
        M_view_config.setSelectedNumber( rcsc::LEFT, number );
    }
    else
    {
        M_view_config.setSelectedNumber( rcsc::RIGHT, number - 11 );
    }

    M_player_choice->setCurrentItem( orig_number );
    M_select_fix_rb->setChecked( true );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::choicePlayer( int number )
{
    int orig_number = number;
    if ( M_view_config.reverseSide() )
    {
        if ( number <= 11 )
        {
            number += 11;
        }
        else
        {
            number -= 11;
        }
    }

    if ( number <= 11
         && M_view_config.isSelectedPlayer( rcsc::LEFT, number ) )
    {
        return;
    }

    if ( number > 11
         && M_view_config.isSelectedPlayer( rcsc::RIGHT, number - 11 ) )
    {
        return;
    }

    selectPlayer( orig_number );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editBallTraceStart( const QString & text )
{
    bool ok = true;
    long value = text.toLong( & ok );

    if ( ok )
    {
        //std::cerr << "editBallTraceStart " << value << std::endl;
        M_view_config.setBallTraceStart( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editBallTraceEnd( const QString & text )
{
    bool ok = true;
    long value = text.toLong( & ok );

    if ( ok )
    {
        //std::cerr << "editBallTraceEnd " << value << std::endl;
        M_view_config.setBallTraceEnd( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickBallTraceAll()
{
    const std::vector< MonitorViewPtr > & view
        = M_main_data.viewHolder().monitorViewCont();

    if ( ! view.empty() )
    {
        M_ball_trace_cb->setChecked( false );

        if ( M_view_config.ballTraceStart() == view.front()->cycle()
                  && M_view_config.ballTraceEnd() == view.back()->cycle() )
        {
            M_ball_trace_start->setText( QString::number( 0 ) );
            M_ball_trace_end->setText( QString::number( 0 ) );

            M_view_config.setBallTraceStart( 0 );
            M_view_config.setBallTraceEnd( 0 );
        }
        else
        {
            M_ball_trace_start->setText( QString::number( view.front()->cycle() ) );
            M_ball_trace_end->setText( QString::number( view.back()->cycle() ) );

            M_view_config.setBallTraceStart( view.front()->cycle() );
            M_view_config.setBallTraceEnd( view.back()->cycle() );
        }

        if ( M_view_config.isBallAutoTrace() )
        {
            M_view_config.toggleBallAutoTrace();
        }

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickAutoBallTrace( bool checked )
{
    if ( M_view_config.isBallAutoTrace() != checked )
    {
        M_view_config.toggleBallAutoTrace();

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editPlayerTraceStart( const QString & text )
{
    bool ok = true;
    long value = text.toLong( & ok );

    if ( ok )
    {
        std::cerr << "editPlayerTraceStart " << value << std::endl;
        M_view_config.setPlayerTraceStart( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editPlayerTraceEnd( const QString & text )
{
    bool ok = true;
    long value = text.toLong( & ok );

    if ( ok )
    {
        std::cerr << "editPlayerTraceEnd " << value << std::endl;
        M_view_config.setPlayerTraceEnd( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickPlayerTraceAll()
{
    const std::vector< MonitorViewPtr > & view
        = M_main_data.viewHolder().monitorViewCont();
    if ( ! view.empty() )
    {
        M_player_trace_cb->setChecked( false );

        if ( M_view_config.playerTraceStart() == view.front()->cycle()
             && M_view_config.playerTraceEnd() == view.back()->cycle() )
        {
            M_player_trace_start->setText( QString::number( 0 ) );
            M_player_trace_end->setText( QString::number( 0 ) );

            M_view_config.setPlayerTraceStart( 0 );
            M_view_config.setPlayerTraceEnd( 0 );
        }
        else
        {
            M_player_trace_start->setText( QString::number( view.front()->cycle() ) );
            M_player_trace_end->setText( QString::number( view.back()->cycle() ) );

            M_view_config.setPlayerTraceStart( view.front()->cycle() );
            M_view_config.setPlayerTraceEnd( view.back()->cycle() );
        }

        if ( M_view_config.isPlayerAutoTrace() )
        {
            M_view_config.togglePlayerAutoTrace();
        }

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickAutoPlayerTrace( bool checked )
{
    if ( M_view_config.isPlayerAutoTrace() != checked )
    {
        M_view_config.togglePlayerAutoTrace();

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::editAutoTraceStart( const QString & text )
{
    bool ok = true;
    long value = text.toLong( & ok );

    if ( ok )
    {
        //std::cerr << "editAutTraceStart " << value << std::endl;
        M_view_config.setAutoTraceStart( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::changeAutoTracePeriod( int value )
{
    //std::cerr << "changedAutoTracePeriod " << value << std::endl;
    if ( value >= 1 )
    {
        M_view_config.setAutoTracePeriod( value );

        emit configured();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::clickLinePointButton()
{
    M_view_config.toggleLineTrace();

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::changeBallFutureCycle( int value )
{
    M_view_config.setBallFutureCycle( value );

    emit configured();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ViewConfigDialog::changePlayerFutureCycle( int value )
{
    M_view_config.setPlayerFutureCycle( value );

    emit configured();
}
