// -*-c++-*-

/*!
  \file plaeyr_painter_rcssmonito.cpp
  \brief rcssmonitor style: player painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "player_painter_rcss.h"

#include "player_painter.h"
#include "draw_config.h"

// model
#include "main_data.h"

#include <rcsc/common/player_type.h>
#include <rcsc/common/server_param.h>
#include <rcsc/geom/angle_deg.h>

#include <cstring>
#include <cstdio>
#include <cmath>


const QPen PlayerPainterRCSS::PLAYER_PEN( QColor( 0, 0, 0 ),
                                          0, Qt::SolidLine );
const QPen PlayerPainterRCSS::NECK_PEN( QColor( "#ff0000" ),
                                        0, Qt::SolidLine );

const QPen PlayerPainterRCSS::LEFT_TEAM_PEN( QColor( "#ff0000" ),
                                             0, Qt::SolidLine );
const QBrush PlayerPainterRCSS::LEFT_TEAM_BRUSH( QColor( "#ffff00" ),
                                                 Qt::SolidPattern );
const QBrush PlayerPainterRCSS::LEFT_GOALIE_BRUSH( QColor( "#00ff00" ),
                                                   Qt::SolidPattern );

const QPen PlayerPainterRCSS::RIGHT_TEAM_PEN( QColor( "#00008b" ),
                                              0, Qt::SolidLine );
const QBrush PlayerPainterRCSS::RIGHT_TEAM_BRUSH( QColor( "#00ffff" ),
                                                  Qt::SolidPattern );
const QBrush PlayerPainterRCSS::RIGHT_GOALIE_BRUSH( QColor( "#ff99ff" ),
                                                    Qt::SolidPattern );

const QBrush PlayerPainterRCSS::BALL_COLLIDE_BRUSH( QColor( "#ff0000" ),
                                                    Qt::SolidPattern );

const QBrush PlayerPainterRCSS::VIEW_AREA_TOOFAR_BRUSH( QColor( "#00b400" ),
                                                        Qt::SolidPattern );
const QBrush PlayerPainterRCSS::VIEW_AREA_BRUSH( QColor( "#00aa00" ),
                                                 Qt::SolidPattern );

const QBrush PlayerPainterRCSS::KICK_FAULT_BRUSH( QColor( "#0000ff" ),
                                                  Qt::SolidPattern );

/*-------------------------------------------------------------------*/
/*

*/
inline
PlayerPainterRCSS::Param::Param( const Player & player,
                                 const Ball & ball,
                                 const ViewConfig & view_conf,
                                 const rcsc::PlayerType & ptype )
    : x_( view_conf.screenX( player.x() ) )
    , y_( view_conf.screenY( player.y() ) )
    , body_( player.body() + ( view_conf.reverseSide() ? 180.0 : 0.0 ) )
    , head_( player.head() + ( view_conf.reverseSide() ? 180.0 : 0.0 ) )
    , body_radius_( view_conf.scale( ptype.playerSize() ) )
    , kick_radius_( view_conf.scale( ptype.kickableArea() ) )
    , have_full_effort_( player.hasFullEffort( ptype.effortMax() ) )
    , player_( player )
    , ball_( ball )
    , player_type_( ptype )
{

    if ( body_radius_ < 1 ) body_radius_ = 1;
    if ( kick_radius_ < 5 ) kick_radius_ = 5;

    draw_radius_ =  kick_radius_;
}

/*-------------------------------------------------------------------*/
/*

*/
PlayerPainterRCSS::PlayerPainterRCSS( const MainData & main_data )
    : M_main_data( main_data )
{

}

/*-------------------------------------------------------------------*/
/*

*/
void
PlayerPainterRCSS::draw( QPainter & painter )
{
    if ( ! M_main_data.viewConfig().isShownPlayers() )
    {
        return;
    }

    MonitorViewConstPtr view = M_main_data.getViewData( M_main_data.viewIndex() );

    if ( ! view )
    {
        return;
    }

    const ViewConfig & vconf = M_main_data.viewConfig();

    const Ball & ball = view->ball();

    if ( vconf.selectedNumber() != 0 )
    {
        const std::vector< Player >::const_iterator end = view->players().end();
        for ( std::vector< Player >::const_iterator it = view->players().begin();
              it != end;
              ++it )
        {
            if ( ! it->hasView() )
            {
                break;
            }

            if ( vconf.isSelectedPlayer( it->side(), it->unum() ) )
            {
                drawViewArea( painter, *it );
                break;
            }
        }
    }

    if ( vconf.playerReverseDraw() )
    {
        const std::vector< Player >::const_reverse_iterator end = view->players().rend();
        for ( std::vector< Player >::const_reverse_iterator it = view->players().rbegin();
              it != end;
              ++it )
        {
            drawAll( painter, *it, ball );
        }
    }
    else
    {
        const std::vector< Player >::const_iterator end = view->players().end();
        for ( std::vector< Player >::const_iterator it = view->players().begin();
              it != end;
              ++it )
        {
            drawAll( painter, *it, ball );
        }
    }
}

/*-------------------------------------------------------------------*/
/*

*/
void
PlayerPainterRCSS::drawAll(  QPainter & painter,
                             const Player & player,
                             const Ball & ball ) const
{
    const Param param( player,
                       ball,
                       M_main_data.viewConfig(),
                       M_main_data.viewHolder().playerType( player.type() ) );

    drawBody( painter, param );
    //drawShadow( painter, param );
    drawEdge( painter, param );

    if ( M_main_data.viewConfig().isSelectedPlayer( player.side(), player.unum() )
         && M_main_data.viewConfig().playerFutureCycle() > 0
         && player.hasDelta() )
    {
        drawFuture( painter, param );
    }

    if ( player.hasView()
         && M_main_data.viewConfig().isShownViewCone() )
    {
        drawViewCone( painter, param );
    }

    if ( M_main_data.viewConfig().isShownControlArea() )
    {
        drawControlArea( painter, param );
    }

    drawText( painter, param );
}

/*-------------------------------------------------------------------*/
/*

*/
void
PlayerPainterRCSS::drawBody( QPainter & painter,
                             const PlayerPainterRCSS::Param & param ) const
{
    const DrawConfig & dconf = DrawConfig::instance();

    // decide base color
    painter.setPen( PLAYER_PEN );

    rcsc::SideID side = param.player_.side();
    if ( M_main_data.viewConfig().reverseSide() )
    {
        side = static_cast< rcsc::SideID >( -1 * side );
    }

    switch ( side ) {
    case rcsc::LEFT:
        if ( param.player_.isGoalie() )
        {
            painter.setBrush( LEFT_GOALIE_BRUSH );
        }
        else
        {
            painter.setBrush( LEFT_TEAM_BRUSH );
        }
        break;
    case rcsc::RIGHT:
        if ( param.player_.isGoalie() )
        {
            painter.setBrush( RIGHT_GOALIE_BRUSH );
        }
        else
        {
            painter.setBrush( RIGHT_TEAM_BRUSH );
        }
        break;
    case rcsc::NEUTRAL:
        painter.setBrush( dconf.shadowBrush() );
        break;
    default:
        painter.setBrush( dconf.shadowBrush() );
        break;
    }


    // decide status color
    if ( ! param.player_.isAlive() )
    {
        painter.setBrush( dconf.shadowBrush() );
    }
    if ( param.player_.isKicking() )
    {
        painter.setPen( dconf.kickPen() );
    }
    if ( param.player_.isKickingFault() )
    {
        painter.setPen( dconf.kickFaultPen() );
        painter.setBrush( KICK_FAULT_BRUSH );
    }
    if ( param.player_.isCatching() )
    {
        painter.setBrush( dconf.catchBrush() );
    }
    if ( param.player_.isCatchingFault() )
    {
        painter.setBrush( dconf.catchFaultBrush() );
    }
    if ( param.player_.isTackling() )
    {
        painter.setPen( dconf.tacklePen() );
        painter.setBrush( dconf.tackleBrush() );
    }
    if ( param.player_.isTacklingFault() )
    {
        painter.setPen( dconf.tacklePen() );
        painter.setBrush( dconf.tackleFaultBrush() );
    }
    if ( param.player_.isCollidedBall() )
    {
        painter.setBrush( BALL_COLLIDE_BRUSH );
    }
//     if ( param.player_.isCollidedPlayer() )
//     {
//         painter.setBrush( dconf.collidePlayerBrush() );
//     }

    painter.drawEllipse( param.x_ - param.kick_radius_ ,
                         param.y_ - param.kick_radius_ ,
                         param.kick_radius_ * 2 ,
                         param.kick_radius_ * 2 );

}

/*-------------------------------------------------------------------*/
/*

*/
void
PlayerPainterRCSS::drawShadow( QPainter & painter,
                               const PlayerPainterRCSS::Param & param ) const
{

}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainterRCSS::drawEdge( QPainter & painter,
                             const PlayerPainterRCSS::Param & param ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    // draw body edge
    {

        // draw real body edge
        painter.setPen( PLAYER_PEN );
        painter.setBrush( dconf.transparentBrush() );
        painter.drawEllipse( param.x_ - param.body_radius_,
                             param.y_ - param.body_radius_,
                             param.body_radius_ * 2 ,
                             param.body_radius_ * 2  );
    }

    // draw stamina status if effort or recovery is decayed.
    if ( param.player_.hasStamina() )
    {
        if ( param.player_.stamina() < 1500.0 )
        {
            painter.setBrush( dconf.shadowBrush() );
            painter.drawEllipse( param.x_ - param.body_radius_,
                                 param.y_ - param.body_radius_,
                                 param.body_radius_ * 2 ,
                                 param.body_radius_ * 2  );
        }
    }

    // body direction line
    {
        int end_x = vconf.absScreenX( param.player_.x() * vconf.reverseValue()
                                      + param.player_type_.kickableArea()
                                      * std::cos( param.body_ * rcsc::AngleDeg::DEG2RAD ) );
        int end_y = vconf.absScreenY( param.player_.y() * vconf.reverseValue()
                                      + param.player_type_.kickableArea()
                                      * std::sin( param.body_ * rcsc::AngleDeg::DEG2RAD ) );

        painter.setPen( PLAYER_PEN );
        painter.setBrush( dconf.transparentBrush() );
        painter.drawLine( param.x_, param.y_, end_x, end_y );

    }
}


/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainterRCSS::drawFuture( QPainter & painter,
                               const PlayerPainterRCSS::Param & param ) const
{
    const DrawConfig & dconf = DrawConfig::instance();

    // draw future state
    rcsc::Vector2D ppos( param.player_.x(), param.player_.y() );
    rcsc::Vector2D pvel( param.player_.deltaX(), param.player_.deltaY() );
    if ( M_main_data.viewConfig().reverseSide() )
    {
        ppos *= -1.0;
        pvel *= -1.0;
    }

    const int last = M_main_data.viewConfig().playerFutureCycle();

    QPoint first_point( M_main_data.viewConfig().absScreenX( ppos.x ),
                        M_main_data.viewConfig().absScreenY( ppos.y ) ) ;
    QPoint last_point = first_point;

    // draw kickable area edge
    rcsc::SideID side = param.player_.side();
    if ( M_main_data.viewConfig().reverseSide() )
    {
        side = static_cast< rcsc::SideID >( -1 * side );
    }

    if ( side == rcsc::LEFT )
    {
        painter.setPen( dconf.rightTeamPen() );
    }
    else
    {
        painter.setPen( dconf.leftTeamPen() );
    }

    painter.setBrush( dconf.transparentBrush() );

    for ( int i = 0; i < last; ++i )
    {
        ppos += pvel;
        pvel *= param.player_type_.playerDecay();

        QPoint pt( M_main_data.viewConfig().absScreenX( ppos.x ),
                   M_main_data.viewConfig().absScreenY( ppos.y ) );
        if ( std::abs( last_point.x() - pt.x() ) < 1
             && std::abs( last_point.y() - pt.y() ) < 1 )
        {
            break;
        }

        painter.drawEllipse( pt.x() - 1,
                             pt.y() - 1,
                             3,
                             3 );
        painter.drawEllipse( pt.x() - param.kick_radius_,
                             pt.y() - param.kick_radius_,
                             param.kick_radius_ * 2,
                             param.kick_radius_ * 2 );

        last_point = pt;
    }

    // draw move line
    painter.setPen( dconf.debugTargetPen() );
    painter.drawLine( first_point, last_point );
}


/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainterRCSS::drawViewArea( QPainter & painter,
                                 const Player & player ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const int x = vconf.screenX( player.x() );
    const int y = vconf.screenY( player.y() );
    const int visible_radius = vconf.scale( rcsc::ServerParam::i().visibleDistance() );
    const int view_start_angle
        = static_cast< int >
        ( rint( ( - player.head() - player.viewWidth()*0.5
                  + ( vconf.reverseSide() ? 180.0 : 0.0 ) ) * 16 ) );
    const int view_width
        = static_cast< int >( rint( player.viewWidth() * 16 ) );

    const int TEAM_FAR = vconf.scale( 40.0 );
    const int TEAM_TOOFAR = vconf.scale( 60.0 );

    painter.setRasterOp( Qt::OrROP );
    painter.setPen( dconf.transparentPen() );

    painter.setBrush( VIEW_AREA_TOOFAR_BRUSH );

    painter.drawPie( x - TEAM_TOOFAR,
                     y - TEAM_TOOFAR,
                     TEAM_TOOFAR * 2,
                     TEAM_TOOFAR * 2,
                     view_start_angle,
                     view_width );

    painter.drawEllipse( x - visible_radius,
                         y - visible_radius,
                         visible_radius * 2,
                         visible_radius * 2 );

    painter.setBrush( VIEW_AREA_BRUSH );

    painter.drawPie( x - TEAM_FAR,
                     y - TEAM_FAR,
                     TEAM_FAR * 2,
                     TEAM_FAR * 2,
                     view_start_angle,
                     view_width );

    painter.setRasterOp( Qt::CopyROP );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainterRCSS::drawViewCone( QPainter & painter,
                                 const PlayerPainterRCSS::Param & param ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    // draw face direction
    painter.setPen( NECK_PEN );
    painter.setBrush( dconf.transparentBrush() );

    int end_x = vconf.absScreenX( param.player_.x() * vconf.reverseValue()
                                  + param.player_type_.kickableArea()
                                  * std::cos( param.head_ * rcsc::AngleDeg::DEG2RAD ) );
    int end_y = vconf.absScreenY( param.player_.y() * vconf.reverseValue()
                                  + param.player_type_.kickableArea()
                                  * std::sin( param.head_ * rcsc::AngleDeg::DEG2RAD ) );;

    painter.drawLine( param.x_, param.y_, end_x, end_y );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainterRCSS::drawControlArea( QPainter & painter,
                                    const PlayerPainterRCSS::Param & param ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    if ( param.player_.isGoalie() )
    {
        rcsc::SideID side = param.player_.side();
        if ( M_main_data.viewConfig().reverseSide() )
        {
            side = static_cast< rcsc::SideID >( -1 * side );
        }

        int catchable = vconf.scale( rcsc::ServerParam::i().catchableArea() );
        painter.setPen( ( side == rcsc::LEFT )
                        ? dconf.leftGoaliePen()
                        : dconf.rightGoaliePen() );
        painter.setBrush( dconf.transparentBrush() );

        painter.drawEllipse( param.x_ - catchable,
                             param.y_ - catchable,
                             catchable * 2,
                             catchable * 2 );
    }

    rcsc::Vector2D ppos( param.player_.x(),
                         param.player_.y() );
    rcsc::Vector2D bpos( param.ball_.x(),
                         param.ball_.y() );
    if ( vconf.reverseSide() )
    {
        ppos *= -1.0;
        bpos *= -1.0;
    }

    rcsc::Vector2D player_to_ball = bpos - ppos;
    player_to_ball.rotate( - param.body_ );

    // draw tackle area & probability
    double tackle_dist = ( player_to_ball.x > 0.0
                           ? rcsc::ServerParam::i().tackleDist()
                           : rcsc::ServerParam::i().tackleBackDist() );
    double tackle_prob = ( std::pow( player_to_ball.absX() / tackle_dist,
                                     rcsc::ServerParam::i().tackleExponent() )
                           + std::pow( player_to_ball.absY() / rcsc::ServerParam::i().tackleWidth(),
                                       rcsc::ServerParam::i().tackleExponent() ) );
    if ( tackle_prob < 1.0 )
    {
        rcsc::AngleDeg body_angle = param.body_;
        rcsc::AngleDeg body_angle_side = body_angle + 90.0;
        double body_x = body_angle.cos();
        double body_y = body_angle.sin();
        double forward_x = rcsc::ServerParam::i().tackleDist() * body_x;
        double forward_y = rcsc::ServerParam::i().tackleDist() * body_y;
        double back_x = rcsc::ServerParam::i().tackleBackDist() * -body_x;
        double back_y = rcsc::ServerParam::i().tackleBackDist() * -body_y;
        double right_x = rcsc::ServerParam::i().tackleWidth() * body_angle_side.cos();
        double right_y = rcsc::ServerParam::i().tackleWidth() * body_angle_side.sin();

        QPointArray pts( 4 );
        pts[0].setX( vconf.absScreenX( ppos.x + forward_x + right_x ) );
        pts[0].setY( vconf.absScreenY( ppos.y + forward_y + right_y ) );
        pts[1].setX( vconf.absScreenX( ppos.x + forward_x - right_x ) );
        pts[1].setY( vconf.absScreenY( ppos.y + forward_y - right_y ) );
        pts[2].setX( vconf.absScreenX( ppos.x + back_x - right_x ) );
        pts[2].setY( vconf.absScreenY( ppos.y + back_y - right_y ) );
        pts[3].setX( vconf.absScreenX( ppos.x + back_x + right_x ) );
        pts[3].setY( vconf.absScreenY( ppos.y + back_y + right_y ) );
        pts[4] = pts[0];

        painter.setPen( dconf.tackleAreaPen() );
        painter.setBrush( dconf.transparentBrush() );

        //painter.drawConvexPolygon( pts, 0, 4 );
        painter.drawPolyline( pts, 0, 5 );

        int text_radius = param.draw_radius_;
        if ( text_radius > 40 )
        {
            text_radius = 40;
        }

        painter.setFont( dconf.playerFont() );
        painter.setPen( dconf.tacklePen() );

        char msg[32];
        std::snprintf( msg, 32, "TakleProb=%.3f", 1.0 - tackle_prob );
        painter.drawText( param.x_ + text_radius,
                          param.y_ + 4 + painter.fontMetrics().ascent(),
                          QString::fromAscii( msg ) );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
PlayerPainterRCSS::drawText( QPainter & painter,
                             const PlayerPainterRCSS::Param & param ) const
{
    const ViewConfig & vconf = M_main_data.viewConfig();

    char main_buf[64];
    std::memset( main_buf, 0, 64 );

    if ( vconf.isShownPlayerNumber() )
    {
        char buf[8];
        std::snprintf( buf, 8, "%X", param.player_.unum() );
        std::strcat( main_buf, buf );
    }

    if ( param.player_.hasStamina()
         && vconf.isShownStamina() )
    {
        char buf[16];
        std::snprintf( buf, 16, "%4.0f", param.player_.stamina() );
        if ( main_buf[0] != '\0' ) std::strcat( main_buf, " " );
        std::strcat( main_buf, buf );
    }

    if ( vconf.isShownHeteroNumber() )
    {
        char buf[8];
        std::snprintf( buf, 8, "t%d", param.player_.type() );
        if ( main_buf[0] != '\0' ) std::strcat( main_buf, ", " );
        strcat( main_buf, buf );
    }

    if ( main_buf[0] != '\0' )
    {
        static bool first = true;
        static QFont font( "6x13bold", 9, QFont::Bold );
        if ( first )
        {
            font.setPointSize( 9 );
            font.setBold( true );
            //font.setStyleHint( QFont::System, QFont::PreferBitmap );
            font.setBold( true );
            font.setFixedPitch( true );
            first = false;
        }

        painter.setFont( font );

        rcsc::SideID side = param.player_.side();
        if ( M_main_data.viewConfig().reverseSide() )
        {
            side = static_cast< rcsc::SideID >( -1 * side );
        }

        painter.setPen( side == rcsc::LEFT
                        ? LEFT_TEAM_PEN
                        : RIGHT_TEAM_PEN );
        painter.drawText( param.x_,
                          param.y_,
                          QString::fromAscii( main_buf ) );
    }
}
