// -*-c++-*-

/*!
  \file debug_log_painter.cpp
  \brief debug log_info painter class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "debug_log_painter.h"

#include "draw_config.h"
#include "main_data.h"
#include "debug_log_data.h"
#include "debug_log_holder.h"

#include <iostream>

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugLogPainter::draw( QPainter & painter )
{
    const ViewConfig & vconf = M_main_data.viewConfig();

    if ( ! vconf.isShownDebugLogObjects() )
    {
        return;
    }

    const int level = M_main_data.debugLogHolder().level();
    if ( level == 0 )
    {
        return;
    }

    const int number = vconf.selectedNumber();
    if ( number == 0 )
    {
        return;
    }

    const int unum = std::abs( number );

    const DebugLogHolder::DataList * data_list
        = M_main_data.debugLogHolder().getDataList( unum );

    if ( ! data_list )
    {
        return;
    }

    const rcsc::SideID player_side = ( number < 0 ? rcsc::RIGHT : rcsc::LEFT );

    MonitorViewConstPtr monitor_view
        = M_main_data.getViewData( M_main_data.viewIndex() );
    if ( ! monitor_view )
    {
        return;
    }

    const long cycle = monitor_view->cycle();

    const DebugLogHolder::DataList::const_iterator end = data_list->end();
    for ( DebugLogHolder::DataList::const_iterator it = data_list->begin();
          it != end;
          ++it )
    {
        if ( (*it)->cycle() == cycle )
        {
            drawPoints( painter, player_side, **it );
            drawLines( painter, player_side, **it );
            drawCircles( painter, player_side, **it );
            drawTriangles( painter, player_side, **it );
            drawRectangles( painter, player_side, **it );
            drawMessages( painter, player_side, **it );
            break;
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugLogPainter::drawPoints( QPainter & painter,
                             const rcsc::SideID player_side,
                             const DebugLogData & log_data ) const
{
    const int level = M_main_data.debugLogHolder().level();

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::LEFT
                             ? 1.0
                             : -1.0 );

    painter.setBrush( dconf.transparentBrush() );

    const DebugLogData::PointCont::const_iterator end = log_data.pointCont().end();
    for ( DebugLogData::PointCont::const_iterator it = log_data.pointCont().begin();
          it != end;
          ++it )
    {
        if ( level & it->level_ )
        {
            QColor col( it->color_.c_str() );
            if ( col.isValid() )
            {
                painter.setPen( col );
            }
            else
            {
                painter.setPen( dconf.debugFigurePen() );
            }

            painter.drawPoint( vconf.screenX( it->x_ * reverse ),
                               vconf.screenY( it->y_ * reverse ) );
        }
    }

}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugLogPainter::drawLines( QPainter & painter,
                            const rcsc::SideID player_side,
                            const DebugLogData & log_data ) const
{
    const int level = M_main_data.debugLogHolder().level();

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::LEFT
                             ? 1.0
                             : -1.0 );

    painter.setBrush( dconf.transparentBrush() );

    const DebugLogData::LineCont::const_iterator end = log_data.lineCont().end();
    for ( DebugLogData::LineCont::const_iterator it = log_data.lineCont().begin();
          it != end;
          ++it )
    {
        if ( level & it->level_ )
        {
            QColor col( it->color_.c_str() );
            if ( col.isValid() )
            {
                painter.setPen( col );
            }
            else
            {
                painter.setPen( dconf.debugFigurePen() );
            }

            painter.drawLine( vconf.screenX( it->x1_ * reverse ),
                              vconf.screenY( it->y1_ * reverse ),
                              vconf.screenX( it->x2_ * reverse ),
                              vconf.screenY( it->y2_ * reverse ) );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugLogPainter::drawCircles( QPainter & painter,
                              const rcsc::SideID player_side,
                              const DebugLogData & log_data ) const
{
    const int level = M_main_data.debugLogHolder().level();

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::LEFT
                             ? 1.0
                             : -1.0 );

    painter.setBrush( dconf.transparentBrush() );

    const DebugLogData::CircleCont::const_iterator end = log_data.circleCont().end();
    for ( DebugLogData::CircleCont::const_iterator it = log_data.circleCont().begin();
          it != end;
          ++it )
    {
        if ( level & it->level_ )
        {
            QColor col( it->color_.c_str() );
            if ( col.isValid() )
            {
                painter.setPen( col );
            }
            else
            {
                painter.setPen( dconf.debugFigurePen() );
            }

            int r = vconf.scale( it->r_ );
            painter.drawEllipse( vconf.screenX( it->x_ * reverse ) - r,
                                 vconf.screenY( it->y_ * reverse ) - r,
                                 r * 2, r * 2 );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugLogPainter::drawTriangles( QPainter & painter,
                                const rcsc::SideID player_side,
                                const DebugLogData & log_data ) const
{
    const int level = M_main_data.debugLogHolder().level();

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::LEFT
                             ? 1.0
                             : -1.0 );

    painter.setBrush( dconf.transparentBrush() );

    const DebugLogData::TriangleCont::const_iterator end = log_data.triangleCont().end();
    for ( DebugLogData::TriangleCont::const_iterator it = log_data.triangleCont().begin();
          it != end;
          ++it )
    {
        if ( level & it->level_ )
        {
            QColor col( it->color_.c_str() );
            if ( col.isValid() )
            {
                painter.setPen( col );
            }
            else
            {
                painter.setPen( dconf.debugFigurePen() );
            }

            QPointArray points( 3 );
            points.setPoint( 0,
                             vconf.screenX( it->x1_ * reverse ),
                             vconf.screenY( it->y1_ * reverse ) );
            points.setPoint( 1,
                             vconf.screenX( it->x2_ * reverse ),
                             vconf.screenY( it->y2_ * reverse ) );
            points.setPoint( 2,
                             vconf.screenX( it->x3_ * reverse ),
                             vconf.screenY( it->y3_ * reverse ) );

            painter.drawPolygon( points );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugLogPainter::drawRectangles( QPainter & painter,
                                 const rcsc::SideID player_side,
                                 const DebugLogData & log_data ) const
{
    const int level = M_main_data.debugLogHolder().level();

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::LEFT
                             ? 1.0
                             : -1.0 );

    painter.setBrush( dconf.transparentBrush() );

    const DebugLogData::RectCont::const_iterator end = log_data.rectCont().end();
    for ( DebugLogData::RectCont::const_iterator it = log_data.rectCont().begin();
          it != end;
          ++it )
    {
        if ( level & it->level_ )
        {
            QColor col( it->color_.c_str() );
            if ( col.isValid() )
            {
                painter.setPen( col );
            }
            else
            {
                painter.setPen( dconf.debugFigurePen() );
            }

            painter.drawRect( vconf.screenX( it->left_ * reverse ),
                              vconf.screenY( it->top_ * reverse ),
                              vconf.scale( it->width_ ),
                              vconf.scale( it->height_ ) );
        }
    }

}

/*-------------------------------------------------------------------*/
/*!

*/
void
DebugLogPainter::drawMessages( QPainter & painter,
                               const rcsc::SideID player_side,
                               const DebugLogData & log_data ) const
{
    const int level = M_main_data.debugLogHolder().level();

    const ViewConfig & vconf = M_main_data.viewConfig();
    const DrawConfig & dconf = DrawConfig::instance();

    const double reverse = ( player_side == rcsc::LEFT
                             ? 1.0
                             : -1.0 );

    painter.setFont( dconf.debugLogMessageFont() );
    painter.setBrush( dconf.transparentBrush() );

    const DebugLogData::MessageCont::const_iterator end = log_data.messageCont().end();
    for ( DebugLogData::MessageCont::const_iterator it = log_data.messageCont().begin();
          it != end;
          ++it )
    {
        if ( level & it->level_ )
        {
            QColor col( it->color_.c_str() );
            if ( col.isValid() )
            {
                painter.setPen( col );
            }
            else
            {
                painter.setPen( dconf.debugLogMessageFontPen() );
            }

            painter.drawText( vconf.screenX( it->x_ * reverse ),
                              vconf.screenY( it->y_ * reverse ),
                              it->message_ );
        }
    }

}
