// -*-c++-*-

/*!
  \file view_config.h
  \brief field canvas configuration class Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef SOCCERWINDOW2_MODEL_VIEW_CONFIG_H
#define SOCCERWINDOW2_MODEL_VIEW_CONFIG_H

#include "point.h"

#include <rcsc/geom/vector_2d.h>
#include <rcsc/types.h>

#include <cmath>

//! drawing parameter holder
class ViewConfig {
public:
    enum GrassType {
        GRASS_LINES,
        GRASS_CHECKER,
        GRASS_NORMAL,
    };

    enum DrawType {
        DRAW_DEFAULT,
        DRAW_RCSSMONITOR,
    };

    enum PlayerSelectType {
        SELECT_FIX,
        SELECT_AUTO_LEFT,
        SELECT_AUTO_RIGHT,
        SELECT_AUTO_ALL,
        SELECT_UNSELECT,
    };

    enum FocusType {
        FOCUS_BALL,
        FOCUS_PLAYER,
        FOCUS_POINT, // include center
    };

    //! minimal field scale
    static const double MIN_FIELD_SCALE;
    //! maximal field scale
    static const double MAX_FIELD_SCALE;

    static const double ZOOM_RATIO;

private:

    //---------------------------------------------------------
    // canvas draw parameters

    int M_canvas_width; //!< pixel of draw area width
    int M_canvas_height; //!< pixel of draw area height

    bool M_anti_aliasing;
    bool M_gradient; //!< use gradient mode

    bool M_cursor_hide; //!< automatic cursor hide or not

    Point M_field_center; //!< the screen point of field center
    double M_field_scale; //!< field scale rate

    bool M_reverse_side; //!< reverse left/right
    bool M_player_reverse_draw; //!< reverse the order of player draw

    //    bool M_score_board_auto_fit;
    int M_score_board_font_size;
    int M_score_board_height; //!< screen height of score board

    // zoom
    bool M_zoomed; //!< true if canvas scale is changed.

    // object size
    bool M_enlarge; //!< true if enlarge mode
    double M_ball_size; //!< enlarged ball size
    double M_player_size; //!< enlarged and fixed player size

    // players detail
    bool M_show_player_number; //!< true if uniform number is drawn
    bool M_show_hetero_number; //!< true if hetero ID is drawon
    bool M_show_stamina; //!< true if stamina value is drawn
    bool M_show_view_cone; //!< true if player's view cone is drawn
    bool M_show_body_shadow;
    bool M_show_control_area; //!< true if control area is drawn

    // show/hide
    bool M_anonymous_mode;
    bool M_show_score_board; //!< true if score board is drawon
    bool M_show_team_logo;
    bool M_show_ball; //!< true if ball is drawin
    bool M_show_players; //!< true if player is drawin
    bool M_show_flags; //!< true if marker flags are drawn
    bool M_show_offside_line; //!< true if offside lines are drawn

    // field
    GrassType M_grass_type; //!< field grass type
    DrawType M_draw_type;
    bool M_keepaway_mode;
    bool M_show_grid_coord;
    int M_grid_step;

    // computational geometry
    bool M_show_voronoi_diagram; //!< true if voronoi diagram is drawn
    bool M_show_delaunay_trianglation; //!< true if delaunay triangle is drawn
    rcsc::SideID M_voronoi_target;

    // focus
    FocusType M_focus_type; //!< focus type ID
    rcsc::Vector2D M_focus_point; //!< real coordinates

    // player selection
    int M_selected_number; //!< selected player's uniform number.
    PlayerSelectType M_player_select_type; //!< flag for player auto selection

    // ball trace
    long M_ball_trace_start; //!< cycle of ball trace start
    long M_ball_trace_end; //!< cycle of ball trace end
    bool M_ball_auto_trace; //! if true, trace is drawn automatically

    // player trace
    bool M_player_auto_trace; //! if true, trace is drawn automatically
    long M_player_trace_start; //!< log index of player trace start
    long M_player_trace_end; //!< log index of player trace end

    // auto trace
    long M_auto_trace_start;
    int M_auto_trace_period;

    // trace line type
    bool M_line_trace; //! if true, trace is line, else only trace point is drawn

    // inertia move
    int M_ball_future_cycle; //!< specify the cycle to draw ball future point
    int M_player_future_cycle; //!< specify the cycle to draw ball future point

    // debug view
    bool M_show_debug_view;
    bool M_show_debug_view_ball;
    bool M_show_debug_view_self;
    bool M_show_debug_view_players;
    bool M_show_debug_view_comment;
    bool M_show_debug_view_figure;
    bool M_show_debug_view_target;
    bool M_show_debug_view_message;

    // debug log objects
    bool M_show_debug_log_objects;

public:
    //! constructor
    ViewConfig();

    //! reset all settings to default
    void reset();

    /*!
      \brief update parameters with canvas size
      \param canvas_width pixel of canvas width
      \param canvas_height pixel of canvas height
    */
    void updateFieldSize( const int canvas_width,
                          const int canvas_height );

private:

    void updateFieldCenter( const int canvas_width,
                            const int canvas_height );

    /*
      \brief update scoreboard size
      \param canvas_width canvas window width
      \param canvas_height canvas window height
    */
    void updateScoreBoardSize( const int canvas_width,
                               const int canvas_height );

public:

    const
    Point & fieldCenter() const
      {
          return M_field_center;
      }

    void toggleAntiAliasing()
      {
          M_anti_aliasing = ! M_anti_aliasing;
      }
    bool antiAliasing() const
      {
          return M_anti_aliasing;
      }

    void toggleGradient()
      {
          M_gradient = ! M_gradient;
      }
    bool gradient() const
      {
          return M_gradient;
      }

    void toggleCursorHide()
      {
          M_cursor_hide = ! M_cursor_hide;
      }
    bool cursorHide() const
      {
          return M_cursor_hide;
      }

//     bool scoreBoardAutoFit() const
//       {
//           return M_score_board_auto_fit;
//       }
//     void toggleScoreBoardAutoFit()
//       {
//           M_score_board_auto_fit = ! M_score_board_auto_fit;
//       }

    int scoreBoardFontSize() const
      {
          return M_score_board_font_size;
      }

    const
    int & scoreBoardHeight() const
      {
          return M_score_board_height;
      }

    bool reverseSide() const
      {
          return M_reverse_side;
      }
    void toggleReverseSide()
      {
          M_reverse_side = ! M_reverse_side;
      }
    double reverseValue() const
      {
          return ( M_reverse_side ? -1.0 : 1.0 );
      }

    bool playerReverseDraw() const
      {
          return M_player_reverse_draw;
      }
    void togglePlayerReverseDraw()
      {
          M_player_reverse_draw = ! M_player_reverse_draw;
      }

    //------------------------------------------------------

    void zoomIn();
    void zoomOut();
    void unzoom();
    bool isZoomed() const
      {
          return M_zoomed;
      }

    bool isEnlarged() const
      {
          return M_enlarge;
      }
    void toggleEnlarge()
      {
          M_enlarge = ! M_enlarge;
      }

    const
    double & ballSize() const
      {
          return M_ball_size;
      }
    void setBallSize( const double & size )
      {
          if ( size <= 0.001 ) return;
          M_ball_size = size;
      }

    const
    double & playerSize() const
      {
          return M_player_size;
      }
    void setPlayerSize( const double & size )
      {
          if ( size < 0.0 ) return;
          M_player_size = size;
      }

    void setFieldScale( const double & scale );
    const
    double & fieldScale() const
      {
          return M_field_scale;
      }

    int canvasWidth() const
      {
          return M_canvas_width;
      }
    int canvasHeight() const
      {
          return M_canvas_height;
      }

    void setFocusPoint( const int screen_x,
                        const int screen_y );
    void updateFocusPoint( const double & x,
                           const double & y )
      {
          M_focus_point.x = x;
          M_focus_point.y = y;
      }
    const
    rcsc::Vector2D & focusPoint() const
      {
          return M_focus_point;
      }

    // players' detail

    void toggleShowPlayers()
      {
          M_show_players = ! M_show_players;
      }
    bool isShownPlayers() const
      {
          return M_show_players;
      }

    void toggleShowPlayerNumber()
      {
          M_show_player_number = ! M_show_player_number;
      }
    bool isShownPlayerNumber() const
      {
          return M_show_player_number;
      }

    void toggleShowHeteroNumber()
      {
          M_show_hetero_number = ! M_show_hetero_number;
      }
    bool isShownHeteroNumber() const
      {
          return M_show_hetero_number;
      }

    void toggleShowStamina()
      {
          M_show_stamina = ! M_show_stamina;
      }
    bool isShownStamina() const
      {
          return M_show_stamina;
      }

    void toggleShowViewCone()
      {
          M_show_view_cone = ! M_show_view_cone;
      }
    bool isShownViewCone() const
      {
          return M_show_view_cone;
      }

    void toggleShowBodyShadow()
      {
          M_show_body_shadow = ! M_show_body_shadow;
      }
    bool isShownBodyShadow() const
      {
          return M_show_body_shadow;
      }

    void toggleShowControlArea()
      {
          M_show_control_area = ! M_show_control_area;
      }
    bool isShownControlArea() const
      {
          return M_show_control_area;
      }

    // show/hide field objects

    void toggleAnonymousMode()
      {
          M_anonymous_mode = ! M_anonymous_mode;
      }
    bool anonymousMode() const
      {
          return M_anonymous_mode;
      }

    void toggleShowScoreBoard()
      {
          M_show_score_board = ! M_show_score_board;
      }
    bool isShownScoreBoard() const
      {
          return M_show_score_board;
      }

    void toggleShowTeamLogo()
      {
          M_show_team_logo = ! M_show_team_logo;
      }
    bool isShownTeamLogo() const
      {
          return M_show_team_logo;
      }

    void toggleShowBall()
      {
          M_show_ball = ! M_show_ball;
      }
    bool isShownBall() const
      {
          return M_show_ball;
      }

    void toggleShowFlags()
      {
          M_show_flags = ! M_show_flags;
      }
    bool isShownFlags() const
      {
          return M_show_flags;
      }

    void toggleShowOffsideLine()
      {
          M_show_offside_line = ! M_show_offside_line;
      }
    bool isShownOffsideLine() const
      {
          return M_show_offside_line;
      }

    // computational geometory

    void toggleShowVoronoiDiagram()
      {
          M_show_voronoi_diagram = ! M_show_voronoi_diagram;
      }
    bool isShownVoronoiDiagram() const
      {
          return M_show_voronoi_diagram;
      }

    void toggleShowDelaunayTrianglation()
      {
          M_show_delaunay_trianglation = ! M_show_delaunay_trianglation;
      }
    bool isShownDelaunayTrianglation() const
      {
          return M_show_delaunay_trianglation;
      }

    void setVoronoiTarget( const rcsc::SideID side )
      {
          M_voronoi_target = side;
      }
    rcsc::SideID voronoiTarget() const
      {
          return M_voronoi_target;
      }


    // player selection

    void unselectPlayer();
    void setPlayerSelectType( const ViewConfig::PlayerSelectType type );
    PlayerSelectType playerSelectType() const
      {
          return M_player_select_type;
      }
    bool isPlayerAutoSelect() const
      {
          return ( M_player_select_type != SELECT_FIX
                   && M_player_select_type != SELECT_UNSELECT );
      }
    void setSelectedNumber( rcsc::SideID side,
                            int unum )
      {
          M_selected_number = ( side == rcsc::LEFT ? unum : -unum );
      }
    int selectedNumber() const
      {
          return M_selected_number;
      }
    bool isSelectedPlayer( rcsc::SideID side,
                           int unum ) const
      {
          return ( M_selected_number
                   == ( side == rcsc::LEFT ? unum : -unum ) );
      }



    // field grass

    void setGrassType( GrassType type )
      {
          M_grass_type = type;
      }
    GrassType grassType() const
      {
          return M_grass_type;
      }

    void setDrawType( DrawType type )
      {
          M_draw_type = type;
      }
    DrawType drawType() const
      {
          return M_draw_type;
      }

    void toggleKeepawayMode()
      {
          M_keepaway_mode = ! M_keepaway_mode;
      }
    bool keepawayMode() const
      {
          return M_keepaway_mode;
      }

    void toggleShowGridCoord()
      {
          M_show_grid_coord = ! M_show_grid_coord;
      }
    bool isShownGridCoord() const
      {
          return M_show_grid_coord;
      }

    void setGridStep( const int step )
      {
          M_grid_step = step;
      }
    int gridStep() const
      {
          return M_grid_step;
      }

    void setFocusType( const ViewConfig::FocusType type )
      {
          M_focus_type = type;
      }
    FocusType focusType() const
      {
          return M_focus_type;
      }


    // ball trace

    bool isShownBallTrace() const
      {
          return ( isBallAutoTrace()
                   || ( M_ball_trace_start < M_ball_trace_end ) );
      }
    void setBallTraceStart( const long & cycle );
    const
    long & ballTraceStart() const
      {
          return M_ball_trace_start;
      }
    void setBallTraceEnd( const long & cycle );
    const
    long & ballTraceEnd() const
      {
          return M_ball_trace_end;
      }
    void toggleBallAutoTrace()
      {
          M_ball_auto_trace = ! M_ball_auto_trace;
      }
    bool isBallAutoTrace() const
      {
          return M_ball_auto_trace;
      }

    // player trace

    bool isShownPlayerTrace() const
      {
          return ( isPlayerAutoTrace()
                   || ( M_player_trace_start < M_player_trace_end ) );
      }
    void setPlayerTraceStart( const long & cycle );
    const
    long & playerTraceStart() const
      {
          return M_player_trace_start;
      }
    void setPlayerTraceEnd( const long & cycle );
    const
    long & playerTraceEnd() const
      {
          return M_player_trace_end;
      }
    void togglePlayerAutoTrace()
      {
          M_player_auto_trace = ! M_player_auto_trace;
      }
    bool isPlayerAutoTrace() const
      {
          return M_player_auto_trace;
      }

    // auto trace

    void setAutoTraceStart( const long & start )
      {
          M_auto_trace_start = start;
      }
    const
    long & autoTraceStart() const
      {
          return M_auto_trace_start;
      }
    void setAutoTracePeriod( const int period )
      {
          M_auto_trace_period = period;
      }
    int autoTracePeriod() const
      {
          return M_auto_trace_period;
      }

    // mode of the trace line

    void toggleLineTrace()
      {
          M_line_trace = ! M_line_trace;
      }
    bool isLineTrace() const
      {
          return M_line_trace;
      }

    // inertia move

    void setBallFutureCycle( const int cycle );
    int ballFutureCycle() const
      {
          return M_ball_future_cycle;
      }

    void setPlayerFutureCycle( const int cycle );
    int playerFutureCycle() const
      {
          return M_player_future_cycle;
      }

    //////////////////////////////////////////////////////////
    // debug view options

    void toggleShowDebugView()
      {
          M_show_debug_view = ! M_show_debug_view;
      }
    bool isShownDebugView() const
      {
          return M_show_debug_view;
      }

    void toggleShowDebugViewSelf()
      {
          M_show_debug_view_self = ! M_show_debug_view_self;
      }
    bool isShownDebugViewSelf() const
      {
          return M_show_debug_view_self;
      }

    void toggleShowDebugViewBall()
      {
          M_show_debug_view_ball = ! M_show_debug_view_ball;
      }
    bool isShownDebugViewBall() const
      {
          return M_show_debug_view_ball;
      }

    void toggleShowDebugViewPlayers()
      {
          M_show_debug_view_players = ! M_show_debug_view_players;
      }
    bool isShownDebugViewPlayers() const
      {
          return M_show_debug_view_players;
      }

    void toggleShowDebugViewComment()
      {
          M_show_debug_view_comment = ! M_show_debug_view_comment;
      }
    bool isShownDebugViewComment() const
      {
          return M_show_debug_view_comment;
      }

    void toggleShowDebugViewFigure()
      {
          M_show_debug_view_figure = ! M_show_debug_view_figure;
      }
    bool isShownDebugViewFigure() const
      {
          return M_show_debug_view_figure;
      }

    void toggleShowDebugViewTarget()
      {
          M_show_debug_view_target = ! M_show_debug_view_target;
      }
    bool isShownDebugViewTarget() const
      {
          return M_show_debug_view_target;
      }

    void toggleShowDebugViewMessage()
      {
          M_show_debug_view_message = ! M_show_debug_view_message;
      }
    bool isShownDebugViewMessage() const
      {
          return M_show_debug_view_message;
      }

    //////////////////////////////////////////////////////////
    // debug log view options

    void toggleShowDebugLogObjects()
      {
          M_show_debug_log_objects = ! M_show_debug_log_objects;
      }
    bool isShownDebugLogObjects() const
      {
          return M_show_debug_log_objects;
      }


    //////////////////////////////////////////////////////////
    /*!
      \brief scale input length to screen length.
      \param len real length
      \return screen pixel length
    */
    int scale( const double & len ) const
      {
          return static_cast< int >( rint( len * fieldScale() ) );
          //double dlen = len * M_field_scale;
          //if ( std::fabs( dlen ) < 0.5 ) return 0;
          //dlen += ( dlen > 0.0 ) ? 0.5 : -0.5;
          //return static_cast< int >( dlen );
      }

    /*!
      \brief convert 'x' to the screen coordinate X.
      \param abs_x real point
      \return screen point
    */
    int absScreenX( const double & abs_x ) const
      {
          return fieldCenter().x + scale( abs_x );
      }
    /*!
      \brief convert 'x' to the screen coordinate X with reverse mode.
      \param x real point (may be reversed)
      \return screen point
    */
    int screenX( const double & x ) const
      {
          return ( M_reverse_side
                   ? fieldCenter().x + scale( -x )
                   : fieldCenter().x + scale( x ) );
      }

    /*!
      \brief convert 'y' to the screen coordinate X.
      \param y real point
      \return screen point
    */
    int absScreenY( const double & abs_y ) const
      {
          return fieldCenter().y + scale( abs_y );
      }
    /*!
      \brief convert 'y' to the screen coordinate X.
      \param y real point (may be reversed)
      \return screen point
    */
    int screenY( const double & y ) const
      {
          return ( M_reverse_side
                   ? fieldCenter().y + scale( -y )
                   : fieldCenter().y + scale( y ) );
      }

    /*!
      \brief convert 'x' of screen to field coordinate x
      \param x screen point value
      \return field real point value
     */
    double fieldX( const int x ) const
      {
          return ( x - fieldCenter().x ) / fieldScale();
      }

    /*!
      \brief convert 'y' of screen to field coordinate y
      \param y screen point value
      \return field real point value
     */
    double fieldY( const int y ) const
      {
          return ( y - fieldCenter().y ) / fieldScale();
      }

    /*!
      \brief convert screen point to the field real coordinate
      \param point screen point value
      \return field real point value
     */
    rcsc::Vector2D fieldPoint( const Point & point ) const
      {
          return rcsc::Vector2D( fieldX( point.x ),
                                 fieldY( point.y ) );
      }
};

#endif
