// -*-c++-*-

/*!
  \file app_config.cpp
  \brief Application configuration class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#ifndef PACKAGE_NAME
#define PACKAGE_NAME "soccerwindow2"
#endif
#ifndef VERSION
#define VERSION "unknown-version"
#endif

#include "app_config.h"

#include <rcsc/param/param_map.h>
#include <rcsc/param/cmd_line_parser.h>

#include <iostream>
#include <cmath>
#include <cstdio>


const int AppConfig::MIN_SCORE_BOARD_HEIGHT = 20;
const int AppConfig::MAX_SCORE_BOARD_HEIGHT = 40;

//! default logplayer timer step (ms).
const int AppConfig::DEFAULT_TIMER_INTERVAL = 100;

/*-------------------------------------------------------------------*/
/*!
  singleton interface
*/
AppConfig &
AppConfig::instance()
{
    static AppConfig s_instance;
    return s_instance;
}

/*-------------------------------------------------------------------*/
/*!

*/
AppConfig::AppConfig()
    // monitor client options
    : M_connect( false )
    , M_host( "localhost" )
    , M_port( 6000 )
    , M_client_version( 3 )
    , M_wait_seconds( 5 )
    , M_auto_quit_mode( false )
    , M_kill_server( false )
    , M_server_pid( 0 )
    , M_server_path( "rcssserver" )
    , M_time_shift_replay( true )
      // logplayer options
    , M_game_log_file_path( "" )
    , M_game_log_dir( "" )
    , M_auto_loop_mode( false )
    , M_timer_interval( DEFAULT_TIMER_INTERVAL )
      // debug server options
    , M_debug_server_mode( false )
    , M_debug_server_port( 6000 + 32 )
    , M_debug_log_dir( "" )
      // window options
    , M_pos_x( -1 )
    , M_pos_y( -1 )
    , M_width( -1 )
    , M_height( -1 )
    , M_maximize( false )
    , M_full_screen( false )
    , M_canvas_width( -1 )
    , M_canvas_height( -1 )
    , M_tool_bar_area( "top" )
    , M_hide_menu_bar( false )
    , M_hide_tool_bar( false )
    , M_hide_status_bar( false )
      // view options
    , M_anonymous_mode( false )
    , M_draw_type( "default" )
    , M_field_grass_type( "mono" )
    , M_keepaway( false )

    , M_show_score_board( true )
    , M_show_team_logo( true )

    , M_anti_aliasing( true )
    , M_gradient( false )

    , M_reverse_side( false )
    , M_player_reverse_draw( false )

    , M_show_player_number( true )
    , M_show_player_type( false )
    , M_show_view_cone( true )
    , M_show_stamina( true )

    , M_enlarged_ball_size( 0.35 )
    , M_fixed_player_size( 0.0 )
      // image save options
    , M_auto_image_save( false )
    , M_image_save_dir( "" )
    , M_image_name_prefix( "image-" )
    , M_image_save_format( "PNG" )
      // other options
    , M_monitor_client_mode( false )
{

}

/*-------------------------------------------------------------------*/
/*!

*/
AppConfig::~AppConfig()
{

}

/*-------------------------------------------------------------------*/
/*!
  analyze command line options
*/
bool
AppConfig::parseCmdLine( int argc,
                         char ** argv )
{
    rcsc::ParamMap system_options( "System Options" );
    rcsc::ParamMap client_options( "Monitor Client Options" );
    rcsc::ParamMap logplayer_options( "Log Player Options" );
    rcsc::ParamMap debug_server_options( "Debug Server Options" );
    rcsc::ParamMap window_options( "Window Options" );
    rcsc::ParamMap view_options( "View Preference Options" );
    rcsc::ParamMap image_options( "Image Save Options" );

    bool help = false;
    bool version = false;
    std::string geometry;
    std::string canvas_size;

    system_options.add()
        ( "help", "h",
          rcsc::BoolSwitch( &help ),
          "print this message." )
        ( "version", "v",
          rcsc::BoolSwitch( &version ),
          "print version." )
        ;

    client_options.add()
        ( "connect", "c",
          rcsc::BoolSwitch( &M_connect ),
          "start as a soccer monitor." )
        ( "host", "",
          &M_host,
          "set host name to be connected." )
        ( "port", "",
          &M_port,
          "set port number for the monitor client." )
        ( "client-version", "",
          &M_client_version,
          "set a monitor client protocol version." )
        ( "wait-seconds", "",
          &M_wait_seconds,
          "set maximal seconds to wait a message from rcssserver." )
        ( "auto-quit-mode", "",
          rcsc::BoolSwitch( &M_auto_quit_mode ),
          "enable automatic quit after game end." )
        ( "kill-server", "",
          rcsc::BoolSwitch( &M_kill_server ),
          "kill rcssserver when soccerwindow2 is quit." )
        ( "server-pid", "",
          &M_server_pid,
          "set rcssserver process ID to kill it." )
        ( "server-path", "",
          &M_server_path,
          "set a rcssserver command line." )
        ( "time-shift-replay", "",
          &M_time_shift_replay,
          "enable time shift replay mode." )
        ;

    logplayer_options.add()
        ( "game-log", "l",
          &M_game_log_file_path,
          "set the path to Game Log file(.rcg) to be loaded.")
        ( "game-log-dir", "",
          &M_game_log_dir,
          "set a default path where game log files exist." )
        ( "auto-loop-mode", "",
          rcsc::BoolSwitch( &M_auto_loop_mode ),
          "enable automatic replay loop." )
        ( "timer-interval", "",
          &M_timer_interval,
          "set the logplayer timer interval." )
        ;

    debug_server_options.add()
        ( "debug-server-mode", "d",
          rcsc::BoolSwitch( &M_debug_server_mode ),
          "start as a debug server." )
        ( "debug-server-port", "",
          &M_debug_server_port,
          "set port number for the debug server." )
        ( "debug-log-dir", "",
          &M_debug_log_dir,
          "set a default path where debug log files exist." )
        ;

    window_options.add()
        ( "geometry", "",
          &geometry,
          "specifies the window geometry ([WxH][+X+Y]). e.g. --geometry=1024x768+1+1" )
//         ( "pos-x", "",
//           &M_pos_x,
//           "set left x position of the main window." )
//         ( "pos-y", "",
//           &M_pos_y,
//           "set top y position of the main window." )
//         ( "width", "",
//           &M_width,
//           "set width of the main window." )
//         ( "height", "",
//           &M_height,
//           "set height of the main window." )
        ( "maximize", "",
          rcsc::BoolSwitch( &M_maximize ),
          "start with a maximized frame." )
        ( "full-screen", "",
          rcsc::BoolSwitch( &M_full_screen ),
          "start with a full screen frame." )
        ( "canvas-size", "",
          &canvas_size,
          "specifies the canvas size(WxH). e.g. --canvas-size=1024x768" )
        ( "tool-bar-area", "",
          &M_tool_bar_area,
          "specify the initial tool bar position {top,bottom,left,right}." )
        ( "hide-menu-bar", "",
          rcsc::BoolSwitch( &M_hide_menu_bar ),
          "start without a menu bar (available only by Qt3 version)." )
        ( "hide-tool-bar", "",
          rcsc::BoolSwitch( &M_hide_tool_bar ),
          "start without a tool bar." )
        ( "hide-status-bar", "",
          rcsc::BoolSwitch( &M_hide_status_bar ),
          "start without a status bar." )
        ;

    view_options.add()
        ( "anonymous-mode", "",
          rcsc::BoolSwitch( &M_anonymous_mode ),
          "hide team names." )
        ( "draw-type", "",
          &M_draw_type,
          "set a draw style type {default,rcssmonitor}." )
        ( "field-grass-type", "",
          &M_field_grass_type,
          "set a field grass type {mono,lines,checker}." )
        ( "keepaway", "",
          rcsc::BoolSwitch( &M_keepaway ),
          "start as a keepaway mode monitor." )
        ( "show-score-board", "",
          &M_show_score_board,
          "show score board." )
        ( "show-team-logo", "",
          &M_show_team_logo,
          "show team logo." )
        ( "anti-aliasing", "",
          &M_anti_aliasing,
          "set an anti-aliasing draw mode (only for Qt4 version)." )
        ( "gradient", "",
          &M_gradient,
          "set a gradient draw mode (only for Qt4 version)." )
        ( "reverse-side", "",
          rcsc::BoolSwitch( &M_reverse_side ),
          "reverse left<->right." )
        ( "player-reverse-draw", "",
          rcsc::BoolSwitch( &M_player_reverse_draw ),
          "reverse the player draw order." )
        ( "show-player-number", "",
          &M_show_player_number,
          "show player\'s uniform number." )
        ( "show-player-type", "",
          &M_show_player_type,
          "show player\'s heterogeneous type ID." )
        ( "show-view-cone", "",
          &M_show_view_cone,
          "show player\'s view cone." )
        ( "show-stamina", "",
          &M_show_stamina,
          "show player\'s stamina." )
        ( "enlarged-ball-size", "",
          &M_enlarged_ball_size,
          "set a ball radius in enlarge mode." )
        ( "fixed-player-size", "",
          &M_fixed_player_size,
          "set a fixed player radius in enlarge mode." )
        ;

    image_options.add()
        ( "auto-image-save", "",
          rcsc::BoolSwitch( &M_auto_image_save ),
          "automatically saved image and quit application." )
        ( "image-save-dir", "",
          &M_image_save_dir,
          "set a default path where image files are saved." )
        ( "image-name-prefix", "",
          &M_image_name_prefix,
          "set a default prefix of the image file name." )
        ( "image-save-format", "",
          &M_image_save_format,
          "set a default image format type." )
        ;

    rcsc::CmdLineParser parser( argc, argv );

    parser.parse( system_options );
    parser.parse( client_options );
    parser.parse( logplayer_options );
    parser.parse( debug_server_options );
    parser.parse( window_options );
    parser.parse( view_options );
    parser.parse( image_options );


    if ( help
         || parser.failed()
         || parser.positionalOptions().size() > 1 )
    {
        std::cout << "Usage: " << PACKAGE_NAME
                  << " [options ... ] [<GameLogFile>]\n";
        system_options.printHelp( std::cout, false );
        client_options.printHelp( std::cout );
        logplayer_options.printHelp( std::cout );
        debug_server_options.printHelp( std::cout );
        window_options.printHelp( std::cout );
        view_options.printHelp( std::cout );
        image_options.printHelp( std::cout );
        return false;
    }

    if ( version )
    {
        std::cout << PACKAGE_NAME << " Version " << VERSION
                  << std::endl;
        return false;
    }

    if ( M_game_log_file_path.empty()
         && ! parser.positionalOptions().empty() )
    {

        M_game_log_file_path = parser.positionalOptions().front();
    }

    if ( M_timer_interval < 0 )
    {
        std::cerr << "Illegal timer interval " << M_timer_interval
                  << ". set default value." << std::endl;
        M_timer_interval = DEFAULT_TIMER_INTERVAL;
    }

    if ( M_timer_interval < 5 )
    {
        std::cerr << "Too small timer interval " << M_timer_interval
                  << ".  replaced by 5." << std::endl;
        M_timer_interval = 5;
    }

    if ( M_timer_interval > 5000 )
    {
        std::cerr << "Too huge timer interval " << M_timer_interval
                  << ".  replaced by 5000." << std::endl;
        M_timer_interval = 5000;
    }

    if ( ! geometry.empty() )
    {
        int w = -1, h = -1;
        int x = -1, y = -1;

        int num = std::sscanf( geometry.c_str(),
                               " %d x %d %d %d " ,
                               &w, &h, &x, &y );
        //std::cerr << "geometry = " << geometry
        //          << "  param_num=" << num
        //          << " width=" << w << " height=" << h
        //          << " x=" << x << " y=" << y
        //          << std::endl;
        if ( num == 4 || num == 2 )
        {
            if ( w <= 0 || h <= 0 )
            {
                std::cerr << "Illegal window size [" << geometry
                          << "]" << std::endl;
            }
            else
            {
                M_width = w;
                M_height = h;
                M_pos_x = x;
                M_pos_y = y;
            }
        }
        else if ( std::sscanf( geometry.c_str(),
                               " %d %d " ,
                               &x, &y ) == 2 )
        {
            //std::cerr << "only pos = "
            //          << " x=" << x
            //          << " y=" << y
            //          << std::endl;
            M_pos_x = x;
            M_pos_y = y;
        }
        else
        {
            std::cerr << "Illegal geometry format [" << geometry
                      << "]" << std::endl;
        }
    }

    if ( ! canvas_size.empty() )
    {
        int w = -1, h = -1;
        if ( std::sscanf( canvas_size.c_str(),
                          " %d x %d ",
                          &w, &h ) == 2
             && w > 1
             && h > 1 )
        {
            M_canvas_width = w;
            M_canvas_height = h;
        }
        else
        {
            std::cerr << "Illegal canvas size format [" << canvas_size
                      << "]" << std::endl;
        }
    }

    return true;
}

#if 0
{
    namespace po = boost::program_options;

    po::options_description visibles( "Allowed options:" );

    visibles.add_options()
        ( "help,h",
          "generates this message." )
        ( "version,v",
          "print version." )
        ( "host",
          po::value< std::string >( &M_host )->default_value( "localhost", "localhost" ),
          "set host name to be connected." )
        ( "port",
          po::value< int >( &M_port )->default_value( 6000, "6000" ),
          "set port number for the monitor client." )
        ( "client-version",
          po::value< int >( &M_client_version )->default_value( 2, "2" ),
          "set a monitor client protocol version." )
        ( "wait-seconds",
          po::value< long >( &M_wait_seconds )->default_value( 5, "5" ),
          "set maximal seconds to wait a message from rcssserver." )
        ( "auto-quit-mode",
          po::bool_switch( &M_auto_quit_mode )->default_value( false ),
          "enable automatic quit after game end." )
        ( "kill-server",
          po::bool_switch( &M_kill_server )->default_value( false ),
          "kill rcssserver when soccerwindow2 is quit." )
        ( "server-pid",
          po::value< long >( &M_server_pid )->default_value( 0 ),
          "set rcssserver process ID to kill it." )
        ( "server-path",
          po::value< std::string >( &M_server_path )->default_value( "rcssserver", "rcssserver" ),
          "set a rcssserver command line." )
        ( "connect,c",
          po::bool_switch( &M_connect )->default_value( false ),
          "start as soccer monitor." )
        ( "debug-server-mode,d",
          po::bool_switch( &M_debug_server_mode )->default_value( false ),
          "start as a debug server." )
        ( "debug-server-port",
          po::value< long >( &M_debug_server_port )->default_value( 6000 + 32, "6032" ),
          "set port number for the debug server." )
        ( "game-log-dir",
          po::value< std::string >( &M_game_log_dir ), //->default_value( "", "$HOME" ),
          "set a default path where game log files exist." )
        ( "debug-log-dir",
          po::value< std::string >( &M_debug_log_dir ), //->default_value( "", "$HOME" ),
          "set a default path where debug log files exist." )
        ( "time-shift-replay",
          po::value< bool >( &M_time_shift_replay )->default_value( true, "on" ),
          "enable time shift replay mode." )
        ( "auto-loop-mode",
          po::bool_switch( &M_auto_loop_mode )->default_value( false ),
          "enable automatic replay loop." )
        ( "replay-speed-ratio",
          po::value< double >( &M_replay_speed_ratio )->default_value( 1.0, "1.0" ),
          "set a logplayer's replay speed ratio." )
        ( "field-grass-type",
          po::value< std::string >( &M_field_grass_type )->default_value( "mono", "mono" ),
          "set a field grass type {mono,lines,checker}." )
        ( "show-player-number",
          po::value< bool >( &M_show_player_number )->default_value( true, "on" ),
          "show player\'s uniform number." )
        ( "show-player-type",
          po::value< bool >( &M_show_player_type )->default_value( false, "off" ),
          "show player\'s heterogeneous type ID." )
        ( "show-view-cone",
          po::value< bool >( &M_show_view_cone )->default_value( true, "on" ),
          "show player\'s view cone." )
        ( "show-stamina",
          po::value< bool >( &M_show_stamina )->default_value( true, "on" ),
          "show player\'s stamina." )
        ( "keepaway-mode",
          po::bool_switch( &M_keepaway )->default_value( false ),
          "start as a keepaway mode monitor" )
        ( "maximize",
          po::bool_switch( &M_maximize )->default_value( false ),
          "start with a maximized frame." )
        ( "full-screen",
          po::bool_switch( &M_full_screen )->default_value( false ),
          "start with a full screen frame." )
        ( "hide-tool-bar",
          po::bool_switch( &M_hide_tool_bar )->default_value( false ),
          "start without a tool bar." )
        ( "hide-status-bar",
          po::bool_switch( &M_hide_status_bar )->default_value( false ),
          "start without a status bar." )
        ( "pos-x",
          po::value< long >( &M_pos_x )->default_value( -1, "" ),
          "set left x position of a main frame." )
        ( "pos-y",
          po::value< long >( &M_pos_y )->default_value( -1, "" ),
          "set top y position of a main frame." )
        ( "width",
          po::value< long >( &M_width )->default_value( -1, "" ),
          "set width of a main frame." )
        ( "height",
          po::value< long >( &M_height )->default_value( -1, "" ),
          "set height of a main frame." )
        ;

    po::options_description invisibles( "Invisibles" );
    invisibles.add_options()
        ( "rcg-file",
          po::value< std::string >( &M_game_log_file_path )->default_value( "" ),
          "set the path to Game Log file(.rcg) to be loaded."  )
        ;

    po::options_description all_desc( "All options:" );
    all_desc.add( visibles ).add( invisibles );

    po::positional_options_description pdesc;
    pdesc.add( "rcg-file", 1 ); // allowd only one rcg file

    bool help = false;
    try
    {
        po::variables_map vm;
        po::command_line_parser parser( argc, argv );
        parser.options( all_desc ).positional( pdesc );
        po::store( parser.run(), vm );
        po::notify( vm );

        if ( vm.count( "help" ) )
        {
            help = true;
        }
        else if ( vm.count( "version" ) )
        {
            std::cout << PACKAGE_NAME << " Version " << VERSION
                      << std::endl;
            return false;
        }
    }
    catch ( std::exception & e )
    {
        std::cerr << e.what() << std::endl;
        help = true;
    }

    if ( help )
    {
        std::cout << "Usage: " << PACKAGE_NAME
                  << " [options ... ] [<GameLogFile>]\n";
        std::cout << visibles << std::endl;
        return false;
    }

    return true;
}
#endif
