
// simple remover

#include <rcsc/formation/formation_factory.h>
#include <rcsc/formation/formation_dt.h>
#include <rcsc/geom/delaunay_triangulation.h>
#include <rcsc/geom/triangle_2d.h>
#include <rcsc/geom/vector_2d.h>

#include <list>
#include <algorithm>
#include <string>
#include <sstream>
#include <fstream>
#include <iostream>
#include <ctime>

using namespace rcsc;

/*!
  \brief simple vertex remover for FormationDT
 */
class SimpleRemover {
private:

    typedef DelaunayTriangulation::Triangle Triangle;
    typedef Triangle* TrianglePtr;

    typedef std::list< rcsc::Formation::Snapshot > DataList;

    static const double PITCH_LENGTH;
    static const double PITCH_WIDTH;

    static const double BOUNDING_RECT_LENGTH;
    static const double BOUNDING_RECT_WIDTH;

    static const double MIN_SAMPLE_DIST;

    static const double MIN_ERROR;
    static const double ERROR_RATE;

    rcsc::FormationDT M_formation;

    DataList M_training_data;

    rcsc::Rect2D M_rect; //!< bounding box

public:

    SimpleRemover();
    ~SimpleRemover();


    void execute( const std::string & conf_path,
                  const std::string & data_path );

private:
    bool openConf( const std::string & conf_path );
    bool openData( const std::string & data_path );

    void saveConf( const std::string & orig_conf_path );
    void saveData( const std::string & orig_data_path );

    void updateBoundingBox();
    int removeLoop();


};

/*-------------------------------------------------------------------*/
const double SimpleRemover::PITCH_LENGTH = 105.0;
const double SimpleRemover::PITCH_WIDTH = 68.0;

const double SimpleRemover::BOUNDING_RECT_LENGTH = SimpleRemover::PITCH_LENGTH + (2.01 * 2.0);
const double SimpleRemover::BOUNDING_RECT_WIDTH = SimpleRemover::PITCH_WIDTH + (2.01 * 2.0);

const double SimpleRemover::MIN_SAMPLE_DIST = 2.0;

const double SimpleRemover::MIN_ERROR = 0.5;
const double SimpleRemover::ERROR_RATE = 0.09;


/*-------------------------------------------------------------------*/
SimpleRemover::SimpleRemover()
{


}

/*-------------------------------------------------------------------*/
SimpleRemover::~SimpleRemover()
{


}

/*-------------------------------------------------------------------*/
void
SimpleRemover::execute( const std::string & conf_path,
                        const std::string & data_path )
{
    if ( ! openConf( conf_path ) )
    {
        std::cerr << "failed to open the conf file " << conf_path
                  << std::endl;
        return;
    }

    if ( ! openData( data_path ) )
    {
        std::cerr << "failed to open the data file " << data_path
                  << std::endl;
        return;
    }

    if ( M_formation.methodName() != rcsc::FormationDT::name()
         || M_training_data.empty() )
    {
        std::cerr << "No data." << std::endl;
        return;
    }

    updateBoundingBox();

    int loop = 0;
    while ( removeLoop() > 0 )
    {
        std::cerr << "---------- loop " << ++loop << " ----------"
                  << std::endl;
    }

    saveConf( conf_path );
    saveData( data_path );
}

/*-------------------------------------------------------------------*/
bool
SimpleRemover::openConf( const std::string & conf_path )
{
    std::ifstream fin( conf_path.c_str() );
    if ( ! fin.is_open() )
    {
        std::cerr << "Failed to open formation file [" << conf_path << "]"
                  << std::endl;
        return false;
    }

    M_training_data.clear();

    {
        std::string temp, type;
        fin >> temp >> type; // read training method type name
        if ( type != rcsc::FormationDT::name() )
        {
            std::cerr << "Unsupported formation format in the file ["
                      << conf_path << "]\n"
                      << "You must write the training method type name"
                      << " at the top of the formation file"
                      << std::endl;
            return false;
        }
    }

    fin.seekg( 0 );
    if ( ! M_formation.read( fin ) )
    {
        fin.close();
        return false;
    }

    fin.close();

    return true;
}

/*-------------------------------------------------------------------*/
bool
SimpleRemover::openData( const std::string & data_path )
{
    std::ifstream fin( data_path.c_str() );
    if ( ! fin.is_open() )
    {
        std::cerr << "Failed to open training data file [" << data_path
                  << "]" << std::endl;
        return false;
    }

    // clear current data list
    M_training_data.clear();

    bool success = true;
    std::string line_buf;
    while ( std::getline( fin, line_buf ) )
    {
        rcsc::Formation::Snapshot snap;
        std::istringstream istr( line_buf );
        istr >> snap.ball_.x >> snap.ball_.y;
        for ( int i = 0; i < 11; ++i )
        {
            if ( ! istr.good() )
            {
                success = false;
                break;
            }
            rcsc::Vector2D v;
            istr >> v.x >> v.y;
            snap.players_.push_back( v );
        }

        M_training_data.push_back( snap );
    }

    fin.close();

    if ( ! success )
    {
        M_training_data.clear();
        return false;
    }

    return true;
}

/*-------------------------------------------------------------------*/
void
SimpleRemover::saveConf( const std::string & orig_conf_path )
{
    std::string new_conf_path = orig_conf_path;
    if ( new_conf_path.length() >= 5
         && new_conf_path.compare( new_conf_path.length() - 5, 5, ".conf" ) == 0 )
    {
        new_conf_path.erase( new_conf_path.length() - 5, 5 );
    }
    new_conf_path += ".removed.conf";

    std::ofstream fout( new_conf_path.c_str() );
    if ( ! fout.is_open() )
    {
        std::cerr << "failed to opent the new file " << new_conf_path
                  << std::endl;
        fout.close();
        return;
    }

    M_formation.print( fout );

    fout.close();
}

/*-------------------------------------------------------------------*/
void
SimpleRemover::saveData( const std::string & orig_data_path )
{
    std::string new_data_path = orig_data_path;
    if ( new_data_path.length() >= 5
         && new_data_path.compare( new_data_path.length() - 4, 4, ".dat" ) == 0 )
    {
        new_data_path.erase( new_data_path.length() - 4, 4 );
    }
    new_data_path += ".removed.dat";

    std::ofstream fout( new_data_path.c_str() );

    if ( ! fout.is_open() )
    {
        std::cerr << "failed to opent the new file " << new_data_path
                  << std::endl;
        fout.close();
        return;
    }

    for ( DataList::iterator it = M_training_data.begin();
          it != M_training_data.end();
          ++it )
    {
        fout << it->ball_.x << ' ' << it->ball_.y << ' ';
        for ( std::vector< rcsc::Vector2D >::iterator p = it->players_.begin();
              p != it->players_.end();
              ++p )
        {
            fout << p->x << ' ' << p->y << ' ';
        }
        fout << std::endl;
    }

    fout.close();
}

/*-------------------------------------------------------------------*/
void
SimpleRemover::updateBoundingBox()
{
    double min_x = 100000.0;
    double min_y = 100000.0;
    double max_x = -100000.0;
    double max_y = -100000.0;

    for ( DataList::iterator it = M_training_data.begin();
          it != M_training_data.end();
          ++it )
    {
        if ( it->ball_.x < min_x ) min_x = it->ball_.x;
        if ( it->ball_.x > max_x ) max_x = it->ball_.x;
        if ( it->ball_.y < min_y ) min_y = it->ball_.y;
        if ( it->ball_.y > max_y ) max_y = it->ball_.y;
    }

    std::cerr << " min_x = " << min_x
              << " max_x = " << max_x
              << " min_y = " << min_y
              << " max_y = " << max_y
              << std::endl;
    M_rect.assign( min_x, min_y,
                   max_x - min_x, max_y - min_y );

    std::cerr << " top_left = " << M_rect.topLeft()
              << " top_right = " << M_rect.topRight()
              << " bottom_left = " << M_rect.bottomLeft()
              << " bottom_right = " << M_rect.bottomRight()
              << std::endl;
}

/*-------------------------------------------------------------------*/
int
SimpleRemover::removeLoop()
{
    int total_removed = 0;

    rcsc::FormationDT tmp_formation;
    tmp_formation.createDefaultParam();

    DataList::iterator checked_data = M_training_data.begin();
    while ( checked_data != M_training_data.end() )
    {
        if ( checked_data->ball_.dist( M_rect.topLeft() ) < 0.01
             || checked_data->ball_.dist( M_rect.topRight() ) < 0.01
             || checked_data->ball_.dist( M_rect.bottomLeft() ) < 0.01
             || checked_data->ball_.dist( M_rect.bottomRight() ) < 0.01 )
        {
            std::cerr << "skip bounding box vertex "
                      << checked_data->ball_
                      << std::endl;
            ++checked_data;
            continue;
        }

        DataList tmp_data;

        for ( DataList::iterator it = M_training_data.begin();
              it != M_training_data.end();
              ++it )
        {
            if ( checked_data == it )
            {
                continue;
            }

            tmp_data.push_back( *it );
        }

        tmp_formation.train( tmp_data );

        bool very_near = true;
        for ( int p = 0; p < 11; ++p )
        {
            rcsc::Vector2D pos = tmp_formation.getPosition( p + 1, checked_data->ball_ );
            double ball_dist = checked_data->ball_.dist( pos );
            if ( checked_data->players_[p].dist( pos )
                 > 0.4 + ball_dist * 0.05 )
            {
                very_near = false;
                break;
            }
        }

        if ( very_near )
        {
            ++total_removed;
            std::cerr << "Remove " << total_removed
                      << " index=" << std::distance( M_training_data.begin(),
                                                                 checked_data )
                      << " ball=" << checked_data->ball_
                      << std::endl;
            //M_training_data.erase( checked_data );
            //checked_data = M_training_data.begin();
            checked_data = M_training_data.erase( checked_data );
        }
        else
        {
            ++checked_data;
        }
    }

    M_formation.train( M_training_data );

    return total_removed;
}


/*-------------------------------------------------------------------*/
int
main( int argc, char ** argv )
{
    std::cout << "start simple remover." << std::endl;

    std::string conf_file;
    std::string data_file;

    for ( int i = 1; i < argc; ++i )
    {
        std::string arg = argv[i];
        if ( arg.length() > 5
             && arg.compare( arg.length() - 5, 5, ".conf" ) == 0 )
        {
            conf_file = arg;
        }
        else if ( arg.length() > 4
                  && arg.compare( arg.length() - 4, 4, ".dat" ) == 0 )
        {
            data_file = arg;
        }
    }

    if ( conf_file.empty() )
    {
        std::cerr << "empty conf file name" << std::endl;
        return 1;
    }

    if ( data_file.empty() )
    {
        std::cerr << "empty data file name" << std::endl;
        return 1;
    }

    SimpleRemover remover;

    remover.execute( conf_file, data_file );

    std::cout << "end simple remover." << std::endl;
    return 0;
}
