// -*-c++-*-

/*!
	\file mesh_creator.cpp
	\brief meshed data creator Source file
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.	If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <rcsc/formation/formation_factory.h>
#include <rcsc/formation/formation_dt.h>
#include <rcsc/math_util.h>

#include <list>
#include <algorithm>
#include <fstream>


class MeshCreator {
private:

    static const double PITCH_LENGTH;
    static const double PITCH_WIDTH;
    static const double PITCH_MARGIN;

    static const double MAX_X;
    static const double MAX_Y;

    double M_mesh_size;

    //boost::shared_ptr< rcsc::Formation > M_formation;
    boost::shared_ptr< rcsc::Formation > M_offense_formation;
    boost::shared_ptr< rcsc::Formation > M_defense_formation;

    rcsc::FormationDT M_formation_dt;
    std::list< rcsc::Formation::Snapshot > M_training_data;

    rcsc::DelaunayTriangulation M_triangulation;

public:

    MeshCreator();

    void setMeshSize( const double & mesh_size )
      {
          if ( mesh_size < 1.0 )
          {
              std::cerr << "Too small mesh size " << mesh_size
                        << std::endl;
              return;
          }
          M_mesh_size = mesh_size;
      }

    bool open( const std::string & confpath );

    bool open( const std::string & offense_conf_path,
               const std::string & defense_conf_path );

    bool save( const std::string & confpath,
               const std::string & datapath );

    void execute();

private:

    bool saveConfFile( const std::string & filepath );
    bool saveDataFile( const std::string & filepath );

    boost::shared_ptr< rcsc::Formation >
    createFormation( const std::string & type_name );

    void createMeshData();
};

/*-------------------------------------------------------------------*/
const double MeshCreator::PITCH_LENGTH = 105.0;
const double MeshCreator::PITCH_WIDTH = 68.0;
const double MeshCreator::PITCH_MARGIN = 5.0;

const double MeshCreator::MAX_X = MeshCreator::PITCH_LENGTH * 0.5 + 2.0;;
const double MeshCreator::MAX_Y = MeshCreator::PITCH_WIDTH * 0.5 + 2.0;

/*-------------------------------------------------------------------*/
/*!

*/
MeshCreator::MeshCreator()
    : M_mesh_size( 2.0 )
{
    M_formation_dt.createDefaultParam();
}

/*-------------------------------------------------------------------*/
/*!

*/
boost::shared_ptr< rcsc::Formation >
MeshCreator::createFormation( const std::string & type_name )
{
    boost::shared_ptr< rcsc::Formation > f;

    if ( type_name == rcsc::FormationDT::name() )
    {
        f = boost::shared_ptr< rcsc::Formation >( new rcsc::FormationDT );
    }
    else
    {
        std::cerr << __FILE__ << ':' << __LINE__
                  << " ***RRROR*** Unknown formation method name. ["
                  << type_name << "]"
                  << std::endl;
    }

    return f;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
MeshCreator::open( const std::string & confpath )
{
    M_offense_formation = rcsc::make_formation( confpath );
    if ( ! M_offense_formation )
    {
        std::cerr << "failed to create formation from conf file"
                  << confpath
                  << std::endl;
        return false;
    }

    std::ifstream fin( confpath.c_str() );
    if ( ! fin.is_open() )
    {
        std::cerr << "Failed to open formation file [" << confpath << "]"
                  << std::endl;
        return false;
    }

    if ( ! M_offense_formation->read( fin ) )
    {
        fin.close();
        M_offense_formation.reset();
        return false;
    }

    fin.close();

    M_defense_formation = M_offense_formation;

    for ( int unum = 1; unum <= 11; ++unum )
    {
        M_formation_dt.updateRole( unum,
                                   M_offense_formation->getSynmetryNumber( unum ),
                                   M_offense_formation->getRoleName( unum ) );
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
MeshCreator::open( const std::string & offense_conf_path,
                   const std::string & defense_conf_path )
{
    M_offense_formation = rcsc::make_formation( offense_conf_path );
    if ( ! M_offense_formation )
    {
        std::cerr << "failed to create formation from conf file"
                  << offense_conf_path
                  << std::endl;
        return false;
    }

    M_defense_formation = rcsc::make_formation( offense_conf_path );
    if ( ! M_defense_formation )
    {
        std::cerr << "failed to create formation from conf file"
                  << defense_conf_path
                  << std::endl;
        return false;
    }

    {
        std::ifstream fin( offense_conf_path.c_str() );
        if ( ! fin.is_open() )
        {
            std::cerr << "Failed to open formation file ["
                      << offense_conf_path << "]"
                      << std::endl;
            return false;
        }

        fin.seekg( 0 );
        if ( ! M_offense_formation->read( fin ) )
        {
            fin.close();
            M_offense_formation.reset();
            return false;
        }
        fin.close();
    }

    {
        std::ifstream fin( defense_conf_path.c_str() );
        if ( ! fin.is_open() )
        {
            std::cerr << "Failed to open formation file ["
                      << defense_conf_path << "]"
                      << std::endl;
            return false;
        }

        fin.seekg( 0 );
        if ( ! M_defense_formation->read( fin ) )
        {
            fin.close();
            M_defense_formation.reset();
            return false;
        }
        fin.close();
    }

    for ( int unum = 1; unum <= 11; ++unum )
    {
        M_formation_dt.updateRole( unum,
                                   M_offense_formation->getSynmetryNumber( unum ),
                                   M_offense_formation->getRoleName( unum ) );
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
MeshCreator::save( const std::string & confpath,
                 const std::string & datapath )
{
    return ( saveConfFile( confpath )
             && saveDataFile( datapath ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
MeshCreator::saveConfFile( const std::string & filepath )
{
    std::ofstream fout( filepath.c_str() );
    if ( ! fout.is_open() )
    {
        fout.close();
        return false;
    }

    //if ( ! M_formation->print( fout ) )
    if ( ! M_formation_dt.print( fout ) )
    {
        return false;
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
MeshCreator::saveDataFile( const std::string & filepath )
{
    std::ofstream fout( filepath.c_str() );
    if ( ! fout.is_open() )
    {
        fout.close();
        return false;
    }

    // put data to the stream
    for ( std::list< rcsc::Formation::Snapshot >::iterator it = M_training_data.begin();
          it != M_training_data.end();
          ++it )
    {
        fout << it->ball_.x << ' ' << it->ball_.y << ' ';
        for ( std::vector< rcsc::Vector2D >::iterator p = it->players_.begin();
              p != it->players_.end();
              ++p )
        {
            fout << p->x << ' ' << p->y << ' ';
        }
        fout << std::endl;
    }

    if ( ! fout )
    {
        return false;
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MeshCreator::createMeshData()
{
    if ( ! M_offense_formation
         || ! M_defense_formation )
    {
        return;
    }

    //const double MESH_SIZE = 2.0;

    const int mesh_x = static_cast< int >( std::floor( MAX_X / M_mesh_size ) );
    const double x_step = MAX_X / mesh_x;
    const int mesh_y = static_cast< int >( std::floor( MAX_Y / M_mesh_size ) );
    const double y_step = MAX_Y / mesh_y;

//     rcsc::Vector2D bounding_points[4] = { rcsc::Vector2D( - x_step * mesh_x,
//                                                           - y_step * mesh_y ),
//                                           rcsc::Vector2D(   x_step * mesh_x,
//                                                           - y_step * mesh_y ),
//                                           rcsc::Vector2D( - x_step * mesh_x,
//                                                             y_step * mesh_y ),
//                                           rcsc::Vector2D(   x_step * mesh_x,
//                                                             y_step * mesh_y ),
//     };

    rcsc::Formation::Snapshot snapshot;
    snapshot.players_.resize( 11 );

    for ( int x = -mesh_x; x <= mesh_x; ++x )
    {
        for ( int y = 0; y <= mesh_y; ++y )
        {
            snapshot.ball_.assign( x_step * x,
                                   y_step * y );

            boost::shared_ptr< rcsc::Formation > f = ( snapshot.ball_.x > 0.0
                                                       ? M_offense_formation
                                                       : M_defense_formation );

            for ( int unum = 1; unum <= 11; ++unum )
            {
                rcsc::Vector2D pos = f->getPosition( unum, snapshot.ball_ );
                pos.x = rcsc::min_max( -53.0, pos.x, 53.0 );
                pos.y = rcsc::min_max( -35.0, pos.y, 35.0 );

                snapshot.players_[unum - 1] = pos;
            }

            M_training_data.push_back( snapshot );

            // create symmetry data
            if ( y != 0 )
            {
                snapshot.ball_.y *= -1.0;

                for ( int unum = 1; unum <= 11; ++unum )
                {
                    rcsc::Vector2D pos = f->getPosition( unum, snapshot.ball_ );
                    pos.x = rcsc::min_max( -53.0, pos.x, 53.0 );
                    pos.y = rcsc::min_max( -35.0, pos.y, 35.0 );

                    snapshot.players_[unum - 1] = pos;
                }

                M_training_data.push_back( snapshot );
            }
        }

    }

    std::cout << "created " << M_training_data.size() << " data." << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MeshCreator::execute()
{
    if ( ! M_offense_formation
         || ! M_defense_formation )
    {
        return;
    }

    createMeshData();

    M_formation_dt.train( M_training_data );
}


/*-------------------------------------------------------------------*/

int
main( int argc, char ** argv )
{
    double mesh_size = 2.0;
    std::string conf_file;
    std::string off_conf_file;
    std::string def_conf_file;

    for ( int i = 1; i < argc; ++i )
    {
        std::string arg = argv[i];

        if ( arg == "-m"
             && i + 1 < argc )
        {
            mesh_size = std::atof( argv[++i] );
        }
        else if ( arg.length() > 5
                  && arg.compare( arg.length() - 5, 5, ".conf" ) == 0 )
        {
            if ( arg.find( "offense-formation" ) != std::string::npos )
            {
                off_conf_file = arg;
            }
            else if ( arg.find( "defense-formation" ) != std::string::npos )
            {
                def_conf_file = arg;
            }
            else
            {
                conf_file = arg;
            }
        }
    }

    if ( mesh_size < 1.0 )
    {
        std::cerr << "Too small mesh size " << mesh_size
                  << std::endl;
        return 1;
    }

    MeshCreator creator;

    creator.setMeshSize( mesh_size );

    if ( ! conf_file.empty() )
    {
        if ( ! creator.open( conf_file ) )
        {
            return 1;
        }
    }
    else if ( ! off_conf_file.empty()
              && ! def_conf_file.empty() )
    {
        conf_file = off_conf_file;
        if ( ! creator.open( off_conf_file, def_conf_file ) )
        {
            return 1;
        }
    }
    else
    {
        std::cerr << "empty conf file name" << std::endl;
        return 1;
    }

    creator.execute();

    std::string meshed_conf = conf_file;
    std::string meshed_data = conf_file;;
    if ( conf_file.length() >= 5
         && conf_file.compare( conf_file.length() - 5, 5, ".conf" ) == 0 )
    {
        meshed_conf.erase( conf_file.length() - 5, 5 );
        meshed_data.erase( conf_file.length() - 5, 5 );
    }
    meshed_conf += ".meshed.conf";
    meshed_data += ".meshed.dat";

    creator.save( meshed_conf, meshed_data );

    return 0;
}
