// -*-c++-*-

/*!
  \file main_window.cpp
  \brief main application window class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "main_window.h"
#include "fedit_canvas.h"
#include "fedit_config.h"
#include "fedit_data.h"
#include "fedit_dialog.h"

#include <rcsc/formation/formation_bpn.h>
#include <rcsc/formation/formation_dt.h>
#include <rcsc/formation/formation_knn.h>
#include <rcsc/formation/formation_ngnet.h>
//#include <rcsc/formation/formation_sbsp.h>
#include <rcsc/formation/formation_rbf.h>
//#include <rcsc/formation/formation_static.h>
//#include <rcsc/formation/formation_uva.h>

#include <string>
#include <iostream>
#include <cstring>

#include "xpm/soccerwindow2.xpm"
#include "xpm/soccerwindow2-nostr.xpm"

#include "xpm/chase.xpm"
#include "xpm/delete.xpm"
#include "xpm/hand.xpm"
#include "xpm/new.xpm"
#include "xpm/open.xpm"
#include "xpm/train.xpm"
#include "xpm/record.xpm"
#include "xpm/insert.xpm"
#include "xpm/replace.xpm"
#include "xpm/reverse.xpm"
#include "xpm/save.xpm"
#include "xpm/symmetry.xpm"

#ifndef PACKAGENAME
#define PACKAGENAME "soccerwindow2"
#endif

/*-------------------------------------------------------------------*/
/*!

*/
MainWindow::MainWindow()
{
    //    QPixmapCache::setCacheLimit( 1024 * 24 );
    loadSettings();


    this->setIcon( QPixmap( soccerwindow2_xpm ) );
    this->setCaption( tr( "FormationEditor" ) );

    //this->setUsesBigPixmaps( true );
    this->setUsesTextLabel( true );

    createFieldCanvas();
    createEditDialog();
    createActions();
    createMenus();
    createToolBars();
    createStatusBar();

    this->setMinimumSize( 320, 240 );
    this->resize( FEditConfig::instance().frameWidth() > 0
                  ? FEditConfig::instance().frameWidth()
                  : 640,
                  FEditConfig::instance().frameHeight() > 0
                  ? FEditConfig::instance().frameHeight()
                  : 480 );
    this->move( FEditConfig::instance().framePosX() > 0
                ? FEditConfig::instance().framePosX()
                : this->x(),
                FEditConfig::instance().framePosY() > 0
                ? FEditConfig::instance().framePosY()
                : this->y() );

    if ( FEditConfig::instance().maximize() )
    {
        this->showMaximized();
    }

    if ( FEditConfig::instance().fullScreen() )
    {
        this->showFullScreen();
    }

    connect( M_editor_dialog, SIGNAL( viewUpdated() ),
             M_editor_canvas, SLOT( update() ) );
    connect( M_editor_canvas, SIGNAL( objectMoved() ),
             M_editor_dialog, SLOT( updateData() ) );
    connect( M_editor_canvas, SIGNAL( objectMoved() ),
             this, SLOT( updateDataIndex() ) );
    connect( M_editor_dialog, SIGNAL( shown( bool ) ),
             M_show_edit_dialog_act, SLOT( setOn( bool ) ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
MainWindow::~MainWindow()
{
    //std::cerr << "delete MainWindow" << std::endl;

    saveSettings();
}


/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::init()
{
//     M_editor_dialog->show();

//     QSize size = M_editor_dialog->size();
//     M_editor_dialog->setMinimumSize( size );
//     M_editor_dialog->setMaximumSize( size );

    if ( ! FEditConfig::instance().confFile().empty() )
    {
        if ( ! FEditConfig::instance().dataFile().empty() )
        {
            openConfFile( FEditConfig::instance().confFile() );
        }
        else
        {
            open( FEditConfig::instance().confFile() );
        }
    }

    if ( ! FEditConfig::instance().dataFile().empty() )
    {
        openDataFile( FEditConfig::instance().dataFile() );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::loadSettings()
{
#if 0
    QSettings settings( QSettings::Ini );
    settings.insertSearchPath( QSettings::Unix,
                               QDir::homeDirPath() );

    settings.beginGroup( "/.soccerwindow2-qt3./FEditConfig" );

    val = settings.readEntry( "debugLogDir" );
    if ( ! val.isNull()
         && FEditConfig::instance().debugLogDir().empty() )
    {
        FEditConfig::instance().setDebugLogDir( val.ascii() );
    }

    settings.endGroup();
#endif
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::saveSettings()
{
#if 0
    QSettings settings( QSettings::Ini );
    settings.insertSearchPath( QSettings::Unix,
                               QDir::homeDirPath() );

    settings.beginGroup( "/.soccerwindow2-qt3./FEditConfig" );

    settings.writeEntry( "gameLogDir",
                         QString( FEditConfig::instance().gameLogDir().c_str() ) );
    settings.writeEntry( "debugLogDir",
                         QString( FEditConfig::instance().debugLogDir().c_str() ) );

    settings.endGroup();
#endif
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createActions()
{
    createActionsFile();
    createActionsMode();
    createActionsData();
    createActionsView();
    createActionsHelp();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createActionsFile()
{
    M_new_file_act = new QAction( QIconSet( QPixmap( new_xpm ) ),
                                  tr( "New formation" ),
                                  QKeySequence(),
                                  this );
    M_new_file_act->setAccel( tr( "Ctrl+N" ) );
    M_new_file_act->setStatusTip( tr( "Create new formation data" ) );
    connect( M_new_file_act, SIGNAL( activated() ), this, SLOT( newFile() ) );
    //
    M_open_act = new QAction( QIconSet( QPixmap( open_xpm ) ),
                              tr( "&Open formation" ),
                              QKeySequence(),
                              this );
    M_open_act->setAccel( tr( "Ctrl+O" ) );
    M_open_act->setStatusTip( tr( "Open formation file" ) );
    connect( M_open_act, SIGNAL( activated() ), this, SLOT( open() ) );
    //
    M_open_train_act = new QAction( QIconSet( QPixmap( open_xpm ) ),
                                    tr( "&Open training data." ),
                                    QKeySequence(),
                                    this );
    M_open_train_act->setAccel( tr( "Ctrl+Alt+O" ) );
    M_open_train_act->setStatusTip( tr( "Open training data file" ) );
    connect( M_open_train_act, SIGNAL( activated() ),
             this, SLOT( openData() ) );
    //
    M_save_all_act = new QAction( QIconSet( QPixmap( save_xpm ) ),
                                  tr( "&Save all" ),
                                  QKeySequence(),
                                  this );
    M_save_all_act->setAccel( tr( "Ctrl+S" ) );
    M_save_all_act->setStatusTip( tr( "Save all data" ) );
    connect( M_save_all_act, SIGNAL( activated() ), this, SLOT( saveAll() ) );
    //
    M_save_act = new QAction( tr( "&Save formation" ),
                              QKeySequence(),
                              this );
    M_save_act->setStatusTip( tr( "Save formation data" ) );
    connect( M_save_act, SIGNAL( activated() ), this, SLOT( save() ) );
    //
    M_save_as_act = new QAction( tr( "&Save formation as..." ),
                              QKeySequence(),
                              this );
    M_save_as_act->setStatusTip( tr( "Save formation data to the new file" ) );
    connect( M_save_as_act, SIGNAL( activated() ), this, SLOT( saveAs() ) );
    //
    M_save_train_act = new QAction( tr( "&Save training data" ),
                              QKeySequence(),
                              this );
    M_save_train_act->setStatusTip( tr( "Save training data" ) );
    connect( M_save_train_act, SIGNAL( activated() ),
             this, SLOT( saveTrainingData() ) );
    //
    M_save_train_as_act = new QAction( tr( "&Save training data as..." ),
                              QKeySequence(),
                              this );
    M_save_train_as_act->setStatusTip( tr( "Save training data to the new file" ) );
    connect( M_save_train_as_act, SIGNAL( activated() ),
             this, SLOT( saveTrainingDataAs() ) );
    //
    M_exit_act = new QAction( tr( "&Quit" ),
                              QKeySequence(),
                              this );
    M_exit_act->setAccel( tr( "Ctrl+Q" ) );
    M_exit_act->setStatusTip( tr( "Exit the application." ) );
    connect( M_exit_act, SIGNAL( activated() ), this, SLOT( close() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createActionsMode()
{
    M_toggle_ball_drag_act = new QAction( QIconSet( QPixmap( hand_xpm ) ),
                                          tr( "&Ball Drag" ),
                                          QKeySequence(),
                                          this );
    M_toggle_ball_drag_act->setAccel( Qt::Key_B );
    M_toggle_ball_drag_act->setStatusTip( tr( "Toggle ball drag mode." ) );
    M_toggle_ball_drag_act->setToggleAction( true );
    M_toggle_ball_drag_act->setOn( true );
    connect( M_toggle_ball_drag_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleBallDrag( bool ) ) );
    //
    M_toggle_player_auto_move_act = new QAction( QIconSet( QPixmap( chase_xpm ) ),
                                                 tr( "&Auto Move" ),
                                                 QKeySequence(),
                                                 this );
    //M_toggle_player_auto_move_act->setAccel( tr( "Ctrl+P" ) );
    M_toggle_player_auto_move_act->setAccel( Qt::Key_A );
    M_toggle_player_auto_move_act->setStatusTip( tr( "Toggle player auto move mode." ) );
    M_toggle_player_auto_move_act->setToggleAction( true );
    M_toggle_player_auto_move_act->setOn( true );
    connect( M_toggle_player_auto_move_act, SIGNAL( toggled( bool ) ),
             this, SLOT( togglePlayerAutoMove( bool ) ) );
    //
    M_toggle_data_auto_select_act = new QAction( tr( "&Data Auto Select" ),
                                                 QKeySequence(),
                                                 this );
    M_toggle_data_auto_select_act->setAccel( Qt::Key_D );
    M_toggle_data_auto_select_act->setStatusTip( tr( "Toggle data is automatically select or not when ball is moved." ) );
    M_toggle_data_auto_select_act->setToggleAction( true );
    M_toggle_data_auto_select_act->setOn( true );
    connect( M_toggle_data_auto_select_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleDataAutoSelect( bool ) ) );
    //
    M_toggle_symmetric_mode_act = new QAction( QIconSet( QPixmap( symmetry_xpm ) ),
                                               tr( "Symmetry" ),
                                               QKeySequence(),
                                               this );
    M_toggle_symmetric_mode_act->setAccel( Qt::Key_S );
    M_toggle_symmetric_mode_act->setStatusTip( tr( "Toggle symmetrical mode." ) );
    M_toggle_symmetric_mode_act->setToggleAction( true );
    M_toggle_symmetric_mode_act->setOn( true );
    connect( M_toggle_symmetric_mode_act, SIGNAL( toggled( bool ) ),
             this, SLOT( toggleSymmetricMode( bool ) ) );
    //
    M_show_edit_dialog_act = new QAction( tr( "Show Edit Dialog" ),
                                          QKeySequence(),
                                          this );
    M_show_edit_dialog_act->setAccel( tr( "Ctrl+D" ) );
    M_show_edit_dialog_act->setStatusTip( tr( "Show/Hide edit dialog box." ) );
    M_show_edit_dialog_act->setToggleAction( true );
    M_show_edit_dialog_act->setOn( false );
    connect( M_show_edit_dialog_act, SIGNAL( toggled( bool ) ),
             this, SLOT( showEditDialog( bool ) ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createActionsData()
{
    M_insert_data_act = new QAction( QIconSet( QPixmap( insert_xpm ) ),
                                      tr( "Insert" ),
                                      QKeySequence(),
                                      this );
    M_insert_data_act->setStatusTip( tr( "Insert the screen data after the current index" ) );
    connect( M_insert_data_act, SIGNAL( activated() ),
             this, SLOT( insertData() ) );
    //
    M_replace_data_act = new QAction( QIconSet( QPixmap( replace_xpm ) ),
                                      tr( "Replace" ),
                                      QKeySequence(),
                                      this );
    M_replace_data_act->setStatusTip( tr( "Replace a current indexed training"
                                          " data by the screen status" ) );
    connect( M_replace_data_act, SIGNAL( activated() ),
             this, SLOT( replaceData() ) );
    //
    M_delete_data_act = new QAction( QIconSet( QPixmap( delete_xpm ) ),
                                     tr( "Delete" ),
                                     QKeySequence(),
                                     this );
    M_delete_data_act->setAccel( Qt::Key_Delete );
    M_delete_data_act->setStatusTip( tr( "Delete a current indexed training data" ) );
    connect( M_delete_data_act, SIGNAL( activated() ),
             this, SLOT( deleteData() ) );
    //
    M_reverse_y_act = new QAction( QIconSet( QPixmap( reverse_xpm ) ),
                                   tr( "ReverseY" ),
                                   QKeySequence(),
                                   this );
    M_reverse_y_act->setAccel( Qt::Key_R );
    M_reverse_y_act->setStatusTip( tr( "Reverse Y positions." ) );
    connect( M_reverse_y_act, SIGNAL( activated() ),
             this, SLOT( reverseY() ) );
    //
    M_train_act = new QAction( QIconSet( QPixmap( train_xpm ) ),
                               tr( "Train" ),
                               QKeySequence(),
                               this );
    M_train_act->setStatusTip( tr( "Execute train using current trainig data set" ) );
    connect( M_train_act, SIGNAL( activated() ),
             this, SLOT( train() ) );
    //
    M_record_data_act = new QAction( QIconSet( QPixmap( record_xpm ) ),
                               tr( "Record" ),
                               QKeySequence(),
                               this );
    M_record_data_act->setAccel( tr( "Ctrl+R" ) );
    M_record_data_act->setStatusTip( tr( "Record current field status as a training data" ) );
    connect( M_record_data_act, SIGNAL( activated() ),
             this, SLOT( recordData() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createActionsView()
{
    M_zoom_in_act = new QAction( tr( "Zoom In" ),
                                 QKeySequence(),
                                 this );
    M_zoom_in_act->setAccel( Qt::Key_Z );
    M_zoom_in_act->setStatusTip( tr( "Zoom In" ) );
    connect( M_zoom_in_act, SIGNAL( activated() ),
             M_editor_canvas, SLOT( zoomIn() ) );

    M_zoom_out_act = new QAction( tr( "Zoom Out" ),
                                  QKeySequence(),
                                  this );
    M_zoom_out_act->setAccel( Qt::Key_X );
    M_zoom_out_act->setStatusTip( tr( "Zoom Out" ) );
    connect( M_zoom_out_act, SIGNAL( activated() ),
             M_editor_canvas, SLOT( zoomOut() ) );

    M_fit_act = new QAction( tr( "Fit Size" ),
                                  QKeySequence(),
                                  this );
    M_fit_act->setAccel( Qt::Key_I );
    M_fit_act->setStatusTip( tr( "Fit Size" ) );
    connect( M_fit_act, SIGNAL( activated() ),
             M_editor_canvas, SLOT( fitSize() ) );

    M_toggle_enlarge_act = new QAction( tr( "Toggle Enlarge" ),
                                        QKeySequence(),
                                        this );
    M_toggle_enlarge_act->setAccel( Qt::Key_E );
    M_toggle_enlarge_act->setStatusTip( tr( "Toggle Enlarge Status" ) );
    M_toggle_enlarge_act->setToggleAction( true );
    connect( M_toggle_enlarge_act, SIGNAL( toggled( bool ) ),
             M_editor_canvas, SLOT( toggleEnlarge( bool ) ) );
    M_toggle_enlarge_act->setOn( true );

    M_toggle_show_index_act = new QAction( tr( "Show Index" ),
                                           QKeySequence(),
                                           this );
    M_toggle_show_index_act->setStatusTip( tr( "Show/Hide Index" ) );
    M_toggle_show_index_act->setToggleAction( true );
    connect( M_toggle_show_index_act, SIGNAL( toggled( bool ) ),
             M_editor_canvas, SLOT( toggleShowIndex( bool ) ) );
    M_toggle_show_index_act->setOn( true );

    M_toggle_show_circumcircle_act = new QAction( tr( "Show Circumcircle" ),
                                                  QKeySequence(),
                                                  this );
    M_toggle_show_circumcircle_act->setStatusTip( tr( "Show/Hide Circumcircle" ) );
    M_toggle_show_circumcircle_act->setToggleAction( true );
    connect( M_toggle_show_circumcircle_act, SIGNAL( toggled( bool ) ),
             M_editor_canvas, SLOT( toggleShowCircumcircle( bool ) ) );
    M_toggle_show_circumcircle_act->setOn( false );

}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createActionsHelp()
{
    M_about_act = new QAction( QIconSet( QPixmap( soccerwindow2_nostr_xpm ) ),
                               tr( "&About" ),
                               QKeySequence(),
                               this );
    M_about_act->setStatusTip( tr( "Show the about dialog." ) );
    connect( M_about_act, SIGNAL( activated() ), this, SLOT( about() ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createMenus()
{
    createMenuFile();
    createMenuEdit();
    createMenuMode();
    createMenuView();
    createMenuHelp();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createMenuFile()
{
    QPopupMenu * menu = new QPopupMenu( this );
    menu->insertTearOffHandle();

    M_new_file_act->addTo( menu );
    menu->insertSeparator();

    M_open_act->addTo( menu );
    M_open_train_act->addTo( menu );
    menu->insertSeparator();

    M_save_all_act->addTo( menu );
    M_save_act->addTo( menu );
    M_save_train_act->addTo( menu );
    menu->insertSeparator();

    M_save_as_act->addTo( menu );
    M_save_train_as_act->addTo( menu );
    menu->insertSeparator();

    M_exit_act->addTo( menu );

    menuBar()->insertItem( tr( "&File" ), menu );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createMenuEdit()
{
    QPopupMenu * menu = new QPopupMenu( this );


    M_record_data_act->addTo( menu );
    M_insert_data_act->addTo( menu );
    M_replace_data_act->addTo( menu );
    M_train_act->addTo( menu );

    menu->insertSeparator();

    M_delete_data_act->addTo( menu );

    menu->insertSeparator();

    M_reverse_y_act->addTo( menu );


    menuBar()->insertItem( tr( "&Edit" ), menu );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createMenuMode()
{
    QPopupMenu * menu = new QPopupMenu( this );

    M_toggle_ball_drag_act->addTo( menu );
    M_toggle_player_auto_move_act->addTo( menu );
    M_toggle_data_auto_select_act->addTo( menu );
    M_toggle_symmetric_mode_act->addTo( menu );

    menu->insertSeparator();

    M_show_edit_dialog_act->addTo( menu );

    menuBar()->insertItem( tr( "&Mode" ), menu );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createMenuView()
{
    QPopupMenu * menu = new QPopupMenu( this );

    M_zoom_in_act->addTo( menu );
    M_zoom_out_act->addTo( menu );
    M_fit_act->addTo( menu );

    menu->insertSeparator();

    M_toggle_enlarge_act->addTo( menu );
    M_toggle_show_index_act->addTo( menu );
    M_toggle_show_circumcircle_act->addTo( menu );

    menuBar()->insertItem( tr( "&View" ), menu );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createMenuHelp()
{
    QPopupMenu * menu = new QPopupMenu( this );
    M_about_act->addTo( menu );

    menuBar()->insertItem( tr( "&Help" ), menu );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createToolBars()
{
    M_tool_bar = new QToolBar( this );

    M_save_all_act->addTo( M_tool_bar );
    //M_toggle_ball_drag_act->addTo( M_tool_bar );
    M_toggle_player_auto_move_act->addTo( M_tool_bar );
    M_toggle_symmetric_mode_act->addTo( M_tool_bar );

    M_tool_bar->addSeparator();

    M_delete_data_act->addTo( M_tool_bar );
    M_reverse_y_act->addTo( M_tool_bar );

    M_index_spin_box = new QSpinBox( M_tool_bar );
    M_index_spin_box->setWrapping( true );
    M_index_spin_box->setRange( 0, 0 );
    M_index_spin_box->setMaximumSize( 64, 24 );
    //M_index_spin_box->setMinimumSize( 16, 16 );
    connect( M_index_spin_box, SIGNAL( valueChanged( int ) ),
             this, SLOT( changeDataIndex( int ) ) );

    M_tool_bar->addSeparator();

    M_train_act->addTo( M_tool_bar );
    M_replace_data_act->addTo( M_tool_bar );
    M_insert_data_act->addTo( M_tool_bar );
    M_record_data_act->addTo( M_tool_bar );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createStatusBar()
{
    this->statusBar()->message( tr( "Ready" ) );

    M_position_label = new QLabel( tr( "(0.0, 0.0)" ), this->statusBar() );

    int min_width
        = M_position_label->fontMetrics().width(  tr( "(-60.0, -30.0)" ) )
        + 16;
    M_position_label->setMinimumWidth( min_width );
    M_position_label->setAlignment( Qt::AlignRight );

    //this->statusBar()->addPermanentWidget( M_position_label ); // qt4
    this->statusBar()->addWidget( M_position_label, 0, true ); // permanent
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createFieldCanvas()
{
    M_editor_canvas = new FEditCanvas( this );
    this->setCentralWidget( M_editor_canvas );
    M_editor_canvas->setFocus();

    connect( this, SIGNAL( viewUpdated() ),
             M_editor_canvas, SLOT( update() ) );

    connect( M_editor_canvas, SIGNAL( pointerMoved( const QString & ) ),
             this, SLOT( updatePositionLabel( const QString & ) ) );

}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::createEditDialog()
{
    M_editor_dialog = new FEditDialog( this );

}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::closeEvent( QCloseEvent * event )
{
    if ( ! saveChanged() )
    {
        return;
    }

    event->ignore();

    qApp->quit();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::resizeEvent( QResizeEvent * event )
{
    //event->accept(); qt4

}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::wheelEvent( QWheelEvent * event )
{
    if ( M_index_spin_box->maxValue() <= 0 )
    {
        event->accept();
        return;
    }

    int idx = M_index_spin_box->value();

    if ( event->delta() > 0 )
    {
        --idx;
        if ( idx < 1 )
        {
            idx = M_index_spin_box->maxValue();
        }

        M_index_spin_box->setValue( idx );
    }
    else
    {
        ++idx;
        if ( idx > M_index_spin_box->maxValue() )
        {
            idx = 0;
        }

        M_index_spin_box->setValue( idx );
    }

    event->accept();
}


/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::about()
{
    QString msg( tr( "fedit-" VERSION ) ); //QString msg( "soccerwindow2-2.0.0" );
    msg += tr( "\n\n" );
    msg += tr( "fedit is a is a formation construction tool for agent2d.\n"
               "This program is a part of soccerwindow2.\n"
               "\n"
               "Web Site:\n"
               "  http://sourceforge.jp/projects/rctools/\n"
               "Author:\n"
               "  Hidehisa Akiyama <akky@users.sourceforge.jp>" );

    QMessageBox::about( this,
                        tr( "About fedit" ),
                        msg );

    // from Qt 4.1 documents
    /*
      about() looks for a suitable icon in four locations:

      1. It prefers parent->icon() if that exists.
      2. If not, it tries the top-level widget containing parent.
      3. If that fails, it tries the active window.
      4. As a last resort it uses the Information icon.

      The about box has a single button labelled "OK".
    */
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::toggleToolBar()
{
    if ( M_tool_bar->isVisible() )
    {
        M_tool_bar->hide();
    }
    else
    {
        M_tool_bar->show();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::toggleStatusBar()
{
    if ( this->statusBar()->isVisible() )
    {
        this->statusBar()->hide();
    }
    else
    {
        this->statusBar()->show();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::toggleFullScreen()
{
    if ( this->isFullScreen() )
    {
        this->showNormal();
    }
    else
    {
        this->showFullScreen();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::changeStyle( const QString & style )
{
    QApplication::setStyle( style );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::resizeCanvas( const QSize & size )
{
    if ( size.width() < this->minimumWidth()
         || size.height() < this->minimumHeight() )
    {
        std::cerr << "Too small canvas size ("
                  << size.width() << " "
                  << size.height() << ")"
                  << std::endl;
        return;
    }

    if ( centralWidget() )
    {
        if ( this->isMaximized()
             || this->isFullScreen() )
        {
            this->showNormal();
        }

        QRect rect = this->geometry();

        int width_diff = rect.width() - centralWidget()->width();
        int height_diff = rect.height() - centralWidget()->height();

        rect.setWidth( size.width() + width_diff );
        rect.setHeight( size.height() + height_diff );

        this->setGeometry( rect );

        //std::cerr << "centralWidget width = " << centralWidget()->width()
        //          << " height = " << centralWidget()->height()
        //          << std::endl;
    }
}


/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::updatePositionLabel( const QString & str )
{
    if ( M_position_label
         && statusBar()
         && statusBar()->isVisible()
         )
    {
        M_position_label->setText( str );
    }
}


/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::newFile()
{
    if ( ! saveChanged() )
    {
        // data is changed, but save operation is cancelled.
        return;
    }

    QStringList names;
    names.push_back( QString::fromAscii( rcsc::FormationDT::name().c_str() ) );
    names.push_back( QString::fromAscii( rcsc::FormationKNN::name().c_str() ) );
    names.push_back( QString::fromAscii( rcsc::FormationBPN::name().c_str() ) );
    names.push_back( QString::fromAscii( rcsc::FormationRBF::name().c_str() ) );
    names.push_back( QString::fromAscii( rcsc::FormationNGNet::name().c_str() ) );
    //names.push_back( QString( rcsc::FormationSBSP::name().c_str() ) );

    bool ok = false;
    QString name = QInputDialog::getItem( tr( "Create New Formation" ),
                                          tr( "Choose a training method:" ),
                                          names,
                                          0,
                                          false, // no editable
                                          &ok,
                                          this );
    if ( ! ok )
    {
        return;
    }

    std::string type_name = name.ascii();

    if ( M_editor_data )
    {
        M_editor_data.reset();
    }

    this->setCaption( tr( "FormationEditor - New Formation -" ) );


    // create new data

    M_editor_data = boost::shared_ptr< FEditData >( new FEditData() );

    M_editor_data->setTrainingTypeName( type_name );
    M_editor_data->createDefaultParam();

    if ( ! M_editor_data->formation() )
    {
        std::cerr << "***ERROR*** Failed to initialize formation data"
                  << std::endl;
        M_editor_data.reset();
        return;
    }

    M_editor_canvas->setData( M_editor_data );
    M_editor_dialog->setData( M_editor_data );

    M_editor_data->moveBallTo( 0.0, 0.0 );
    M_editor_dialog->updateData();

    emit viewUpdated();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::open()
{
    if ( ! saveChanged() )
    {
        // data is changed, but save operation is cancelled.
        return;
    }

    QString filter( tr( "Formation file (*.conf);;"
                        "All files (*)" ) );
    QString filepath = QFileDialog::getOpenFileName( QString::null,
                                                     filter,
                                                     this,
                                                     tr( "Open Formation" ),
                                                     tr( "Choose a file name to open" ) );

    open( filepath );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::open( const QString & filepath )
{
    if ( ! openConfFile( filepath ) )
    {
        return;
    }

    int result = QMessageBox::information( this,
                                           tr( "Option" ),
                                           tr( "Do you open a training data, too?" ),
                                           QMessageBox::Yes,
                                           QMessageBox::No );
    if ( result == QMessageBox::Yes )
    {
        openData();
    }

    emit viewUpdated();
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
MainWindow::openConfFile( const QString & filepath )
{
    if ( filepath.isEmpty() )
    {
        QMessageBox::warning( this,
                              tr( "Warning" ),
                              tr( "Empty file path." ),
                              QMessageBox::Ok,
                              QMessageBox::NoButton );
        return false;
    }

    QFileInfo fileinfo( filepath );

    if ( ! fileinfo.exists()
         || ! fileinfo.isReadable() )
    {
        QMessageBox::warning( this,
                              tr( "Warning" ),
                              tr( "No such a file or not readable. " ) + filepath,
                              QMessageBox::Ok,
                              QMessageBox::NoButton );
        return false;
    }

    M_editor_data = boost::shared_ptr< FEditData >( new FEditData() );
    if ( ! M_editor_data->open( filepath.ascii() ) )
    {
        M_editor_data.reset();
        QMessageBox::warning( this,
                              tr( "Error" ),
                              tr( "Failed to open the file " ) + filepath,
                              QMessageBox::Ok,
                              QMessageBox::NoButton );
        return false;
    }

    if ( ! M_editor_data->formation() )
    {
        M_editor_data.reset();
        QMessageBox::warning( this,
                              tr( "Error" ),
                              tr( "Failed to create a formation from " ) + filepath,
                              QMessageBox::Ok,
                              QMessageBox::NoButton );
        return false;
    }

    // set window title
    this->setCaption( tr( "FormationEditor - " )
                      + fileinfo.baseName()
                      + tr( " -") );

    M_editor_canvas->setData( M_editor_data );
    M_editor_dialog->setData( M_editor_data );

    M_editor_data->moveBallTo( 0.0, 0.0 );
    M_editor_dialog->updateData();

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::openData()
{
    if ( ! M_editor_data )
    {
        QMessageBox::warning( this,
                              tr( "Warning" ),
                              tr( "Data specified, but no formation." ),
                              QMessageBox::Ok,
                              QMessageBox::NoButton );
        return;
    }

    QString filter( tr( "Formation file (*.dat);;"
                        "All files (*)" ) );
    QString filepath = QFileDialog::getOpenFileName( QString::null,
                                                     filter,
                                                     this,
                                                     tr( "Open Training Data" ),
                                                     tr( "Choose a file name to open" ) );
    openDataFile( filepath );
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
MainWindow::openDataFile( const QString & filepath )
{
    if ( filepath.isEmpty() )
    {
        QMessageBox::warning( this,
                              tr( "Warning" ),
                              tr( "Empty file path." ),
                              QMessageBox::Ok,
                              QMessageBox::NoButton );
        return false;
    }

    if ( ! M_editor_data )
    {
        QMessageBox::warning( this,
                              tr( "Warning" ),
                              tr( "Data specified, but no formation." ),
                              QMessageBox::Ok,
                              QMessageBox::NoButton );
        return false;
    }

    QFileInfo fileinfo( filepath );

    if ( ! fileinfo.exists()
         || ! fileinfo.isReadable() )
    {
        QMessageBox::warning( this,
                              tr( "Warning" ),
                              tr( "No such a file or not readable. " ) + filepath,
                              QMessageBox::Ok,
                              QMessageBox::NoButton );
        return false;
    }

    if ( ! M_editor_data->openTrainingData( filepath.ascii() ) )
    {
        QMessageBox::warning( this,
                              tr( "Error" ),
                              tr( "Failed to read the file " ) + filepath,
                              QMessageBox::Ok,
                              QMessageBox::NoButton );
        return false;

    }

    setTrainingDataSize();

    emit viewUpdated();

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
MainWindow::saveChanged()
{
    if ( ! M_editor_data )
    {
        return true;
    }

    if ( ! M_editor_data->isModified()
         && ! M_editor_data->isTrainingDataChanged() )
    {
        return true;
    }

    int result = QMessageBox::question( this,
                                        tr( "Save Notify" ),
                                        tr( "Data is changed.\nSave?" ),
                                        QMessageBox::Cancel,
                                        QMessageBox::Ok,
                                        QMessageBox::No );
    if ( result == QMessageBox::Cancel )
    {
        return false;
    }

    if ( result == QMessageBox::Ok )
    {
        if ( M_editor_data->isModified() )
        {
            save();
        }

        if ( M_editor_data->isTrainingDataChanged() )
        {
            saveTrainingData();
        }
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::saveAll()
{
    if ( ! M_editor_data )
    {
        return;
    }

    save();
    saveTrainingData();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::save()
{
    if ( ! M_editor_data
         || ! M_editor_data->formation() )
    {
        return;
    }

    std::cerr << "save" << std::endl;

    if ( ! M_editor_data->save() )
    {
        saveAs();
    }

}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::saveAs()
{
    if ( ! M_editor_data
         || ! M_editor_data->formation() )
    {
        return;
    }

    QString filter( tr( "Formation file (*.conf);;"
                        "All files (*)" ) );
    QString filepath = QFileDialog::getSaveFileName( QString::null,
                                                     filter,
                                                     this,
                                                     tr( "Save Formation" ),
                                                     tr( "Choose a file name to save under" ) );

    if ( filepath.length() <= 5
         || filepath.right( 5 ) != tr( ".conf" ) )
    {
        filepath += ".conf";
    }

    if ( M_editor_data->save( filepath.ascii() ) )
    {
        QFileInfo fileinfo( filepath );
        this->setCaption( tr( "FormationEditor - " )
                          + fileinfo.baseName()
                          + tr( " -") );

    }
    else
    {
        QMessageBox::critical( this,
                               tr( "Error" ),
                               tr( "Failed to save the file " ) + filepath,
                               QMessageBox::Ok,
                               QMessageBox::NoButton );
        std::cerr << "Error. Failed to open file output steramr" << std::endl;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::saveTrainingData()
{
    if ( ! M_editor_data
         || M_editor_data->trainingData().empty() )
    {
        return;
    }

    if ( ! M_editor_data->saveTrainingData() )
    {
        saveTrainingDataAs();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::saveTrainingDataAs()
{
    if ( ! M_editor_data
         || M_editor_data->trainingData().empty() )
    {
        return;
    }

    QString filter( tr( "Formation file (*.dat);;"
                        "All files (*)" ) );
    QString filepath = QFileDialog::getSaveFileName( QString::null,
                                                     filter,
                                                     this,
                                                     tr( "Save Training Data" ),
                                                     tr( "Choose a file name to save under" ) );

    if ( filepath.length() <= 4
         || filepath.right( 4 ) != tr( ".dat" ) )
    {
        filepath += ".dat";
    }

    if ( ! M_editor_data->saveTrainingData( filepath.ascii() ) )
    {
        QMessageBox::critical( this,
                               tr( "Error" ),
                               tr( "Failed to save the file " ) + filepath,
                               QMessageBox::Ok,
                               QMessageBox::NoButton );
        std::cerr << "Error. Failed to open file output steramr" << std::endl;
    }

}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::toggleBallDrag( bool on )
{
    //std::cerr << "toggle ball drag " << on << std::endl;

    if ( M_editor_data )
    {
        M_editor_data->setBallDraggable( on );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::togglePlayerAutoMove( bool on )
{
    //std::cerr << "toggle player auto move " << on << std::endl;

    if ( M_editor_data )
    {
        M_editor_data->setPlayerAutoMove( on );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::toggleDataAutoSelect( bool on )
{
    //std::cerr << "toggle player auto move " << on << std::endl;

    if ( M_editor_data )
    {
        M_editor_data->setDataAutoSelect( on );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::toggleSymmetricMode( bool on )
{
    //std::cerr << "toggle symmetric mode " << std::endl;

    if ( M_editor_data )
    {
        M_editor_data->setSymmetricMode( on );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::showEditDialog( bool on )
{
    if ( on )
    {
        M_editor_dialog->show();

        QSize size = M_editor_dialog->size();
        M_editor_dialog->setMinimumSize( size );
        M_editor_dialog->setMaximumSize( size );
    }
    else
    {
        M_editor_dialog->hide();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::recordData()
{
    std::cerr << "recordData" << std::endl;

    if ( ! M_editor_data )
    {
        return;
    }

    std::string msg = M_editor_data->recordTrainingData();

    if ( ! msg.empty() )
    {
        QMessageBox::warning( this,
                              tr( "Warning" ),
                              QString::fromAscii( msg.c_str() ),
                              QMessageBox::Ok,
                              QMessageBox::NoButton );
        return;
    }

    setTrainingDataSize();
    M_index_spin_box->setValue( M_editor_data->trainingData().size() );

    emit viewUpdated();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::insertData()
{
    if ( ! M_editor_data )
    {
        return;
    }

    int cur = M_index_spin_box->value();

    std::string msg = M_editor_data->insertTrainingData( cur );

    if ( ! msg.empty() )
    {
        QMessageBox::warning( this,
                              tr( "Warning" ),
                              QString::fromAscii( msg.c_str() ),
                              QMessageBox::Ok,
                              QMessageBox::NoButton );
        return;
    }

    setTrainingDataSize();
    M_index_spin_box->setValue( cur + 1 );

    emit viewUpdated();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::replaceData()
{
    if ( ! M_editor_data )
    {
        return;
    }

    if ( M_editor_data->replaceTrainingData( M_index_spin_box->value() ) )
    {
        emit viewUpdated();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::deleteData()
{
    if ( ! M_editor_data )
    {
        return;
    }

    if ( M_editor_data->deleteTrainingData( M_index_spin_box->value() ) )
    {
        M_index_spin_box->setValue( 0 );
        setTrainingDataSize();
        emit viewUpdated();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::reverseY()
{
    if ( M_editor_data )
    {
        M_editor_data->reverseY();
        updateDataIndex();
        emit viewUpdated();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::train()
{
    if ( M_editor_data )
    {
        M_editor_data->train();
        emit viewUpdated();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::setTrainingDataSize()
{
    if ( M_editor_data
         && ! M_editor_data->trainingData().empty() )
    {
        M_index_spin_box->setRange( 0, M_editor_data->trainingData().size() );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::changeDataIndex( int idx )
{
    if ( M_index_spin_box->value() != idx
         && 0 < idx
         && idx <= M_index_spin_box->maxValue() )
    {
        M_index_spin_box->setValue( idx );
    }

    if ( M_editor_data
         && M_editor_data->focusTrainingData( idx ) )
    {
        M_editor_dialog->updateData();
        emit viewUpdated();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
MainWindow::updateDataIndex()
{
    if ( M_editor_data
         && M_editor_data->isPlayerAutoMove() )
    {
        int idx = M_editor_data->currentIndex();
        if ( M_index_spin_box->value() != idx
             && 0 < idx
             && idx <= M_index_spin_box->maxValue() )
        {
            M_index_spin_box->setValue( idx );
        }
    }
}
