// -*-c++-*-

/*!
	\file fedit_data.h
	\brief formation editor data class Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.	If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef FEDIT_QT3_FEDIT_DATA_H
#define FEDIT_QT3_FEDIT_DATA_H

#include <rcsc/formation/formation.h>
#include <rcsc/geom/delaunay_triangulation.h>
#include <rcsc/geom/vector_2d.h>

#include <boost/shared_ptr.hpp>
#include <list>
#include <string>

///////////////////////////////////////////////////////////////////////
//! formation editor data class
class FEditData {
public:

    static const int MAX_TRAINING_DATA_SIZE;

    enum SelectType {
        SELECT_BALL,
        SELECT_OUR,
        SELECT_OPP,
        NO_SELECT,
    };

    enum {
        X_DIV = 23,
        Y_DIV = 13,
    };

private:

    //! formation training type name
    std::string M_training_type_name;

    //! file path to save the formation parameters
    std::string M_filepath;

    //! file path to save the training data
    std::string M_filepath_train;

    //! flag to check whether any formation paramters are changed
    bool M_modified;

    //! flag to check whether any training data are changed
    bool M_training_data_changed;

    //----------------------------------------
    // field status

    //! current ball position
    rcsc::Vector2D M_ball;

    //! flag to check whether our player is dragged or not.
    std::vector< int > M_our_dragged;

    //! current our players position
    std::vector< rcsc::Vector2D > M_our_players;

    //! current opponent players position
    std::vector< rcsc::Vector2D > M_opp_players;

    //! current triangulation
    rcsc::DelaunayTriangulation M_triangulation;

    //----------------------------------------
    // interaction info

    //! symmetrically handle the data or not
    bool M_symmetric_mode;

    //! object selection information
    SelectType M_select_type;

    //! selected player index, usualy unum-1
    std::size_t M_select_index;

    //! ball is draggable or not
    bool M_ball_draggable;

    //! player is automatically moved or not.
    bool M_player_auto_move;

    //! exist data is automatically selected or not when ball is moved
    bool M_data_auto_select;

    //! stored training data
    std::list< rcsc::Formation::Snapshot > M_training_data;

    //! current data index
    int M_current_index;

    //----------------------------------------
    //! editingformation parameters
    boost::shared_ptr< rcsc::Formation > M_formation;

public:

    FEditData();

		~FEditData();

private:

    void init();

    void backup( const std::string & filepath );

public:

    /*!
      \brief initialize formation param.
      pre-defined roles are assigned and formation parameters are
      randomized.
     */
    void createDefaultParam();


    bool isModified() const
      {
          return M_modified;
      }

    bool isTrainingDataChanged() const
      {
          return M_training_data_changed;
      }

    /*!
      \brief get current ball position
      \return coordinate of ball position
     */
    const
    rcsc::Vector2D & ball() const
      {
          return M_ball;
      }

    /*!
      \brief get our players' position container
      \return const reference to the position container
     */
    const
    std::vector< rcsc::Vector2D > & ourPlayers() const
      {
          return M_our_players;
      }

    /*!
      \brief get opponent players' position container
      \return const reference to the position container
     */
    const
    std::vector< rcsc::Vector2D > & oppPlayers() const
      {
          return M_opp_players;
      }

    /*!
      \brief get triangulation data
      \return const reference to the data
     */
    rcsc::DelaunayTriangulation & triangulation()
      {
          return M_triangulation;
      }

    /*!
      \brief get triangulation data
      \return const reference to the data
     */
    const
    rcsc::DelaunayTriangulation & triangulation() const
      {
          return M_triangulation;
      }

    /*!
      \brief get training data container
      \return const reference to the data container
     */
    const
    std::list< rcsc::Formation::Snapshot > & trainingData() const
      {
          return M_training_data;
      }

    int currentIndex() const
      {
          return M_current_index;
      }

    void setSymmetricMode( bool on )
      {
          M_symmetric_mode = on;
      }

    bool symmetricMode() const
      {
          return M_symmetric_mode;
      }

    /*!
      \brief get type of current selected object type
      \return object selection type
    */
    SelectType getSelectType() const
      {
          return M_select_type;
      }

    /*!
      \brief get player index if selected object is player.
      \return player index
     */
    std::size_t getSelectIndex() const
      {
          return M_select_index;
      }

    void setBallDraggable( bool on )
      {
          M_ball_draggable = on;
      }

    bool isBallDraggable() const
      {
          return M_ball_draggable;
      }

    void setPlayerAutoMove( bool on )
      {
          M_player_auto_move = on;
      }

    bool isPlayerAutoMove() const
      {
          return M_player_auto_move;
      }

    void setDataAutoSelect( bool on )
      {
          M_data_auto_select = on;
      }

    bool isDataAutoSelect() const
      {
          return M_data_auto_select;
      }

    /*!
      \brief get formation param pointer
      \return pointer to the formation paramter instance
     */
    boost::shared_ptr< rcsc::Formation > formation()
      {
          return M_formation;
      }

    /*!
      \brief get formation param const pointer
      \return const pointer to the formation paramter instance
     */
    boost::shared_ptr< const rcsc::Formation > formation() const
      {
          return M_formation;
      }

    bool open( const std::string & filepath );
    bool openTrainingData( const std::string & filepath );

    bool save();
    bool save( const std::string & filepath );
    bool save( std::ostream & os );

    bool saveTrainingData();
    bool saveTrainingData( const std::string & filepath );
    bool saveTrainingData( std::ostream & os );


private:

    void updatePlayerPosition();

    void updateTriangulation();

public:

    void reverseY();
    void reverseY( std::vector< rcsc::Vector2D > & our_players );

    /*!
      \brief move the ball to the specified position
      \param x target x
      \param y target y
     */
    void moveBallTo( const double & x,
                     const double & y );

    /*!
      \brief select object nearest to the specifiend position
      We can choose a ball, and players
      \param x mouse pointed position x
      \param y mouse pointed position y
     */
    void selectObject( const double & x,
                       const double & y );

    /*!
      \brief unselect(drop) object at (x, y)
      \param x released point x
      \param y released point y
     */
    void releaseObject( const double & x,
                        const double & y );

    /*!
      \brief update dragged object position. object position is
      restricted within pitch.
      \param x new object position x
      \param y new object position y
     */
    void setObjectDragPoint( double x,
                             double y );

    void setTrainingTypeName( const std::string & type_name );

    void updateRoleData( const int unum,
                         const int synmetry_unum,
                         const std::string & role_name );

    void setBallPosition( const double & x,
                          const double & y );
    void setPlayerPosition( const int unum,
                            const double & x,
                            const double & y );

    bool focusTrainingData( const int idx );

    std::string recordTrainingData();
    std::string recordTrainingData( const rcsc::Formation::Snapshot & snapshot,
                                    const bool symmetry );

    std::string insertTrainingData( const int idx );
    std::string insertTrainingData( const int idx,
                                    const rcsc::Formation::Snapshot & snapshot );

    bool replaceTrainingData( const int idx );
    bool replaceTrainingData( const int idx,
                              const rcsc::Formation::Snapshot & snapshot,
                              const bool symmetry );

    bool deleteTrainingData( const int idx );

    void train();

private:

    bool recordSymmetricData( const rcsc::Formation::Snapshot & snapshot );

    bool replaceSymmetricData( const rcsc::Formation::Snapshot & snapshot );
};

#endif
