// -*-c++-*-

/*!
	\file feditor_data.cpp
	\brief formation editor data class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.	If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "fedit_data.h"

#include "fedit_config.h"

#include <rcsc/formation/formation_factory.h>
#include <rcsc/formation/formation_dt.h>
#include <rcsc/formation/formation_knn.h>
#include <rcsc/math_util.h>

#include <algorithm>
#include <sstream>
#include <fstream>
#include <iostream>

#include <cassert>
#include <ctime>
#include <cstdio>
#include <cstdarg>

const int FEditData::MAX_TRAINING_DATA_SIZE = 128;


/*-------------------------------------------------------------------*/
/*!

*/
FEditData::FEditData()
    : M_training_type_name( "" )
    , M_filepath( "" )
    , M_filepath_train( "" )
    , M_modified( false )
    , M_training_data_changed( false )
    , M_our_dragged( 11, 0 )
    , M_our_players( 11 )
    , M_opp_players( 11 )
    , M_symmetric_mode( true )
    , M_select_type( NO_SELECT )
    , M_select_index( 0 )
    , M_ball_draggable( true )
    , M_player_auto_move( true )
    , M_data_auto_select( true )
{
    init();
}

/*-------------------------------------------------------------------*/
/*!

*/
FEditData::~FEditData()
{
    //saveChanged();
    //std::cerr << "delete FEditData" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::init()
{
    M_modified = false;
    M_training_data_changed = false;

    M_ball.assign( FEditConfig::instance().initialBallX(),
                   FEditConfig::instance().initialBallY() );

    M_our_dragged.assign( 11, 0 );
    for ( int i = 1; i <= 11; ++i )
    {
        M_our_players[i-1].setPolar( -3.0 * i, -37.0 );
        M_opp_players[i-1].assign( 3.0 * i, -37.0 );
    }

    M_select_type = NO_SELECT;
    M_select_index = 0;
    M_ball_draggable = true;
    M_player_auto_move = true;

    M_triangulation.init( rcsc::Rect2D( - FEditConfig::PITCH_LENGTH * 0.5,
                                        - FEditConfig::PITCH_WIDTH * 0.5,
                                        FEditConfig::PITCH_LENGTH,
                                        FEditConfig::PITCH_WIDTH ) );
    updateTriangulation();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::backup( const std::string & filepath )
{
    if ( filepath.empty() )
    {
        return;
    }

    char time_str[64];
    std::time_t current = std::time( NULL );
    std::tm * local_time = std::localtime( &current );
    if ( local_time )
    {
        std::strftime( time_str, 64, ".%Y%m%d-%H%M", local_time );
    }
    else
    {
        std::snprintf( time_str, 64, ".time-%ld", current );
    }

    std::string backup_filepath = ".";
    backup_filepath += filepath;
    backup_filepath += time_str;

    std::ifstream fin( filepath.c_str() );
    std::ofstream fout( backup_filepath.c_str() );

    if ( fin && fout )
    {
        std::istreambuf_iterator< char > in_itr( fin );
        std::ostreambuf_iterator< char > out_itr( fout );
        std ::copy( in_itr, std::istreambuf_iterator< char >(),
                    out_itr );

        fin.close();
        fout.close();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::createDefaultParam()
{
    M_formation = rcsc::make_formation( M_training_type_name );

    if ( ! M_formation )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " ***ERROR*** Failed to create formation."
                  << std::endl;
        return;
    }

    rcsc::Formation::Snapshot default_data = M_formation->createDefaultParam();

    M_training_data.push_back( default_data );

    train();
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::open( const std::string & filepath )
{
    std::ifstream fin( filepath.c_str() );
    if ( ! fin.is_open() )
    {
        std::cerr << "Failed to open formation file [" << filepath << "]"
                  << std::endl;
        return false;
    }

    M_training_data.clear();

    {
        std::string temp, type;
        fin >> temp >> type; // read training method type name
        M_formation = rcsc::make_formation( type );

        if ( ! M_formation )
        {
            std::cerr << "Invalid formation format in the file ["
                      << filepath << "]\n"
                      << "You must write the training method type name"
                      << " at the top of the formation file"
                      << std::endl;
            return false;
        }
    }

    fin.seekg( 0 );
    if ( ! M_formation->read( fin ) )
    {
        fin.close();
        M_formation.reset();
        return false;
    }

    fin.close();

    M_filepath = filepath;

    init();
    updatePlayerPosition();
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::openTrainingData( const std::string & filepath )
{
    std::ifstream fin( filepath.c_str() );
    if ( ! fin.is_open() )
    {
        std::cerr << "Failed to open training data file [" << filepath << "]"
                  << std::endl;
        return false;
    }

    // clear current data list
    M_training_data.clear();

    bool success = true;
    std::string line_buf;
    while ( std::getline( fin, line_buf ) )
    {
        rcsc::Formation::Snapshot snap;
        std::istringstream istr( line_buf );
        istr >> snap.ball_.x >> snap.ball_.y;
        for ( int i = 0; i < 11; ++i )
        {
            if ( ! istr.good() )
            {
                success = false;
                break;
            }
            rcsc::Vector2D v;
            istr >> v.x >> v.y;
            snap.players_.push_back( v );
        }

        M_training_data.push_back( snap );
    }

    fin.close();

    if ( ! success )
    {
        M_training_data.clear();
        return false;
    }


    M_filepath_train = filepath;
    M_modified = false;
    M_training_data_changed = false;

    updateTriangulation();

    moveBallTo( FEditConfig::instance().initialBallX(),
                FEditConfig::instance().initialBallY() );
    updatePlayerPosition();

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::save()
{
    if ( M_filepath.empty() )
    {
        return false;
    }

    return save( M_filepath );
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::save( const std::string & filepath )
{
    // create backup file
    if ( FEditConfig::instance().autoBackup() )
    {
        backup( M_filepath );
    }

    std::ofstream fout( filepath.c_str() );
    if ( ! fout.is_open() )
    {
        fout.close();
        return false;
    }

    if ( ! save( fout ) )
    {
        return false;
    }

    M_filepath = filepath;
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::save( std::ostream & os )
{
    if ( ! M_formation )
    {
        return false;
    }

    if ( ! M_formation->print( os ) )
    {
        return false;
    }

    M_modified = false;
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::saveTrainingData()
{
    if ( M_filepath_train.empty() )
    {
        return false;
    }

    return saveTrainingData( M_filepath_train );
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::saveTrainingData( const std::string & filepath )
{
    // create backup file
    if ( FEditConfig::instance().autoBackup() )
    {
        backup( M_filepath_train );
    }

    std::ofstream fout( filepath.c_str() );
    if ( ! fout.is_open() )
    {
        fout.close();
        return false;
    }

    if ( ! saveTrainingData( fout ) )
    {
        return false;
    }

    M_filepath_train = filepath;
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::saveTrainingData( std::ostream & os )
{
    // put data to the stream
    for ( std::list< rcsc::Formation::Snapshot >::iterator it = M_training_data.begin();
          it != M_training_data.end();
          ++it )
    {
        os << it->ball_.x << ' ' << it->ball_.y << ' ';
        for ( std::vector< rcsc::Vector2D >::iterator p = it->players_.begin();
              p != it->players_.end();
              ++p )
        {
            os << p->x << ' ' << p->y << ' ';
        }
        os << std::endl;
    }

    if ( ! os )
    {
        return false;
    }

    M_training_data_changed = false;

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::updatePlayerPosition()
{
    if ( ! M_player_auto_move )
    {
        return;
    }

    if ( ! M_formation )
    {
        //std::cerr << "No formation param set" << std::endl;
        return;
    }

    // update all players' position using formation param

    const rcsc::Vector2D & bpos = M_ball;

    for ( int unum = 1; unum <= 11; ++unum )
    {
        rcsc::Vector2D pos = M_formation->getPosition( unum, bpos );
        pos.x = rcsc::min_max( -53.0, pos.x, 53.0 );
        pos.y = rcsc::min_max( -35.0, pos.y, 35.0 );

        M_our_players[unum - 1] = pos;
    }

    M_our_dragged.assign( 11, 0 );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::updateTriangulation()
{
    const rcsc::Rect2D pitch( - FEditConfig::PITCH_LENGTH * 0.5,
                              - FEditConfig::PITCH_WIDTH * 0.5,
                              FEditConfig::PITCH_LENGTH,
                              FEditConfig::PITCH_WIDTH );
#if 0
    if ( M_formation
         && M_formation->methodName() == rcsc::FormationDT::name() )
    {
        rcsc::FormationDT tmp_formation;
        tmp_formation.createDefaultParam();

        for ( int idx = 0; idx < trainingData().size(); ++idx )
        {
            const std::list< rcsc::Formation::Snapshot >::iterator end
                = M_training_data.end();

            std::list< rcsc::Formation::Snapshot > tmp_data;

            int check_idx = 0;
            std::list< rcsc::Formation::Snapshot >::iterator check_it = end;
            for ( std::list< rcsc::Formation::Snapshot >::iterator it
                      = M_training_data.begin();
                  it != end;
                  ++it, ++check_idx )
            {
                if ( check_idx == idx )
                {
                    check_it = it;
                    continue;
                }

                tmp_data.push_back( *it );
            }

            if ( check_it == end )
            {
                std::cerr << __FILE__ << ":" << __LINE__
                          << " ***ERROR*** index = " << idx
                          << std::endl;
                continue;
            }

            tmp_formation.train( tmp_data );

            bool very_near = true;
            for ( int p = 0; p < 11; ++p )
            {
                rcsc::Vector2D pos = tmp_formation.getPosition( p + 1, check_it->ball_ );
                if ( pos.dist( check_it->players_[p] ) > 0.5 )
                {
                    very_near = false;
                    break;
                }
            }

            if ( very_near )
            {
                std::cerr << "Removed data. index=" << idx
                          << " ball=" << check_it->ball_
                          << std::endl;
                M_training_data.erase( check_it );
                --idx;
            }
        }
    }
#endif
    {
        M_triangulation.init( pitch );

        const std::list< rcsc::Formation::Snapshot >::const_iterator end
            = trainingData().end();
        for ( std::list< rcsc::Formation::Snapshot >::const_iterator it
                  = trainingData().begin();
              it != end;
              ++it )
        {
            M_triangulation.addVertex( it->ball_ );
        }

        M_triangulation.compute();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::reverseY()
{
    if ( ! M_formation )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " ***ERROR*** No formation!!"
                  << std::endl;
        return;
    }

    M_ball.y *= -1.0;

    reverseY( M_our_players );

    // TODO: right side players

    //
    // update index
    //
    if ( M_data_auto_select )
    {
        int index = 1;
        const std::list< rcsc::Formation::Snapshot >::iterator end = M_training_data.end();
        for ( std::list< rcsc::Formation::Snapshot >::iterator it = M_training_data.begin();
              it != end;
              ++it, ++index )
        {
            if ( it->ball_.dist2( M_ball ) < 1.0*1.0 )
            {
                M_ball = it->ball_;
                M_current_index = index;
                break;
            }
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::reverseY( std::vector< rcsc::Vector2D > & our_players )
{
    if ( ! M_formation )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " ***ERROR*** No formation!!"
                  << std::endl;
        return;
    }

    std::vector< rcsc::Vector2D > old_positions = our_players;

    int unum = 1;
    for ( std::vector< rcsc::Vector2D >::iterator it = our_players.begin();
          it != our_players.end();
          ++it )
    {
        if ( M_formation->isCenterType( unum ) )
        {
            it->y *= -1.0;
        }
        else if ( M_formation->isSynmetryType( unum ) )
        {
            int synmetry_unum = M_formation->getSynmetryNumber( unum );
            if ( synmetry_unum == 0 ) continue;
            it->x = old_positions[synmetry_unum - 1].x;
            it->y = old_positions[synmetry_unum - 1].y * -1.0;
        }
        else if ( M_formation->isSideType( unum ) )
        {
            it->y *= -1.0;
            for ( int iunum = 1; iunum <= 11; ++iunum )
            {
                if ( M_formation->getSynmetryNumber( iunum ) == unum )
                {
                    it->x = old_positions[iunum - 1].x;
                    it->y = old_positions[iunum - 1].y * -1.0;
                }
            }
        }
        ++unum;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::moveBallTo( const double & x,
                       const double & y )
{
    rcsc::Vector2D pos( x, y );
    pos.y = std::max( - FEditConfig::PITCH_WIDTH * 0.5 - 2.0, pos.y );
    pos.y = std::min( + FEditConfig::PITCH_WIDTH * 0.5 + 2.0, pos.y );
    pos.x = std::max( - FEditConfig::PITCH_LENGTH * 0.5 - 2.0, pos.x );
    pos.x = std::min( + FEditConfig::PITCH_LENGTH * 0.5 + 2.0, pos.x );

    M_ball = pos;
    if ( pos.absY() < 1.0 )
    {
        M_ball.y = 0.0;
    }

    if ( M_data_auto_select )
    {
        int index = 1;
        const std::list< rcsc::Formation::Snapshot >::iterator end = M_training_data.end();
        for ( std::list< rcsc::Formation::Snapshot >::iterator it = M_training_data.begin();
              it != end;
              ++it, ++index )
        {
            if ( it->ball_.dist2( M_ball ) < 1.0*1.0 )
            {
                M_ball = it->ball_;
                M_current_index = index;
                break;
            }
        }
    }

    updatePlayerPosition();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::selectObject( const double & x,
                         const double & y )
{
    const rcsc::Vector2D pos( x, y );
    const double dist2_thr = 1.5 * 1.5;

    SelectType old_selection = M_select_type;

    double mindist2 = 200.0 * 200.0;

    if ( M_ball_draggable )
    {
        double d2 = M_ball.dist2( pos );
        if ( d2 < dist2_thr )
        {
            //std::cerr << "selection update ball" << std::endl;
            M_select_type = SELECT_BALL;
            mindist2 = d2;
        }
    }
    //mindist2 = M_ball.dist2( pos );

#if 0
    // check oppnent
    for ( std::vector< rcsc::Vector2D >::iterator it = M_opp_players.begin();
          it != M_opp_players.end();
          ++it )
    {
        double d2 = it->dist2( pos );
        if ( d2 < dist2_thr
             && d2 < mindist2 )
        {
            M_select_type = SELECT_OPP;
            M_select_index = std::distance( M_opp_players.begin(), it );
            mindist2 = d2;
        }
    }
#endif

    // check our players
    int unum = 1;
    for ( std::vector< rcsc::Vector2D >::iterator it = M_our_players.begin();
          it != M_our_players.end();
          ++it, ++unum )
    {
        double d2 = it->dist2( pos );
        if ( d2 < dist2_thr
             && d2 < mindist2 )
        {
            M_select_type = SELECT_OUR;
            M_select_index = std::distance( M_our_players.begin(), it );
            mindist2 = d2;
        }
    }

    if ( old_selection != M_select_type )
    {
        if ( M_select_type == SELECT_BALL )
        {
            M_ball = pos;
        }
        else if ( M_select_type == SELECT_OUR )
        {
            M_our_players[M_select_index] = pos;
        }
        else if ( M_select_type == SELECT_OPP )
        {
            M_opp_players[M_select_index] = pos;
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::releaseObject( const double & x,
                          const double & y )
{
    M_select_type = NO_SELECT;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::setObjectDragPoint( double x,
                               double y )
{
    if ( M_select_type == SELECT_BALL )
    {
        moveBallTo( x, y );
        return;
    }

    y = std::max( - FEditConfig::PITCH_WIDTH * 0.5 - 2.0, y );
    y = std::min( + FEditConfig::PITCH_WIDTH * 0.5 + 2.0, y );
    x = std::max( - FEditConfig::PITCH_LENGTH * 0.5 - 2.0, x );
    x = std::min( + FEditConfig::PITCH_LENGTH * 0.5 + 2.0, x );

    switch ( M_select_type ) {
    case SELECT_OUR:
        M_our_dragged[M_select_index] = 1;
        M_our_players[M_select_index].assign( x, y );

        if ( M_symmetric_mode
             && M_ball.absY() < 0.1
             && M_formation )
        {
            if ( M_formation->isSynmetryType( M_select_index + 1 ) )
            {
                int synmetry_unum = M_formation->getSynmetryNumber( M_select_index + 1 );
                if ( synmetry_unum != 0 )
                {
                    M_our_players[synmetry_unum-1].x = x;
                    M_our_players[synmetry_unum-1].y = -y;
                }
            }
            else if ( M_formation->isSideType( M_select_index + 1 ) )
            {
                for ( int unum = 1; unum <= 11; ++unum )
                {
                    if ( M_formation->getSynmetryNumber( unum )
                         == static_cast< int >( M_select_index + 1 ) )
                    {
                        M_our_players[unum-1].x = x;
                        M_our_players[unum-1].y = -y;
                    }
                }
            }
        }

        break;
    case SELECT_OPP:
        M_opp_players[M_select_index].assign( x, y );
        break;
    default:
        break;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::setTrainingTypeName( const std::string & type_name )
{
    M_training_type_name = type_name;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::updateRoleData( const int unum,
                           const int synmetry_unum,
                           const std::string & role_name )
{
    if ( ! M_formation )
    {
        return;
    }

    if ( M_formation->updateRole( unum, synmetry_unum, role_name ) )
    {
        M_modified = true;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::setBallPosition( const double & x,
                            const double & y )
{
    M_ball.assign( x, y );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::setPlayerPosition( const int unum,
                              const double & x,
                              const double & y )
{
    M_our_players[unum - 1].assign( x, y );
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::focusTrainingData( const int idx )
{
    if ( idx <= 0 )
    {
        return false;
    }

    std::list< rcsc::Formation::Snapshot >::iterator it = M_training_data.begin();

    int count = 1;
    while ( it != M_training_data.end() )
    {
        if ( idx == count )
        {
            M_ball = it->ball_;

            for ( std::size_t i = 0; i < 11; ++i )
            {
                M_our_players[i] = it->players_[i];
            }

            M_current_index = idx;
            return true;
        }
        ++count;
        ++it;
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::string
FEditData::recordTrainingData()
{
    return recordTrainingData( rcsc::Formation::Snapshot( M_ball, M_our_players ),
                               M_symmetric_mode );
}

/*-------------------------------------------------------------------*/
/*!

*/
std::string
FEditData::recordTrainingData( const rcsc::Formation::Snapshot & snapshot,
                               const bool symmetry )
{
    if ( ! M_formation )
    {
        return std::string( "No formation data." );
    }

    if ( static_cast< int >( M_training_data.size() ) >= MAX_TRAINING_DATA_SIZE )
    {
        return std::string( "Too many recorded positin!!" );
    }

    for ( std::list< rcsc::Formation::Snapshot >::iterator it = M_training_data.begin();
          it != M_training_data.end();
          ++it )
    {
        if ( it->ball_.dist( snapshot.ball_ ) < 0.5 )
        {
            return std::string( "*** WARNING ***"
                                " Exist too near recorded position!" );
        }
    }

    M_training_data.push_back( snapshot );
    std::cerr << "Save positin. current data size = " << M_training_data.size()
              << std::endl;

    M_training_data_changed = true;

    if ( symmetry )
    {
        recordSymmetricData( snapshot );
    }
    else
    {
        M_ball = snapshot.ball_;
        M_our_players = snapshot.players_;
    }

    updateTriangulation();

    if ( M_formation->methodName() == rcsc::FormationDT::name()
         || M_formation->methodName() == rcsc::FormationKNN::name() )
    {
        train();
    }

    return std::string();
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::recordSymmetricData( const rcsc::Formation::Snapshot & snapshot )
{
    if ( snapshot.ball_.absY() < 0.5 )
    {
        return false;
    }

    const rcsc::Vector2D ball( snapshot.ball_.x, - snapshot.ball_.y );

    for ( std::list< rcsc::Formation::Snapshot >::iterator it = M_training_data.begin();
          it != M_training_data.end();
          ++it )
    {
        if ( it->ball_.dist( ball ) < 0.5 )
        {
            std::cerr << "***WARNING*** recordSymmetricData() exist too near data."
                      << std::endl;
            return false;
        }
    }

    rcsc::Formation::Snapshot reversed = snapshot;

    reversed.ball_.y *= -1.0;
    reverseY( reversed.players_ );

    M_training_data.push_back( reversed );
    std::cerr << "Save symmetric positin. current data size = "
              << M_training_data.size()
              << std::endl;

    M_ball = reversed.ball_;
    M_our_players = reversed.players_;

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
std::string
FEditData::insertTrainingData( const int idx )
{
    return insertTrainingData( idx,
                               rcsc::Formation::Snapshot( M_ball, M_our_players ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
std::string
FEditData::insertTrainingData( const int idx,
                               const rcsc::Formation::Snapshot & snapshot )
{
    if ( static_cast< int >( M_training_data.size() ) >= MAX_TRAINING_DATA_SIZE )
    {
        return std::string( "Too many recorded positin!!" );
    }

    if ( idx < 0
         || static_cast< int >( M_training_data.size() ) < idx )
    {
        return std::string( "Index is over the range." );
    }

    std::list< rcsc::Formation::Snapshot >::iterator it = M_training_data.begin();
    for ( int i = 0; i < idx; ++ i )
    {
        ++it;
    }

    M_training_data.insert( it, snapshot );
    std::cerr << "Inserted. current data size = "
              << M_training_data.size()
              << std::endl;

    updateTriangulation();

    if ( M_formation->methodName() == rcsc::FormationDT::name()
         || M_formation->methodName() == rcsc::FormationKNN::name() )
    {
        train();
    }

    return std::string();
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::replaceTrainingData( const int idx )
{
    if ( idx <= 0
         || static_cast< int >( M_training_data.size() ) < idx )
    {
        return false;
    }

    return replaceTrainingData( idx,
                                rcsc::Formation::Snapshot( M_ball, M_our_players ),
                                M_symmetric_mode );
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::replaceTrainingData( const int idx,
                                const rcsc::Formation::Snapshot & snapshot,
                                const bool symmetry )
{
    int count = 1;
    for ( std::list< rcsc::Formation::Snapshot >::iterator it = M_training_data.begin();
          it != M_training_data.end();
          ++it, ++count )
    {
        if ( idx == count )
        {
            *it = snapshot;
            M_training_data_changed = true;

            std::cerr << "Replaced data. index = " << idx
                      << std::endl;

            if ( symmetry )
            {
                replaceSymmetricData( snapshot );
            }
            else
            {
                M_ball = snapshot.ball_;
                M_our_players = snapshot.players_;
            }

            updateTriangulation();

            if ( M_formation
                 && ( M_formation->methodName() == rcsc::FormationDT::name()
                      || M_formation->methodName() == rcsc::FormationKNN::name() )
                 )
            {
                train();
            }

            return true;
        }
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::replaceSymmetricData( const rcsc::Formation::Snapshot & snapshot )
{
    if ( snapshot.ball_.absY() < 0.5 )
    {
        return false;
    }

    const rcsc::Vector2D ball( snapshot.ball_.x, - snapshot.ball_.y );

    int index = 1;

    for ( std::list< rcsc::Formation::Snapshot >::iterator it = M_training_data.begin();
          it != M_training_data.end();
          ++it, ++index )
    {
        if ( it->ball_.dist( ball ) < 0.1 )
        {
            rcsc::Formation::Snapshot reversed = snapshot;

            reversed.ball_.y *= -1.0;
            reverseY( reversed.players_ );

            *it = reversed;
            std::cerr << "Replaced symmetric data. index = " << index
                      << std::endl;

            M_ball = reversed.ball_;
            M_our_players = reversed.players_;
            return true;
        }
    }

    return false;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
FEditData::deleteTrainingData( const int idx )
{
    if ( idx <= 0
         || static_cast< int >( M_training_data.size() ) < idx )
    {
        return false;
    }

    bool deleted = false;
    int count = 1;
    for ( std::list< rcsc::Formation::Snapshot >::iterator it = M_training_data.begin();
          it != M_training_data.end();
          ++it, ++count )
    {
        if ( idx == count )
        {
            M_training_data.erase( it );
            M_training_data_changed = true;
            deleted = true;
            break;
        }
    }

    if ( deleted )
    {
        M_current_index = 0;
        updateTriangulation();

        if ( M_formation
             && ( M_formation->methodName() == rcsc::FormationDT::name()
                  || M_formation->methodName() == rcsc::FormationKNN::name() )
             )
        {
            M_formation->train( M_training_data );
            M_modified = true;
        }
    }

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditData::train()
{
    if ( ! M_formation )
    {
        return;
    }

    M_formation->train( M_training_data );

    M_modified = true;

    updatePlayerPosition();
}
