// -*-c++-*-

/*!
	\file feditor_canvas.cpp
	\brief formation editor canvas class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.	If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qt.h>

#include "fedit_canvas.h"

#include "main_window.h"
#include "fedit_data.h"
#include "fedit_config.h"

#include <rcsc/formation/formation_dt.h>
#include <rcsc/formation/formation_knn.h>

#include <rcsc/geom/triangle_2d.h>

#include <algorithm>
#include <iostream>
#include <cassert>

/*-------------------------------------------------------------------*/
/*!

*/
FEditCanvas::FEditCanvas( MainWindow * parent )
    : QWidget( parent, 0, Qt::WNoAutoErase )
    , M_field_scale( 1.0 )
    , M_zoomed( false )
    , M_field_center( 0, 0 )
    , M_focus_point( 0.0, 0.0 )
    , M_enlarged( true )
    , M_show_index( true )
    , M_show_circumcircle( false )
    , M_field_brush( QColor( 31, 160, 31 ), Qt::SolidPattern )
    , M_field_dark_brush( QColor( 15, 143, 15 ), Qt::SolidPattern )
    , M_line_pen( QColor( 255, 255, 255 ), 1, Qt::SolidLine )
      //, M_triangle_pen( QColor( 255, 127, 0 ), 1, Qt::SolidLine )
    , M_triangle_pen( QColor( 255, 0, 0 ), 1, Qt::SolidLine )
      //, M_triangle_pen( QColor( 0, 0, 255 ), 1, Qt::SolidLine )
    , M_triangle_font( "Sans Serif", 10 )
    , M_area_pen( QColor( 127, 127, 127 ), 1, Qt::SolidLine )
    , M_ball_pen( QColor( 255, 255, 255 ), 1, Qt::SolidLine )
    , M_ball_brush( QColor( 255, 255, 255 ), Qt::SolidPattern )
    , M_player_pen( QColor( 0, 0, 0 ), 1, Qt::SolidLine )
    , M_select_pen( Qt::white, 2, Qt::SolidLine )
    , M_left_team_brush( QColor( 255, 215, 0 ), Qt::SolidPattern )
    , M_right_team_brush( QColor( 240, 20, 20 ), Qt::SolidPattern )
    , M_synmetry_brush( QColor( 0, 255, 95 ), Qt::SolidPattern )
    , M_player_font( "Sans Serif", 10 )
{
    assert( parent );

    // need for the MouseMoveEvent
    this->setMouseTracking( true );

    this->setFocusPolicy( QWidget::WheelFocus );

    updateSize();

    M_canvas_pixmap.resize( this->size() );
}

/*-------------------------------------------------------------------*/
/*!

*/
FEditCanvas::~FEditCanvas()
{
    //std::cerr << "delete FEditCanvas" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::updateSize()
{
    M_field_center.setX( this->width() / 2 - scale( focusPoint().x ) );
    M_field_center.setY( this->height() / 2 - scale( focusPoint().y ) );

    const double pitch_l = ( FEditConfig::PITCH_LENGTH
                             + FEditConfig::PITCH_MARGIN * 2.0 );
    const double pitch_w = ( FEditConfig::PITCH_WIDTH
                             + FEditConfig::PITCH_MARGIN * 2.0 );

    if ( ! M_zoomed )
    {
        M_field_scale = this->width() / pitch_l;
        if ( pitch_w * M_field_scale > this->height() )
        {
            M_field_scale = this->height() / pitch_w;
        }

        M_field_scale = std::max( 2.0, M_field_scale );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::paintEvent( QPaintEvent * event )
{
    QWidget::paintEvent( event );

    if ( M_canvas_pixmap.isNull()
         || M_canvas_pixmap.size() != this->size() )
    {
        M_canvas_pixmap.resize( this->size() );
        updateSize();
    }

    QPainter painter;
    painter.begin( &M_canvas_pixmap, this );

    drawField( painter );
    drawTrainingData( painter );
    drawPlayers( painter );
    drawBall( painter );

    painter.end();

    bitBlt( this, 0, 0,
            &M_canvas_pixmap, 0, 0, this->width(), this->height() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::drawField( QPainter & painter )
{
    //--------------------------------------
    // fill background
    painter.fillRect( painter.window(), M_field_brush );

    //--------------------------------------
    drawContainedArea( painter );

    // set screen coordinates of field
    const int left_x   = screenX( - FEditConfig::PITCH_LENGTH * 0.5 );
    const int right_x  = screenX( + FEditConfig::PITCH_LENGTH * 0.5 );
    const int top_y    = screenY( - FEditConfig::PITCH_WIDTH * 0.5 );
    const int bottom_y = screenY( + FEditConfig::PITCH_WIDTH * 0.5 );

    //--------------------------------------
    // draw lines
    painter.setPen( M_line_pen );
    painter.setBrush( Qt::NoBrush );

    // side lines & goal lines
    painter.drawLine( left_x, top_y, right_x, top_y );
    painter.drawLine( right_x, top_y, right_x, bottom_y );
    painter.drawLine( right_x, bottom_y, left_x, bottom_y );
    painter.drawLine( left_x, bottom_y, left_x, top_y );
    // center line
    painter.drawLine( M_field_center.x(), top_y, M_field_center.x(), bottom_y );
    // center circle
    const int center_radius = scale( FEditConfig::CENTER_CIRCLE_R );
    painter.drawEllipse( M_field_center.x() - center_radius,
                         M_field_center.y() - center_radius,
                         center_radius * 2,
                         center_radius * 2 );
    // draw penalty area box
    const int pen_top_y    = screenY( - FEditConfig::PENALTY_AREA_WIDTH / 2 );
    const int pen_bottom_y = screenY( + FEditConfig::PENALTY_AREA_WIDTH / 2 );
    // left penalty box
    int pen_x = screenX( -( FEditConfig::PITCH_LENGTH*0.5
                            - FEditConfig::PENALTY_AREA_LENGTH ) );
    painter.drawLine( left_x, pen_top_y, pen_x, pen_top_y );
    painter.drawLine( pen_x, pen_top_y, pen_x, pen_bottom_y );
    painter.drawLine( pen_x, pen_bottom_y, left_x, pen_bottom_y );
    // right penalty box
    pen_x = screenX( +( FEditConfig::PITCH_LENGTH*0.5
                        - FEditConfig::PENALTY_AREA_LENGTH ) );
    painter.drawLine( right_x, pen_top_y, pen_x, pen_top_y );
    painter.drawLine( pen_x, pen_top_y, pen_x, pen_bottom_y );
    painter.drawLine( pen_x, pen_bottom_y, right_x, pen_bottom_y );
    // draw goal area box
    const int goal_area_y_abs = scale( FEditConfig::GOAL_AREA_WIDTH*0.5 );
    const int goal_area_top_y = M_field_center.y() - goal_area_y_abs;
    const int goal_area_bottom_y = M_field_center.y() + goal_area_y_abs;
    // left goal area
    int goal_area_x = screenX( - FEditConfig::PITCH_LENGTH*0.5
                               + FEditConfig::GOAL_AREA_LENGTH );
    painter.drawLine( left_x, goal_area_top_y, goal_area_x, goal_area_top_y );
    painter.drawLine( goal_area_x, goal_area_top_y, goal_area_x, goal_area_bottom_y );
    painter.drawLine( goal_area_x, goal_area_bottom_y, left_x, goal_area_bottom_y );
    // right goal area
    goal_area_x = screenX( FEditConfig::PITCH_LENGTH*0.5
                           - FEditConfig::GOAL_AREA_LENGTH );
    painter.drawLine( right_x, goal_area_top_y, goal_area_x, goal_area_top_y );
    painter.drawLine( goal_area_x, goal_area_top_y, goal_area_x, goal_area_bottom_y );
    painter.drawLine( goal_area_x, goal_area_bottom_y, right_x, goal_area_bottom_y );

    //--------------------------------------
    // draw goal boxes
    painter.setPen( Qt::black );
    painter.setBrush( Qt::black );
    const int goal_top_y = screenY( - FEditConfig::GOAL_WIDTH*0.5 );
    const int goal_size_x = scale( FEditConfig::GOAL_DEPTH );
    const int goal_size_y = scale( FEditConfig::GOAL_WIDTH );
    // left goal
    painter.drawRect( screenX( - FEditConfig::PITCH_LENGTH*0.5
                               - FEditConfig::GOAL_DEPTH ),
                      goal_top_y,
                      goal_size_x, goal_size_y );
    // right goal
    painter.drawRect( screenX( FEditConfig::PITCH_LENGTH*0.5 ) + 1,
                      goal_top_y,
                      goal_size_x + 1, goal_size_y );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::drawContainedArea( QPainter & painter )
{
    boost::shared_ptr< FEditData > ptr = M_data_ptr.lock();
    if ( ! ptr )
    {
        return ;
    }

    boost::shared_ptr< const rcsc::Formation > f = ptr->formation();
    if ( ! f )
    {
        return;
    }

    if ( f->methodName() == rcsc::FormationDT::name() )
    {
        if ( ptr->triangulation().vertices().size() < 3 )
        {
            return;
        }

        //std::cerr << "triangle size = " << ptr->triangulation().triangleMap().size()
        //          << std::endl;

        // draw polygon
        const rcsc::DelaunayTriangulation::TrianglePtr tri
            = ptr->triangulation().findTriangleContains( ptr->ball() );
        if ( tri )
        {
            painter.setPen( Qt::NoPen );
            painter.setBrush( M_field_dark_brush );

            QPointArray vertices( 3 );
            vertices.setPoint( 0,
                               screenX( tri->vertex( 0 )->pos().x ),
                               screenY( tri->vertex( 0 )->pos().y ) );
            vertices.setPoint( 1,
                               screenX( tri->vertex( 1 )->pos().x ),
                               screenY( tri->vertex( 1 )->pos().y ) );
            vertices.setPoint( 2,
                               screenX( tri->vertex( 2 )->pos().x ),
                               screenY( tri->vertex( 2 )->pos().y ) );

            painter.drawConvexPolygon( vertices );

            // draw center point
            rcsc::Vector2D center
                = rcsc::Triangle2D::centroid( tri->vertex( 0 )->pos(),
                                              tri->vertex( 1 )->pos(),
                                              tri->vertex( 2 )->pos() );


            painter.setPen( Qt::red );
            painter.setBrush( Qt::red );
            painter.drawRect( screenX( center.x ) - 1,
                              screenY( center.y ) - 1,
                              3, 3 );
#if 0
            center = rcsc::Triangle2D::incenter( tri->vertex( 0 )->pos(),
                                                 tri->vertex( 1 )->pos(),
                                                 tri->vertex( 2 )->pos() );
            painter.setPen( Qt::red );
            painter.setBrush( Qt::red );
            painter.drawRect( screenX( center.x ) - 1,
                              screenY( center.y ) - 1,
                              3, 3 );
#endif

            // draw circum circle of this triangle

            if ( M_show_circumcircle )
            {
                int x = screenX( tri->circumcenter().x );
                int y = screenY( tri->circumcenter().y );
                int r = scale( tri->circumradius() );

                painter.setPen( Qt::cyan );
                painter.setBrush( Qt::cyan );
                painter.drawRect( x - 1, y - 1,
                                  3, 3 );
                painter.setBrush( Qt::NoBrush );
                painter.drawEllipse( x - r, y - r,
                                     r * 2, r * 2 );
            }
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::drawTrainingData( QPainter & painter )
{
    boost::shared_ptr< FEditData > ptr = M_data_ptr.lock();
    if ( ! ptr )
    {
        return;
    }

    boost::shared_ptr< const rcsc::Formation > f = ptr->formation();
    if ( ! f )
    {
        return;
    }

    if ( f->methodName() == rcsc::FormationDT::name()
         || f->methodName() == rcsc::FormationKNN::name() )
    {
        // draw delaunay triangulation

        if ( ptr->triangulation().vertices().size() < 3 )
        {
            // too few kernel points
            // no valid triangulation
            return;
        }

        painter.setPen( M_triangle_pen );
        painter.setBrush( Qt::NoBrush );

        const std::map< int, rcsc::DelaunayTriangulation::EdgePtr >::const_iterator edge_end
            = ptr->triangulation().edgeMap().end();
        for ( std::map< int, rcsc::DelaunayTriangulation::EdgePtr >::const_iterator it
                  = ptr->triangulation().edgeMap().begin();
              it != edge_end;
              ++it )
        {
            painter.drawLine( screenX( it->second->vertex( 0 )->pos().x ),
                              screenY( it->second->vertex( 0 )->pos().y ),
                              screenX( it->second->vertex( 1 )->pos().x ),
                              screenY( it->second->vertex( 1 )->pos().y ) );
        }

        painter.setFont( M_player_font );
        painter.setPen( Qt::red );
        //painter.setPen( Qt::black );
        painter.setBackgroundMode( Qt::TransparentMode );

        const int radius = scale( 0.7 );
        int count = 0;
        const std::list< rcsc::Formation::Snapshot >::const_iterator data_end
            = ptr->trainingData().end();
        for ( std::list< rcsc::Formation::Snapshot >::const_iterator it
                  = ptr->trainingData().begin();
              it != data_end;
              ++it )
        {
            int ix = screenX( it->ball_.x );
            int iy = screenY( it->ball_.y );
            //char buf[4];
            //std::snprintf( buf, 4, "%2d", ++count );
            if ( M_show_index )
            {
                painter.drawText( ix + radius, iy - radius,
                                  QString().sprintf( "%2d", ++count ) ); //QString::fromAscii( buf ) );
            }
        }
    }
    else
    // if method type is not a FormationDT
    // only sample ball points are drawn
    {
        painter.setPen( Qt::red );
        //painter.setPen( Qt::black );
        painter.setBrush( Qt::NoBrush );
        painter.setFont( M_player_font );
        //painter.setBackgroundMode( Qt::TransparentMode );

        const int r = scale( 0.5 );

        int count = 0;
        const std::list< rcsc::Formation::Snapshot >::const_iterator data_end
            = ptr->trainingData().end();
        for ( std::list< rcsc::Formation::Snapshot >::const_iterator it
                  = ptr->trainingData().begin();
              it != data_end;
              ++it )
        {
            int x = screenX( it->ball_.x );
            int y = screenY( it->ball_.y );
            //painter.DrawCircle( x, y, r );
            painter.drawLine( x - r, y - r,
                              x + r, y + r );
            painter.drawLine( x - r, y + r,
                              x + r, y - r );
            //char buf[4];
            //std::snprintf( buf, 4, "%2d", ++count );
            if ( M_show_index )
            {
                painter.drawText( x + r, y - r,
                                  QString().sprintf( "%2d", ++count ) );// QString::fromAscii( buf ) );
            }
        }

    }

    if ( f->methodName() == rcsc::FormationKNN::name() )
    {
        const int k = std::min( size_t( 3 ), ptr->trainingData().size() );

        std::vector< rcsc::Vector2D > points( k );
        std::vector< double > dists( k, std::numeric_limits< double >::max() );

        const std::list< rcsc::Formation::Snapshot >::const_iterator end
            = ptr->trainingData().end();
        for ( std::list< rcsc::Formation::Snapshot >::const_iterator it
                  = ptr->trainingData().begin();
              it != end;
              ++it )
        {
            double d2 = it->ball_.dist2( ptr->ball() );
            for ( int i = 0; i < k; ++i )
            {
                if ( dists[i] > d2 )
                {
                    for ( int j = k - 2; j >= i; --j )
                    {
                        dists[j+1] = dists[j];
                        points[j+1] = points[j];
                    }
                    dists[i] = d2;
                    points[i] = it->ball_;
                    break;
                }
            }
        }

        painter.setPen( Qt::blue );
        painter.setBrush( Qt::NoBrush );
        const int r = scale( 1.0 );

        for ( int i = 0; i < k; ++i )
        {
            //std::cerr << "  " << i << " : " << points[i] << std::endl;
            int x = screenX( points[i].x );
            int y = screenY( points[i].y );
            painter.drawEllipse( x - r, y - r, r*2, r*2 );
//             painter.drawLine( x - r, y - r,
//                               x + r, y + r );
//             painter.drawLine( x - r, y + r,
//                               x + r, y - r );
        }
    }

}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::drawPlayers( QPainter & painter )
{
    boost::shared_ptr< FEditData > ptr = M_data_ptr.lock();
    if ( ! ptr )
    {
        return;
    }

    boost::shared_ptr< const rcsc::Formation > f = ptr->formation();
    if ( ! f )
    {
        return;
    }

    const int radius = ( M_enlarged
                         ? scale( 1.1 )
                         : scale( 0.3 ) );
    const int diameter = radius * 2;;

    const int kickable_radius = scale( 1.1 );
    const int kickable_diameter = kickable_radius * 2;

    const int catchable_radius = scale( 2.0 );

    const int selected_radius = radius + 3;

    ////////////////////////////////////////////////////////////
    painter.setFont( M_player_font );
    painter.setBackgroundMode( Qt::TransparentMode );
    ////////////////////////////////////////////////////////////
    // draw right players
#if 0
    {
        painter.setBrush( M_right_team_brush );

        const std::vector< rcsc::Vector2D >::const_iterator
            selection = ( ptr->getSelectType() == FEditData::SELECT_OPP
                          ? ( ptr->getOppPlayersPos().begin()
                              + ptr->getSelectIndex() )
                          : ptr->getOppPlayersPos().end() );
        int unum = 1;
        for ( std::vector< rcsc::Vector2D >::const_iterator
                  it = ptr->getOppPlayersPos().begin();
              it != ptr->getOppPlayersPos().end();
              ++it, ++unum )
        {
            int ix = screenX( it->x );
            int iy = screenY( it->y );
            if ( selection == it )
            {
                painter.setPen( M_select_pen );
                painter.drawEllipse( ix - selected_radius,
                                     iy - selected_radius,
                                     selected_radius * 2,
                                     selected_radius * 2 );
                painter.setPen( M_player_pen );
            }
            else
            {
                painter.drawEllipse( ix - radius,
                                     iy - radius,
                                     diameter,
                                     diameter );
            }
            //char buf[4];
            //std::snprintf( buf, 4, "%2d", unum );
            painter.DrawText( ix + radius, iy - radius,
                              QString().sprintf( "%2d", unum ) );
            //QString::fromAscii( buf ) );

        }
    }
#endif
    ////////////////////////////////////////////////////////////
    // draw left players
    {
        const std::vector< rcsc::Vector2D >::const_iterator
            selection = ( ptr->getSelectType() == FEditData::SELECT_OUR
                          ? ( ptr->ourPlayers().begin()
                              + ptr->getSelectIndex() )
                          : ptr->ourPlayers().end() );
        int unum = 1;
        for ( std::vector< rcsc::Vector2D >::const_iterator
                  it = ptr->ourPlayers().begin();
              it != ptr->ourPlayers().end();
              ++it, ++unum )
        {
            int ix = screenX( it->x );
            int iy = screenY( it->y );
            if ( selection == it )
            {
                painter.setPen( M_select_pen );
                painter.setBrush( f->isSynmetryType( unum )
                                  ? M_synmetry_brush
                                  : M_left_team_brush );
                painter.drawEllipse( ix - selected_radius,
                                     iy - selected_radius,
                                     selected_radius * 2,
                                     selected_radius * 2 );
            }
            else if ( f->isSynmetryType( unum ) )
            {
                painter.setPen( M_player_pen );
                painter.setBrush( M_synmetry_brush );
                painter.drawEllipse( ix - radius,
                                     iy - radius,
                                     diameter,
                                     diameter );
            }
            else
            {
                painter.setPen( M_player_pen );
                painter.setBrush( M_left_team_brush );
                painter.drawEllipse( ix - radius,
                                     iy - radius,
                                     diameter,
                                     diameter );
            }

            bool goalie = ( f->getRoleName( unum ) == "Goalie" );
            if ( ! M_enlarged
                 && ! goalie )
            {
                painter.setBrush( NoBrush );
                painter.drawEllipse( ix - kickable_radius,
                                     iy - kickable_radius,
                                     kickable_diameter,
                                     kickable_diameter );
            }
            else if ( goalie )
            {
                painter.setBrush( NoBrush );
                painter.drawEllipse( ix - catchable_radius,
                                     iy - catchable_radius,
                                     catchable_radius * 2,
                                     catchable_radius * 2 );
            }

            painter.setPen( Qt::white );
            painter.drawText( ix + radius, iy - radius,
                              QString().sprintf( "%2d", unum ) );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::drawBall( QPainter & painter )
{
    boost::shared_ptr< FEditData > ptr = M_data_ptr.lock();
    if ( ! ptr )
    {
        return;
    }

    painter.setPen( M_ball_pen );

    if ( ptr->isBallDraggable() )
    {
        painter.setBrush( M_ball_brush );
    }
    else
    {
        painter.setBrush( Qt::lightGray );
    }

    int radius = 1;
    if ( M_enlarged )
    {
        radius = ( ptr->getSelectType() == FEditData::SELECT_BALL
                   ? scale( 1.0 )
                   : scale( 0.7 ) );
    }
    else
    {
        radius = ( ptr->getSelectType() == FEditData::SELECT_BALL
                   ? scale( 0.2 )
                   : scale( 0.085 ) );
    }


    int ix = screenX( ptr->ball().x );
    int iy = screenY( ptr->ball().y );
    painter.drawEllipse( ix - radius,
                         iy - radius,
                         radius * 2,
                         radius * 2 );

    if ( ! M_enlarged
         && ptr->getSelectType() != FEditData::SELECT_BALL )
    {
        painter.setBrush( Qt::NoBrush );

        int kickable_radius = scale( 1.085 );
        painter.drawEllipse( ix - kickable_radius,
                             iy - kickable_radius,
                             kickable_radius * 2,
                             kickable_radius * 2 );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::setFocusPoint( const QPoint & pos )
{
    M_focus_point.x = fieldX( pos.x() );
    M_focus_point.y = fieldY( pos.y() );

    updateSize();

    this->update();

}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::zoomIn()
{
    M_field_scale *= 1.5;
    M_zoomed = true;
    updateSize();

    this->update();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::zoomOut()
{
    M_field_scale /= 1.5;
    M_zoomed = true;
    updateSize();

    this->update();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::fitSize()
{
    M_zoomed = false;
    M_focus_point.assign( 0.0, 0.0 );
    updateSize();

    this->update();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::toggleEnlarge( bool on )
{
    M_enlarged = on;

    this->update();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::toggleShowIndex( bool on )
{
    M_show_index = on;

    this->update();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::toggleShowCircumcircle( bool on )
{
    M_show_circumcircle = on;

    this->update();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::mouseDoubleClickEvent( QMouseEvent * event )
{
    if ( event->button() == Qt::LeftButton )
    {
        setFocusPoint( event->pos() );
        this->update();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::mousePressEvent( QMouseEvent * event )
{
    if ( event->button() == Qt::LeftButton )
    {
        M_mouse_state[0].pressed( event->pos() );

        if ( event->state() & Qt::ControlButton )
        {

        }
        else
        {
            if ( boost::shared_ptr< FEditData > ptr = M_data_ptr.lock() )
            {
                ptr->selectObject( fieldX( event->pos().x() ),
                                   fieldY( event->pos().y() ) );
                this->update();
            }
        }
    }
    else if ( event->button() == Qt::MidButton )
    {
        M_mouse_state[1].pressed( event->pos() );
    }
    else if ( event->button() == Qt::RightButton )
    {
        M_mouse_state[2].pressed( event->pos() );

        if ( boost::shared_ptr< FEditData > ptr = M_data_ptr.lock() )
        {
            if ( ptr->isBallDraggable() )
            {
                ptr->moveBallTo( fieldX( event->pos().x() ),
                                 fieldY( event->pos().y() ) );
                this->update();
                emit objectMoved();
            }
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::mouseReleaseEvent( QMouseEvent * event )
{
    if ( event->button() == Qt::LeftButton )
    {
        M_mouse_state[0].released();

        if ( boost::shared_ptr< FEditData > ptr = M_data_ptr.lock() )
        {
            ptr->releaseObject( fieldX( event->pos().x() ),
                                fieldY( event->pos().y() ) );
            this->update();
        }
    }
    else if ( event->button() == Qt::MidButton )
    {
        M_mouse_state[1].released();
    }
    else if ( event->button() == Qt::RightButton )
    {
        M_mouse_state[2].released();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FEditCanvas::mouseMoveEvent( QMouseEvent * event )
{
    QPoint point = event->pos();

    double field_x = fieldX( point.x() );
    double field_y = fieldY( point.y() );
    //char buf[32];
    //std::snprintf( buf, 32, "(%.2f %.2f)", field_x, field_y );

    if ( M_mouse_state[0].isDragged() )
    {
        boost::shared_ptr< FEditData > ptr = M_data_ptr.lock();

        if ( event->state() & Qt::ControlButton
             //|| ( ptr
             //&& ptr->getSelectType() == FEditData::NO_SELECT )
             )
        {
            int new_x = screenX( focusPoint().x );
            int new_y = screenY( focusPoint().y );
            new_x -= ( point.x() - M_mouse_state[0].draggedPoint().x() );
            new_y -= ( point.y() - M_mouse_state[0].draggedPoint().y() );
            setFocusPoint( QPoint( new_x, new_y ) );
            this->update();
        }
        else if ( ptr
                  && ptr->getSelectType() != FEditData::NO_SELECT )
        {
            ptr->setObjectDragPoint( field_x, field_y );

            this->update();
            emit objectMoved();
        }
    }

    for ( int i = 0; i < 3; ++i )
    {
        M_mouse_state[i].moved( point );
    }

    //emit pointerMoved( QString::fromAscii( buf ) );
    emit pointerMoved( QString().sprintf( "(%.2f %.2f)", field_x, field_y ) );

}
