// -*-c++-*-

/*!
  \file scheduler.cpp
  \brief league scheduler program source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa Akiyama

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <cstring>
#include <iostream>
#include <fstream>
#include <vector>
#include <list>
#include <set>
#include <utility>
#include <string>
#include <algorithm>

///////////////////////////////////////////////////////////

class TeamNameLoader {
private:
    std::vector< std::string > M_teams;
public:
    TeamNameLoader()
      { }

    explicit
    TeamNameLoader( std::istream & is );

    const
    std::vector< std::string > & teams() const
      {
          return M_teams;
      }

    std::ostream & print( std::ostream & os ) const;
};

///////////////////////////////////////////////////////////

/*---------------------------------------------------------------*/
/*!

*/
TeamNameLoader::TeamNameLoader( std::istream & is )
{
    const std::string valid_chars
        ( "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789._-/" );

    std::string buf;

    while ( std::getline( is, buf ) )
    {
        std::replace( buf.begin(), buf.end(), '\t', ' ' );
        buf.erase( std::remove( buf.begin(), buf.end(), ' ' ),
                   buf.end() );

        if ( buf.empty()
             || buf.find_first_not_of( valid_chars ) != std::string::npos )
        {
            continue;
        }

        M_teams.push_back( buf );
    }

    // remove same string
    std::unique( M_teams.begin(), M_teams.end() );
}

/*---------------------------------------------------------------*/
/*!

*/
std::ostream &
TeamNameLoader::print( std::ostream & os ) const
{
    for ( std::vector< std::string >::const_iterator it = M_teams.begin();
          it != M_teams.end();
          ++it )
    {
        os << *it << std::endl;
    }
    return os;
}


///////////////////////////////////////////////////////////

class Scheduler {
public:
    typedef std::pair< int, int > MatchId;
    typedef std::vector< MatchId > MatchCont;

private:
    int M_total_teams;
    MatchCont M_match_list;

public:
    Scheduler()
        : M_total_teams( 0 )
      { }

    explicit
    Scheduler( int total_teams )
      {
          create( total_teams );
      }

    void create( int total_teams );

    const MatchCont & matchList() const
      {
          return M_match_list;
      }

    std::ostream & print( std::ostream & os ) const;
};

/*---------------------------------------------------------------*/
/*!

*/
void
Scheduler::create( const int total_teams )
{
    M_total_teams = total_teams;
#if 0
    for ( int left = 1; left < total_teams; ++left )
    {
        MatchCont::iterator insert_point = M_match_list.begin();

        for ( int right = left + 1; right <= total_teams; ++right )
        {
            M_match_list.push_back( std::make_pair( left, right ) );
        }
    }
#endif

    const int total_matches = total_teams * (total_teams - 1) / 2;
    for ( int match = 1; match <= total_matches; ++match )
    {
        int k[2];
        k[0] = 1;
        k[1] = total_teams;

        int i = 0;
        int down = 0;

        int m = 1;
        while ( m < match )
        {
            if ( i + k[down] < total_teams - 1 )
            {
                ++i;
            }
            else
            {
                i = 0;
                if ( down == 0 )
                {
                    down = 1;
                    --k[down];
                } else {
                    down = 0;
                    ++k[down];
                }
            }
            ++m;
        }

        int left = i;
        int right = k[down] + i;

        M_match_list.push_back( std::make_pair( left, right ) );
    }
}

/*---------------------------------------------------------------*/
/*!

*/
std::ostream &
Scheduler::print( std::ostream & os ) const
{
    //int count = 0;
    for ( MatchCont::const_iterator it = M_match_list.begin();
          it != M_match_list.end();
          ++it )
    {
        // os << ++count << ": " << it->first << " vs " << it->second << std::endl;
        os << it->first << " vs " << it->second << std::endl;
    }
    return os;
}

///////////////////////////////////////////////////////////

/*---------------------------------------------------------------*/
/*
  Usage:
  $ rclmscheduler < TeamListedFile
*/
void
usage()
{
    std::cerr << "Usage: rclmscheduler < TeamListedFile"
              << std::endl;
}


///////////////////////////////////////////////////////////

int
main( int argc, char** argv )
{
    for ( int i = 1; i < argc; ++i )
    {
        if ( ! std::strcmp( argv[i], "--help" )
             || ! std::strcmp( argv[i], "-h" ) )
        {
            usage();
            return 1;
        }
    }

    TeamNameLoader loader( std::cin );
    //loader.print( std::cout );
    //std::cout << " -------------------------------- " << std::endl;

    if( loader.teams().empty() )
    {
        std::cerr << "Team size is empty!" << std::endl;
        usage();
        return 1;
    }

    Scheduler scheduler( loader.teams().size() );

    //scheduler.print( std::cout );
    //std::cout << " -------------------------------- " << std::endl;

    std::cerr << "The number of teams : " << loader.teams().size() << std::endl;
    std::cerr << "The number of matches : " << scheduler.matchList().size() << std::endl;

    //int count = 0;
    const Scheduler::MatchCont::const_iterator end = scheduler.matchList().end();
    for ( Scheduler::MatchCont::const_iterator it =  scheduler.matchList().begin();
          it != end;
          ++it )
    {
        std::cout
            // << ++count << ": "
            << loader.teams().at( it->first )
            << " vs "
            << loader.teams().at( it->second )
            << std::endl;
    }

    return 0;
}
