// -*-c++-*-

/*!
  \file main_frame.h
  \brief main application frame class Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef SOCCER_WINDOW_MAIN_FRAME_H
#define SOCCER_WINDOW_MAIN_FRAME_H

#include <boost/any.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/shared_ptr.hpp>

#include "event_handler.h"
#include "main_data.h"

class DebugMessageFrame;
class DebugServer;
class DetailDialog;
class FieldCanvas;
class ImageSaveDialog;
class LogPlayer;
class MainToolBar;
class MonitorClient;
class ViewConfigControl;
class ViewConfigDialog;

//! main frame class
class MainFrame
    : public wxFrame {
private:

    EventHandler M_handler;

    //! main data supplier
    MainData M_data;

    //! toolbar window
    MainToolBar * M_main_tbar;

    //! main child screen window
    FieldCanvas * M_field_canvas;

    boost::shared_ptr< ViewConfigControl > M_view_config_ctrl;

    ViewConfigDialog * M_view_config_dlg;

    DetailDialog * M_detail_dlg;

    ImageSaveDialog * M_image_save_dlg;

    //! debug message frame
    DebugMessageFrame * M_debug_message;

    //! monitor connection interface
    boost::shared_ptr< MonitorClient > M_monitor_client;

    //! debug connection interface
    boost::shared_ptr< DebugServer > M_debug_server;

    //! logplayer module
    boost::scoped_ptr< LogPlayer > M_logplayer;

    //! not used
    MainFrame();

public:
    //! constructor
    explicit
    MainFrame( const wxString & title );

    //! destructor
    ~MainFrame();


    template < typename Handler >
    void connect( const int event_id,
                  Handler * handler,
                  void (Handler::*memfun)( const boost::any * ) )
      {
          M_handler.connect( event_id, handler, memfun );
      }

    void disconnect( const int event_id )
      {
          M_handler.disconnect( event_id );
      }

    void handle( const EventMessage & ev )
      {
          M_handler.handle( ev );
      }

private:

    //--------------------------------------------------
    //! load settings from config file
    void loadConfig();
    //! save settings to config file
    void saveConfig();

    //! set event table
    void connectEvent();

    //--------------------------------------------------
    // create window objects

    //! overrided method
    wxToolBar * OnCreateToolBar( long style, wxWindowID id,
                                 const wxString & name );

    //! realize menu bar.
    void createMenuBar();
    //! menu factory
    wxMenu * createMenuFile();
    //! menu factory
    wxMenu * createMenuMonitor();
    //! menu factory
    wxMenu * createMenuView();
    //! menu factory
    wxMenu * createMenuTool();
    //! menu factory
    wxMenu * createMenuHelp();

    /*!
      \brief realize tool bar.
    */
    void createToolBar();

    /*!
      \brief realize status bar.
    */
    void createStatusBar();

    /*!
      \brief realizes all windows
    */
    void createWindows();

    //--------------------------------------------------
    // operations

    /*!
      \brief realize about dialog
     */
    void showAboutDialog();

    /*!
      \brief destron main frame.
     */
    void quit();

    /*!
      \brief toggle menu bar
    */
    void toggleMenuBar();

    /*!
      \brief toggle tool bar
    */
    void toggleToolBar();

    /*!
      \brief toggle status bar
    */
    void toggleStatusBar();

    /*!
      \brief toggle full screen status
     */
    void toggleFullScreen();

    /*!
      \brief toggle maximize status
     */
    void toggleMaximize();

    /*!
      \brief realize file dialog
     */
    void showOpenRCGDialog();

    /*!
      \brief realize dialog
    */
    void showHeteroParamDialog();

    //--------------------------------------------------
    // wx event handlers

    //! called when main frame is closed.
    void handleCloseEvent( wxCloseEvent & event );

    //! called when window size changed
    void handleSizeEvent( wxSizeEvent & event );

    //! called when any command event occured.
    void handleMenuEvent( wxCommandEvent & event );

    //! called when any command event occured.
    void handleCommandEvent( wxCommandEvent & event );

    //--------------------------------------------------
    // event message handlers

    void recvQuit( const boost::any * )
      {
          quit();
      }

    void recvShowAboutDialog( const boost::any * )
      {
          showAboutDialog();
      }

    void recvToggleMenuBar( const boost::any * )
      {
          toggleMenuBar();
      }

    void recvToggleToolBar( const boost::any * )
      {
          toggleToolBar();
      }

    void recvToggleStatusBar( const boost::any * )
      {
          toggleStatusBar();
      }

    void recvToggleFullScreen( const boost::any * )
      {
          toggleFullScreen();
      }

    void recvShowOpenRCGDialog( const boost::any * )
      {
          showOpenRCGDialog();
      }

    //! realize text enty dialog
    void recvShowConnectDialog( const boost::any * );

    void recvShowPlayerTypeDialog( const boost::any * );

    void recvShowDetailDialog( const boost::any * );

    void recvShowImageSaveDialog( const boost::any * );

    void recvShowViewConfigDialog( const boost::any * );

    void recvShowDebugMessageFrame( const boost::any * );

    // monitor client related events
    void recvMonitorConnect( const boost::any * );

    void recvMonitorConnectTo( const boost::any * data );

    void recvMonitorDisconnect( const boost::any * )
      {
          disconnect();
      }

    void recvRequestRestartServer( const boost::any * )
      {
          restartServer();
      }

    void recvRequestLiveMode( const boost::any * );

    void recvReceiveMonitorPacket( const boost::any * );

    // debug server related events
    void recvDebugServerStart( const boost::any * )
      {
          startDebugServer();
      }
    void recvDebugServerStop( const boost::any * );
public:

    void updateView();

    bool loadRCG( const std::string & filepath );

    // create monitor connection
    void connectMonitorClient( const char * hostname );

    void disconnect();

    /*!
      \brief restart rcssserver and connect to it.
      if monitor client already exists, it kills rcssserver and is
      also destructed.
     */
    void restartServer();

    void startDebugServer();

    void updateStatusText( const double & x,
                           const double & y );

    bool isMonitorConnected() const;

    void saveImage( const int start_cycle,
                    const int end_cycle,
                    const std::string & saved_dir,
                    const std::string & name_prefix,
                    const int format_type );
};

#endif
