// -*-c++-*-

/*!
  \file logplayer.cpp
  \brief log player Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif

#include <iostream>
#include <cassert>

#include "id.h"
#include "main_data.h"
#include "main_frame.h"
#include "monitor_view_data.h"
#include "view_holder.h"

#include "logplayer.h"

//! default logplayer timer step (ms).
const int DEFAULT_TIMER_INTERVAL = 100;

/*-------------------------------------------------------------------*/
/*!
  constructor.
*/
LogPlayer::LogPlayer( MainFrame * main_frame,
                      const MainData & data )
    : M_main_frame( main_frame )
    , M_data( data )
    , M_timer( this, SWID_TIMER_LOGPLAYER )
    , M_forward( true )
    , M_live_mode( false )
{
    assert( main_frame );

    //std::cerr << "create Logplayer" << std::endl;
    // connect event handler
    Connect( SWID_TIMER_LOGPLAYER, wxEVT_TIMER,
             wxTimerEventHandler( LogPlayer::handleTimer ) );

    // register to event handler
    M_main_frame->connect( SWID_LOGPLAYER_STOP,
                           this, &LogPlayer::recvLogPlayerStop );
    M_main_frame->connect( SWID_LOGPLAYER_PLAY_BACK,
                           this, &LogPlayer::recvLogPlayerPlayBack );
    M_main_frame->connect( SWID_LOGPLAYER_PLAY_FORWARD,
                           this, &LogPlayer::recvLogPlayerPlayForward );
    M_main_frame->connect( SWID_LOGPLAYER_PLAY_OR_STOP,
                           this, &LogPlayer::recvLogPlayerPlayOrStop );
    M_main_frame->connect( SWID_LOGPLAYER_STEP_BACK,
                           this, &LogPlayer::recvLogPlayerStepBack );
    M_main_frame->connect( SWID_LOGPLAYER_STEP_FORWARD,
                           this, &LogPlayer::recvLogPlayerStepForward );
    M_main_frame->connect( SWID_LOGPLAYER_GOTO_FIRST,
                           this, &LogPlayer::recvLogPlayerGoToFirst );
    M_main_frame->connect( SWID_LOGPLAYER_GOTO_LAST,
                           this, &LogPlayer::recvLogPlayerGoToLast );
    M_main_frame->connect( SWID_LOGPLAYER_GOTO_PREV_SCORE,
                           this, &LogPlayer::recvLogPlayerGoToPrevScore );
    M_main_frame->connect( SWID_LOGPLAYER_GOTO_NEXT_SCORE,
                           this, &LogPlayer::recvLogPlayerGoToNextScore );
    M_main_frame->connect( SWID_LOGPLAYER_DECELERATE,
                           this, &LogPlayer::recvLogPlayerDecelerate );
    M_main_frame->connect( SWID_LOGPLAYER_ACCELERATE,
                           this, &LogPlayer::recvLogPlayerAccelerate );

    M_main_frame->connect( SWID_LOGPLAYER_GOTO_INDEX,
                           this, &LogPlayer::recvLogPlayerGoToIndex );
    M_main_frame->connect( SWID_LOGPLAYER_GOTO_CYCLE,
                           this, &LogPlayer::recvLogPlayerGoToCycle );
}

/*-------------------------------------------------------------------*/
/*!
  timer event handler.
*/
void
LogPlayer::handleTimer( wxTimerEvent & event )
{
    if ( M_forward )
    {
        stepForward();
    }
    else
    {
        stepBack();
    }
}

/*-------------------------------------------------------------------*/
/*!
  play forward one step
*/
void
LogPlayer::stepForward()
{
    M_main_frame->handle( EventMessage( SWID_STEP_VIEW_DATA_FORWARD ) );
}

/*-------------------------------------------------------------------*/
/*!
  play backward
*/
void
LogPlayer::stepBack()
{
    M_main_frame->handle( EventMessage( SWID_STEP_VIEW_DATA_BACK ) );
}

/*-------------------------------------------------------------------*/
/*!
  stop to play
*/
void
LogPlayer::stop()
{
    M_timer.Stop();
}

/*-------------------------------------------------------------------*/
/*!
  start to play.
  First, timer is stopped.
  Second, timer interval is reset to default value.
  Final, start timer.
*/
void
LogPlayer::playForward()
{
    M_forward = true;
    M_timer.Stop();
    M_timer.Start( DEFAULT_TIMER_INTERVAL );
}

/*-------------------------------------------------------------------*/
/*!
  start to play backward.
  First, timer is stopped.
  Second, timer interval is reset to default value.
  Final, start timer.
*/
void
LogPlayer::playBack()
{
    M_forward = false;
    M_timer.Stop();
    M_timer.Start( DEFAULT_TIMER_INTERVAL );
}

/*-------------------------------------------------------------------*/
/*!
  PAC event message handler.
  toggle play or stop
*/
void
LogPlayer::recvLogPlayerPlayOrStop( const boost::any * )
{
    if ( M_timer.IsRunning() )
    {
        stop();
    }
    else
    {
        playForward();
    }
}

/*-------------------------------------------------------------------*/
/*!
  accerelate to play.
*/
void
LogPlayer::recvLogPlayerAccelerate( const boost::any * )
{
    if ( M_timer.IsRunning() )
    {
        if ( M_timer.GetInterval() / 2 >= 10 )
        {
            M_timer.Stop();
            M_timer.Start( M_timer.GetInterval() / 2 );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!
  decerelate to play.
*/
void
LogPlayer::recvLogPlayerDecelerate( const boost::any * )
{
    if ( M_timer.IsRunning() )
    {
        if ( M_timer.GetInterval() * 2 <= 4000 )
        {
            M_timer.Stop();
            M_timer.Start( M_timer.GetInterval() * 2 );
        }
    }
}

/*-------------------------------------------------------------------*/
/*!
  jump to first view data.
*/
void
LogPlayer::recvLogPlayerGoToFirst( const boost::any * )
{
    M_live_mode = false;

    stop();
    M_main_frame->handle( EventMessage( SWID_SET_VIEW_DATA_INDEX_FIRST ) );
}

/*-------------------------------------------------------------------*/
/*!
  jump to last view data.
*/
void
LogPlayer::recvLogPlayerGoToLast( const boost::any * )
{
    M_live_mode = false;

    stop();
    M_main_frame->handle( EventMessage( SWID_SET_VIEW_DATA_INDEX_LAST ) );
}

/*-------------------------------------------------------------------*/
/*!
  jump to specific index.
*/
void
LogPlayer::recvLogPlayerGoToIndex( const boost::any * data )
{
    M_live_mode = false;

    const std::size_t * idx = boost::any_cast< std::size_t >( data );
    if ( ! idx )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " bad any_cast" << std::endl;
        return;
    }

    M_main_frame->handle( EventMessage( SWID_SET_VIEW_DATA_INDEX, *idx ) );
}

/*-------------------------------------------------------------------*/
/*!
  jump to specific cycle.
*/
void
LogPlayer::recvLogPlayerGoToCycle( const boost::any * data )
{
    M_live_mode = false;

    const long * cycle = boost::any_cast< long >( data );
    if ( ! cycle )
    {
        std::cerr << __FILE__ << ":" << __LINE__
                  << " bad any_cast" << std::endl;
        return;
    }

    M_main_frame->handle( EventMessage( SWID_SET_VIEW_DATA_CYCLE,
                                        *cycle ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
LogPlayer::recvLogPlayerGoToPrevScore( const boost::any * )
{
    M_live_mode = false;

    const ViewHolder & holder = M_data.getViewHolder();

    const std::size_t cur_idx = M_data.getViewIndex();
    std::vector< std::size_t >::const_reverse_iterator rend = holder.getScoreChangeIndexes().rend();
    for ( std::vector< std::size_t >::const_reverse_iterator it = holder.getScoreChangeIndexes().rbegin();
          it != rend;
          ++it )
    {
        if ( *it < cur_idx )
        {
            std::size_t new_idx = *it;
            new_idx -= ( new_idx < 50 ? new_idx : 50 );
            M_main_frame->handle( EventMessage( SWID_SET_VIEW_DATA_INDEX, new_idx ) );
            return;
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
LogPlayer::recvLogPlayerGoToNextScore( const boost::any * )
{
    M_live_mode = false;

    const ViewHolder & holder = M_data.getViewHolder();

    const std::size_t cur_idx = M_data.getViewIndex();
    std::vector< std::size_t >::const_iterator end = holder.getScoreChangeIndexes().end();
    for ( std::vector< std::size_t >::const_iterator it = holder.getScoreChangeIndexes().begin();
          it != end;
          ++it )
    {
        if ( 50 < *it && cur_idx < *it - 50 )
        {
            std::size_t new_idx = *it;
            new_idx -= ( new_idx < 50 ? 0 : 50 );
            M_main_frame->handle( EventMessage( SWID_SET_VIEW_DATA_INDEX, new_idx ) );
            return;
        }
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
LogPlayer::setLiveMode()
{
    M_live_mode = true;
    M_timer.Stop();
    M_main_frame->handle( EventMessage( SWID_SET_VIEW_DATA_INDEX_LAST ) );
}
