// -*-c++-*-

/*!
  \file image_save_dialog.cpp
  \brief Image save dialog class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

// For compilers that support precompilation, includes "wx/wx.h".
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

#ifndef WX_PRECOMP
#include <wx/wx.h>
#include <wx/spinctrl.h>
#include <wx/radiobut.h>
#endif

#include <cassert>
#include <cstdlib> // getenv
#include <iostream>

#include "id.h"
#include "main_data.h"
#include "main_frame.h"
#include "view_holder.h"

#include "image_save_dialog.h"

namespace {
wxString format_choices[] = { wxT( "BMP" ),
                              wxT( "PNG" ),
                              wxT( "JPEG" ),
                              wxT( "XPM" ) };
}

/*-------------------------------------------------------------------*/
/*!

*/
ImageSaveDialog::ImageSaveDialog( wxWindow * parent,
                                  MainFrame * main_frame,
                                  const MainData & data )
    : wxDialog( parent,
                SWID_DIALOG_IMAGE_SAVE, wxT( "Image Save" ),
                wxDefaultPosition,
                wxDefaultSize,
                wxCAPTION | wxSYSTEM_MENU | wxCLOSE_BOX )
    , M_main_frame( main_frame )
    , M_data( data )

{
    assert( main_frame );

    createControls();
    connectEvents();

    wxRect cli_rect = ::wxGetClientDisplayRect();
    this->SetSize( 0, cli_rect.GetHeight() - this->GetSize().y - 32,
                   -1, -1 );
}

/*-------------------------------------------------------------------*/
/*!

*/
ImageSaveDialog::~ImageSaveDialog()
{
    //std::cerr << "delete ImageSaveDialog" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
ImageSaveDialog::Show( bool show )
{
    if ( show )
    {
        const std::vector< MonitorViewPtr > & view = M_data.getViewHolder().getMonitorViewCont();
        int min_cycle = 0;
        int max_cycle = 0;
        if ( ! view.empty() )
        {
            min_cycle = static_cast< int >( view.front()->cycle() );
            max_cycle = static_cast< int >( view.back()->cycle() );
        }
        M_start_cycle->SetRange( min_cycle, max_cycle );
        M_end_cycle->SetRange( min_cycle, max_cycle );
    }

    return wxDialog::Show( show );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ImageSaveDialog::connectEvents()
{
    // close frame
    Connect( wxID_ANY, wxEVT_CLOSE_WINDOW,
             wxCloseEventHandler( ImageSaveDialog::handleCloseEvent ) );

    // command event : button
    Connect( SWID_IMAGE_SAVE_SELECT_ALL, wxEVT_COMMAND_BUTTON_CLICKED,
             wxCommandEventHandler( ImageSaveDialog::handleSelectAllButton ) );
    Connect( SWID_IMAGE_SAVE_SELECT_DIR, wxEVT_COMMAND_BUTTON_CLICKED,
             wxCommandEventHandler( ImageSaveDialog::handleSelectDirButton ) );
    Connect( SWID_IMAGE_SAVE_EXECUTE, wxEVT_COMMAND_BUTTON_CLICKED,
             wxCommandEventHandler( ImageSaveDialog::handleExecuteButton ) );

}

/*-------------------------------------------------------------------*/
/*!

*/
void
ImageSaveDialog::createControls()
{
    wxBoxSizer * panel_sizer = new wxBoxSizer( wxVERTICAL );

    panel_sizer->Add( createCycleSelectControls( this ),
                      0, wxLEFT | wxRIGHT, 4 );

    panel_sizer->Add( createFileNameControls( this ),
                      0, wxLEFT | wxRIGHT, 4 );

    panel_sizer->Add( createDirSelectControls( this ),
                      0, wxLEFT | wxRIGHT, 4 );

    panel_sizer->Add( createExecuteControls( this ),
                      0, wxLEFT | wxRIGHT, 4 );

    panel_sizer->SetSizeHints( this );
    this->SetAutoLayout( true );
    this->SetSizer( panel_sizer );
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ImageSaveDialog::createCycleSelectControls( wxWindow * parent )
{
    wxStaticBox * sbox = new wxStaticBox( parent, -1,
                                          wxT( "Cycle Range" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxHORIZONTAL );

    // create input spinctrls & labels
    {
        //wxBoxSizer * sub_sizer = new wxBoxSizer( wxHORIZONTAL );
        const std::vector< MonitorViewPtr > & view = M_data.getViewHolder().getMonitorViewCont();
        int min_cycle = 0;
        int max_cycle = 0;
        if ( ! view.empty() )
        {
            min_cycle = static_cast< int >( view.front()->cycle() );
            max_cycle = static_cast< int >( view.back()->cycle() );
        }

        sizer->Add( new wxStaticText( parent, -1, wxT( "Start: " ) ),
                    0, wxALIGN_CENTER_VERTICAL, 0 );

        M_start_cycle = new wxSpinCtrl( parent, -1 );
        M_start_cycle->SetRange( min_cycle, max_cycle );
        sizer->Add( M_start_cycle,
                    0, wxALIGN_CENTER_VERTICAL, 0 );

        sizer->Add( new wxStaticText( parent, -1, wxT( " - End: " ) ),
                    0, wxALIGN_CENTER_VERTICAL, 0 );

        M_end_cycle = new wxSpinCtrl( parent, -1 );
        M_end_cycle->SetRange( min_cycle, max_cycle );
        sizer->Add( M_end_cycle,
                    0, wxALIGN_CENTER_VERTICAL, 0 );

        M_select_all_btn = new wxButton( parent, SWID_IMAGE_SAVE_SELECT_ALL,
                                         wxT( "Select All" ) );
        sizer->Add( M_select_all_btn,
                    0, wxLEFT | wxALIGN_CENTER_VERTICAL, 5 );

        //sizer->Add( sub_sizer,
        //0, wxLEFT | wxRIGHT, 2 );
    }

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ImageSaveDialog::createFileNameControls( wxWindow * parent )
{
    wxStaticBox * sbox = new wxStaticBox( parent, -1,
                                          wxT( "File" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxHORIZONTAL );

    {
        sizer->Add( new wxStaticText( parent, -1, wxT( "Name Prefix: " ) ),
                    0, wxALIGN_CENTER_VERTICAL, 0 );


        M_name_prefix = new wxTextCtrl( parent, -1,
                                        wxT( "image-" ),
                                        wxDefaultPosition,
                                        wxSize( 100, -1 ) );
        sizer->Add( M_name_prefix,
                    0, wxALIGN_CENTER_VERTICAL, 0 );

        sizer->Add( new wxStaticText( parent, -1, wxT( "Format: " ) ),
                    0, wxALIGN_CENTER_VERTICAL | wxLEFT, 12 );

        M_format_choice = new wxChoice( parent, -1,
                                        wxDefaultPosition, wxDefaultSize,
                                        WXSIZEOF( format_choices ),
                                        format_choices );
        M_format_choice->SetSelection( 0 );
        sizer->Add( M_format_choice,
                    0, wxALIGN_CENTER_VERTICAL | wxLEFT, 5 );
    }

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ImageSaveDialog::createDirSelectControls( wxWindow * parent )
{
    wxStaticBox * sbox = new wxStaticBox( parent, -1,
                                          wxT( "Save Directry" ) );
    wxStaticBoxSizer * sizer = new wxStaticBoxSizer( sbox, wxHORIZONTAL );

    {
        //sizer->Add( new wxStaticText( parent, -1, wxT( "Dir: " ) ),
        //wxALIGN_CENTER_VERTICAL, 0 );

        M_saved_dir = new wxTextCtrl( parent, -1,
                                      ::wxGetCwd(), // current working directory
                                      wxDefaultPosition,
                                      wxSize( 320, -1 ) );
        sizer->Add( M_saved_dir,
                    0, wxALIGN_CENTER_VERTICAL, 0 );

        int width, height;
        this->GetTextExtent( wxT( "..." ), &width, &height );

        M_select_dir_btn = new wxButton( parent, SWID_IMAGE_SAVE_SELECT_DIR,
                                         wxT( "..." ),
                                         wxDefaultPosition,
                                         wxSize( width + 12, -1 ) );
        sizer->Add( M_select_dir_btn,
                    0, wxALIGN_CENTER_VERTICAL, 0 );
    }

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
wxSizer *
ImageSaveDialog::createExecuteControls( wxWindow * parent )
{
    wxBoxSizer * sizer = new wxBoxSizer( wxHORIZONTAL );

    sizer->AddStretchSpacer();

    M_execute_btn = new wxButton( parent, SWID_IMAGE_SAVE_EXECUTE,
                                  wxT( "Create!" ) );
    sizer->Add( M_execute_btn,
                1, wxALL, 6 );

    sizer->AddStretchSpacer();

    return sizer;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ImageSaveDialog::handleCloseEvent( wxCloseEvent & WXUNUSED( event ) )
{
    //this->Destroy();
    this->Hide();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ImageSaveDialog::handleSelectAllButton( wxCommandEvent & WXUNUSED( event ) )
{
    const std::vector< MonitorViewPtr > & view = M_data.getViewHolder().getMonitorViewCont();
    int min_cycle = 0;
    int max_cycle = 0;
    if ( ! view.empty() )
    {
        min_cycle = static_cast< int >( view.front()->cycle() );
        max_cycle = static_cast< int >( view.back()->cycle() );
    }

    M_start_cycle->SetValue( min_cycle );
    M_end_cycle->SetValue( max_cycle );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ImageSaveDialog::handleSelectDirButton( wxCommandEvent & WXUNUSED( event ) )
{
    wxDirDialog dlg( this, // parent parent
                     wxT( "Choose a save directory" ),
                     M_saved_dir->GetValue(),  // default path
                     wxDEFAULT_DIALOG_STYLE );

    if ( dlg.ShowModal() != wxID_OK )
    {
        return;
    }

    M_saved_dir->SetValue( dlg.GetPath() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
ImageSaveDialog::handleExecuteButton( wxCommandEvent & WXUNUSED( event ) )
{
    /*
    std::cout << "Create button pressed" << std::endl;
    std::cout << "  start cycle = " << M_start_cycle->GetValue()
              << "  end cycle = " << M_end_cycle->GetValue()
              << std::endl;
    std::cout << "  name prefix = " << M_name_prefix->GetValue()
              << std::endl;
    std::cout << "  format = " << format_choices[M_format_choice->GetSelection()]
              << std::endl;
    std::cout << "  save dir = " << M_saved_dir->GetValue()
              << std::endl;
    */
    const wxString & format_str = format_choices[M_format_choice->GetSelection()];
    int format_type = wxBITMAP_TYPE_BMP;
    if ( format_str == wxT( "BMP" ) )
    {
        format_type = wxBITMAP_TYPE_BMP;
    }
    else if ( format_str == wxT( "PNG" ) )
    {
        format_type = wxBITMAP_TYPE_PNG;
    }
    else if ( format_str == wxT( "XPM" ) )
    {
        format_type = wxBITMAP_TYPE_XPM;
    }
    else if ( format_str == wxT( "JPEG" ) )
    {
        format_type = wxBITMAP_TYPE_JPEG;
    }

    this->Hide();

    M_main_frame->saveImage( M_start_cycle->GetValue(),
                             M_end_cycle->GetValue(),
#ifdef UNICODE
                             (const char*)M_saved_dir->GetValue().mbc_str(),
                             (const char*)M_name_prefix->GetValue().mbc_str(),
#else
                             (const char*)M_saved_dir->GetValue().c_str(),
                             (const char*)M_name_prefix->GetValue().c_str(),
#endif
                             format_type );
}
