// -*-c++-*-

/*!
  \file field_canvas.cpp
  \brief main field canvas class Source File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.  If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

// for compliers supporting precompling
#include <wx/wxprec.h>

#ifdef __BORLANDC__
#pragma hdrstop
#endif

// for compliers NOT supporting precompling
#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif

#include <iostream>
#include <cassert>
#include <cmath>
#include <vector>
#include <utility>
#include <algorithm>

#include <boost/scoped_ptr.hpp>

//#include <rcsc/timer.h>
#include <rcsc/geom/vector_2d.h>
#include <rcsc/param/player_type.h>
#include <rcsc/param/server_param.h>

#include "id.h"
#include "app_config.h"

#include "monitor_view_data.h"

#include "gdi_config.h"
#include "view_config.h"

#include "ball_painter.h"
#include "ball_trace_painter.h"
#include "debug_painter.h"
#include "field_painter.h"
#include "offside_line_painter.h"
#include "player_control_painter.h"
#include "player_painter.h"
#include "player_trace_painter.h"
#include "score_board_painter.h"
#include "voronoi_diagram_painter.h"

#include "view_holder.h"
#include "main_data.h"
#include "main_frame.h"

#include "field_canvas.h"

/*-------------------------------------------------------------------*/
/*!
  \brief constructor
  \param parent pointer to the parent window
  \param model pointer to the model module
*/
FieldCanvas::FieldCanvas( wxWindow * parent,
                          MainFrame * main_frame,
                          MainData & data )
    : wxWindow( parent, -1, wxDefaultPosition, wxDefaultSize,
                wxSIMPLE_BORDER /*| wxNO_FULL_REPAINT_ON_RESIZE*/ )
    , M_main_frame( main_frame )
    , M_data( data )
    , M_normal_menu( new wxMenu )
    , M_system_menu( new wxMenu )
    , M_monitor_menu( new wxMenu )
{
    assert( parent );
    assert( main_frame );

    createWindows();
    connectEvents();

    wxSize cli_size = this->GetClientSize();
    M_canvas_bmp.Create( cli_size.GetWidth(), cli_size.GetHeight() );
}

/*-------------------------------------------------------------------*/
/*!
  destructor
*/
FieldCanvas::~FieldCanvas()
{
    //std::cerr << "delete FieldCanvas" << std::endl;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::createWindows()
{
    //----------------------------------------------------
    M_normal_menu->Append( SWID_SHOW_OPEN_RCG_DIALOG,
                           _( "Open\tctrl-o" ) );
    M_normal_menu->Append( SWID_MONITOR_CONNECT,
                           _( "Connect\tctrl-c" ) );
    M_normal_menu->Append( SWID_REQUEST_RESTART_SERVER,
                           _( "Start server" ) );
    //----------------------------------------------------
    M_system_menu->Append( SWID_SHOW_OPEN_RCG_DIALOG,
                           _( "Open\tctrl-o" ) );
    M_system_menu->Append( SWID_MONITOR_CONNECT,
                           _( "Connect\tctrl-c" ) );
    M_system_menu->AppendSeparator();
    M_system_menu->Append( SWID_REQUEST_KILL_SERVER,
                           _( "Kill server" ) );
    M_system_menu->Append( SWID_REQUEST_RESTART_SERVER,
                           _( "Restart server" ) );
    //----------------------------------------------------
    M_monitor_menu->Append( SWID_MONITOR_KICKOFF,
                            _( "Kick Off\tctrl-k" ) );
    M_monitor_menu->AppendSeparator();
    M_monitor_menu->Append( SWID_MENU_MONITOR_DROPBALL,
                            _( "Drop Ball" ) );
    M_monitor_menu->Append( SWID_MENU_MONITOR_FREEKICK_LEFT,
                            _( "Free Kick Left" ) );
    M_monitor_menu->Append( SWID_MENU_MONITOR_FREEKICK_RIGHT,
                            _( "Free Kick Right" ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::connectEvents()
{
    //////////////////////////////////////////////////////////////
    // paint event
    Connect( wxID_ANY, wxEVT_PAINT,
             wxPaintEventHandler( FieldCanvas::handlePaint ) );
    // command event : menu
    Connect( wxID_ANY, wxEVT_COMMAND_MENU_SELECTED,
             wxCommandEventHandler( FieldCanvas::handleMenuEvent ) );
    // mouse event
    Connect( wxID_ANY, wxEVT_LEFT_DOWN,
             wxMouseEventHandler( FieldCanvas::handleLeftDown ) );
    Connect( wxID_ANY, wxEVT_LEFT_UP,
             wxMouseEventHandler( FieldCanvas::handleLeftUp ) );
    Connect( wxID_ANY, wxEVT_MIDDLE_DOWN,
             wxMouseEventHandler( FieldCanvas::handleMiddleDown ) );
    Connect( wxID_ANY, wxEVT_MIDDLE_UP,
             wxMouseEventHandler( FieldCanvas::handleMiddleUp ) );
    Connect( wxID_ANY, wxEVT_RIGHT_DOWN,
             wxMouseEventHandler( FieldCanvas::handleRightDown ) );
    Connect( wxID_ANY, wxEVT_RIGHT_UP,
             wxMouseEventHandler( FieldCanvas::handleRightUp ) );
    Connect( wxID_ANY, wxEVT_MOTION,
             wxMouseEventHandler( FieldCanvas::handleMouseMotion ) );

    // key event
    Connect( wxID_ANY, wxEVT_KEY_DOWN,
             wxKeyEventHandler( FieldCanvas::handleKeyDown ) );
    Connect( wxID_ANY, wxEVT_CHAR,
             wxCharEventHandler( FieldCanvas::handleChar ) );
    //////////////////////////////////////////////////////////////
    // register to event handler

    M_main_frame->connect( SWID_DISPLAY_DEBUG_MESSAGE,
                           this, &FieldCanvas::recvDisplayDebugMessage );

    M_main_frame->connect( SWID_MENU_MONITOR_DROPBALL,
                           this, &FieldCanvas::recvMenuMonitorDropBall );
    M_main_frame->connect( SWID_MENU_MONITOR_FREEKICK_LEFT,
                           this, &FieldCanvas::recvMenuMonitorFreeKickLeft );
    M_main_frame->connect( SWID_MENU_MONITOR_FREEKICK_RIGHT,
                           this, &FieldCanvas::recvMenuMonitorFreeKickRight );
}

/*-------------------------------------------------------------------*/
/*!
  event handler for EVT_PAINT.
*/
void
FieldCanvas::handlePaint( wxPaintEvent & WXUNUSED( event ) )
{
    wxPaintDC dc( this );
    draw( dc );
}

/*-------------------------------------------------------------------*/
/*!
  redraw all.
  called from external event handlers.
*/
void
FieldCanvas::draw()
{
    wxClientDC dc( this );
    draw( dc );
}

/*-------------------------------------------------------------------*/
/*!
  draw all data.
  create memory DC, draw the data to it and blit memoryDC to 'dc'.
*/
void
FieldCanvas::draw( wxDC & dc )
{
    // check canvas size
    //const wxSize cur_size = this->GetClientSize();
    const wxSize cur_size = this->GetSize();
    if ( M_canvas_bmp.GetWidth() != cur_size.GetWidth()
         || M_canvas_bmp.GetHeight() != cur_size.GetHeight() )
    {
        //std::cerr << "recreate field canbas bitmap memory" << std::endl;
        //M_measure_bmp.Create( cur_size.GetWidth(), cur_size.GetHeight() );
        //M_measure_bmp.SetMask( new wxMask( M_measure_bmp, *wxBLACK ) );
        M_canvas_bmp.Create( cur_size.GetWidth(), cur_size.GetHeight() );
    }

    M_data.update( cur_size.GetWidth(), cur_size.GetHeight() );

    // create buffer DC
    wxMemoryDC mem_dc;
    mem_dc.SelectObject( M_canvas_bmp );

    drawAll( mem_dc );

    if ( M_mouse_state[2].isDown()
         && M_mouse_state[2].isDragged() )
    {
        drawMouseMeasure( mem_dc );
    }

    // blit buffer DC to 'dc'
    dc.Blit( 0, 0,
             cur_size.GetWidth(), cur_size.GetHeight(),
             &mem_dc, 0, 0 );

    mem_dc.SelectObject( wxNullBitmap );
}


/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::drawAll( wxDC & mem_dc )
{
    // draw view data to buffer DC.
    FieldPainter( M_data.gdi(), M_data.config() ).draw( mem_dc );

    if ( ! M_data.getCurrentViewData() )
    {
        return;
    }

    if ( M_data.config().isShownBallTrace() )
    {
        BallTracePainter( M_data ).draw( mem_dc );
    }

    if ( M_data.config().isShownPlayerTrace() )
    {
        PlayerTracePainter( M_data ).draw( mem_dc );
    }

    if ( M_data.config().isShownPlayers() )
    {
        PlayerPainter( M_data ).draw( mem_dc );
    }

    if ( M_data.config().isShownBall() )
    {
        BallPainter( M_data ).draw( mem_dc );
    }

    if ( M_data.config().isShownOffsideLine() )
    {
        OffsideLinePainter( M_data ).draw( mem_dc );
    }

    if ( M_data.config().isShownPlayers() )
    {
        PlayerControlPainter( M_data ).draw( mem_dc );
    }

    if ( M_data.config().isShownVoronoiDiagram()
         || M_data.config().isShownDelaunayTriangle() )
    {
        VoronoiDiagramPainter( M_data ).draw( mem_dc );
    }

    if ( M_data.config().isShownDebugView() )
    {
        DebugPainter( M_data ).draw( mem_dc );
    }

    if ( M_data.config().isShownScoreBoard() )
    {
        ScoreBoardPainter( M_data ).draw( mem_dc );
    }
}

/*-------------------------------------------------------------------*/
/*!
  draw the line of mouse measure on 'mem_dc'.
*/
void
FieldCanvas::drawMouseMeasure( wxDC & mem_dc )
{
    wxPoint start_point = M_mouse_state[2].getDownPoint();
    wxPoint end_point = M_mouse_state[2].getDragPoint();

    // draw straight line
    mem_dc.SetPen( M_data.gdi().getMeasurePen() );
    mem_dc.SetBrush( *wxTRANSPARENT_BRUSH );
    mem_dc.DrawLine( start_point, end_point );

    mem_dc.SetPen( *wxRED_PEN );
    mem_dc.DrawCircle( start_point, 1 );
    mem_dc.DrawCircle( end_point, 1 );

    rcsc::Vector2D start_real = M_data.config().getFieldPoint( start_point );
    rcsc::Vector2D end_real = M_data.config().getFieldPoint( end_point );

    // draw distance & angle text
    mem_dc.SetFont( M_data.gdi().getMeasureFont() );
    mem_dc.SetTextForeground( M_data.gdi().getMeasureFontColor() );
    mem_dc.SetBackgroundMode( wxTRANSPARENT );

    wxString temp_str;
    temp_str.Printf( wxT( "(%.2f,%.2f)" ), start_real.x, start_real.y );
    mem_dc.DrawText( temp_str,
                     start_point.x,
                     start_point.y );

    if ( std::abs( start_point.x - end_point.x ) < 1
         && std::abs( start_point.y - end_point.y ) < 1 )
    {
        return;
    }

    temp_str.Printf( wxT( "(%.2f,%.2f)" ), end_real.x, end_real.y );
    // get the dimension of the string with current font
    wxCoord string_w, string_h;
    mem_dc.GetTextExtent( temp_str, &string_w, &string_h );
    mem_dc.DrawText( temp_str,
                     end_point.x,
                     end_point.y - string_h );

    mem_dc.SetTextForeground( wxColor( 224, 224, 192 ) );
    rcsc::Vector2D rel( end_real - start_real );
    temp_str.Printf( wxT( "rel(%.2f,%.2f) r%.2f th%.1f" ),
                     rel.x, rel.y, rel.r(), rel.th().degree() );

    mem_dc.DrawText( temp_str,
                     end_point.x,
                     end_point.y - string_h * 2 - 5 );

}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleMenuEvent( wxCommandEvent & event )
{
    //std::cerr << "FieldCanvas::handleMenuEvent" << std::endl;
    M_main_frame->handle( EventMessage( event.GetId() ) );
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleLeftDown( wxMouseEvent & event )
{
    M_mouse_state[0].down( event.GetPosition() );

    if ( event.ControlDown() )
    {
        //std::cerr << "Ctrl down" << std::endl;
        M_main_frame->handle( EventMessage( SWID_CANVAS_FOCUS_POINT,
                                            event.GetPosition() ) );
    }
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleLeftUp( wxMouseEvent & event )
{
    M_mouse_state[0].up();

    if ( M_main_frame->isMonitorConnected() )
    {
        PopupMenu( M_monitor_menu.get(), event.GetPosition() );
    }
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleMiddleDown( wxMouseEvent & event )
{
    M_mouse_state[1].down( event.GetPosition() );
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleMiddleUp( wxMouseEvent & event )
{
    M_mouse_state[1].up();
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleRightDown( wxMouseEvent & event )
{
    M_mouse_state[2].down( event.GetPosition() );
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleRightUp( wxMouseEvent & event )
{
    wxPoint up_point = event.GetPosition();

    if ( M_mouse_state[2].isDragged() )
    {

    }
    else
    {
        if ( M_main_frame->isMonitorConnected() )
        {
            PopupMenu( M_system_menu.get(), up_point );
        }
        else
        {
            PopupMenu( M_normal_menu.get(), up_point );
        }
    }

    M_mouse_state[2].up();
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleMouseMotion( wxMouseEvent & event )
{
    wxPoint motion_point = event.GetPosition();

    for ( int i = 0; i < 3; ++i )
    {
        M_mouse_state[i].motion( motion_point );
    }

    if ( M_mouse_state[2].isDragged() )
    {
        // draw measure
        draw();
    }

    M_main_frame->updateStatusText( M_data.config().getFieldX( motion_point.x ),
                                    M_data.config().getFieldY( motion_point.y ) );
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleChar( wxKeyEvent & event )
{
#if 0
    std::cerr << "handleChar: key code="
              << event.GetKeyCode()
              << std::endl;
#endif

    if ( doHandleKeyCode( event ) )
    {
        return;
    }
    event.Skip();
}

/*-------------------------------------------------------------------*/
/*!
  event handler.
*/
void
FieldCanvas::handleKeyDown( wxKeyEvent & event )
{
#if 0
    std::cerr << "handleKeyDown: key code="
              << event.GetKeyCode()
              << std::endl;
#endif
    if ( doHandleKeyCode( event ) )
    {
        return;
    }
    event.Skip();
}

/*-------------------------------------------------------------------*/
/*!
  internal key code handling method.
*/
bool
FieldCanvas::doHandleKeyCode( wxKeyEvent & event )
{
    //std::cerr << "handleKeyCod: key code="
    //          << event.GetKeyCode()
    //          << std::endl;

    // if Ctrl is down, the key code value of char event of 'a' becomes 1.
    // case 1:

    switch ( event.GetKeyCode() ) {
    case wxT( '1' ):
    case wxT( '2' ):
    case wxT( '3' ):
    case wxT( '4' ):
    case wxT( '5' ):
    case wxT( '6' ):
    case wxT( '7' ):
    case wxT( '8' ):
    case wxT( '9' ):
        {
            int unum = event.GetKeyCode() - wxT( '0' );
            M_main_frame->handle( event.ControlDown()
                                  ? EventMessage( SWID_CANVAS_SELECT_PLAYER, -unum )
                                  : EventMessage( SWID_CANVAS_SELECT_PLAYER, unum ) );
        }
        break;
    case wxT( '0' ):
        M_main_frame->handle( event.ControlDown()
                              ? EventMessage( SWID_CANVAS_SELECT_PLAYER, -10 )
                              : EventMessage( SWID_CANVAS_SELECT_PLAYER, 10 ) );
        break;
    case wxT( '-' ):
        M_main_frame->handle( event.ControlDown()
                              ? EventMessage( SWID_CANVAS_SELECT_PLAYER, -11 )
                              : EventMessage( SWID_CANVAS_SELECT_PLAYER, 11 ) );
        break;
    case wxT( 'a' ):
        M_main_frame->handle( EventMessage( SWID_CANVAS_SELECT_AUTO_ALL ) );
        break;
    case wxT( 'b' ):
        if ( event.ControlDown() )
        {
            M_main_frame->handle( EventMessage( SWID_CANVAS_DRAW_BALL ) );
        }
        else
        {
            M_main_frame->handle( EventMessage( SWID_CANVAS_FOCUS_BALL ) );
        }
        break;
    case wxT( 'c' ):
        if ( event.ControlDown() )
        {
            return false;
        }
        else
        {
            M_main_frame->handle( EventMessage( SWID_CANVAS_DRAW_CONTROL_AREA ) );
        }
        break;
    case wxT( 'e' ):
        M_main_frame->handle( EventMessage( SWID_CANVAS_ENLARGE ) );
        break;
    case wxT( 'f' ):
        M_main_frame->handle( EventMessage( SWID_CANVAS_DRAW_FLAGS ) );
        break;
    case wxT( 'g' ):
        if ( event.ControlDown() )
        {
            M_main_frame->handle( EventMessage( SWID_LOGPLAYER_GOTO_PREV_SCORE ) );
        }
        else
        {
            M_main_frame->handle( EventMessage( SWID_LOGPLAYER_GOTO_NEXT_SCORE ) );
        }
        break;
    case wxT( 'i' ):
        if ( event.ControlDown() )
        {
            return false;
        }
        else
        {
            M_main_frame->handle( EventMessage( SWID_CANVAS_UNZOOM ) );
        }
        break;
    case wxT( 'h' ):
        M_main_frame->handle( EventMessage( SWID_CANVAS_DRAW_HETERO_NUMBER ) );
        break;
    case wxT( 'l' ):
        if ( event.ControlDown() )
        {
            return false;
        }
        else
        {
            M_main_frame->handle( EventMessage( SWID_CANVAS_SELECT_AUTO_LEFT ) );
        }
        break;
    case wxT( 'n' ):
        M_main_frame->handle( EventMessage( SWID_CANVAS_DRAW_PLAYER_NUMBER ) );
        break;
    case wxT( 'o' ):
        if ( event.ControlDown() )
        {
            return false;
        }
        else
        {
            M_main_frame->handle( EventMessage( SWID_CANVAS_DRAW_OFFSIDE_LINE ) );
        }
        break;
    case wxT( 'p' ):
        if ( event.ControlDown() )
        {
            M_main_frame->handle( EventMessage( SWID_CANVAS_DRAW_PLAYER ) );
        }
        else
        {
            M_main_frame->handle( EventMessage( SWID_CANVAS_FOCUS_PLAYER ) );
        }
        break;
    case wxT( 'r' ):
        M_main_frame->handle( EventMessage( SWID_CANVAS_SELECT_AUTO_RIGHT ) );
        break;
    case wxT( 's' ):
        M_main_frame->handle( EventMessage( SWID_CANVAS_DRAW_STAMINA ) );
        break;
    case wxT( 't' ):
        M_main_frame->handle( EventMessage( SWID_CANVAS_DRAW_SCORE_BOARD ) );
        break;
    case wxT( 'u' ):
        M_main_frame->handle( EventMessage( SWID_CANVAS_UNSELECT_PLAYER ) );
        break;
    case wxT( 'v' ):
        M_main_frame->handle( EventMessage( SWID_CANVAS_DRAW_VIEW_CONE ) );
        break;
        //case 1 + ('z' - 'a'): // Ctrl + char event of 'z'
    case wxT( 'x' ):
        M_main_frame->handle( EventMessage( SWID_CANVAS_ZOOMOUT ) );
        break;
    case wxT( 'z' ):
        M_main_frame->handle( EventMessage( SWID_CANVAS_ZOOMIN ) );
        break;
    case WXK_LEFT:
        if ( event.ControlDown() )
        {
            M_main_frame->handle( EventMessage( SWID_LOGPLAYER_DECELERATE ) );
        }
        else
        {
            M_main_frame->handle( EventMessage( SWID_LOGPLAYER_STEP_BACK ) );
        }
        break;
    case WXK_RIGHT:
        if ( event.ControlDown() )
        {
            M_main_frame->handle( EventMessage( SWID_LOGPLAYER_ACCELERATE ) );
        }
        else
        {
            M_main_frame->handle( EventMessage( SWID_LOGPLAYER_STEP_FORWARD ) );
        }
        break;
    case WXK_UP:
        M_main_frame->handle( EventMessage( SWID_LOGPLAYER_PLAY_FORWARD ) );
        break;
    case WXK_DOWN:
        M_main_frame->handle( EventMessage( SWID_LOGPLAYER_STOP ) );
        break;
    case WXK_HOME:
        M_main_frame->handle( EventMessage( SWID_LOGPLAYER_GOTO_FIRST ) );
        break;
    case WXK_END:
        M_main_frame->handle( EventMessage( SWID_LOGPLAYER_GOTO_LAST ) );
        break;
    case WXK_SPACE:
        M_main_frame->handle( EventMessage( SWID_LOGPLAYER_PLAY_OR_STOP ) );
        break;
    default:
        return false;
        break;
    }
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::recvMenuMonitorDropBall( const boost::any * )
{
    M_main_frame->handle( EventMessage( SWID_MONITOR_DROPBALL,
                                        M_data.config().getFieldPoint( M_mouse_state[0].getDownPoint() ) ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::recvMenuMonitorFreeKickLeft( const boost::any * )
{
    M_main_frame->handle( EventMessage( SWID_MONITOR_FREEKICK_LEFT,
                                        M_data.config().getFieldPoint( M_mouse_state[0].getDownPoint() ) ) );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
FieldCanvas::recvMenuMonitorFreeKickRight( const boost::any * )
{
    M_main_frame->handle( EventMessage( SWID_MONITOR_FREEKICK_RIGHT,
                                        M_data.config().getFieldPoint( M_mouse_state[0].getDownPoint() ) ) );
}
