// -*-c++-*-

/*!
	\file feditor_data.h
	\brief formation editor data class Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.	If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef SOCCERWINDOW2_FORMATION_EDITOR_DATA_H
#define SOCCERWINDOW2_FORMATION_EDITOR_DATA_H

#include <list>
#include <boost/shared_ptr.hpp>

#include <rcsc/geom/vector_2d.h>
#include <rcsc/geom/delaunay_triangulation.h>

#include "formation.h"

class FEditorFrame;
class FEditorCanvas;
class FEditorDialog;

///////////////////////////////////////////////////////////////////////
//! formation editor data class
class FEditorData {
public:

    static const int MAX_TRAINING_DATA_SIZE;

    enum SelectType {
        SELECT_BALL,
        SELECT_OUR,
        SELECT_OPP,
        NO_SELECT,
    };

    enum {
        X_DIV = 23,
        Y_DIV = 13,
    };

private:

    //! reference pointer to the main frame
    FEditorFrame * M_frame;

    //! reference pointer to the main view
    FEditorCanvas * M_canvas;

    //! reference pointer to the dialog
    FEditorDialog * M_dialog;

    //! formation training type name
    std::string M_training_type_name;

    //! file path to save the formation parameters
    std::string M_filepath;

    //! file path to save the training data
    std::string M_filepath_train;

    //! flag to check whether any formation paramters are changed
    bool M_modified;

    //! flag to check whether any training data are changed
    bool M_train_data_changed;

    //----------------------------------------
    // field status

    //! current ball position
    rcsc::Vector2D M_ball;

    //! flag to check whether our player is dragged or not.
    std::vector< int > M_our_dragged;

    //! current our players position
    std::vector< rcsc::Vector2D > M_our_players;

    //! current opponent players position
    std::vector< rcsc::Vector2D > M_opp_players;

    //! current triangulation
    rcsc::DelaunayTriangulation M_triangulation;

    //----------------------------------------
    // interaction info

    //! object selection information
    SelectType M_select_type;

    //! selected player index, usualy unum-1
    std::size_t M_select_index;

    //! status flag whether ball is draggable or not
    bool M_ball_draggable;

    //! status flag whether player is auto move mode or not.
    bool M_player_auto_move;


    //! stored training data
    std::list< Formation::Snapshot > M_train_data;

    //----------------------------------------
    //! editingformation parameters
    boost::shared_ptr< Formation > M_formation;

    // not used
    FEditorData()
      { }
public:
		/*!
      \brief data is constructed with main frame
      \param frame pointer to the main frame instance
      \param training_type_name the name of training method
    */
    explicit
		FEditorData( FEditorFrame * frame );

		/*!
      \brief destructor, but nothing to do
     */
		~FEditorData();

    /*!
      \brief set reference to the View
      \param canvas pointer to the canvas instance
     */
    void setCanvas( FEditorCanvas * canvas );

    /*!
      \brief set reference to the View
      \param canvas pointer to the dialog instance.
     */
    void setDialog( FEditorDialog * dialog );

    /*!
      \brief create new formation instance according to the training method name
      \param type_name training method name
      \return new formation parameter
     */
    static
    boost::shared_ptr< Formation >
    create_formation( const std::string & type_name );

private:

    /*!
      \brief init all member variables.
     */
    void init();

    /*!
      \brief print operation log
     */
    void printMessageWithTime( char * msg, ... );

public:

    /*!
      \brief initialize formation param.
      pre-defined roles are assigned and formation parameters are
      randomized.
     */
    void createDefaultParam();

    /*!
      \brief get current ball position
      \return coordinate of ball position
     */
    const
    rcsc::Vector2D & getBall() const
      {
          return M_ball;
      }

    /*!
      \brief get our players' position container
      \return const reference to the position container
     */
    const
    std::vector< rcsc::Vector2D > & getOurPlayers() const
      {
          return M_our_players;
      }

    /*!
      \brief get opponent players' position container
      \return const reference to the position container
     */
    const
    std::vector< rcsc::Vector2D > & getOppPlayersPos() const
      {
          return M_opp_players;
      }

    /*!
      \brief get triangulation data
      \return const reference to the data
     */
    rcsc::DelaunayTriangulation & getTriangulation()
      {
          return M_triangulation;
      }

    /*!
      \brief get triangulation data
      \return const reference to the data
     */
    const
    rcsc::DelaunayTriangulation & getTriangulation() const
      {
          return M_triangulation;
      }

    /*!
      \brief get training data container
      \return const reference to the data container
     */
    const
    std::list< Formation::Snapshot > & getTrainingData() const
      {
          return M_train_data;
      }

    /*!
      \brief get type of current selected object type
      \return object selection type
    */
    SelectType getSelectType() const
      {
          return M_select_type;
      }

    /*!
      \brief get player index if selected object is player.
      \return player index
     */
    std::size_t getSelectIndex() const
      {
          return M_select_index;
      }

    /*!
      \brief get ball draggable status
      \return true if ball is draggable
     */
    bool isBallDraggable() const
      {
          return M_ball_draggable;
      }

    /*!
      \brief get player auto move status
      \return true if player auto move mode is on
     */
    bool isPlayerAutoMove() const
      {
          return M_player_auto_move;
      }

    /*!
      \brief get formation param pointer
      \return pointer to the formation paramter instance
     */
    boost::shared_ptr< Formation > getFormation()
      {
          return M_formation;
      }

    /*!
      \brief get formation param const pointer
      \return const pointer to the formation paramter instance
     */
    boost::shared_ptr< const Formation > formation() const
      {
          return M_formation;
      }

    /*!
      \brief open the specified formation conf file
      \param filepath path to the formation file
      \return status of operation result
     */
    bool open( const std::string & filepath );

    /*!
      \brief open the specified training data file
      \param filepath path to the training data file
      \return status of operation result
     */
    bool openTrainData( const std::string & filepath );

    /*!
      \brief check changed status. if necessary, save notify window is
      realized.
      \return return status of the save notify dialog.
     */
    int saveChanged();

    /*!
      \brief save current formation to the current opened file.
      \return status of operation result
     */
    bool save();

    /*!
      \brief save current formation to the new file.
      \return status of operation result
     */
    bool saveAs();

    /*!
      \brief save current training data to the current opened file.
      \return status of operation result
     */
    bool saveTrainData();

    /*!
      \brief save current training data to the new file.
      \return status of operation result
     */
    bool saveTrainDataAs();

private:

    /*!
      \brief put current formation to the stream.
      \return status of operation result
     */
    bool save( std::ostream & os );

    /*!
      \brief put current training data to the stream.
      \return status of operation result
     */
    bool saveTrainData( std::ostream & os );

    /*!
      \brief realize formation save notify dialog
      \return answer value from message box
     */
    int showSaveNotifyDialog();

    /*!
      \brief realize training data save notify dialog
      \return answer value from message box
     */
    int showTrainDataSaveNotifyDialog();

    /*!
      \brief when data is totally changed, this method is called
      and notify changed status to the other modules.
    */
    void notifyChanged();

    /*!
      \brief update player position depending on the current ball
      position.
     */
    void updatePlayerPosition();

    /*!
      \brief update delaunay triangulation using current training data set
     */
    void updateTriangulation();

public:

    /*!
      \brief when draw info is changed, this method is called
      and notify changed status to the other modules.
     */
    void notifyDraw();

    /*!
      \brief reverse all Y coordinate
     */
    void reverseY();

    /*!
      \brief move the ball to the specified position
      \param x target x
      \param y target y
     */
    void moveBallTo( const double & x,
                     const double & y );

    /*!
      \brief same as moveBallTo()
      \param x target x
      \param y target y
     */
    void dropBallTo( const double & x,
                     const double & y );

    /*!
      \brief select object nearest to the specifiend position
      We can choose a ball, right side players or left side players
      that is not mirrored.
      \param x mouse pointed position x
      \param y mouse pointed position y
     */
    void selectObject( const double & x,
                       const double & y );

    /*!
      \brief unselect(drop) object at (x, y)
      \param x released point x
      \param y released point y
     */
    void releaseObject( const double & x,
                        const double & y );

    /*!
      \brief update dragged object position. object position is
      restricted within pitch.
      \param x new object position x
      \param y new object position y
     */
    void setObjectDragPoint( double x,
                             double y );

    /*!
      \brief toggle ball draggable status
     */
    void toggleBallDraggable();

    /*!
      \brief toggle player auto move mode status
     */
    void togglePlayerAutoMove();

    /*!
      \brief set new formation training method type name
      \param name formation name to be set
     */
    void setTrainingTypeName( const std::string & type_name );

    /*!
      \brief set player's role data. if necessary, new parameter is created.
      \param unum status changed player's uniform number
      \param refered_unum 0 means type becomes CENTER, <0 means type becomes
      SIDE, >0 means type becomes MIRROR
      \param name new role name string
     */
    void updateRoleData( const int unum,
                         const int mirror_unum,
                         const std::string & role_name );

    /*!
      \brief set the ball position but canvas is not updated.
      \param x moved position x
      \param y moved position y
     */
    void setBallPosition( const double & x,
                          const double & y );

    /*!
      \brief set the specified player's position.
      but canvas is not updated.
      \param unum target player's uniform number
      \param x moved position x
      \param y moved position y
     */
    void setPlayerPosition( const int unum,
                            const double & x,
                            const double & y );

    /*!
      \brief record current object's positions as tarining data
     */
    void recordCurrentPosition();

    /*!
      \brief call the specified tarining data.
      \param idx index of target training data
     */
    void focusTrainData( const int idx );

    /*!
      \brief replace the specified tarining data by current field status.
      \param idx index of target training data
     */
    void replaceTrainData( const int idx );

    /*!
      \brief delete the specified tarining data.
      \param idx index of target training data
     */
    void deleteTrainData( const int idx );

    /*!
      \brief perform training using current stored training data set.
     */
    void train();
};

#endif
