// -*-c++-*-

/*!
	\file feditor_canvas.h
	\brief formation editor canvas class Header File.
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2, or (at your option)
 any later version.

 This code is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this code; see the file COPYING.	If not, write to
 the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifndef SOCCERWINDOW2_EDITOR_CANVAS_H
#define SOCCERWINDOW2_EDITOR_CANVAS_H

#include <boost/weak_ptr.hpp>

#include "mouse_state.h"

class FEditorFrame;
class FEditorData;

///////////////////////////////////////////////////////////////////////
//! formation editor canvas class
class FEditorCanvas
		: public wxWindow {
private:

    //! parent frame
    FEditorFrame * M_parent;

    //! weak reference pointer
    boost::weak_ptr< FEditorData > M_data_ptr;

		//! bitmap buffer of draw area.
		wxBitmap M_canvas_bmp;

    //! scaling magnitude
    double M_scale;

    //! field center point on the screen
    wxPoint M_center;

    //! 0: left, 1: middle, 2: right
    MouseState M_mouse_state[3];

    //---------------------------------------------------
    // GDI objects
    wxBrush M_field_brush;
    wxBrush M_field_dark_brush;
    wxPen M_line_pen;
    wxPen M_triangle_pen;
    wxFont M_triangle_font;
    wxPen M_area_pen;
    wxPen M_ball_pen;
    wxBrush M_ball_brush;
    wxPen M_player_pen;
    wxPen M_select_pen;
    wxBrush M_left_team_brush;
    wxBrush M_right_team_brush;
    wxBrush M_mirror_brush;
    wxFont M_player_font;

public:
		//! constructor
		FEditorCanvas( FEditorFrame * parent );

		//! destructor
		~FEditorCanvas();

    void setData( boost::shared_ptr< FEditorData > ptr )
      {
          M_data_ptr = ptr;
      }

		//----------------------------------------------
		/*!
      \brief redraw all data. called from external event handlers.
      \param dc reference to the device context
     */
		void draw();

private:

    /*!
      \brief create child window objects
     */
    void createWindows();

    /*!
      \brief associate event Id with handler method
     */
    void connectEvents();

		//----------------------------------------------
		/*!
      \brief event handler for EVT_PAINT
     */
		void handlePaint( wxPaintEvent & event );

		/*!
      \brief mouse event handler
     */
		void handleLeftDown( wxMouseEvent & event );

		/*!
      \brief mouse event handler
     */
		void handleLeftUp( wxMouseEvent & event );

		/*!
      \brief mouse event handler
     */
		void handleMiddleDown( wxMouseEvent & event );

		/*!
      \brief mouse event handler
     */
		void handleMiddleUp( wxMouseEvent & event );

		/*!
      \brief mouse event handler
     */
		void handleRightDown( wxMouseEvent & event );

		/*!
      \brief mouse event handler
     */
		void handleRightUp( wxMouseEvent & event );

		/*!
      \brief mouse event handler
     */
		void handleMouseMotion( wxMouseEvent & event );

		//----------------------------------------------
		/*!
      \brief redraw all data.
      create memory DC, draw the data to it and blit memoryDC to 'dc'.
      \param dc reference to the device context
     */
		void draw( wxDC & dc );

		/*!
      \brief redraw field
      \param dc reference to the device context
     */
    void drawField( wxDC & dc );

    /*!
      \brief fill ball contained area with dark brush
      \param dc reference to the device context
     */
    void drawContainedArea( wxDC & dc );

		/*!
      \brief redraw training data point with delaunay triangulation.
      \param dc reference to the device context
     */
    void drawTrainingData( wxDC & dc );

		/*!
      \brief redraw players
      \param dc reference to the device context
     */
		void drawPlayers( wxDC & dc );

		/*!
      \brief redraw ball
      \param dc reference to the device context
     */
		void drawBall( wxDC & dc );

		/*!
      \brief update canvas scale using frame size
     */
    void updateSize();

    /*!
      \brief get screen size of len
      \param len calculated length
      \return screen size
     */
    int scale( const double & len ) const
      {
          return static_cast< int >( ::rint( len * M_scale ) );
      }

    /*!
      \brief convert the field x-coordinate to the screen-x coordinate.
      \param x field x-coordinate value
      \return screen x-coordinate value
     */
    int getScreenX( const double & x ) const
      {
          return M_center.x + scale( x );
      }

    /*!
      \brief convert the field y-coordinate to the screen-x coordinate.
      \param y field y-coordinate value
      \return screen y-coordinate value
     */
    int getScreenY( const double & y ) const
      {
          return M_center.y + scale( y );
      }
};

#endif
