// -*-c++-*-

/*!
  \file world_model.cpp
  \brief world model Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <cmath>
#include <cassert>
#include <algorithm>
#include <iterator>

#include <rcsc/math_util.h>
#include <rcsc/soccer_math.h>

#include <rcsc/param/player_param.h>
#include <rcsc/param/server_param.h>

#include "action_effector.h"
#include "body_sensor.h"
#include "visual_sensor.h"
#include "fullstate_sensor.h"
#include "debug_client.h"
#include "intercept_table.h"
#include "localization.h"
#include "logger.h"
#include "penalty_kick_state.h"
#include "player_command.h"

#include "world_model.h"


namespace rcsc {

class WMImpl {
public:

    /*!
      
    */    
    inline
    static
    void create_player_set( rcsc::PlayerCont & players,
                            rcsc::PlayerPtrCont & players_from_self,
                            rcsc::PlayerPtrCont & players_from_ball,
                            const rcsc::Vector2D & self_pos,
                            const rcsc::Vector2D & ball_pos )
        
      {
          const rcsc::PlayerCont::iterator end = players.end();
          for ( rcsc::PlayerCont::iterator it = players.begin();
                it != end;
                ++it )
          {
              it->updateSelfBallRelated( self_pos, ball_pos );
              players_from_self.push_back( &( *it ) );
              players_from_ball.push_back( &( *it ) );
          }
      }

    /*!
      
    */
    inline
    static
    bool check_player_kickable( rcsc::PlayerPtrCont::iterator first,
                                const rcsc::PlayerPtrCont::iterator last,
                                const int ball_count )
      {
          for ( ; first != last; ++first )
          {
              if ( (*first)->posCount() > ball_count )
              {
                  continue;
              }
              
              if ( (*first)->isKickable() )
              {
                  return true;
              }
              
              return false;
          }
          
          return false;
      }

    typedef boost::shared_ptr< rcsc::Localization::PlayerT > LPPtr;

    /*!
      
    */
    inline
    static
    void localize_players( rcsc::VisualSensor::PlayerCont::const_iterator first,
                           const rcsc::VisualSensor::PlayerCont::const_iterator last,
                           const rcsc::Localization * localize,
                           const rcsc::SelfObject & self,
                           std::vector< LPPtr > & localized_players )
      {
          for ( ; first != last; ++first )
          {
              rcsc::Localization::PlayerT * player = new rcsc::Localization::PlayerT();
              
              localize->localizePlayer( *first,
                                        self.face().degree(),
                                        self.faceError(),
                                        self.pos(),
                                        self.vel(),
                                        player );
              
              localized_players.push_back( LPPtr( player ) );
          }
          
      }
};


/////////////////////////////////////////////////////////////////////

//! view cone area
struct ViewArea {
    double width_half_;
    Vector2D origin_;
    double angle_;
    GameTime time_;

    ViewArea( const double & width,
              const Vector2D & origin,
              const double & angle,
              const GameTime & t )
        : width_half_( width * 0.5 )
        , origin_( origin )
        , angle_( angle )
        , time_( t )
      { }

    bool contains( const Vector2D & p,
                   const double & dir_thr,
                   const double & vis_dist2 ) const
      {
          Vector2D rpos( p - origin_ );
          if ( rpos.r2() < vis_dist2 )
          {
              return true;
          }
          AngleDeg angle_diff = rpos.th() - angle_;
          //rpos.rotate( -angle_ );
          //if ( rpos.th().abs() < width_half_ - dir_thr )
          if ( angle_diff.abs() < width_half_ - dir_thr )
          {
              return true;
          }
          return false;
      }
};

/*-------------------------------------------------------------------*/


const double WorldModel::DIR_STEP = 360.0 / static_cast< double >( DIR_CONF_DIVS );

/*-------------------------------------------------------------------*/
/*!

*/
WorldModel::WorldModel()
    : M_localize( static_cast< Localization * >( 0 ) )
    , M_intercept_table( new InterceptTable( *this ) )
    , M_penalty_kick_state( new PenaltyKickState )
    , M_our_side( NEUTRAL )
    , M_time( -1, 0 )
    , M_sense_body_time( -1, 0 )
    , M_see_time( -1, 0 )
    , M_last_set_play_start_time( 0, 0 )
    , M_setplay_count( 0 )
    , M_game_mode()
    , M_self()
    , M_ball()
    , M_offside_line_x( 0.0 )
    , M_defense_line_x( 0.0 )
    , M_exist_kickable_teammate( false )
    , M_exist_kickable_opponent( false )
    , M_heard_receiver_number( 0 )
    , M_heard_receive_pos( 0.0, 0.0 )
    , M_heard_pass_time( -1, 0 )
    , M_heard_goalie_pos( 50.0, 0.0 )
    , M_heard_goalie_time( -1, 0 )
    , M_heard_offside_line_x( 100.0 )
    , M_heard_offside_line_time( -100, 0 )
    , M_heard_defense_line_x( 100.0 )
    , M_heard_defense_line_time( -100, 0 )
    , M_heard_wait_request_time( -100, 0 )
    , M_heard_attack_request_time( -100, 0 )
{
    assert( M_intercept_table );
    assert( M_penalty_kick_state );

    for ( int i = 0; i < DIR_CONF_DIVS; i++ )
    {
        M_dir_count[i] = 1000;
    }

}

/*-------------------------------------------------------------------*/
/*!

*/
WorldModel::~WorldModel()
{
    if ( M_localize )
    {
        delete M_localize;
        M_localize = static_cast< Localization * >( 0 );
    }
    if ( M_intercept_table )
    {
        delete M_intercept_table;
        M_intercept_table = static_cast< InterceptTable * >( 0 );
    }
    if ( M_penalty_kick_state )
    {
        delete M_penalty_kick_state;
        M_penalty_kick_state = static_cast< PenaltyKickState * >( 0 );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
const
InterceptTable *
WorldModel::getInterceptTable() const
{
    assert( M_intercept_table );
    return M_intercept_table;
}

/*-------------------------------------------------------------------*/
/*!

*/
const
PenaltyKickState *
WorldModel::getPenaltyKickState() const
{
    assert( M_penalty_kick_state );
    return M_penalty_kick_state;
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
WorldModel::initTeamInfo( const std::string & teamname,
                          const SideID ourside,
                          const int my_unum,
                          const bool my_goalie )
{
    M_localize = new Localization( ourside );

    if ( ! M_localize )
    {
        return false;
    }

    M_teamname = teamname;
    M_our_side = ourside;
    M_self.init( my_unum, my_goalie );

    PlayerType default_type( ServerParam::i() );

    for ( int i = 0; i < 11; i++ )
    {
        M_teammate_types[i] = Hetero_Default;
        M_opponent_types[i] = Hetero_Default;
    }

    M_self.setPlayerType( default_type );

    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::setTeammatePlayerType( const int unum,
                                   const HeteroID id )
{
    M_teammate_types[unum - 1] = id;

    if ( unum == self().unum() )
    {
        const PlayerType * tmp = PlayerTypeSet::i().get( id );
        if ( ! tmp )
        {
            std::cerr << teamName() << " : " << self().unum()
                      << "WorldModel: Illega player type id??"
                      << " player type param not found, id = "
                      << id << std::endl;
            return;
        }
        M_self.setPlayerType( *tmp );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::setOpponentPlayerType( const int unum,
                                   const HeteroID id )
{
    if ( unum < 1 || 11 < unum )
    {
        std::cerr << teamName() << " : " << self().unum()
                  << "WorldModel:: change_player_type opponent : unum error"
                  << unum << std::endl;
        return;
    }
    M_opponent_types[unum - 1] = id;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::update( const ActionEffector & act,
                    const GameTime & current )
{
    // this function called from update_after_sense
    // or, if player could not get sense_body and update, this is called

    // simplest estimational update
    if ( time() == current )
    {
        std::cerr << teamName() << " : " << self().unum()
                  << current << "internal update called twice ??"
                  << std::endl;
        return;
    }

    M_time = current;

    // playmode is updated in updateJustBeforeDecision

    // update each object
    M_self.update( act, current );
    M_ball.update( act, gameMode(), current );

    dlog.addText( Logger::WORLD,
                  "world.update. internal update. bpos=(%.2f, %.2f)"
                  " brpos=(%.2f, %.2f) bvel=(%.2f, %.2f)",
                  M_ball.pos().x, M_ball.pos().y,
                  M_ball.rpos().x, M_ball.rpos().y,
                  M_ball.vel().x, M_ball.vel().y );

    // clear pointer reference container
    M_teammates_from_self.clear();
    M_opponents_from_self.clear();
    M_teammates_from_ball.clear();
    M_opponents_from_ball.clear();

    // unum known teammates
    std::for_each( M_teammates.begin(),
                   M_teammates.end(),
                   PlayerObject::UpdateOp() );
    M_teammates.remove_if( PlayerObject::IsInvalidOp() );

    // only side known teammates
    std::for_each( M_unknown_teammates.begin(),
                   M_unknown_teammates.end(),
                   PlayerObject::UpdateOp() );
    M_unknown_teammates.remove_if( PlayerObject::IsInvalidOp() );

    // unum known opponents
    std::for_each( M_opponents.begin(),
                   M_opponents.end(),
                   PlayerObject::UpdateOp() );
    M_opponents.remove_if( PlayerObject::IsInvalidOp() );

    // only side known opponents
    std::for_each( M_unknown_opponents.begin(),
                   M_unknown_opponents.end(),
                   PlayerObject::UpdateOp() );
    M_unknown_opponents.remove_if( PlayerObject::IsInvalidOp() );

    // unknown
    std::for_each( M_unknown_players.begin(),
                   M_unknown_players.end(),
                   PlayerObject::UpdateOp() );
    M_unknown_players.remove_if( PlayerObject::IsInvalidOp() );

    for ( int i = 0; i < DIR_CONF_DIVS; i++ )
    {
        M_dir_count[i] += 1;
        //dlog.addText( Logger::WORLD,
        //            "  world.dirConf: %4.0f -> %d",
        //            (double)i * 360.0 / static_cast<double>(DIR_CONF_DIVS) - 180.0,
        //            M_dir_conf[i] );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::updateAfterSense( const BodySensor & sense,
                              const ActionEffector & act,
                              const GameTime & current )
{
    // called just after sense_body

    // if I could not get sense_body & could get see before action decision,
    // this method is called before update(VisualSensor &, GameTime &)

    // if I could not get sense_body & see before action decision,
    // this method is called just before action decision.

    if ( M_sense_body_time == current )
    {
        std::cerr << teamName() << " : " << self().unum()
                  << current
                  << " world.updateAfterSense: called twice"
                  << std::endl;
        dlog.addText( Logger::WORLD,
                      "world.updateAfterSense. called twide" );
        return;
    }

    M_sense_body_time = sense.time();

    dlog.addText( Logger::WORLD,
                  "*************** updateAfterSense ***************" );

    if ( sense.time() == current )
    {
        dlog.addText( Logger::WORLD,
                      "world.updateAfterSense. update self" );
        M_self.updateAfterSense( sense, act, current );
    }

    if ( time() != current )
    {
        dlog.addText( Logger::WORLD,
                      "world.updateAfterSense. call internal update" );
        // internal update
        update( act, current );
        // check collision
        updateCollision();
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::updateCollision()
{
    // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    // called in updateAfterSense()
    // not use visual info

    if ( ! ball().posValid()
         || ! ball().velValid()
         || ! self().posValid()
         || ! self().velValid() )
    {
        return;
    }

    // internally updated positions
    const double self_ball_dist
        = ( ball().pos() - self().pos() ).r();

    if ( ( self().collision()
           && self_ball_dist < ( self().playerType().playerSize()
                                 + ServerParam::i().ballSize()
                                 + 0.1 )
           )
         || ( self_ball_dist
              < ( self().playerType().playerSize()
                  + ServerParam::i().ballSize()
                  - 0.2 ) ) )
    {
        dlog.addText( Logger::WORLD,
                      "world.updateCollision. detected. ball_dist= %.3f",
                      self_ball_dist );

        Vector2D mid = ball().pos() + self().pos();
        mid *= 0.5;

        Vector2D mid2ball = ball().pos() - mid;
        Vector2D mid2self = self().pos() - mid;
        double ave_size = ( ServerParam::i().ballSize()
                            + self().playerType().playerSize() ) * 0.5;
        mid2ball.setLength( ave_size );
        mid2self.setLength( ave_size );

        Vector2D col_pos = mid + mid2ball;
        Vector2D addv = col_pos - ball().pos();
        Vector2D col_rpos = ball().rpos() + addv;
        Vector2D col_vel = ball().vel() * -0.1;

        M_ball.updateByCollision( col_pos, col_rpos, col_vel );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::updateAfterSee( const VisualSensor & see,
                            const BodySensor & sense,
                            const ActionEffector & act,
                            const GameTime & current )
{
    //////////////////////////////////////////////////////////////////
    // check internal update time
    if ( time() != current )
    {
        update( act, current );
    }

    //////////////////////////////////////////////////////////////////
    // check last sight update time
    if ( M_see_time == current )
    {
        std::cerr << teamName() << " : " << self().unum()
                  << current << " updateAfterSee : called twice "
                  << std::endl;
        return;
    }
    //////////////////////////////////////////////////////////////////
    // time update
    M_see_time = current;

    dlog.addText( Logger::WORLD,
                  "*************** updateAfterSee *****************" );

    if ( M_fullstate_time == current )
    {
        dlog.addText( Logger::WORLD,
                      " already updated by fullstate, just update view history" );
        // stored info
        ViewArea varea( self().viewWidth().getWidth(),
                        self().pos(),
                        self().face().degree(),
                        current );

        // update dir accuracy
        updateDirCount( varea );
        return;
    }

    //////////////////////////////////////////////////////////////////
    // self localization
    localizeSelf( see.lines(),
                  see.markers(),
                  see.behindMarkers(),
                  current );

    // correct vel dir using seen my angle & sense_body's speed magnitude
    M_self.updateVelDirAfterSee( sense, current );

    //////////////////////////////////////////////////////////////////
    // ball localization
    localizeBall( see.balls(), current );

    dlog.addText( Logger::WORLD,
                  "  +++++ ball pos=(%.2f, %.2f) err=(%.3f, %.3f)"
                  " rpos=(%.2f, %.2f) rpos_err=(%.3f, %.3f)",
                  ball().pos().x, ball().pos().y,
                  ball().posError().x, ball().posError().y,
                  ball().rpos().x, ball().rpos().y,
                  ball().rposError().x, ball().rposError().y );

    //////////////////////////////////////////////////////////////////
    // player localization & matching

    localizePlayers( see.teammates(),
                     see.unknownTeammates(),
                     see.opponents(),
                     see.unknownOpponents(),
                     see.unknownPlayers(),
                     current );

    //////////////////////////////////////////////////////////////////
    // debug output
    dlog.addText( Logger::WORLD,
                  "  ***** mypos=(%.2f, %.2f) err=(%.3f, %.3f) vel=(%.2f, %.2f)",
                  self().pos().x, self().pos().y,
                  self().posError().x, self().posError().y,
                  self().vel().x, self().vel().y );
    dlog.addText( Logger::WORLD,
                  "  seen players t= %d:  ut= %d:  o= %d:  uo= %d:  u= %d",
                  see.teammates().size(),
                  see.unknownTeammates().size(),
                  see.opponents().size(),
                  see.unknownOpponents().size(),
                  see.unknownPlayers().size() );
    dlog.addText( Logger::WORLD,
                  "  internal players  t= %d:  ut= %d:  o= %d:  uo= %d:  u= %d",
                  M_teammates.size(),
                  M_unknown_teammates.size(),
                  M_opponents.size(),
                  M_unknown_opponents.size(),
                  M_unknown_players.size() );

    //////////////////////////////////////////////////////////////////
    // view cone & ghost check

    // my global position info is successfully updated.
    if ( self().posCount() == 0 )
    {
        // stored info
        ViewArea varea( self().viewWidth().getWidth(),
                        self().pos(),
                        self().face().degree(),
                        current );
        dlog.addText( Logger::WORLD,
                      "world.viewArea. pos=(%.2f, %.2f) angle= %.1f, width_half= %.1f",
                      varea.origin_.x, varea.origin_.y,
                      varea.angle_, varea.width_half_ );

        // check ghost object
        checkGhost( varea );
        // update dir accuracy
        updateDirCount( varea );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::updateAfterFullstate( const FullstateSensor & fullstate,
                                  const ActionEffector & act,
                                  const GameTime & current )
{
    // internal update
    if ( time() != current )
    {
        update( act, current );
    }

    if ( M_fullstate_time == current )
    {
        std::cerr << teamName() << " : " << self().unum()
                  << current << " updateAfterFullstate : called twice "
                  << std::endl;
        return;
    }

    M_fullstate_time = current;

    dlog.addText( Logger::WORLD,
                  "*************** updateAfterFullstate ***************" );

    // update players
    const FullstateSensor::PlayerCont & our_players 
        = ( isOurLeft()
            ? fullstate.leftTeam()
            : fullstate.rightTeam() );
    const FullstateSensor::PlayerCont & opp_players 
        = ( isOurLeft()
            ? fullstate.rightTeam()
            : fullstate.leftTeam() );
    
    // clean unkown players
    M_unknown_teammates.clear();
    M_unknown_opponents.clear();
    M_unknown_players.clear();

    // update self
    for ( FullstateSensor::PlayerCont::const_iterator fp
              = our_players.begin();
          fp != our_players.end();
          ++fp )
    {
        // update self
        if ( fp->unum_ == self().unum() )
        {
            dlog.addText( Logger::WORLD,
                          "FS updated self" );
            M_self.updateAfterFullstate( *fp, act, current );
            break;
        }
    }

    PlayerCont teammates_temp;
    PlayerCont opponents_temp;

    teammates_temp.splice( teammates_temp.end(),
                           M_teammates );
    opponents_temp.splice( opponents_temp.end(),
                           M_opponents );

    // update teammates
    for ( FullstateSensor::PlayerCont::const_iterator fp
              = our_players.begin();
          fp != our_players.end();
          ++fp )
    {
        // update self
        if ( fp->unum_ == self().unum() )
        {
            continue;
        }

        PlayerObject * player = static_cast< PlayerObject * >( 0 );
        for ( PlayerCont::iterator p = teammates_temp.begin();
              p != teammates_temp.end();
              ++p )
        {
            if ( p->unum() == fp->unum_ )
            {
                M_teammates.splice( M_teammates.end(),
                                    teammates_temp,
                                    p );
                player = &(*p);
                break;
            }
        }

        if ( ! player )
        {
            M_teammates.push_back( PlayerObject() );
            player = &(M_teammates.back());
        }

        dlog.addText( Logger::WORLD,
                      "FS updated teammate %d",
                      fp->unum_ );
        player->updateByFullstate( *fp, self().pos(), fullstate.ball().pos_ );
    }

    // update opponents
    for ( FullstateSensor::PlayerCont::const_iterator fp
              = opp_players.begin();
          fp != opp_players.end();
          ++fp )
    {
        PlayerObject * player = static_cast< PlayerObject * >( 0 );
        for ( PlayerCont::iterator p = opponents_temp.begin();
              p != opponents_temp.end();
              ++p )
        {
            if ( p->unum() == fp->unum_ )
            {
                M_opponents.splice( M_opponents.end(),
                                    opponents_temp,
                                    p );
                player = &(*p);
                break;
            }
        }

        if ( ! player )
        {
            M_opponents.push_back( PlayerObject() );
            player = &(M_opponents.back());
        }

        dlog.addText( Logger::WORLD,
                      "FS updated opponent %d",
                      fp->unum_ );
        player->updateByFullstate( *fp, self().pos(), fullstate.ball().pos_ );
    }

    // update ball
    M_ball.updateByFullstate( fullstate.ball().pos_,
                              fullstate.ball().vel_,
                              self().pos() );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::updateGameMode( const GameMode & game_mode,
                            const GameTime & current )
{
    bool pk_mode = game_mode.isPenaltyKickMode();

    if ( ! pk_mode
         && game_mode.type() != GameMode::PlayOn )// not play_on
    {
        // playmode is changed
        if ( gameMode().type() != game_mode.type() )
        {
            if ( game_mode.type() == GameMode::FreeKick_
                 && ( gameMode().type() == GameMode::OffSide_
                      || gameMode().type() == GameMode::BackPass_
                      || gameMode().type() == GameMode::FreeKickFault_
                      || gameMode().type() == GameMode::CatchFault_
                      || gameMode().type() == GameMode::IndFreeKick_
                      )
                 )
            {
                // nothing to do
            }
            else
            {
                M_last_set_play_start_time = current;
                M_setplay_count = 0;
            }
        }
        // this check supports human referee's interaction
        if ( gameMode().type() == game_mode.type()
             && game_mode.type() == GameMode::FreeKick_ )
        {
            M_last_set_play_start_time = current;
            M_setplay_count = 0;
        }
    }

    // substitute new game mode to member variable
    M_game_mode = game_mode;

    // update penalty kick status
    if ( pk_mode )
    {
        M_penalty_kick_state->update( game_mode, ourSide(), current );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::setHeardBallInfo( const Vector2D & ball_pos,
                              const Vector2D & ball_vel,
                              const GameTime & current )
{
    dlog.addText( Logger::WORLD,
                  "world.setHeardBallInfo. pos=(%.3f, %.3f) vel=(%.2f, %.2f)",
                  ball_pos.x, ball_pos.y, ball_vel.x, ball_vel.y );

    if ( M_fullstate_time == current )
    {
        return;
    }

    M_ball.setHeardInfo(ball_pos, ball_vel, current);
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::setHeardPassInfo( const int receiver,
                              const Vector2D & recv_pos,
                              const GameTime & current )
{
    dlog.addText( Logger::WORLD,
                  "world.setHear. heard pass info: receiver= %d, pos=(%.2f, %.2f)",
                  receiver, recv_pos.x, recv_pos.y );

    M_heard_receiver_number = receiver;
    M_heard_receive_pos = recv_pos;
    M_heard_pass_time = current;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::setHeardGoalieInfo( const int goalie_number,
                                const Vector2D & goalie_pos,
                                const GameTime & current )
{
    dlog.addText( Logger::WORLD,
                  "world.setHearGoalie. unum %d. pos=(%.2f, %.2f)",
                  goalie_number, goalie_pos.x, goalie_pos.y );

    M_heard_goalie_pos = goalie_pos;
    M_heard_goalie_time = current;

    if ( M_fullstate_time == current )
    {
        return;
    }

    PlayerObject * goalie = static_cast< PlayerObject * >( 0 );
    PlayerCont::iterator it_end = M_opponents.end();
    for( PlayerCont::iterator it = M_opponents.begin();
         it != it_end;
         ++it )
    {
        if ( it->goalie() )
        {
            goalie = &(*it);
            break;
        }
    }

    if ( ! goalie ) // not found
    {
        // register new object
        M_opponents.push_back( PlayerObject() );
        goalie = &(M_opponents.back());
    }

    goalie->updateByHear( goalie_number, true, goalie_pos );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::setHeardOffsideLineInfo( const double & offside_line_x,
                                     const GameTime & current )
{
    M_heard_offside_line_x = offside_line_x;
    M_heard_offside_line_time = current;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::setHeardDefenseLineInfo( const double & defense_line_x,
                                     const GameTime & current )
{
    M_heard_defense_line_x = defense_line_x;
    M_heard_defense_line_time = current;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::setHeardWaitRequest( const GameTime & current )
{
    M_heard_wait_request_time = current;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::setHeardAttackRequest( const GameTime & current )
{
    M_heard_attack_request_time = current;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::updateJustBeforeDecision( const ActionEffector & act,
                                      const GameTime & current )
{
    if ( time() != current )
    {
        update( act, current );
    }

    if ( M_heard_wait_request_time == current )
    {
        M_setplay_count = 0;
    }
    else
    {
        // always increment
        ++M_setplay_count;
    }

    // update using sharing network information
    M_ball.updateByHeardInfo( current );

    // update positional info concerned with other players
    updateObjectRelation();
    updateOffsideLine();
    updateDefenseLine();

    // update interception table
    M_intercept_table->update();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::setCommandEffect( const ActionEffector & act )
{
    if ( act.getChangeViewCommand() )
    {
        M_self.setViewMode( act.getChangeViewCommand()->getWidth(),
                            act.getChangeViewCommand()->getQuality() );
    }

    if ( act.getPointtoCommand() )
    {
        M_self.setPointto( act.getPointtoPos(),
                           time() );
    }

    const PlayerAttentiontoCommand * attentionto = act.getAttentiontoCommand();
    if ( attentionto )
    {
        if ( attentionto->isOn() )
        {
            if ( attentionto->getSideType() == PlayerAttentiontoCommand::OUR )
            {
                M_self.setAttentionto( ourSide(),
                                       attentionto->getNumber() );
            }
            else
            {
                SideID opp_side = ( isOurLeft()
                                    ? RIGHT
                                    : LEFT );
                M_self.setAttentionto( opp_side,
                                       attentionto->getNumber() );
            }
        }
        else
        {
            // off
            M_self.setAttentionto( NEUTRAL, 0 );
        }
    }

}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::localizeSelf( const VisualSensor::LineCont & lines,
                          const VisualSensor::MarkerCont & markers,
                          const VisualSensor::MarkerCont & behind_markers,
                          const GameTime & current )
{
    double angle_face = -360.0, angle_face_error = 0.0;
    Vector2D my_pos( Vector2D::INVALID );
    Vector2D my_pos_error( 0.0, 0.0 );

    //////////////////////////////////////////////////////////////////
    // localization
    M_localize->localizeSelf( lines, markers, behind_markers,
                              &angle_face, &angle_face_error,
                              &my_pos, &my_pos_error );

    //////////////////////////////////////////////////////////////////
    // set data
    if ( my_pos.valid() )
    {
        M_self.updatePos( my_pos, my_pos_error,
                          angle_face, std::min( angle_face_error, 180.0 ),
                          current );
    }
    else if ( angle_face != -360.0 )
    {
        M_self.updateAngle( angle_face, std::min( angle_face_error, 180.0 ),
                            current );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::localizeBall( const VisualSensor::BallCont & balls,
                          const GameTime & current )
{
    if ( ! self().faceValid() )
    {
        //std::cerr << "localizeBall : my face invalid conf= "
        //          << self().faceCount() << std::endl;
        return;
    }

    //////////////////////////////////////////////////////////////////
    // calc relative info

    Vector2D rpos( Vector2D::INVALID );
    Vector2D rpos_error( 0.0, 0.0 );
    Vector2D rvel( Vector2D::INVALID );
    Vector2D vel_error( 0.0, 0.0 );

    M_localize
        -> localizeBallRelative( balls,
                                 self().face().degree(), self().faceError(),
                                 &rpos, &rpos_error,
                                 &rvel, &vel_error );
    if ( ! rpos.valid() )
    {
        dlog.addText( Logger::WORLD,
                      "world.localizeBall : invalid rpos. cannot calc current seen pos" );
        return;
    }

    //////////////////////////////////////////////////////////////////
    // Case: invalid self localization
    // to estimate ball global position, self localization is required.
    // in this case, we can estimate only relative info
    if ( ! self().posValid() )
    {
        if ( ball().rposCount() == 1
             && ( balls.front().dist_
                  > self().playerType().playerSize() + ServerParam::i().ballSize() + 0.1 )
             && self().getLastMove().valid() )
        {
            Vector2D tvel = ( rpos - ball().rposPrev() ) + self().getLastMove();
            Vector2D tvel_err = rpos_error + self().velError();
            // set only vel
            tvel *= ServerParam::i().ballDecay();
            tvel_err *= ServerParam::i().ballDecay();
            M_ball.updateOnlyVel( tvel, tvel_err );

            dlog.addText( Logger::WORLD,
                          "world.localizeBall : only vel (%.3f %.3f)",
                          tvel.x, tvel.y );
        }

        // set relative pos
        M_ball.updateOnlyRelativePos( rpos, rpos_error );

        dlog.addText( Logger::WORLD,
                      "world.localizeBall : only relative pos (%.3f %.3f)",
                      rpos.x, rpos.y );

        return;
    }

    //////////////////////////////////////////////////////////////////
    // calc global pos & vel using visual

    Vector2D pos = self().pos() + rpos;
    Vector2D pos_error = self().posError() + rpos_error;
    Vector2D gvel( Vector2D::INVALID );

    if ( rvel.valid()
         && self().velValid() )
    {
        gvel = self().vel() + rvel;
        vel_error += self().velError();
    }


    //////////////////////////////////////////////////////////////////
    // calc global velocity using rpos diff (if ball is out of view cone and within vis_dist)

    // collision check.
#if 0
    if ( ( balls.front().dist_
           < self().playerType().playerSize() + ServerParam::i().ballSize() + 0.1 )
         && M_collision_estimated )
    {
        dlog.addText( Logger::WORLD,
                      "world.localizeBall : estimate collision" );
    }
    //else
#endif
    if ( ! gvel.valid() )
    {
        dlog.addText( Logger::WORLD,
                      "world.localizeBall: try to update vel using rpos diff " );
        if ( ball().rposCount() == 1 // we saw the ball at prev cycle, too.
             && balls.front().dist_ < 3.15 // ServerParam::i().visibleDistance()
             && self().velValid()
             && self().getLastMove().valid() )
        {
            Vector2D rpos_diff = rpos - ball().rposPrev();
            gvel = rpos_diff;// - ball().rposPrev();
            gvel += self().getLastMove();
            vel_error = rpos_error + self().velError();
            gvel *= ServerParam::i().ballDecay();
            vel_error *= ServerParam::i().ballDecay();

            if ( ( self().collision()
                   || self().pos().dist( ball().pos() ) < self().playerType().playerSize() + ServerParam::i().ballSize() - 0.2 )
                 && rpos.r() < self().playerType().playerSize() + ServerParam::i().ballSize() + 0.15 )
            {
                gvel *= -0.1;
                dlog.addText( Logger::WORLD,
                              "world.localizeBall: vel update by rpos diff"
                              " Collisition estimated!" );
            }

            dlog.addText( Logger::WORLD,
                          "world.localizeBall: vel update by rpos diff"
                          " (%.3f %.3f) prev_rpos(%.3f %.3f) vel=(%.3f, %3f)"
                          " my_move=(%.3f %.3f)",
                          rpos_diff.x, rpos_diff.y,
                          ball().rposPrev().x, ball().rposPrev().y,
                          gvel.x, gvel.y,
                          self().getLastMove().x,
                          self().getLastMove().y );
        }
    }


    //////////////////////////////////////////////////////////////////
    // set data

    if ( gvel.valid() )
    {
        dlog.addText( Logger::WORLD,
                      "ball.updateAll. p(%.3f %.3f) rel(%.3f %.3f) v(%.3f %.3f)",
                      pos.x, pos.y, rpos.x, rpos.y, gvel.x, gvel.y );
        M_ball.updateAll( pos, pos_error, self().posCount(),
                          rpos, rpos_error,
                          gvel, vel_error );
    }
    else
    {
        dlog.addText( Logger::WORLD,
                      "ball.updatePos. p(%.3f %.3f) rel(%.3f %.3f)",
                      pos.x, pos.y, rpos.x, rpos.y );
        M_ball.updatePos( pos, pos_error, self().posCount(),
                          rpos, rpos_error );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::localizePlayers( const VisualSensor::PlayerCont & teammates,
                             const VisualSensor::PlayerCont & unknown_teammates,
                             const VisualSensor::PlayerCont & opponents,
                             const VisualSensor::PlayerCont & unknown_opponents,
                             const VisualSensor::PlayerCont & unknown_players,
                             const GameTime & current )
{
    if ( ! self().faceValid()
         || ! self().posValid() )
    {
        return;
    }
#if 0
    typedef boost::shared_ptr< Localization::PlayerT > LPPtr;

    std::vector< LPPtr > localized_teammates;
    std::vector< LPPtr > localized_opponents;
    std::vector< LPPtr > localized_unknown_players;
    localized_teammates.reserve( teammates.size() + unknown_teammates.size() );
    localized_opponents.reserve( opponents.size() + unknown_opponents.size() );
    localized_teammates.reserve( unknown_players.size() );

    WMImpl::localize_players( teammates.begin(),
                              teammates.end(),
                              M_localize,
                              self(),
                              localized_teammates );
    WMImpl::localize_players( unknown_teammates.begin(),
                              unknown_teammates.end(),
                              M_localize,
                              self(),
                              localized_teammates );
    WMImpl::localize_players( opponents.begin(),
                              opponents.end(),
                              M_localize,
                              self(),
                              localized_opponents );
    WMImpl::localize_players( unknown_opponents.begin(),
                              unknown_opponents.end(),
                              M_localize,
                              self(),
                              localized_opponents );
    WMImpl::localize_players( unknown_players.begin(),
                              unknown_players.end(),
                              M_localize,
                              self(),
                              localized_unknown_players );
#endif

    ////////////////////////////////////////////////////////////////
    // update policy
    //   for each (seen player objects)
    //       if exist matched player in memory list
    //           -> splice from memory to temporary
    //       else
    //           -> assign new data to temporary list
    //   after loop, copy from temporary to memory again

    // temporary data list
    PlayerCont new_teammates;
    PlayerCont new_unknown_teammates;
    PlayerCont new_opponents;
    PlayerCont new_unknown_opponents;
    PlayerCont new_unknown_players;

#if 0
    for ( std::vector< LPPtr >::iterator it = localized_opponents.begin();
          it != localized_opponents.end();
          ++it )
    {
        checkTeamPlayer( *(*it),
                         ( (*it)->pos_ - self().pos() ).r(),
                         M_opponents,
                         M_unknown_opponents,
                         M_unknown_players,
                         new_opponents,
                         new_unknown_opponents );
    }

    for ( std::vector< LPPtr >::iterator it = localized_unknown_players.begin();
          it != localized_unknown_players.end();
          ++it )
    {
        checkUnknownPlayer( *(*it),
                            ( (*it)->pos_ - self().pos() ).r(),
                            M_teammates,
                            M_unknown_teammates,
                            M_opponents,
                            M_unknown_opponents,
                            M_unknown_players,
                            new_teammates,
                            new_unknown_teammates,
                            new_opponents,
                            new_unknown_opponents,
                            new_unknown_players );
    }

    for ( std::vector< LPPtr >::iterator it = localized_teammates.begin();
          it != localized_teammates.end();
          ++it )
    {
        checkTeamPlayer( *(*it),
                         ( (*it)->pos_ - self().pos() ).r(),
                         M_teammates,
                         M_unknown_teammates,
                         M_unknown_players,
                         new_teammates,
                         new_unknown_teammates );
    }

#else
    const Vector2D MYPOS = self().pos();
    const Vector2D MYVEL = self().vel();
    const double MY_FACE = self().face().degree();
    const double MY_FACE_ERR = self().faceError();

    //////////////////////////////////////////////////////////////////
    // search order is very important !!
    //   If we replace the unknown player to unknown teammate,
    //   it may cause a mistake for pass target selection.

    // current version search order is
    //   [unum opp -> side opp -> unknown -> unum mate -> side mate]

    // if matched, that player is removed from memory list
    // and copy to temporary

    //////////////////////////////////////////////////////////////////
    // localize, matching and splice from memory list to temporary list

    // unum seen opp
    {
        const VisualSensor::PlayerCont::const_iterator o_end = opponents.end();
        for ( VisualSensor::PlayerCont::const_iterator it = opponents.begin();
              it != o_end;
              ++it )
        {
            Localization::PlayerT player;
            // localize
            M_localize->localizePlayer( *it, MY_FACE, MY_FACE_ERR, MYPOS, MYVEL,
                                        &player );
            // matching, splice or create
            dlog.addText( Logger::WORLD,
                          " localized opponent %d pos=(%.2f, %.2f) vel=(%.2f, %.2f)",
                          player.unum_,
                          player.pos_.x, player.pos_.y,
                          player.vel_.x, player.vel_.y );
            checkTeamPlayer( player, it->dist_,
                             M_opponents,
                             M_unknown_opponents,
                             M_unknown_players,
                             new_opponents,
                             new_unknown_opponents );
        }
    }
    // side seen opp
    {
        const VisualSensor::PlayerCont::const_iterator uo_end = unknown_opponents.end();
        for ( VisualSensor::PlayerCont::const_iterator it = unknown_opponents.begin();
              it != uo_end;
              ++it )
        {
            Localization::PlayerT player;
            // localize
            M_localize->localizePlayer( *it, MY_FACE, MY_FACE_ERR, MYPOS, MYVEL,
                                        &player );
            dlog.addText( Logger::WORLD,
                          " localized u-opponent pos=(%.2f, %.2f) vel=(%.2f, %.2f)",
                          player.pos_.x, player.pos_.y,
                          player.vel_.x, player.vel_.y );
            // matching, splice or create
            checkTeamPlayer( player, it->dist_,
                             M_opponents,
                             M_unknown_opponents,
                             M_unknown_players,
                             new_opponents,
                             new_unknown_opponents );
        }
    }
    // unknown player
    {
        const VisualSensor::PlayerCont::const_iterator u_end = unknown_players.end();
        for ( VisualSensor::PlayerCont::const_iterator it = unknown_players.begin();
              it != u_end;
              ++it )
        {
            Localization::PlayerT player;
            // localize
            M_localize->localizePlayer( *it, MY_FACE, MY_FACE_ERR, MYPOS, MYVEL,
                                        &player );
            dlog.addText( Logger::WORLD,
                          " localized unknown pos=(%.2f, %.2f) vel=(%.2f, %.2f)",
                          player.pos_.x, player.pos_.y,
                          player.vel_.x, player.vel_.y );
            // matching, splice or create
            checkUnknownPlayer( player,
                                it->dist_,
                                M_teammates,
                                M_unknown_teammates,
                                M_opponents,
                                M_unknown_opponents,
                                M_unknown_players,
                                new_teammates,
                                new_unknown_teammates,
                                new_opponents,
                                new_unknown_opponents,
                                new_unknown_players );
        }
    }
    // unum seen mate
    {
        const VisualSensor::PlayerCont::const_iterator t_end = teammates.end();
        for ( VisualSensor::PlayerCont::const_iterator it = teammates.begin();
              it != t_end;
              ++it )
        {
            Localization::PlayerT player;
            // localize
            M_localize->localizePlayer( *it, MY_FACE, MY_FACE_ERR, MYPOS, MYVEL,
                                        &player );
            dlog.addText( Logger::WORLD,
                          " localized teammate %d pos=(%.2f, %.2f) vel=(%.2f, %.2f)",
                          player.unum_,
                          player.pos_.x, player.pos_.y,
                          player.vel_.x, player.vel_.y );
            // matching, splice or create
            checkTeamPlayer( player, it->dist_,
                             M_teammates,
                             M_unknown_teammates,
                             M_unknown_players,
                             new_teammates,
                             new_unknown_teammates );
        }
    }
    // side seen mate
    {
        const VisualSensor::PlayerCont::const_iterator ut_end = unknown_teammates.end();
        for ( VisualSensor::PlayerCont::const_iterator it = unknown_teammates.begin();
              it != ut_end;
              ++it )
        {
            Localization::PlayerT player;
            // localize
            M_localize->localizePlayer( *it, MY_FACE, MY_FACE_ERR, MYPOS, MYVEL,
                                        &player );
            dlog.addText( Logger::WORLD,
                          " localized u-teammate pos=(%.2f, %.2f) vel=(%.2f, %.2f)",
                          player.pos_.x, player.pos_.y,
                          player.vel_.x, player.vel_.y );
            // matching, splice or create
            checkTeamPlayer( player, it->dist_,
                             M_teammates,
                             M_unknown_teammates,
                             M_unknown_players,
                             new_teammates,
                             new_unknown_teammates );
        }
    }
#endif

    //////////////////////////////////////////////////////////////////
    // splice temporary seen players to memory list
    // temporary lists are cleared
    M_teammates.splice( M_teammates.end(),
                        new_teammates );
    M_unknown_teammates.splice( M_unknown_teammates.end(),
                                new_unknown_teammates );
    M_opponents.splice( M_opponents.end(),
                        new_opponents );
    M_unknown_opponents.splice( M_unknown_opponents.end(),
                                new_unknown_opponents );
    M_unknown_players.splice( M_unknown_players.end(),
                              new_unknown_players );

    //////////////////////////////////////////////////////////////////
    // create team member pointer vector for sort

    PlayerPtrCont all_teammates_ptr;
    PlayerPtrCont all_opponents_ptr;

    {
        const PlayerCont::iterator end = M_teammates.end();
        for ( PlayerCont::iterator it = M_teammates.begin();
              it != end;
              ++it )
        {
            all_teammates_ptr.push_back( &( *it ) );
        }
    }
    {
        const PlayerCont::iterator end = M_unknown_teammates.end();
        for ( PlayerCont::iterator it = M_unknown_teammates.begin();
              it != end;
              ++it )
        {
            all_teammates_ptr.push_back( &( *it ) );
        }
    }
    {
        const PlayerCont::iterator end = M_opponents.end();
        for ( PlayerCont::iterator it = M_opponents.begin();
              it != end;
              ++it )
        {
            all_opponents_ptr.push_back( &( *it ) );
        }
    }
    {
        const PlayerCont::iterator end = M_unknown_opponents.end();
        for ( PlayerCont::iterator it = M_unknown_opponents.begin();
              it != end;
              ++it )
        {
            all_opponents_ptr.push_back( &( *it ) );
        }
    }


    /////////////////////////////////////////////////////////////////
    // sort by confidence
    std::sort( all_teammates_ptr.begin(), all_teammates_ptr.end(),
               PlayerObject::PtrConfCmp() );
    std::sort( all_opponents_ptr.begin(), all_opponents_ptr.end(),
               PlayerObject::PtrConfCmp() );
    M_unknown_players.sort( PlayerObject::ConfCmp() );


    //////////////////////////////////////////////////////////////////
    // check the number of players
    // if overflow is detected, player is removed based on confidence value

    // remove from teammates
    PlayerPtrCont::size_type mate_count = all_teammates_ptr.size();
    while ( mate_count > 11 - 1 )
    {
        // reset least confidence value player
        dlog.addText( Logger::WORLD,
                      "world.localizePlayers. erase overflow teammate pos=(%.2f, %.2f)",
                      all_teammates_ptr.back()->pos().x,
                      all_teammates_ptr.back()->pos().y );
        all_teammates_ptr.back()->forget();
        all_teammates_ptr.pop_back();
        mate_count--;
    }

    // remove from not-teammates
    PlayerPtrCont::size_type opp_count = all_opponents_ptr.size();
    while ( opp_count > 11 )
    {
        // reset least confidence value player
        dlog.addText( Logger::WORLD,
                      "world.localizePlayers. erase overflow opponent pos=(%.2f, %.2f)",
                      all_opponents_ptr.back()->pos().x,
                      all_opponents_ptr.back()->pos().y );
        all_opponents_ptr.back()->forget();
        all_opponents_ptr.pop_back();
        opp_count--;
    }

    // remove from unknown players
    PlayerCont::size_type n_size_unknown = M_unknown_players.size();
    size_t n_size_total
        = static_cast< size_t >( n_size_unknown )
        + static_cast< size_t >( mate_count )
        + static_cast< size_t >( opp_count );
    while ( n_size_unknown > 0
            && n_size_total > 11 * 2 - 1 )
    {
        dlog.addText( Logger::WORLD,
                      "world.localizePlayers."
                      " erase over flow unknown player. pos=(%.2f, %.2f)",
                      M_unknown_players.back().pos().x,
                      M_unknown_players.back().pos().y );
        if ( M_unknown_players.back().posCount() == 0 )
        {
            // not remove !!!
            break;
        }
        // remove least confidence value player
        M_unknown_players.pop_back();
        n_size_unknown--;
        n_size_total--;
    }


    //////////////////////////////////////////////////////////////////
    // if overflow is detected, instance player must be forget.
    // that player must be removed from memory list.

    // check invalid player
    // if exist, that player is removed from instance list
    M_teammates.remove_if( PlayerObject::IsInvalidOp() );
    M_unknown_teammates.remove_if( PlayerObject::IsInvalidOp() );
    M_opponents.remove_if( PlayerObject::IsInvalidOp() );
    M_unknown_opponents.remove_if( PlayerObject::IsInvalidOp() );

    //////////////////////////////////////////////////////////////////
    // it is not necessary to check the all unknown list
    // because invalid unknown player is already removed.


    //////////////////////////////////////////////////////////////////
    // ghost check is done in checkGhost()
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::checkTeamPlayer( const Localization::PlayerT & player,
                             const double & seen_dist,
                             PlayerCont & old_unum_known,
                             PlayerCont & old_side_known,
                             PlayerCont & old_unknown_players,
                             PlayerCont & new_unum_known,
                             PlayerCont & new_side_known )
{
    // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! //
    //  if matched player is found, that player is removed from old list
    //  and updated data is splice to new container
    // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! //

    static const
        double PLAYER_MAX_MOVE_RADIUS_INC
        = ServerParam::i().defaultPlayerSpeedMax() * 1.1;

    //////////////////////////////////////////////////////////////////
    // pre check
    // unum is seen.
    if ( player.unum_ != -1 )
    {
        // search from old unum known players
        const PlayerCont::iterator end = old_unum_known.end();
        for ( PlayerCont::iterator mem = old_unum_known.begin();
              mem != end;
              ++mem )
        {
            if ( mem->unum() == player.unum_ )
            {
                mem->updateBySee( player ); // update using local vision
                new_unum_known.splice( new_unum_known.end(),
                                       old_unum_known,
                                       mem );
                return; // success to match, exit this function
            }
        }
    }

    //////////////////////////////////////////////////////////////////
    // find nearest player

    double mindist2 = 900.0; // Magic Number. consider only within 30m radius
    PlayerCont::iterator candidate = old_unknown_players.end();
    PlayerCont * target_list = static_cast< PlayerCont * >( 0 );

    //////////////////////////////////////////////////////////////////
    // unum is not seen.
    if ( player.unum_ == -1 )
    {
        // search from old unum known list
        const PlayerCont::iterator end = old_unum_known.end();
        for ( PlayerCont::iterator mem = old_unum_known.begin();
              mem != end;
              ++mem )
        {
            double d2 = ( player.pos_ - mem->pos() ).r2();
            if ( d2 < mindist2 )
            {
                mindist2 = d2;
                candidate = mem;
                target_list = &old_unum_known;
            }
        }
    }

    //////////////////////////////////////////////////////////////////
    // search from old only side known
    {
        const PlayerCont::iterator end = old_side_known.end();
        for ( PlayerCont::iterator mem = old_side_known.begin();
              mem != end;
              ++mem )
        {
            double d2 = ( player.pos_ - mem->pos() ).r2();
            if ( d2 < mindist2 )
            {
                mindist2 = d2;
                candidate = mem;
                target_list = &old_side_known;
            }
        }
    }

    //////////////////////////////////////////////////////////////////
    // search from all unknown
    {
        const PlayerCont::iterator end = old_unknown_players.end();
        for ( PlayerCont::iterator mem = old_unknown_players.begin();
              mem != end;
              ++mem )
        {
            double d2 = ( player.pos_ - mem->pos() ).r2();
            if ( d2 < mindist2 )
            {
                mindist2 = d2;
                candidate = mem;
                target_list = &old_unknown_players;
            }
        }
    }

    //////////////////////////////////////////////////////////////////
    // check player movable radius,
    if ( target_list
         && mindist2 < 10.0 * 10.0
         && ( std::sqrt( mindist2 )
              < ( static_cast< double >( candidate->posCount() ) * PLAYER_MAX_MOVE_RADIUS_INC
                  + unquantize_error( seen_dist, ServerParam::i().distQuantizeStep() )
                  )
              )
         )
    {
        // update & splice to new list
        candidate->updateBySee( player );

        if ( candidate->unum() != -1 )
        {
            new_unum_known.splice( new_unum_known.end(),
                                   *target_list,
                                   candidate );
        }
        else
        {
            new_side_known.splice( new_side_known.end(),
                                   *target_list,
                                   candidate );
        }

        return;
    }

    //////////////////////////////////////////////////////////////////
    // generate new player
    if ( player.unum_ != -1 )
    {
        dlog.addText( Logger::WORLD,
                      "world.checkTeamPlayer. unmatch. min_dist= %.2f"
                      "  generate new unum_seen_player pos=(%.2f, %.2f)",
                      std::sqrt( mindist2 ),
                      player.pos_.x, player.pos_.y );
        new_unum_known.push_back( PlayerObject( player ) ); // push to new UNUM KNOWN
    }
    else
    {
        dlog.addText( Logger::WORLD,
                      "world.checkTeamPlayer. unmatch. min_dist= %.2f"
                      "  generate new unknown_player pos=(%.2f, %.2f)",
                      std::sqrt( mindist2 ),
                      player.pos_.x, player.pos_.y );
        new_side_known.push_back( PlayerObject( player ) ); // push to new SIDE KNOWN
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::checkUnknownPlayer( const Localization::PlayerT & player,
                                const double & seen_dist,
                                PlayerCont & old_teammates,
                                PlayerCont & old_unknown_teammates,
                                PlayerCont & old_opponents,
                                PlayerCont & old_unknown_opponents,
                                PlayerCont & old_unknown_players,
                                PlayerCont & new_teammates,
                                PlayerCont & new_unknown_teammates,
                                PlayerCont & new_opponents,
                                PlayerCont & new_unknown_opponents,
                                PlayerCont & new_unknown_players )
{
    // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! //
    //  if matched player is found, that player is removed from old list
    //  and updated data is splice to new container
    // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!! //

    static const
        double PLAYER_MAX_MOVE_RADIUS_INC
        = ServerParam::i().defaultPlayerSpeedMax() * 1.1;

    // matching start
    // search closest one

    //int found_id = 0;
    double mindist2 = 25.0; //900.0;
    PlayerCont::iterator candidate = old_unknown_players.end();
    PlayerCont * new_list = static_cast< PlayerCont * >( 0 );
    PlayerCont * old_list = static_cast< PlayerCont * >( 0 );

    //////////////////////////////////////////////////////////////////
    // search from old unum known opp
    {
        const PlayerCont::iterator end = old_opponents.end();
        for ( PlayerCont::iterator mem = old_opponents.begin();
              mem != end;
              ++mem )
        {
            double d2 = ( player.pos_ - mem->pos() ).r2();
            if ( d2 < mindist2 )
            {
                mindist2 = d2;
                candidate = mem;
                new_list = &new_opponents;
                old_list = &old_opponents;
            }
        }
    }

    //////////////////////////////////////////////////////////////////
    // search from old only side known opp
    {
        const PlayerCont::iterator end = old_unknown_opponents.end();
        for ( PlayerCont::iterator mem = old_unknown_opponents.begin();
              mem != end;
              ++mem )
        {
            // search closest player
            double d2 = ( player.pos_ - mem->pos() ).r2();
            if ( d2 < mindist2 )
            {
                mindist2 = d2;
                candidate = mem;
                new_list = &new_unknown_opponents;
                old_list = &old_unknown_opponents;
            }
        }
    }

    //////////////////////////////////////////////////////////////////
    // search from old unum known mate
    {
        const PlayerCont::iterator end = old_teammates.end();
        for ( PlayerCont::iterator mem = old_teammates.begin();
              mem != end;
              ++mem )
        {
            double d2 = ( player.pos_ - mem->pos() ).r2();
            if ( d2 < mindist2 )
            {
                mindist2 = d2;
                candidate = mem;
                new_list = &new_teammates;
                old_list = &old_teammates;
            }
        }
    }

    //////////////////////////////////////////////////////////////////
    // search from old only side known mate
    {
        const PlayerCont::iterator end = old_unknown_teammates.end();
        for ( PlayerCont::iterator mem = old_unknown_teammates.begin();
              mem != end;
              ++mem )
        {
            // search closest player
            double d2 = ( player.pos_ - mem->pos() ).r2();
            if ( d2 < mindist2 )
            {
                mindist2 = d2;
                candidate = mem;
                new_list = &old_unknown_teammates;
                old_list = &old_unknown_teammates;
            }
        }
    }

    //////////////////////////////////////////////////////////////////
    // search from old all unknown
    {
        const PlayerCont::iterator end = old_unknown_players.end();
        for ( PlayerCont::iterator mem = old_unknown_players.begin();
              mem != end;
              ++mem )
        {
            double d2 = ( player.pos_ - mem->pos() ).r2();
            if ( d2 < mindist2 )
            {
                mindist2 = d2;
                candidate = mem;
                new_list = &new_unknown_players;
                old_list = &old_unknown_players;
            }
        }
    }

    //////////////////////////////////////////////////////////////////
    // check player movable radius
    if ( new_list
         && old_list
         && mindist2 < 10.0 * 10.0
         && ( std::sqrt( mindist2 )
              < ( static_cast< double >( candidate->posCount() ) * PLAYER_MAX_MOVE_RADIUS_INC
                  + unquantize_error( seen_dist, ServerParam::i().distQuantizeStep() )
                  )
              )
         )
    {
        // update & splice to new list
        candidate->updateBySee( player );
        new_list->splice( new_list->end(),
                          *old_list,
                          candidate );
        return;
    }

    //////////////////////////////////////////////////////////////////
    // generate new player
    dlog.addText( Logger::WORLD,
                  "world.checkUnknownPlayer unmatch. min_dist= %.2f"
                  "  generate new unknown player. pos=(%.2f, %.2f)",
                  std::sqrt( mindist2 ),
                  player.pos_.x, player.pos_.y );
    new_unknown_players.push_back( PlayerObject( player ) ); // push to new ALL UNKNOWN
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::updateObjectRelation()
{
    // update ball matrix
    M_ball.updateSelfRelated( self() );

    // update about ball controll
    M_self.updateBallInfo( ball() );

    // update players matrix
    updatePlayerMatrix();
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::updatePlayerMatrix()
{
    //M_teammates_from_self.clear();
    //M_opponents_from_self.clear();
    //M_teammates_from_ball.clear();
    //M_opponents_from_ball.clear();

    if ( ! self().posValid()
         || ! ball().posValid() )
    {
        return;
    }

    dlog.addText( Logger::WORLD,
                  "WorldModel::updatePlayerMatrix()" );

    WMImpl::create_player_set( M_teammates,
                               M_teammates_from_self, 
                               M_teammates_from_ball,
                               self().pos(),
                               ball().pos() );
    WMImpl::create_player_set( M_unknown_teammates,
                               M_teammates_from_self,
                               M_teammates_from_ball,
                               self().pos(),
                               ball().pos() );
    WMImpl::create_player_set( M_opponents, 
                               M_opponents_from_self,
                               M_opponents_from_ball,
                               self().pos(),
                               ball().pos() );
    WMImpl::create_player_set( M_unknown_opponents, 
                               M_opponents_from_self,
                               M_opponents_from_ball,
                               self().pos(), 
                               ball().pos() );
    WMImpl::create_player_set( M_unknown_players, 
                               M_opponents_from_self,
                               M_opponents_from_ball,
                               self().pos(),
                               ball().pos() );
    
    // sort by distance to self or ball
    std::sort( M_teammates_from_self.begin(),
               M_teammates_from_self.end(),
               PlayerObject::PtrSelfDistCmp() );
    std::sort( M_opponents_from_self.begin(),
               M_opponents_from_self.end(),
               PlayerObject::PtrSelfDistCmp() );
    
    std::sort( M_teammates_from_ball.begin(),
               M_teammates_from_ball.end(),
               PlayerObject::PtrBallDistCmp() );
    std::sort( M_opponents_from_ball.begin(),
               M_opponents_from_ball.end(),
               PlayerObject::PtrBallDistCmp() );

    // check kickable player
    M_exist_kickable_teammate 
        = WMImpl::check_player_kickable( M_teammates_from_ball.begin(),
                                         M_teammates_from_ball.end(),
                                         ball().posCount() );
    M_exist_kickable_opponent
        = WMImpl::check_player_kickable( M_opponents_from_ball.begin(),
                                         M_opponents_from_ball.end(),
                                         ball().posCount() );

    dlog.addText( Logger::WORLD,
                  " size of player set"
                  " ourFMself %d"
                  " ourFMball %d"
                  " oppFMself %d"
                  " oppFMball %d",
                  M_teammates_from_self.size(),
                  M_teammates_from_ball.size(),
                  M_opponents_from_self.size(),
                  M_opponents_from_ball.size() );

}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::updateOffsideLine()
{
    if ( ! ServerParam::i().useOffside() )
    {
        M_offside_line_x = ServerParam::i().pitchHalfLength();
        return;
    }

    if ( gameMode().type() == GameMode::KickIn_
         || gameMode().type() == GameMode::CornerKick_
         || ( gameMode().type() == GameMode::GoalKick_
              && gameMode().side() == ourSide() )
         )
    {
        M_offside_line_x = ServerParam::i().pitchHalfLength();
        return;
    }

    if ( gameMode().side() != ourSide()
         && ( gameMode().type() == GameMode::GoalieCatch_
              || gameMode().type() == GameMode::GoalKick_ )
         )
    {
        M_offside_line_x = ServerParam::i().theirPenaltyAreaLine();
        return;
    }

#if 0
    const double speed_rate
        = ( ball().pos().x < self().pos().x
            ? ServerParam::i().defaultPlayerSpeedMax() * 0.5
            : ServerParam::i().defaultPlayerSpeedMax() * 0.25 );
#else
    const double speed_rate
        = ( ball().vel().x < -1.0
            ? ServerParam::i().defaultPlayerSpeedMax() * 0.8
            : ServerParam::i().defaultPlayerSpeedMax() * 0.25 );
#endif

    //////////////////////////////////////////////////////////////////
    double first = 0.0, second = 0.0;
    int opponent_count = 0;
    {
        const PlayerPtrCont::iterator end = M_opponents_from_self.end();
        for ( PlayerPtrCont::iterator it = M_opponents_from_self.begin();
              it != end;
              ++it )
        {
            ++opponent_count;
            double posx = (*it)->pos().x;
            posx -= speed_rate * std::min( 10, (*it)->posCount() );
            if ( posx > second )
            {
                second = posx;
                if ( second > first )
                {
                    std::swap( first, second );
                }
            }
        }
    }

    //////////////////////////////////////////////////////////////////
    double new_line = second;
#if 0
    // assume that opponent goalie exist in their penalty area
    if ( first < ServerParam::i().theirPenaltyAreaLine() )
    {
        new_line = first;
        dlog.addText( Logger::WORLD,
                      "world.updateOffsideLine.  change candidate %.2f -> %.2f",
                      second, first );
    }
#endif
    // consider old offside line

    if ( opponent_count >= 11 )
    {
        // new_line is used directly
    }
    else if ( new_line < M_offside_line_x - 13.0 )
    {
        // new_line is used directly
    }
    else if ( new_line < M_offside_line_x - 5.0 )
    {
        new_line = M_offside_line_x - 1.0;
    }

    if ( new_line < 0.0 )
    {
        new_line = 0.0;
    }

    // ball is more forward than opponent defender line
    if ( gameMode().type() != GameMode::BeforeKickOff
         && gameMode().type() != GameMode::AfterGoal_
         && ball().posValid()
         && ball().pos().x > new_line )
    {
        new_line = ball().pos().x; // server setting
    }

    M_offside_line_x = new_line;

    dlog.addText( Logger::WORLD,
                  " OffsideLine = %.2f",
                  new_line );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::updateDefenseLine()
{
    const double speed_rate = ServerParam::i().defaultPlayerSpeedMax() * 0.5;

    //////////////////////////////////////////////////////////////////
    double first = 0.0, second = 0.0;
    {
        const PlayerPtrCont::iterator end = M_teammates_from_self.end();
        for ( PlayerPtrCont::iterator it = M_teammates_from_self.begin();
              it != end;
              ++it )
        {
            double posx = (*it)->pos().x;
            posx += speed_rate * std::min( 10, (*it)->posCount() );
            if ( posx < second )
            {
                second = posx;
                if ( second < first )
                {
                    std::swap( first, second );
                }
            }
        }
    }

    //////////////////////////////////////////////////////////////////
    double new_line = second;
    // assume that our goalie exist in their penalty area
    if ( first > ( -ServerParam::i().pitchHalfLength()
                   + ServerParam::i().penaltyAreaLength() ) )
    {
        new_line = first;
    }

    // consider old line
    if ( new_line > M_defense_line_x + 5.0 )
    {
        new_line = M_defense_line_x + 1.0;
    }

    // ball is more forward than opponent defender line
    if ( ball().posValid() && ball().pos().x < new_line )
    {
        new_line = ball().pos().x; // server setting
    }

    M_defense_line_x = new_line;

    dlog.addText( Logger::WORLD,
                  " DefenseLine = %.2f",
                  new_line );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::checkGhost( const ViewArea & varea )
{
    // !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
    //  NOTE: this method is called from updateAfterSee

    const double VIS_DIST2
        = square( ServerParam::i().visibleDistance() * 0.9 );

    //////////////////////////////////////////////////////////////////
    // ball
    dlog.addText( Logger::WORLD,
                  "world.checkGhost. ball_conf= %d, rpos_conf= %d",
                  ball().posCount(), ball().rposCount() );
    if ( ball().posCount() > 0
         && ball().posValid() )
    {
        Vector2D ballrel = ball().pos() - varea.origin_;
        dlog.addText( Logger::WORLD,
                      "world.checkGhost. ball. global_dist= %.2f."
                      "  visdist= %.2f.  ",
                      ballrel.r(), std::sqrt( VIS_DIST2 ) );
        // dir threshold is 5.0 // old version 8.0
        if ( varea.contains( ball().pos(), 5.0, VIS_DIST2 ) )
        {
            dlog.addText( Logger::WORLD,
                          "world.checkGhost. forget ball." );
            M_ball.forget();
        }
    }

    //////////////////////////////////////////////////////////////////
    // players are not checked
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::updateDirCount( const ViewArea & varea )
{
    double dir_range = (varea.width_half_ - 5.0) * 2.0;
    double dir_idx = AngleDeg::normalize_angle( varea.angle_
                                                - varea.width_half_
                                                + 5.0 );

    //dlog.addText( Logger::WORLD,
    //            "world.updateDirConf" );
    while ( dir_range >= 0.0 )
    {
        int idx = static_cast< int >( ( dir_idx - 0.5 + 180.0 ) / DIR_STEP);
        if ( idx > DIR_CONF_DIVS - 1 )
        {
            std::cerr << teamName() << " : " << self().unum()
                      << " DIR_CONF over flow  " << idx << std::endl;
            idx = DIR_CONF_DIVS - 1;
        }
        else if ( idx < 0 )
        {
            std::cerr << teamName() << " : " << self().unum()
                      << " DIR_CONF down flow  " << idx << std::endl;
            idx = 0;
        }
        M_dir_count[idx] = 0;
        dir_idx = AngleDeg::normalize_angle( dir_idx + DIR_STEP );
        dir_range -= DIR_STEP;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
int
WorldModel::getDirRangeCount( const AngleDeg & angle,
                              const int add_span ) const
{
    const int base_idx
        = static_cast< int >( ( angle.degree() - 0.5 + 180.0 )
                              / DIR_STEP );

    int sum = 0;
    for ( int i = -add_span; i <= add_span; i++ )
    {
        int idx = base_idx + i;
        if ( idx < 0 )
        {
            idx += DIR_CONF_DIVS;
        }
        else if ( idx >= DIR_CONF_DIVS )
        {
            idx -= DIR_CONF_DIVS;
        }
        sum += M_dir_count[idx];
    }

    return sum;
}

/*-------------------------------------------------------------------*/
/*!

*/
int
WorldModel::getDirRangeMaxCount( const AngleDeg & left_angle,
                                 const AngleDeg & right_angle ) const
{
    int lowest = 0;
    const AngleDeg real_limit = right_angle + DIR_STEP;

    AngleDeg angle = left_angle;
    while ( angle.isLeftOf( real_limit ) )
    {
        int tmp = getDirCount( angle );
        if ( tmp > lowest )
        {
            lowest = tmp;
        }
        angle += DIR_STEP;
    }

    return lowest;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
WorldModel::getDirRangeCount( const AngleDeg & left_angle,
                              const AngleDeg & right_angle,
                              int * max_count,
                              int * ave_count ) const
{
    int tmp_max_count = 0;
    int total = 0;
    int count = 0;
    const AngleDeg real_limit = right_angle + DIR_STEP;

    AngleDeg angle = left_angle;
    while ( angle.isLeftOf( real_limit ) )
    {
        int tmp = getDirCount( angle );
        total += tmp;
        if ( tmp > tmp_max_count )
        {
            tmp_max_count = tmp;
        }

        ++count;
        angle += DIR_STEP;
    }

    if ( max_count )
    {
        *max_count = tmp_max_count;
    }
    if ( ave_count && count > 0 )
    {
        *ave_count = total / count;
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
const
PlayerObject *
WorldModel::getOpponentGoalie() const
{
    const PlayerCont::const_iterator end = M_opponents.end();
    for ( PlayerCont::const_iterator it = M_opponents.begin();
          it != end;
          ++it )
    {
        if ( it->goalie() )
        {
            return &(*it);
        }
    }

    // return &M_dummy_opponent;
    return static_cast< PlayerObject * >( 0 );
}

/*-------------------------------------------------------------------*/
/*!

*/
const
PlayerObject *
WorldModel::getTeammateNearestTo( const Vector2D & point,
                                  const int count_thr,
                                  double * dist_to_point ) const
{
    const PlayerObject * p = static_cast< PlayerObject * >( 0 );
    double min_dist2 = 40000.0;

    const PlayerPtrCont::const_iterator end = M_teammates_from_self.end();
    for ( PlayerPtrCont::const_iterator it = M_teammates_from_self.begin();
          it != end;
          ++it )
    {
        if ( (*it)->posCount() > count_thr )
        {
            continue;
        }
        double tmp = (*it)->pos().dist2(point);
        if ( tmp < min_dist2 )
        {
            p = *it;
            min_dist2 = tmp;
        }
    }

    if ( dist_to_point )
    {
        *dist_to_point = std::sqrt( min_dist2 );
    }
    return p;
}

/*-------------------------------------------------------------------*/
/*!

*/
const
PlayerObject *
WorldModel::getOpponentNearestTo( const Vector2D & point,
                                  const int count_thr,
                                  double * dist_to_point ) const
{
    const PlayerObject * p = static_cast< PlayerObject * >( 0 );
    double min_dist2 = 40000.0;

    const PlayerPtrCont::const_iterator end = M_opponents_from_self.end();
    for ( PlayerPtrCont::const_iterator it = M_opponents_from_self.begin();
          it != end;
          ++it )
    {
        if ( (*it)->posCount() > count_thr )
        {
            continue;
        }
        double tmp = (*it)->pos().dist2( point );
        if ( tmp < min_dist2 )
        {
            p = *it;
            min_dist2 = tmp;
        }
    }

    if ( dist_to_point )
    {
        *dist_to_point = std::sqrt( min_dist2 );
    }
    return p;
}

}
