// -*-c++-*-

/*!
  \file self_intercept.cpp
  \brief intercept predictor for agent itself Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <rcsc/math_util.h>
#include <rcsc/soccer_math.h>
#include <rcsc/param/server_param.h>
#include <rcsc/param/player_type.h>

#include "logger.h"
#include "self_object.h"
#include "ball_object.h"
#include "world_model.h"

#include "intercept_table.h"
#include "self_intercept.h"

namespace rcsc {

const double SelfIntercept::MIN_TURN_THR = 12.5;
const double SelfIntercept::BACK_DASH_THR_ANGLE = 100.0;

/*-------------------------------------------------------------------*/
/*!

*/
void
SelfIntercept::predict( const int max_cycle,
                        std::vector< InterceptInfo > & self_cache ) const
{
    dlog.addText( Logger::INTERCEPT,
                  "%s:%d: ------------- predict self ---------------"
                  ,__FILE__, __LINE__ );

    predictOneStep( self_cache );
    predictLongStep( max_cycle, self_cache );
}

/*-------------------------------------------------------------------*/
/*!

*/
void
SelfIntercept::predictOneStep( std::vector< InterceptInfo > & self_cache ) const
{
    const Vector2D ball_next = M_world.ball().pos() + M_world.ball().vel();
    const bool goalie_mode
        = ( M_world.self().goalie()
            && ball_next.x < ServerParam::i().ourPenaltyAreaLine()
            && ball_next.absY() < ServerParam::i().penaltyAreaHalfWidth()
            );
    const double control_area = ( goalie_mode
                                  ? ServerParam::i().catchableArea()
                                  : M_world.self().playerType().kickableArea() );
    ///////////////////////////////////////////////////////////
    // current distance is too far. never reach by one dasy
    if ( M_world.ball().distFromSelf()
         > ( ServerParam::i().ballSpeedMax()
             + M_world.self().playerType().realSpeedMax()
             + control_area ) )
    {
#ifdef DEBUG
        dlog.addText( Logger::INTERCEPT,
                      "__1 dash: too far. never reach" );
#endif
        return;
    }


    // get next ball relative pos, and set angle relative to my body angle
    Vector2D next_ball_rel = ball_next;
    next_ball_rel -= M_world.self().pos();
    next_ball_rel -= M_world.self().vel();
    next_ball_rel.rotate( - M_world.self().body() ); // X axis is player body angle

    // Now, next_ball_rel is required player's accel vector
#ifdef DEBUG
    dlog.addText( Logger::INTERCEPT,
                  "__1 dash: ctrl_area=%f  next_ball_rel=(%.2f, %.2f)",
                  control_area, next_ball_rel.x, next_ball_rel.y );
#endif

    ///////////////////////////////////////////////////////////
    // if Y difference is over control_area,
    // I never reach the ball even if max power dash
    if ( next_ball_rel.absY() > control_area - 0.2 )
    {
#ifdef DEBUG
        dlog.addText( Logger::INTERCEPT,
                      "__1 dash: big Y diff. never reach" );
#endif
        return;
    }

    ///////////////////////////////////////////////////////////
    // even if player does nothing, ball will be kickable.
    if ( predictNoDash( goalie_mode,
                        control_area,
                        next_ball_rel,
                        self_cache ) )
    {
        // can get the ball without any dash
        // player can turn to next target point
        return;
    }

    ///////////////////////////////////////////////////////////
    // check one step adjust dash
    predictOneDash( control_area, next_ball_rel, self_cache );
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
SelfIntercept::predictNoDash( const bool goalie_mode,
                              const double & control_area,
                              const Vector2D & next_ball_rel,
                              std::vector< InterceptInfo > & self_cache ) const
{
    const double next_ball_dist = next_ball_rel.r();

    // over control area
    if ( next_ball_dist > control_area - 0.2 )
    {
        return false;
    }

    // if goalie, not needed to consider about collision and kick rate
    // immediately success!!
    if ( goalie_mode )
    {
        self_cache.push_back( InterceptInfo( InterceptInfo::NORMAL,
                                             1, 0, 0.0 ) ); // 1 turn
        dlog.addText( Logger::INTERCEPT,
                      "--->Success! No dash goalie mode: nothing to do. next_dist = %f",
                      next_ball_dist );
        return true;
    }

    // normal player

    // check collision.
    if ( next_ball_dist < ( M_world.self().playerType().playerSize()
                            + ServerParam::i().ballSize()
                            + 0.15 ) )
    {
        // maybe collision ...
        dlog.addText( Logger::INTERCEPT,
                      "____No dash kickable: but may cause collision next_dist = %f",
                      next_ball_dist );
        return false;
    }

    // check kick rate
    Vector2D next_ball_vel
        = M_world.ball().vel()
        * ServerParam::i().ballDecay();

    double kickrate
        = M_world.self().playerType().kickRate( ServerParam::i(),
                                                next_ball_dist,
                                                next_ball_rel.th().degree() );

    if ( ServerParam::i().maxPower() * kickrate
         <= next_ball_vel.r() * ServerParam::i().ballDecay() * 1.01 )
    {
        // it has possibility that player cannot stop the ball
        dlog.addText( Logger::INTERCEPT,
                      "____No dash kickable: but may cause no control" );
        return false;
    }


    // at least, player can stop ball
    self_cache.push_back( InterceptInfo( InterceptInfo::NORMAL,
                                         1, 0, 0.0 ) ); // 1 turn
#ifdef DEBUG
    dlog.addText( Logger::INTERCEPT,
                  "-->Sucess! No dash: nothing to do. next_dist = %f",
                  next_ball_dist );
#endif
    return true;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
SelfIntercept::predictOneDash( const double & control_area,
                               const Vector2D & next_ball_rel,
                               std::vector< InterceptInfo > & self_cache ) const
{
    // check possible max accel
    // possible forward dash accel distance
    double max_forward_accel_x
        = ServerParam::i().maxPower()
        * M_world.self().dashRate();
    M_world.self().playerType().normalizeAccel( M_world.self().vel(),
                                                M_world.self().body(),
                                                &max_forward_accel_x );

    if ( max_forward_accel_x + control_area - 0.2 < next_ball_rel.x )
    {
#ifdef DEBUG
        dlog.addText( Logger::INTERCEPT,
                      "____1 dash: ball is front but over dash capacity."
                      "  max_forward_accel = %f",
                      max_forward_accel_x );
#endif
        return;
    }

    // possible backward dash accel distance
    double max_back_accel_x = max_forward_accel_x;
    // normalize accel magnitude
    M_world.self().playerType().normalizeAccel( M_world.self().vel(),
                                                M_world.self().body() + 180.0,
                                                &max_back_accel_x );
    max_back_accel_x *= -1.0;

    ///////////////////////////////////////////////////////////
    if ( next_ball_rel.x < max_back_accel_x - control_area - 0.2 )

    {
        // over the reachable distance
#ifdef DEBUG
        dlog.addText( Logger::INTERCEPT,
                      "____1 dash: ball is back but over dash capacity."
                      "  max_back_accel = %f",
                      max_back_accel_x );
#endif
        return;
    }

    const double safety_power_forward
        = min_max( 0.0,
                   M_world.self().stamina()
                   - ServerParam::i().recoverDecThrValue()
                   - 5.0,
                   ServerParam::i().maxPower() );
    double safety_forward_accel_x
        = std::min( max_forward_accel_x,
                    safety_power_forward * M_world.self().dashRate() );
    const double safety_power_back
        = min_max( 0.0,
                   M_world.self().stamina()
                   - ServerParam::i().recoverDecThrValue()
                   - 5.0,
                   2.0 * ServerParam::i().maxPower() );
    double safety_back_accel_x
        = std::max( max_back_accel_x,
                    -0.5 * safety_power_back * M_world.self().dashRate() );

#ifdef DEBUG
    dlog.addText( Logger::INTERCEPT,
                  "____1 dash: max_forward_accel = %f max_back_accel = %f",
                  max_forward_accel_x, max_back_accel_x );
    dlog.addText( Logger::INTERCEPT,
                  "____1 dash: safety_forward_accel = %f safety_back_accel = %f",
                  safety_forward_accel_x, safety_back_accel_x );
#endif

    ///////////////////////////////////////////////////////////
    // at first, check stamina safety dash
    if ( predictOneDashAdjustX( control_area,
                                next_ball_rel,
                                safety_back_accel_x,
                                safety_forward_accel_x,
                                true, // stamina safe
                                self_cache ) )
    {
#ifdef DEBUG
        dlog.addText( Logger::INTERCEPT,
                      "____1 dash. Success! and stamina safe" );
#endif
        return;
    }

    // second, check maximal accel dash
    if ( predictOneDashAdjustX( control_area,
                                next_ball_rel,
                                max_back_accel_x,
                                max_forward_accel_x,
                                false, // stamina not safe
                                self_cache ) )
    {
#ifdef DEBUG
        dlog.addText( Logger::INTERCEPT,
                      "____1 dash. Success! but not stamina safe" );
#endif
        return;
    }

#ifdef DEBUG
    dlog.addText( Logger::INTERCEPT,
                  "____1 dash. failure" );
#endif
}

/*-------------------------------------------------------------------*/
/*!

*/
bool
SelfIntercept::predictOneDashAdjustX( const double & control_area,
                                      const Vector2D & next_ball_rel,
                                      const double & back_accel_x,
                                      const double & forward_accel_x,
                                      const bool stamina_safety,
                                      std::vector< InterceptInfo > & self_cache ) const
{
#ifdef DEBUG
    dlog.addText( Logger::INTERCEPT,
                  "______1 dash Adjust. back_acc_x = %.3f  next_ball_rel.x = %.3f"
                  "  forward_acc_x = %.3f",
                  back_accel_x, next_ball_rel.x, forward_accel_x );
#endif

    double dash_power = -1000.0;
    /////////////////////////////////////////////////////
    // x difference is small
    // player can put the ball on his side by this trial dash,
    if ( back_accel_x < next_ball_rel.x
         && next_ball_rel.x < forward_accel_x )
    {
        dash_power = getOneStepBestTrapPower( control_area,
                                              next_ball_rel,
                                              back_accel_x,
                                              forward_accel_x );
#ifdef DEBUG
        dlog.addText( Logger::INTERCEPT,
                      "______1 dash Adjust(1). Best Trap. power = %.1f",
                      dash_power );
#endif
    }
    /////////////////////////////////////////////////////
    // x difference is longer than acccelx. needed to check ball dist after dash
    // ball is front
    if ( dash_power < -999.0
         && forward_accel_x < next_ball_rel.x )
    {
        double enable_ball_dist
            = std::sqrt( std::pow( next_ball_rel.x - forward_accel_x, 2.0 )
                         + std::pow( next_ball_rel.y, 2.0 ) );
        if ( enable_ball_dist < control_area - 0.2 )
        {
            // at least, reach the controllale distance
            dash_power = forward_accel_x / M_world.self().dashRate();
#ifdef DEBUG
            dlog.addText( Logger::INTERCEPT,
                          "_____1 dash Adjust(2). Not Best. next_ball_dist = %.3f  power = %.1f",
                          enable_ball_dist, dash_power );
#endif
        }
    }
    // x difference is longer than acccelx. needed to check ball dist after dash
    // ball is back
    if ( dash_power < -999.0
         && next_ball_rel.x < back_accel_x )
    {
        double enable_ball_dist
            = std::sqrt( std::pow( back_accel_x - next_ball_rel.x, 2.0 )
                         + std::pow( next_ball_rel.y, 2.0 ) );
        if ( enable_ball_dist < control_area - 0.2 )
        {
            dash_power = back_accel_x / M_world.self().dashRate();
#ifdef DEBUG
            dlog.addText( Logger::INTERCEPT,
                          "______1 dash Adjust(3).Not Best next_ball_dist = %.3f  power = %.1f",
                          enable_ball_dist, dash_power );
#endif
        }
    }

    // no solution but at least adjustable
    if ( dash_power < -999.0
         && back_accel_x < next_ball_rel.x
         && next_ball_rel.x < forward_accel_x )
    {
        dash_power = next_ball_rel.x / M_world.self().dashRate();
#ifdef DEBUG
        dlog.addText( Logger::INTERCEPT,
                      "______1 dash Adjust(4).Not Best. just adjust X. dash power = %.1f",
                      dash_power );
#endif
    }

    /////////////////////////////////////////////////////
    // register
    if ( dash_power > -200.0 )
    {
        InterceptInfo::Mode mode = InterceptInfo::NORMAL;
        if ( ! stamina_safety )
        {
            double consume = ( dash_power > 0.0 ? dash_power : dash_power * -2.0 );
            if ( M_world.self().stamina() - consume
                 < ServerParam::i().recoverDecThrValue() )
            {
                mode = InterceptInfo::EXHAUST;
            }
        }
        self_cache.push_back( InterceptInfo( mode, 0, 1, dash_power ) );
#ifdef DEBUG
        dlog.addText( Logger::INTERCEPT,
                      "--->Success! 1 dash Adjust. register new dash power = %f  mode = %d",
                      dash_power, mode );
#endif
        return true;
    }

#ifdef DEBUG
    dlog.addText( Logger::INTERCEPT,
                  "______1 dash Adjust. Failed" );
#endif
    return false;
}

/*-------------------------------------------------------------------*/
/*!

*/
double
SelfIntercept::getOneStepBestTrapPower( const double & control_area,
                                        const Vector2D & next_ball_rel,
                                        const double & max_back_accel_x,
                                        const double & max_forward_accel_x ) const
{
    double best_ctrl_dist
        = M_world.self().playerType().playerSize()
        + 0.5 * ( M_world.self().playerType().kickableMargin()
                  + ServerParam::i().ballSize() );

#ifdef DEBUG
    dlog.addText( Logger::INTERCEPT,
                  "________1 dash Best Trap. BestCtrlDist = %.3f, next_ball_y = %.3f",
                  best_ctrl_dist, next_ball_rel.y );
#endif

    // Y diff is longer than best distance.
    // just put the ball on player's side
    if ( next_ball_rel.absY() > best_ctrl_dist )
    {
#ifdef DEBUG
        dlog.addText( Logger::INTERCEPT,
                      "________1 dash Best Trap.  just put the ball on side" );
#endif
        return next_ball_rel.x / M_world.self().dashRate();
    }

    double required_accel_x[2];
    // case  put ball front
    required_accel_x[0]
        = next_ball_rel.x
        - std::sqrt( std::pow( best_ctrl_dist, 2.0 )
                     - std::pow( next_ball_rel.y, 2.0 ) );
    // case put ball back
    required_accel_x[1]
        = next_ball_rel.x
        + std::sqrt( std::pow( best_ctrl_dist, 2.0 )
                     - std::pow( next_ball_rel.y, 2.0 ) );
    // nearest side has priority
    if ( std::fabs( required_accel_x[1] )
         < std::fabs( required_accel_x[0] ) )
    {
        std::swap( required_accel_x[0], required_accel_x[1] );
    }

#ifdef DEBUG
    dlog.addText( Logger::INTERCEPT,
                  "________1 dash: best_ctrl accel_x[0] = %f  accel_x[1] = %f",
                  required_accel_x[0], required_accel_x[1] );
#endif

    for ( int i = 0; i < 2; ++i )
    {
        if ( required_accel_x[i] >= 0.0
             && max_forward_accel_x > required_accel_x[i] )
        {
#ifdef DEBUG
            dlog.addText( Logger::INTERCEPT,
                          "________1 dash Best Trap.  best trap."
                          " forward dash[%d]. x = %f",
                          i, required_accel_x[i] );
#endif
            return required_accel_x[i] / M_world.self().dashRate();
        }
        if ( required_accel_x[i] < 0.0
             && max_back_accel_x < required_accel_x[i] )
        {
#ifdef DEBUG
            dlog.addText( Logger::INTERCEPT,
                          "________1 dash Best Trap.  best trap."
                          " back dash[%d]. x = %f",
                          i, required_accel_x[i] );
#endif
            return required_accel_x[i] / M_world.self().dashRate();
        }
    }
#ifdef DEBUG
    dlog.addText( Logger::INTERCEPT,
                  "________1 dash Best Trap. Failed" );
#endif

    return -1000.0;
}

/*-------------------------------------------------------------------*/
/*!

*/
void
SelfIntercept::predictLongStep( const int max_cycle,
                                std::vector< InterceptInfo > & self_cache ) const
{
    // calc Y distance from ball line
    Vector2D ball_to_self = M_world.self().pos() - M_world.ball().pos();
    ball_to_self.rotate( - M_world.ball().vel().th() );
    int min_cycle = static_cast< int >
        ( std::ceil( ( ball_to_self.absY()
                       - M_world.self().playerType().kickableArea()
                       - 0.2 )
                     / M_world.self().playerType().realSpeedMax() ) );

    if ( min_cycle < 2 )
    {
        min_cycle = 2;
    }
    if ( max_cycle <= min_cycle )
    {
        dlog.addText( Logger::INTERCEPT,
                      "__Long.  Too big Y difference = %f."
                      "  min_cycle = %d.  max_cycle = %d",
                      ball_to_self.y, min_cycle, max_cycle );
    }

    Vector2D ball_pos
        = inertia_n_step_point( M_world.ball().pos(),
                                M_world.ball().vel(),
                                min_cycle - 1,
                                ServerParam::i().ballDecay() );
    Vector2D ball_vel = M_world.ball().vel()
        * std::pow( ServerParam::i().ballDecay(), min_cycle - 1 );
    bool found = false;

    int max_loop = max_cycle;
    for ( int cycle = min_cycle; cycle < max_loop; ++cycle )
    {
        ball_pos += ball_vel;
        ball_vel *= ServerParam::i().ballDecay();
#ifdef DEBUG
        dlog.addText( Logger::INTERCEPT,
                      "__cycle %d: bpos(%.3f, %.3f) bvel(%.3f, %.3f)",
                      cycle,
                      ball_pos.x, ball_pos.y,
                      ball_vel.x, ball_vel.y );

#endif
        // ball is stopped
        if ( ball_vel.r2() < 0.01 )
        {
#ifdef DEBUG
            dlog.addText( Logger::INTERCEPT,
                          "____ball speed reach ZERO" );
#endif
            break;;
        }
        if ( ball_pos.absX() > ServerParam::i().pitchHalfLength() + 5.0
             || ball_pos.absY() > ServerParam::i().pitchHalfWidth() + 5.0
             )
        {
#ifdef DEBUG
            dlog.addText( Logger::INTERCEPT,
                          "____ball is out of pitch" );
#endif
            break;
        }
        const bool goalie_mode =
            ( M_world.self().goalie()
              && ball_pos.x < ServerParam::i().ourPenaltyAreaLine()
              && ball_pos.absY() < ServerParam::i().penaltyAreaHalfWidth()
              );
        const double control_area = ( goalie_mode
                                      ? ServerParam::i().catchableArea()
                                      : M_world.self().playerType().kickableArea() );

        ///////////////////////////////////////////////////////////
        // reach point is too far. never reach
        if ( control_area + ( M_world.self().playerType().realSpeedMax() * cycle )
             < M_world.self().pos().dist( ball_pos ) )
        {
#ifdef DEBUG
            dlog.addText( Logger::INTERCEPT,
                          "____ball is too far. never reach by %d cycle",
                          cycle );
#endif
            continue;
        }
        ///////////////////////////////////////////////////////////
        bool back_dash = false;
        bool save_recovery = true;
        int n_turn = 0;
        if ( canReachAfterTurnDash( cycle,
                                    ball_pos,
                                    control_area,
                                    &n_turn, &back_dash, &save_recovery ) )
        {
#ifdef DEBUG
            dlog.addText( Logger::INTERCEPT,
                          " --> can reach. cycle = %d, turn = %d",
                          cycle, n_turn);
            if ( ! save_recovery )
            {
                dlog.addText( Logger::INTERCEPT,
                              "____But  cannot save recovery" );

            }
#endif
            double dash_power = ServerParam::i().maxPower();
            if ( back_dash ) dash_power = -dash_power;
            self_cache.push_back( InterceptInfo( ( save_recovery
                                                   ? InterceptInfo::NORMAL
                                                   : InterceptInfo::EXHAUST ),
                                                 n_turn,
                                                 std::max( 0, cycle - n_turn ),
                                                 dash_power ) );
            if ( ! found )
            {
                max_loop = std::min( max_cycle, cycle + 10 );
            }
            found = true;
        }
        ///////////////////////////////////////////////////////////
    }

    // not registered any interception
    if ( ! found )
    {
#ifdef DEBUG
        dlog.addText( Logger::INTERCEPT,
                      "__ Failed to predict? register ball final point" );
#endif
        predictFinal( self_cache );
    }
}

/*-------------------------------------------------------------------*/
/*!

*/
void
SelfIntercept::predictFinal( std::vector< InterceptInfo > & self_cache ) const

{
    Vector2D ball_final_pos
        = inertia_final_point( M_world.ball().pos(),
                               M_world.ball().vel(),
                               ServerParam::i().ballDecay() );
    const bool goalie_mode =
        ( M_world.self().goalie()
          && ball_final_pos.x < ServerParam::i().ourPenaltyAreaLine()
          && ball_final_pos.absY() < ServerParam::i().penaltyAreaHalfWidth()
          );
    const double control_area = ( goalie_mode
                                  ? ServerParam::i().catchableArea() - 0.15
                                  : M_world.self().playerType().kickableArea() );

    AngleDeg dash_angle = M_world.self().body();
    bool back_dash; // dummy
    int n_turn = predictTurnCycle( 100,
                                   ball_final_pos,
                                   control_area,
                                   &dash_angle, &back_dash );
    double dash_dist = M_world.self().pos().dist( ball_final_pos );
    int n_dash = M_world.self().playerType().cyclesToReachDistance( dash_dist );

#ifdef DEBUG
    dlog.addText( Logger::INTERCEPT,
                  "__Final. register ball final point. turn=%d, dash=%d",
                  n_turn, n_dash );
#endif
    self_cache.push_back( InterceptInfo( InterceptInfo::NORMAL,
                                         n_turn, n_dash,
                                         ServerParam::i().maxPower() ) );
}

/*-------------------------------------------------------------------*/
/*!
  \param ball_pos ball position after 'cycle'.
*/
bool
SelfIntercept::canReachAfterTurnDash( const int cycle,
                                      const Vector2D & ball_pos,
                                      const double & control_area,
                                      int * n_turn,
                                      bool * back_dash,
                                      bool * save_recovery ) const
{
    AngleDeg dash_angle = M_world.self().body();

    *n_turn = predictTurnCycle( cycle,
                                ball_pos,
                                control_area,
                                &dash_angle, back_dash );
    if ( *n_turn > cycle )
    {
        return false;
    }

    return canReachAfterDash( *n_turn, std::max( 0, cycle - (*n_turn) ),
                              ball_pos, control_area,
                              dash_angle, *back_dash, save_recovery );
}

/*-------------------------------------------------------------------*/
/*!
  predict & get teammate's ball gettable cycle
*/
int
SelfIntercept::predictTurnCycle( const int cycle,
                                 const Vector2D & ball_pos,
                                 const double & control_area,
                                 AngleDeg * dash_angle,
                                 bool * back_dash ) const
{
    int n_turn = 0;

    ///////////////////////////////////////////////////
    // prepare variables
    const Vector2D inertia_pos
        = M_world.self().playerType().inertiaPoint( M_world.self().pos(),
                                                    M_world.self().vel(),
                                                    cycle );
    const Vector2D target_rel = ball_pos - inertia_pos;
    const AngleDeg target_angle = target_rel.th();

    double angle_diff = ( target_angle - (*dash_angle) ).degree();
    const bool diff_is_positive = ( angle_diff > 0.0 ? true : false );
    angle_diff = std::fabs( angle_diff );
    // atan version
    const double target_dist = target_rel.r();
    double turn_margin = 180.0;
    double control_buf = control_area - 0.25;
    //- M_world.ball().vel().r() * ServerParam::i().ballRand() * 0.5
    //- target_dist * ServerParam::i().playerRand() * 0.5;
    control_buf = std::max( 0.5, control_buf );
    if ( control_buf < target_dist )
    {
        // turn_margin = AngleDeg::atan2_deg( control_buf, target_dist );
        turn_margin = AngleDeg::asin_deg( control_buf / target_dist );
    }
    turn_margin = std::max( turn_margin, MIN_TURN_THR );

    /* case buf == 1.085 - 0.15, asin_deg(buf / dist), atan2_deg(buf, dist)
       dist =  1.0  asin_deg = 69.23  atan2_deg = 43.08
       dist =  2.0  asin_deg = 27.87  atan2_deg = 25.06
       dist =  3.0  asin_deg = 18.16  atan2_deg = 17.31
       dist =  4.0  asin_deg = 13.52  atan2_deg = 13.16
       dist =  5.0  asin_deg = 10.78  atan2_deg = 10.59
       dist =  6.0  asin_deg =  8.97  atan2_deg =  8.86
       dist =  7.0  asin_deg =  7.68  atan2_deg =  7.61
       dist =  8.0  asin_deg =  6.71  atan2_deg =  6.67
       dist =  9.0  asin_deg =  5.96  atan2_deg =  5.93
       dist = 10.0  asin_deg =  5.36  atan2_deg =  5.34
       dist = 11.0  asin_deg =  4.88  atan2_deg =  4.86
       dist = 12.0  asin_deg =  4.47  atan2_deg =  4.46
       dist = 13.0  asin_deg =  4.12  atan2_deg =  4.11
       dist = 14.0  asin_deg =  3.83  atan2_deg =  3.82
       dist = 15.0  asin_deg =  3.57  atan2_deg =  3.57
       dist = 16.0  asin_deg =  3.35  atan2_deg =  3.34
       dist = 17.0  asin_deg =  3.15  atan2_deg =  3.15
       dist = 18.0  asin_deg =  2.98  atan2_deg =  2.97
       dist = 19.0  asin_deg =  2.82  atan2_deg =  2.82
       dist = 20.0  asin_deg =  2.68  atan2_deg =  2.68
    */
    ///////////////////////////////////////////////////
    // check back dash possibility
    if ( canBackDashChase( cycle, target_dist, angle_diff ) )
    {
        *back_dash = true;
        *dash_angle += 180.0;
        angle_diff = 180.0 - angle_diff;
    }

#ifdef DEBUG
    dlog.addText( Logger::INTERCEPT,
                  "______control_buf=%.2f turn_margin=%.1f angle_diff=%.1f",
                  control_buf, turn_margin, angle_diff );
#endif

    ///////////////////////////////////////////////////
    // predict turn cycles
    const double max_moment
        = ServerParam::i().maxMoment()
        * ( 1.0 - ServerParam::i().playerRand() );
    double player_speed = M_world.self().vel().r();
    while ( angle_diff > turn_margin )
    {
        double max_turnable
            = M_world.self().playerType().effectiveTurn( max_moment, player_speed );
        angle_diff -= max_turnable;
        player_speed *= M_world.self().playerType().playerDecay();
        ++n_turn;
    }

    ///////////////////////////////////////////////////
    // update dash angle
    if ( n_turn > 0 )
    {
        angle_diff = std::max( 0.0, angle_diff );
        *dash_angle = target_angle + ( diff_is_positive
                                       ? + angle_diff
                                       : - angle_diff );
    }

    return n_turn;
}

/*-------------------------------------------------------------------*/
/*!
  predict & get teammate's ball gettable cycle
*/
bool
SelfIntercept::canBackDashChase( const int cycle,
                                 const double & target_dist,
                                 const double & angle_diff ) const
{
    ///////////////////////////////////////////////
    // check angle threshold
    if ( angle_diff < BACK_DASH_THR_ANGLE )
    {
        return false;
    }
#if 0
    ///////////////////////////////////////////////
    // check distance threshold
    {
        double back_dash_thr
            = ServerParam::i().visibleDistance()
            + ( M_world.self().goalie() ? 10.0 : 5.0 );

        if ( target_dist > back_dash_thr )
        {
            return false;
        }
    }
#else
    if ( ! M_world.self().goalie() && cycle - 1 >= 6 )
    {
        return false;
    }
#endif

    ///////////////////////////////////////////////
    // check stamina threshold
    // consumed stamina by one step
    double total_consume = ServerParam::i().maxPower() * 2.0 * cycle;
    double total_recover
        = M_world.self().playerType().staminaIncMax() * M_world.self().recovery()
        * ( cycle - 1 );
    double result_stamina
        = M_world.self().stamina()
        - total_consume
        + total_recover;

    if ( result_stamina < ServerParam::i().recoverDecThrValue() + 205.0 )
    {
#ifdef DEBUG
        dlog.addText( Logger::INTERCEPT,
                      "______cycle=%d goalie no stamina. no back. stamina=%f",
                      cycle, result_stamina );
#endif
        return false;
    }

#ifdef DEBUG
    dlog.addText( Logger::INTERCEPT,
                  "______try back dash. result stamina=%f",
                  result_stamina );
#endif

    return true;
}

/*-------------------------------------------------------------------*/
/*!
  assume that players's dash angle is directed to ball reach point
  \param ball_pos ball position after 'cycle'.
*/
bool
SelfIntercept::canReachAfterDash( const int n_turn,
                                  const int n_dash,
                                  const Vector2D & ball_pos,
                                  const double & control_area,
                                  const AngleDeg & dash_angle,
                                  const bool back_dash,
                                  bool * save_recovery ) const
{
    static const double PLAYER_NOISE_RATE
        = ( 1.0 - ServerParam::i().playerRand() * 0.25 );
    static const double MAX_POWER = ServerParam::i().maxPower();

    const PlayerType & my_type = M_world.self().playerType();

    //////////////////////////////////////////////////////////////
    const AngleDeg dash_angle_minus = -dash_angle;
    const Vector2D ball_rel
        = ( ball_pos - M_world.self().pos() ).rotatedVector( dash_angle_minus );
    const double ball_noise
        = M_world.ball().pos().dist( ball_pos )
        * ServerParam::i().ballRand()
        * 0.5;
    const double noised_ball_x = ball_rel.x + ball_noise;


    //////////////////////////////////////////////////////////////
    // prepare loop variables
    // ORIGIN: first player pos.
    // X-axis: dash angle
    //Vector2D tmp_pos = my_type.inertiaPoint( M_world.self().pos(), M_world.self().vel(), n_turn );
    //tmp_pos -= M_world.self().pos();
    Vector2D tmp_pos = my_type.inertiaTravel( M_world.self().vel(), n_turn );
    tmp_pos.rotate( dash_angle_minus );

    Vector2D tmp_vel = M_world.self().vel();
    tmp_vel *= std::pow( my_type.playerDecay(), n_turn );
    tmp_vel.rotate( dash_angle_minus );

    double tmp_stamina = M_world.self().stamina();
    double tmp_effort = M_world.self().effort();
    double tmp_recovery = M_world.self().recovery();
    my_type.predictStaminaAfterWait( ServerParam::i(),
                                     n_turn,
                                     &tmp_stamina,
                                     &tmp_effort,
                                     tmp_recovery );

    double prev_effort = tmp_effort;
    double dash_power_abs = MAX_POWER;
    // only consider about x of dash accel vector,
    // because current orientation is player's dash angle (included back dash case)
    // NOTE: dash_accel_x must be positive value.
    double dash_accel_x
        = dash_power_abs
        * my_type.dashRate( tmp_effort );
#ifdef DEBUG
    dlog.addText( Logger::INTERCEPT,
                  "______Try %d turn: %d dash:"
                  " angle=%.1f first_accel=%.2f first_vel=(%.2f %.2f)",
                  n_turn, n_dash,
                  dash_angle.degree(), dash_accel_x, tmp_vel.x, tmp_vel.y );
#endif

    //////////////////////////////////////////////////////////////
    bool can_over_speed_max = my_type.canOverSpeedMax( dash_power_abs,
                                                       tmp_effort );
    for ( int i = 0; i < n_dash; ++i )
    {
        /////////////////////////////////////////////////////////
        // update dash power & accel
        double available_power = tmp_stamina + my_type.extraStamina();
        if ( back_dash ) available_power *= 0.5;
        available_power = min_max( 0.0, available_power, MAX_POWER );

        bool must_update_power = false;
        if ( available_power < dash_power_abs  // no enough stamina
             || tmp_effort < prev_effort       // effort decreased.
             || ( ! can_over_speed_max         // can inclease dash power
                  && dash_power_abs < available_power )
             )
        {

            must_update_power = true;
#ifdef DEBUG
            dlog.addText( Logger::INTERCEPT,
                          "________dash %d/%d: no enough?  stamina=%f  extra=%f "
                          "  cur_pow=%f  available_pow=%f",
                          i, n_dash,
                          tmp_stamina, my_type.extraStamina(),
                          dash_power_abs, available_power );
            dlog.addText( Logger::INTERCEPT,
                          "________dash %d/%d: effort decayed? %f -> %f",
                          i, n_dash, prev_effort, tmp_effort );
            dlog.addText( Logger::INTERCEPT,
                          "________dash %d/%d: reset max power?. curr_pow=%f"
                          "  available=%f",
                          i, n_dash, dash_power_abs, available_power );
#endif
        }

        if ( must_update_power )
        {
            dash_power_abs = available_power;
            dash_accel_x = dash_power_abs * my_type.dashRate( tmp_effort );
            can_over_speed_max = my_type.canOverSpeedMax( dash_power_abs,
                                                          tmp_effort );
#ifdef DEBUG
            dlog.addText( Logger::INTERCEPT,
                          "________dash %d/%d: update dash_power_abs %f  accel_x =%f",
                          i, n_dash, dash_power_abs, dash_accel_x );
#endif
        }

        /////////////////////////////////////////////////////////
        // update vel
        tmp_vel.x += dash_accel_x;
        // power conservation, update accel magnitude & dash_power
        if ( can_over_speed_max
             && tmp_vel.r2() > my_type.playerSpeedMax2() )
        {
            tmp_vel.x -= dash_accel_x;
            // conserve power & reduce accel
            // sqr(rel_vel.y) + sqr(max_dash_x) == sqr(max_speed);
            // accel_mag = dash_x - rel_vel.x;
            double max_dash_x = std::sqrt( my_type.playerSpeedMax2()
                                           - ( tmp_vel.y * tmp_vel.y ) );
            dash_accel_x = max_dash_x - tmp_vel.x;
            dash_power_abs = std::fabs( dash_accel_x
                                        / my_type.dashRate( tmp_effort ) );
            // re-update vel
            tmp_vel.x += dash_accel_x;
            can_over_speed_max = my_type.canOverSpeedMax( dash_power_abs,
                                                          tmp_effort );
#ifdef DEBUG
            dlog.addText( Logger::INTERCEPT,
                          "________dash %d/%d: power conserve. power=%f accel_x=%f",
                          i, n_dash, dash_power_abs, dash_accel_x );
#endif
        }
        /////////////////////////////////////////////////////////
        // velocity reached max speed
        if ( tmp_vel.x > my_type.realSpeedMax() - 0.005 )
        {
            tmp_vel.x = my_type.realSpeedMax();
            double real_power = dash_power_abs;
            if ( back_dash ) real_power *= -1.0;
            int n_safety_dash
                = my_type.getMaxDashCyclesSavingStamina( ServerParam::i(),
                                                         real_power,
                                                         tmp_stamina,
                                                         tmp_recovery );
            n_safety_dash = std::min( n_safety_dash, n_dash - i );
            n_safety_dash = std::max( 0, n_safety_dash - 1 ); // -1 is very important
#ifdef DEBUG
            dlog.addText( Logger::INTERCEPT,
                          "________dash %d/%d: reach real speed max. safety dash= %d",
                          i, n_dash, n_safety_dash );
#endif
            tmp_pos.x += tmp_vel.x * n_safety_dash;
            double one_cycle_consume = ( real_power > 0.0
                                         ? real_power
                                         : real_power * -2.0 );
            one_cycle_consume -= my_type.staminaIncMax() * tmp_recovery;
            tmp_stamina -= one_cycle_consume * n_safety_dash;
            i += n_safety_dash;
        }

        /////////////////////////////////////////////////////////
        // update pos & vel
        tmp_pos += tmp_vel;
        tmp_vel *= my_type.playerDecay();
        /////////////////////////////////////////////////////////
        // update stamina
        my_type.predictStaminaAfterDash( ServerParam::i(),
                                         dash_power_abs * (back_dash ? -1.0 : 1.0),
                                         &tmp_stamina, &tmp_effort, &tmp_recovery );

        /////////////////////////////////////////////////////////
        // check run over
        // it is not necessary to consider about Y difference,
        // because dash_angle is corrected for ball_reach_point
        if ( tmp_pos.x * PLAYER_NOISE_RATE > noised_ball_x )
        {
#ifdef DEBUG
            dlog.addText( Logger::INTERCEPT,
                          "____dash %d/%d: can run over. rel_move_pos=(%.2f, %.2f)"
                          " ball_x=%f over=%f y_diff=%f",
                          i, n_dash,
                          tmp_pos.x, tmp_pos.y,
                          noised_ball_x,
                          tmp_pos.x * PLAYER_NOISE_RATE - noised_ball_x,
                          std::fabs( tmp_pos.y - ball_rel.y ) );
#endif
            if ( tmp_recovery < M_world.self().recovery() )
            {
#ifdef DEBUG
                dlog.addText( Logger::INTERCEPT,
                              "________stamina exhaust. stamina = %f recover = %f",
                              tmp_stamina,
                              tmp_recovery );
#endif
                *save_recovery = false;
            }
            return true;
        }
    }

#ifdef DEBUG
    dlog.addText( Logger::INTERCEPT,
                  "______dash %d: no run over. pmove=(%.2f(noised=%.2f), %.2f)  ball_x=%f"
                  "  x_diff=%f y_diff=%f",
                  n_dash,
                  tmp_pos.x, tmp_pos.x * PLAYER_NOISE_RATE, tmp_pos.y,
                  noised_ball_x,
                  tmp_pos.x * PLAYER_NOISE_RATE - noised_ball_x,
                  std::fabs( tmp_pos.y - ball_rel.y ) );
#endif

    //////////////////////////////////////////////////////////
    // when cycle is small, do strict check
    if ( n_dash <= 4 )
    {
        // tmp_pos is relative to playerPos() --> tmp_pos.r() == player_travel
        double player_travel = tmp_pos.r();
        double player_noise = player_travel * ServerParam::i().playerRand() * 0.5;
        double last_ball_dist = ball_rel.dist( tmp_pos );
        double buf = 0.0;
        if ( n_turn > 0 ) buf += 0.1;
        buf += player_noise;
        buf += ball_noise;
        if ( last_ball_dist < control_area - buf )//0.2 )
        {
#ifdef DEBUG
            dlog.addText( Logger::INTERCEPT,
                          "____dash %d: can reach. last ball dist=%.3f. mod_ctrl_area=%.3f",
                          n_dash, last_ball_dist, control_area - buf );
#endif
            if ( save_recovery && tmp_recovery < M_world.self().recovery() )
            {
#ifdef DEBUG
                dlog.addText( Logger::INTERCEPT,
                              "______dash %d: stamina exhaust",
                              n_dash );
#endif
                *save_recovery = false;
            }
            return true;
        }
#ifdef DEBUG
        dlog.addText( Logger::INTERCEPT,
                      "____%d Near Dash: failed. last_ball_dist=%.3f player_rel=(%.3f %.3f)  ball_rel=(%.3f %.3f)",
                      n_dash,
                      last_ball_dist,
                      tmp_pos.x, tmp_pos.y,
                      ball_rel.x, ball_rel.y );
        dlog.addText( Logger::INTERCEPT,
                      "______noised_ball_x=%.3f  bnoise=%.3f  pnoise=%.3f",
                      noised_ball_x,
                      ball_noise, player_noise );
#endif
        return false;
    }

#ifdef DEBUG
    dlog.addText( Logger::INTERCEPT,
                  "____dash %d: cannot reach. player_rel=(%f %f)  ball_rel=(%f %f)"
                  "  noise_ball_x=%f",
                  n_dash,
                  tmp_pos.x, tmp_pos.y,
                  ball_rel.x, ball_rel.y,
                  noised_ball_x );
#endif

    return false;
}

}
