// -*-c++-*-

/*!
  \file audio_sensor.cpp
  \brief audio message analyzer Source File
*/

/*
 *Copyright:

 Copyright (C) Hidehisa AKIYAMA

 This code is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Lesser General Public License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 *EndCopyright:
 */

/////////////////////////////////////////////////////////////////////

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <cmath>

#include <rcsc/math_util.h>

#include "logger.h"
#include "audio_sensor.h"

namespace rcsc {

/*-------------------------------------------------------------------*/
/*!

 */
AudioSensor::AudioSensor()
    : M_time( 0, 0 )
    , M_ball_pos( 0.0, 0.0 )
    , M_ball_vel( 0.0, 0.0 )
    , M_receiver_number( 0 )
    , M_receive_pos( 50.0, 0.0 )
    , M_goalie_number( 0 )
    , M_goalie_pos( 50.0, 0.0 )
    , M_offside_line_x( 100000.0 )
    , M_defense_line_x( -100000.0 )
{

}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::parsePlayer( const char * msg,
                          const GameTime & current )
{
    /*
      players' communication audio format

      // from self
      v7-: (hear <TIME> self <MSG>)
      v7+: (hear <TIME> self "<MSG>")

      // from other
      v7-: (hear <TIME> <DIR> <MSG>)
      v7:  (hear <TIME> <DIR> "<MSG>")
      v8+: (hear <TIME> <DIR> our <UNUM> "<MSG>")
      (hear <TIME> <DIR> opp "<MSG>")
      (hear <TIME> our)
      (hear <TIME> opp)
    */

    M_time = current;

    long cyc;
    double dir;
    int num;
    char tmp_msg[16];

    if ( std::sscanf( msg, "(hear %ld %lf our %d %s",
                      &cyc, &dir, &num, tmp_msg ) != 4 )
    {
        std::cerr << "***ERROR*** AudioSensor::parse."
                  << " got partial or opponent audio?? ["
                  << msg << "]"
                  << std::endl;
        return;
    }


    const char * msg_ptr = msg;
    while ( *msg_ptr != '\0' && *msg_ptr != '\"' ) ++msg_ptr; // " skip

    if ( std::strlen( msg_ptr ) <= 3 ) // only ["")]
    {
        std::cerr << "***ERROR*** AudioSensor::parsePlayer."
                  << " Empty message "
                  << "[" << msg << "]"
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Too short message [%s]",
                      msg );
        return;
    }


    M_player_message.time_ = current;
    M_player_message.side_ = NEUTRAL;
    M_player_message.unum_ = num;
    M_player_message.type_ = PlayerMessage::NULL_INFO;
    M_player_message.dir_ = dir;

    M_player_message.message_.erase();

    parseTeammateAudioV8( msg_ptr );
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::parseCoach( const char * msg,
                         const GameTime & current )
{
    // (hear <time> online_coach_{left,right} <msg>) : v7-
    // (hear <time> online_coach_{left,right} (freeform "<msg>")) : v7+
    // (hear <time> online_coach_{left,right} <clang>)   : v7+

    M_time = current;

    // skip "(hear "
    while ( *msg != ' ' && *msg != '\0' ) ++msg;
    while ( *msg == ' ' ) ++msg;

    // skip "<time> "
    while ( *msg != ' ' && *msg != '\0' ) ++msg;
    while ( *msg == ' ' ) ++msg;

    // skip sender
    while ( *msg != ' ' && *msg != '\0' ) ++msg;
    while ( *msg == ' ' ) ++msg;

    if ( *msg != '(' )
    {
        // not a clang message
        M_freeform_message.message_.assign( msg );
        if ( ! M_freeform_message.message_.empty()
             && *M_freeform_message.message_.rbegin() == ')' )
        {
            M_freeform_message.message_.erase( M_freeform_message.message_.length() - 1 );
        }

        return;
    }

    // clang message

    int n_read = 0;
    char msg_type[32];

    if ( std::sscanf( msg, "(%[^ ] %n ",
                      msg_type, &n_read ) != 1 )
    {
        std::cerr << "***ERROR*** failed to parse clang message type. ["
                  << msg
                  << std::endl;
        return;
    }

    msg += n_read;

    if ( std::strcmp( msg_type, "freeform" ) != 0 )
    {
        // not a freeform message
        std::cerr << current << ": "
                  << "recv not a freeform message. "
                  << msg_type
                  << std::endl;
        return;
    }

    bool quoted = false;
    if ( *msg != '\0'
         && *msg == '\"' )
    {
        ++msg;
        quoted = true;
    }

    M_freeform_message.time_ = current;
    M_freeform_message.message_.assign( msg );

    // remove quotation
    if ( quoted )
    {
        std::string::size_type qpos = M_freeform_message.message_.find_last_of( '\"' );
        if ( qpos != std::string::npos )
        {
            M_freeform_message.message_.erase( qpos );
        }

        std::cerr << current << ": "
                  << "recv freeform message. ["
                  << M_freeform_message.message_ << ']'
                  << std::endl;
        return;
    }

    // remove last two paren
    for ( int i = 0; i < 2; ++i )
    {
        std::string::size_type ppos = M_freeform_message.message_.find_last_of( ')' );
        if ( ppos != std::string::npos )
        {
            M_freeform_message.message_.erase( ppos );
        }
    }
}


/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::parseTrainer( const char * msg,
                           const GameTime & current )
{

    // (hear <time> referee <msg>) : v7-
    // (hear <time> coach "<msg>")   : v7+
    // (hear <time> coach <clang>)   : v7+

    M_time = current;
    M_trainer_message.time_ = current;
    M_trainer_message.message_.erase();

    int n_read = 0;
    long cycle;
    char sender[32];

    if ( std::sscanf( msg, "(hear %ld %s %n ",
                      &cycle, sender, &n_read ) != 2 )
    {
        std::cerr << "***ERRORR*** failed to parse trainer message. ["
                  << msg << ']'
                  << std::endl;
        return;
    }

    msg += n_read;

    bool quoted = false;
    if ( *msg != '\0'
         && *msg == '\"' )
    {
        ++msg;
        quoted = true;
    }

    M_trainer_message.message_.assign( msg );

    // remove quotation
    if ( quoted )
    {
        std::string::size_type qpos = M_trainer_message.message_.find_last_of( '\"' );
        if ( qpos != std::string::npos )
        {
            M_trainer_message.message_.erase( qpos );
        }
        return;
    }

    // remove last paren
    std::string::size_type ppos = M_trainer_message.message_.find_last_of( ')' );
    if ( ppos != std::string::npos )
    {
        M_trainer_message.message_.erase( ppos );
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
AudioSensor::parseTeammateAudioV8( const char * msg )
{
    // can process only v8+ audio message

    // go to first doubel quatation position
    while ( *msg != '\0' && *msg != '\"' ) ++msg; // "

    // check if double quatation is exist
    if ( *msg != '\"' ) // "
    {
        std::cerr << "***ERROR*** AudioSensor::parseTeammateAudioV8"
                  << " message is not quated [" << msg << "]"
                  << std::endl;
        return false;
    }

    // skip first double quatation
    ++msg;

    // copy raw audio message
    {
        const char * tmp = msg;
        while ( *tmp != '\0' && *tmp != '\"' ) // "
        {
            M_player_message.message_ += *tmp;
            ++tmp;
        }
    }

    /////////////////////////////////////
    // check message type

    // ball info
    if ( ! std::strncmp( msg, "b ", 2 ) )
    {
        if ( *(msg + 8) != '\"' )
        {
            std::cerr << "***ERROR*** AudioSensor::parseTeammateAudioV8"
                      << " Illega ball info [" << msg << "]"
                      << std::endl;
            dlog.addText( Logger::SENSOR,
                          "AudioSensor: Illegal ball info [%s]",
                          msg );
            return false;
        }
        std::string ball_str( msg + 2, 6 );
        if ( M_codec.decodeL6( ball_str, &M_ball_pos, &M_ball_vel ) )
        {
            M_player_message.type_ = PlayerMessage::BALL_INFO;
            return true;
        }
        std::cerr << "***ERROR*** AudioSensor::parseTeammateAudioV8"
                  << " Failed to decode ball [" << msg
                  << "]  str=["  << ball_str << "]"
                  << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Ball Info [%s]",
                      msg );

        return false;
    }
    ////////////////////////////////////////////////////////
    // pass info
    if ( ! std::strncmp( msg, "p ", 2 ) )
    {
        if ( parsePassInfo( msg ) )
        {
            M_player_message.type_ = PlayerMessage::PASS_INFO;
            return true;
        }
        std::cerr  << "***ERROR*** AudioSensor::parseTeammateAudioV8"
                   << " Failed to decode pass [" << msg << "]"
                   << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Pass Info [%s]",
                      msg );
        return false;
    }

    ////////////////////////////////////////////////////////
    // goalie info
    if ( ! std::strncmp( msg, "g ", 2 ) )
    {
        if ( parseGoalieInfo( msg ) )
        {
            M_player_message.type_ = PlayerMessage::OPPONENT_GOALIE_INFO;
            return true;
        }
        std::cerr  << "***ERROR*** AudioSensor::parseTeammateAudioV8"
                   << " Failed to decode goalie [" << msg << "]"
                   << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Goalie Info [%s]",
                      msg );
        return false;
    }

    ////////////////////////////////////////////////////////
    // offside line
    if ( ! std::strncmp( msg, "o ", 2 ) )
    {
        if ( parseOffsideLineInfo( msg ) )
        {
            M_player_message.type_ = PlayerMessage::OFFSIDE_LINE_INFO;
            return true;
        }
        std::cerr  << "***ERROR*** AudioSensor::parseTeammateAudioV8"
                   << " Failed to decode offside line [" << msg << "]"
                   << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Offside Line Info [%s]",
                      msg );
        return false;
    }
    ////////////////////////////////////////////////////////
    // defense line
    if ( ! std::strncmp( msg, "d ", 2 ) )
    {
        if ( parseDefenseLineInfo( msg ) )
        {
            M_player_message.type_ = PlayerMessage::DEFENSE_LINE_INFO;
            return true;
        }
        std::cerr  << "***ERROR*** AudioSensor::parseTeammateAudioV8"
                   << " Failed to decode M_messagetypine [" << msg << "]"
                   << std::endl;
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: Failed to decode Defense Line Info [%s]",
                      msg );
        return false;
    }

    ////////////////////////////////////////////////////////
    // wait request
    if ( ! std::strncmp( msg, "w", 1 ) )
    {
        M_player_message.type_ = PlayerMessage::WAIT_REQUEST;
        return true;
    }

    ////////////////////////////////////////////////////////
    // attack request
    if ( ! std::strncmp( msg, "a", 1 ) )
    {
        M_player_message.type_ = PlayerMessage::ATTACK_REQUEST;
        return true;
    }

    ////////////////////////////////////////////////////////
    // read default ball info

    // decode ball info
    if ( M_codec.decodeL5( M_player_message.message_, &M_ball_pos, &M_ball_vel ) )
    {
        M_player_message.type_ = PlayerMessage::BALL_INFO;

        dlog.addText( Logger::SENSOR,
                      "AudioSensor: success! sender=%d  ball_pos(%.1f %.1f)"
                      " ball_vel(%.1f %.1f)",
                      M_player_message.unum_,
                      M_ball_pos.x, M_ball_pos.y,
                      M_ball_vel.x, M_ball_vel.y );
        return true;
    }

    //std::cerr << "***ERROR*** AudioSensor::parseTeammateAudioV8"
    //          << " Failed to decode ["
    //          << raw_message << "]" << std::endl;
    dlog.addText( Logger::SENSOR,
                  "AudioSensor: unsupported message [%s]",
                  M_player_message.message_.c_str() );

    return false;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
AudioSensor::parsePassInfo( const char * msg )
{
    if ( *msg != 'p' )
    {
        return false;
    }

    // skip space
    msg += 2;

    ///////////////////////////////////////
    // read recever number
    switch ( *msg ) {
    case '1': M_receiver_number = 1; break;
    case '2': M_receiver_number = 2; break;
    case '3': M_receiver_number = 3; break;
    case '4': M_receiver_number = 4; break;
    case '5': M_receiver_number = 5; break;
    case '6': M_receiver_number = 6; break;
    case '7': M_receiver_number = 7; break;
    case '8': M_receiver_number = 8; break;
    case '9': M_receiver_number = 9; break;
    case 'A': M_receiver_number = 10; break;
    case 'B': M_receiver_number = 11; break;
    default:
        std::cerr << "AudioSensor:: Unexpected receiver number " << msg
                  << std::endl;
        return false;
    }

    // skip space
    msg += 2;

    ///////////////////////////////////////
    // read x coordinate
    int x = 0, y = 0;

    x += 100 * (*msg - '0'); ++msg;
    x += 10 * (*msg - '0'); ++msg;
    x += (*msg - '0'); ++msg;

    y += 100 * (*msg - '0'); ++msg;
    y += 10 * (*msg - '0'); ++msg;
    y += (*msg - '0'); ++msg;

    M_receive_pos.x = static_cast< double >( x ) * 0.5 - 52.5;
    M_receive_pos.y = static_cast< double >( y ) * 0.5 - 34.0;


    dlog.addText( Logger::SENSOR,
                  "AudioSensor: success!  receiver= %d  receive_pos(%f %f)",
                  M_receiver_number,
                  M_receive_pos.x, M_receive_pos.y );
    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::encodeBall( const Vector2D & ball_pos,
                         const Vector2D & ball_vel,
                         std::string & say_buf ) const
{
    say_buf = "b ";
    std::string msg = M_codec.encodeL6( ball_pos, ball_vel );
    if ( msg.empty() )
    {
        say_buf.erase();
    }
    else
    {
        say_buf += msg;
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::encodePass( const int receiver,
                         const Vector2D & receive_pos,
                         std::string & say_buf ) const
{
    say_buf = "p ";

    switch ( receiver ) {
    case 1: say_buf += "1 "; break;
    case 2: say_buf += "2 "; break;
    case 3: say_buf += "3 "; break;
    case 4: say_buf += "4 "; break;
    case 5: say_buf += "5 "; break;
    case 6: say_buf += "6 "; break;
    case 7: say_buf += "7 "; break;
    case 8: say_buf += "8 "; break;
    case 9: say_buf += "9 "; break;
    case 10: say_buf += "A "; break;
    case 11: say_buf += "B "; break;
    default:
        std::cerr << "AudioSensor::encode Unexpected receiver number "
                  << receiver << std::endl;
        say_buf.erase();
        return;
    }

    double tmp = min_max( -52.5, receive_pos.x, 52.5 );
    tmp += 52.5;
    tmp *= 2.0;

    int ix = static_cast< int >( rint( tmp ) );

    tmp = min_max( -34.0, receive_pos.y, 34.0 );
    tmp += 34.0;
    tmp *= 2.0;

    int iy = static_cast< int >( rint( tmp ) );


    char xymsg[8];
    std::snprintf( xymsg, 8, "%03d%03d", ix, iy );
    if ( std::strlen( xymsg ) != 6 )
    {
        std::cerr << "AudioSensor::encode XY error "
                  << receive_pos << " -> " << xymsg << std::endl;
        say_buf.erase();
        return;
    }

    say_buf += xymsg;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
AudioSensor::decodeStr7ToPos( const char* msg,
                              Vector2D* pos )
{
    if ( std::strlen( msg ) <= 7 )
    {
        std::cerr << "AudioSensor::decode length error "
                  << "[" << msg << "]" << std::endl;
        return false;
    }

    int x = 0;
    x += 1000 * (*msg - '0'); ++msg;
    x += 100 * (*msg - '0'); ++msg;
    x += 10 * (*msg - '0'); ++msg;
    x += (*msg - '0'); ++msg;

    int y = 0;
    y += 100 * (*msg - '0'); ++msg;
    y += 10 * (*msg - '0'); ++msg;
    y += (*msg - '0'); ++msg;

    pos->x = static_cast< double >( x ) * 0.1 - 52.5;
    pos->y = static_cast< double >( y ) * 0.1 - 34.0;

    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
AudioSensor::encodePosToStr7( const Vector2D & pos,
                              std::string & msg ) const
{
    double tmp = min_max( -52.5, pos.x, 52.5 );
    tmp += 52.5;
    tmp *= 10.0;
    int ix = static_cast< int >( rint( tmp ) );

    tmp = min_max( -34.0, pos.y, 34.0 );
    tmp += 34.0;
    tmp *= 10.0;
    int iy = static_cast< int >( rint( tmp ) );

    char xymsg[10];
    std::snprintf( xymsg, 10, "%04d%03d", ix, iy );
    if ( std::strlen( xymsg ) != 7 )
    {
        std::cerr << "AudioSensor::encodePosToStr7. "
                  << "XY error(2) "
                  << "( " << pos.x << pos.y << ")"
                  << " -> " << xymsg << std::endl;
        return false;
    }

    msg += xymsg;

    return true;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::encodeGoalie( const int goalie_number,
                           const Vector2D & goalie_pos,
                           std::string & say_buf ) const
{
    say_buf = "g ";

    switch ( goalie_number ) {
    case 1: say_buf += "1"; break;
    case 2: say_buf += "2"; break;
    case 3: say_buf += "3"; break;
    case 4: say_buf += "4"; break;
    case 5: say_buf += "5"; break;
    case 6: say_buf += "6"; break;
    case 7: say_buf += "7"; break;
    case 8: say_buf += "8"; break;
    case 9: say_buf += "9"; break;
    case 10: say_buf += "A "; break;
    case 11: say_buf += "B"; break;
    default:
        std::cerr << "AudioSensor::encodeGoalieInfo. "
                  << "Unexpected goalie number "
                  << goalie_number << std::endl;
        say_buf.erase();
        return;
    }

    if ( ! encodePosToStr7( goalie_pos, say_buf ) )
    {
        std::cerr << "AudioSensor::encodeGoalieInfo. "
                  << "Failed to encode goalie info "
                  << "( " << goalie_pos.x << goalie_pos.y << ")"
                  << std::endl;
        say_buf.erase();
    }
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::encodeOffsideLineX( const double & offside_line_x,
                                 std::string & say_buf ) const
{
    char buf[10];
    std::snprintf( buf, 10, "o %.1f", offside_line_x );
    say_buf = buf;
}

/*-------------------------------------------------------------------*/
/*!

 */
void
AudioSensor::encodeDefenseLineX( const double & defense_line_x,
                                 std::string & say_buf ) const
{
    char buf[10];
    std::snprintf( buf, 10, "d %.1f", defense_line_x );
    say_buf = buf;
}

/*-------------------------------------------------------------------*/
/*!

 */
bool
AudioSensor::parseGoalieInfo( const char* msg )
{
    if ( std::strlen( msg ) <= 10
         || msg[0] != 'g'
         || msg[1] != ' ' )
    {
        std::cerr << "AudioSensor::parseGoalieInfo. "
                  << "Unexpected goalie message ["
                  << msg << "]"
                  << std::endl;
        return false;
    }
    msg += 2;

    switch ( *msg ) {
    case '1': M_goalie_number = 1; break;
    case '2': M_goalie_number = 2; break;
    case '3': M_goalie_number = 3; break;
    case '4': M_goalie_number = 4; break;
    case '5': M_goalie_number = 5; break;
    case '6': M_goalie_number = 6; break;
    case '7': M_goalie_number = 7; break;
    case '8': M_goalie_number = 8; break;
    case '9': M_goalie_number = 9; break;
    case 'A': M_goalie_number = 10; break;
    case 'B': M_goalie_number = 11; break;
    default:
        std::cerr << "AudioSensor::parseGoalieInfo. "
                  << "Unexpected goalie number " << msg
                  << std::endl;
        return false;
    }

    ++msg;

    if ( decodeStr7ToPos( msg, &M_goalie_pos ) )
    {
        dlog.addText( Logger::SENSOR,
                      "AudioSensor: success!  goalie pos = (f %f)",
                      M_goalie_pos.x, M_goalie_pos.y );
        return true;
    }

    std::cerr << "AudioSensor::parseGoalieInfo. "
              << " Failed to decode goalie potiiton"
              << std::endl;
    dlog.addText( Logger::SENSOR,
                  "AudioSensor: Failed to decode goalie potiiton" );
    return false;
}

/*-------------------------------------------------------------------*/
/*!
  "o <x>"
 */
bool
AudioSensor::parseOffsideLineInfo( const char * msg )
{
    if ( msg[0] != 'o'
         || msg[1] != ' ' )
    {
        std::cerr << "AudioSensor::parseOffsideLineInfo. "
                  << "Unexpected value message ["
                  << msg << "]"
                  << std::endl;
        return false;
    }
    msg += 2;


    double val = std::strtod( msg, NULL );
    if ( val == -HUGE_VAL
         || val == HUGE_VAL )
    {
        std::cerr << "AudioSensor::parseOffsideLineInfo. "
                  << " Failed to read offside line"
                  << std::endl;
        return false;
    }


    M_offside_line_x = val;

    return true;
}


/*-------------------------------------------------------------------*/
/*!
  "d <x>"
 */
bool
AudioSensor::parseDefenseLineInfo( const char * msg )
{
    if ( msg[0] != 'd'
         || msg[1] != ' ' )
    {
        std::cerr << "AudioSensor::parseDefenseLineInfo. "
                  << "Unexpected value message ["
                  << msg << "]"
                  << std::endl;
        return false;
    }
    msg += 2;


    double val = std::strtod( msg, NULL );
    if ( val == -HUGE_VAL
         || val == HUGE_VAL )
    {
        std::cerr << "AudioSensor::parseDefenseLineInfo. "
                  << " Failed to read defense line"
                  << std::endl;
        return false;
    }


    M_defense_line_x = val;

    return true;
}

}
