# = Log Factory
# 
#--
# Ruby version 1.8
#
# == Authors
# * Yomei Komiya
# 
# == Copyright
# 2008 the original author or authors.
#
# == License
# Apache License 2.0
# 
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
# 
#  http://www.apache.org/licenses/LICENSE-2.0
# 
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#++
# == Version
# SVN: $Id: log_factory.rb 79 2008-10-16 13:51:47Z whitestar $
#
# == Since
# File available since Release 0.8.0

require 'commons/lang/class_loader'
require 'commons/logging/log_configuration_error'
require 'commons/util/properties'

module Commons
  module Logging

    # Abstract base log factory class.
    #
    # Example: 
    #   require 'commons/ruby/log4r'
    #   
    #   module Example
    #     class Sample
    #       LOG = Log4r::Logger.get_logger(self.name)
    #       #@@log = Log4r::Logger.get_logger(self.name)    # for Ruby 1.9 or later.
    #       # ...
    #       if LOG.debug?
    #         LOG.debug('This is DEBUG log.')
    #       end
    #       LOG.info('This is INFO log.')
    #       # ...
    #     end
    #   end      
    class LogFactory
      # Commons.Logging.LogFactory property name.
      FACTORY_PROPERTY = 'commons.logging.LogFactory'
      
      # Default implementation class name of Commons.Logging.LogFactory.
      FACTORY_DEFAULT = 'Commons::Logging::Impl::LogFactoryImpl'
      
      # Default properties file (relative path on $LOAD_PATH)
      FACTORY_PROPERTIES = 'commons-logging.properties'
      
      
      def initialize
        # do nothing.
      end
      
      
      # Cached factories.
      @@factories = Hash.new
      
      
      # Gets Commons.Logging.LogFactory instance.
      # 
      # throws Commons::Logging::LogConfigurationError
      # if the implementation class is not available or cannot be instantiated.
      def self.get_factory
        factory = nil
        
        factory = get_cached_factory
        unless factory == nil
          return factory
        end
        
        props = nil
        props_path = Commons::Lang::ClassLoader.get_resource(FACTORY_PROPERTIES)
        File.open(props_path, 'r') {|file|
          props = Commons::Util::Properties.new
          props.load(file)
        }
        
        # 1st, try the system property.
        factory_class = ENV[FACTORY_PROPERTY]
        unless factory_class == nil
          factory = new_factory(factory_class)
        end
        
        # 2nd try a properties file.
        if factory == nil && props != nil
          factory_class = props.get_property(FACTORY_PROPERTY)
          unless factory_class == nil
            factory = new_factory(factory_class)
          end
        end
        
        # 3rd, try the fallback implementation class
        if factory == nil
          factory = new_factory(FACTORY_DEFAULT)
        end
        
        # cache the factory
        if factory != nil
          cache_factory(factory)
          if props != nil
            props.property_names.each {|name|
              factory.set_attribute(name, props.get_property(name))
            }
          end
        end
        
        return factory
      end
      
      
      def self.get_log(name)
        return get_factory.get_instance(name)
      end
      
      
      # Obtains cached factory.
      def self.get_cached_factory
        if @@factories.has_key?('defaultKey')
          return @@factories['defaultKey']
        else
          return nil
        end
      end
      class << self
        private :get_cached_factory
      end
      
      
      # Caches log factory instance.
      def self.cache_factory(factory)
        unless factory == nil
          @@factories['defaultKey'] = factory
        end
      end
      class << self
        private :cache_factory
      end
      
      
      # Creates and returns new Commons_Logging_LogFactory instance.
      # 
      # throws Commons::Logging::LogConfigurationError
      def self.new_factory(factory_class)
        begin
          Commons::Lang::ClassLoader::load(factory_class)
          return eval(factory_class).new
        rescue LoadError => le
          raise LogConfigurationError, le.message
        rescue Exception => e
          raise LogConfigurationError, e.message
        end
      end
      class << self
        protected :new_factory
      end
    end

  end
end