# = Duration Format Utilities
#  (based on org.apache.commons.lang.time.DurationFormatUtils)
#
#--
# Ruby version 1.8
#
# == Authors
# * Yomei Komiya
# 
# == Copyright
# 2008 the original author or authors.
#
# == License
# Apache License 2.0
# 
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
# 
#  http://www.apache.org/licenses/LICENSE-2.0
# 
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#++
# == Version
# SVN: $Id: duration_format_utils.rb 76 2008-10-12 11:45:33Z whitestar $
#
# == Since
# File available since Release 0.8.0

require 'commons/lang/time/date_utils'

module Commons
  module Lang
    module Time
      
      class DurationFormatUtils
        ISO_EXTENDED_FORMAT_PATTERN = "'P'yyyy'Y'M'M'd'DT'H'H'm'M's.S'S'"
        
        TOKEN_y = :y
        TOKEN_M = :M
        TOKEN_d = :d
        TOKEN_H = :H
        TOKEN_m = :m
        TOKEN_s = :s
        TOKEN_S = :S
        
        
        def initialize
          super()
        end
        
        
        def self.format_duration_hms(duration_millis)
          return format_duration(duration_millis, 'H:mm:ss.SSS')
        end
        
        
        def self.format_duration(duration_millis, format, pad_with_zeros = true)
          tokens = lexx(format)
          
          days         = 0
          hours        = 0
          minutes      = 0
          seconds      = 0
          milliseconds = 0
          
          if Token.contains_token_with_value?(tokens, TOKEN_d)
            days = duration_millis.div(DateUtils::MILLIS_PER_DAY)
            duration_millis = duration_millis - (days * DateUtils::MILLIS_PER_DAY)
          end
          if Token.contains_token_with_value?(tokens, TOKEN_H)
            hours = duration_millis.div(DateUtils::MILLIS_PER_HOUR)
            duration_millis = duration_millis - (hours * DateUtils::MILLIS_PER_HOUR)
          end
          if Token.contains_token_with_value?(tokens, TOKEN_m)
            minutes = duration_millis.div(DateUtils::MILLIS_PER_MINUTE)
            duration_millis = duration_millis - (minutes * DateUtils::MILLIS_PER_MINUTE)
          end
          if Token.contains_token_with_value?(tokens, TOKEN_s)
            seconds = duration_millis.div(DateUtils::MILLIS_PER_SECOND)
            duration_millis = duration_millis - (seconds * DateUtils::MILLIS_PER_SECOND)
          end
          if Token.contains_token_with_value?(tokens, TOKEN_S)
            milliseconds = duration_millis
          end
          
          return format(
            tokens, 0, 0, days, hours, minutes, seconds, milliseconds, pad_with_zeros)
        end
        
        
        def self.format(
          tokens,
          years, months, days, hours, minutes, seconds, milliseconds,
          pad_with_zeros)
          
          buffer = String.new('')
          last_output_seconds = false
          tokens.each {|token|
            value = token.value
            count = token.count
            if value.kind_of?(String)
              buffer.concat(value.to_s)
            else
              if value.equal?(TOKEN_y)
                buffer.concat(
                  pad_with_zeros ? years.to_s.rjust(count, '0') : years.to_s)
                last_output_seconds = false
              elsif value.equal?(TOKEN_M)
                buffer.concat(
                  pad_with_zeros ? months.to_s.rjust(count, '0') : months.to_s)
                last_output_seconds = false
              elsif value.equal?(TOKEN_d)
                buffer.concat(
                  pad_with_zeros ? days.to_s.rjust(count, '0') : days.to_s)
                last_output_seconds = false
              elsif value.equal?(TOKEN_H)
                buffer.concat(
                  pad_with_zeros ? hours.to_s.rjust(count, '0') : hours.to_s)
                last_output_seconds = false
              elsif value.equal?(TOKEN_m)
                buffer.concat(
                  pad_with_zeros ? minutes.to_s.rjust(count, '0') : minutes.to_s)
                last_output_seconds = false
              elsif value.equal?(TOKEN_s)
                buffer.concat(
                  pad_with_zeros ? seconds.to_s.rjust(count, '0') : seconds.to_s)
                last_output_seconds = true
              elsif value.equal?(TOKEN_S)
                if last_output_seconds
                  milliseconds += 1000
                  str = pad_with_zeros \
                    ? milliseconds.to_s.rjust(count, '0') : milliseconds.to_s
                  buffer.concat(str[1..-1])
                else
                  buffer.concat(pad_with_zeros \
                    ? milliseconds.to_s.rjust(count, '0') : milliseconds.to_s)
                end
                last_output_seconds = false
              end
            end
          }
          
          return buffer
        end
        
        
        # Parses a classic date format string into Tokens
        def self.lexx(format)
          chars = format.split(//)
          tokens = Array.new
          
          in_literal = false
          buffer = nil
          previous = nil
          chars.each {|char|
            if in_literal && char != "'"
              buffer.concat(char)
              next
            end
            value = nil
            case char
              # TODO: Need to handle escaping of '
              when "'"
                if in_literal
                  buffer = nil
                  in_literal = false
                else
                  buffer = String.new('')
                  tokens.push(Token.new(buffer))
                  in_literal = true
                end
              when 'y' then value = TOKEN_y
              when 'M' then value = TOKEN_M
              when 'd' then value = TOKEN_d
              when 'H' then value = TOKEN_H
              when 'm' then value = TOKEN_m
              when 's' then value = TOKEN_s
              when 'S' then value = TOKEN_S
              else
                if buffer == nil
                  buffer = String.new('')
                  tokens.push(Token.new(buffer))
                end
                buffer.concat(char)
            end
        
            if value != nil
              if previous != nil && previous.value.equal?(value)
                previous.increment
              else
                token = Token.new(value)
                tokens.push(token)
                previous = token
              end
              buffer = nil
            end
          }
          
          return tokens
        end
      
      
        class Token
          attr_reader :value, :count
          
          def self.contains_token_with_value?(tokens, value)
            tokens.each {|token|
              if token.value.equal?(value)
                return true
              end
            }
            return false
          end
        
        
          def initialize(value, count = 1)
            super()
            
            @value = value
            @count = count
          end
          
          
          def increment
            @count += 1
          end
          
          
          def ==(other)
            return self.eql?(other)
          end
          
          
          def eql?(other)
            if other.kind_of?(Token)
              if !@value.class.equal?(other.value.class)
                return false
              end
              if @count != other.count
                return false
              end
              if @value.kind_of?(String)
                return @value.to_s == other.value.to_s
              elsif @value.kind_of?(Numeric)
                return @value == other.value
              else
                return @value.equal?(other.value)
              end
            end
  
            return false
          end
          
          
          def hash
            return @value.hash
          end
          
          
          def to_s
            return @value.to_s * @count
          end
          
          
          def to_str
            return self.to_s
          end
        end # Token
        
      end # DurationFormatUtils
      
    end
  end
end
