/*
 * Created on 2005/01/23
 *
 * @author KAMO Masahiko (mkamo@users.sourceforge.jp)
 */
package razgriz.news.ui.views;

import java.util.Iterator;

import org.eclipse.jface.action.Action;
import org.eclipse.jface.action.IMenuListener;
import org.eclipse.jface.action.IMenuManager;
import org.eclipse.jface.action.IToolBarManager;
import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.action.Separator;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Menu;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.ViewPart;

import razgriz.news.NewsException;
import razgriz.news.model.News;
import razgriz.news.ui.NewsUIPlugin;
import razgriz.news.ui.actions.AddGroupAction;
import razgriz.news.ui.actions.AddNewsAction;
import razgriz.news.ui.actions.AddRootGroupAction;
import razgriz.news.ui.actions.EditGroupAction;
import razgriz.news.ui.actions.EditNewsAction;
import razgriz.news.ui.actions.ExportOPMLAction;
import razgriz.news.ui.actions.ImportOPMLAction;
import razgriz.news.ui.actions.MarkAsReadAllAction;
import razgriz.news.ui.actions.RemoveFavoriteAction;
import razgriz.news.visitor.NewsClippingVisitor;

/**
 * 
 */
public class FavoriteListView extends ViewPart {
    public static final String ID = FavoriteListView.class.getName();
    
    private Composite parent;

    private TreeViewer viewer;
    private FavoriteListTreeContentProvider provider;

    private Action addGroupAction;
    private Action addRootGroupAction;
    private Action addNewsAction;
    private Action editGroupAction;
    private Action editNewsAction;
    private Action removeFavoriteAction;
    private Action markAsReadAllAction;
    private Action importOPMLAction;
    private Action exportOPMLAction;

    public void createPartControl(Composite parent) {
        this.parent = parent;

        viewer = new TreeViewer(
            parent,
            SWT.V_SCROLL | SWT.H_SCROLL | SWT.MULTI
        );

        viewer.setContentProvider(new FavoriteListTreeContentProvider());
        viewer.setLabelProvider(new FavoriteListTreeContentProvider());
        viewer.setInput(NewsUIPlugin.getFavoriteList());
        
        makeActions();
        contributeToActionBars();
        hookContextMenu();
        hookDoubleClickAction();
    }

    public void setFocus() {
        viewer.getTree().setFocus();
    }

    private ImageDescriptor getSharedImageDescriptor(String descstr) {
        return PlatformUI.getWorkbench().getSharedImages().
            getImageDescriptor(descstr);
    }

    private void makeActions() {
        addGroupAction = new AddGroupAction(getSite(), viewer);
        addRootGroupAction = new AddRootGroupAction(getSite(), viewer);
        addNewsAction = new AddNewsAction(getSite(), viewer);
        editGroupAction = new EditGroupAction(getSite(), viewer);
        editNewsAction = new EditNewsAction(getSite(), viewer);
        removeFavoriteAction = new RemoveFavoriteAction(getSite(), viewer);
        markAsReadAllAction= new MarkAsReadAllAction(getSite(), viewer);
        importOPMLAction = new ImportOPMLAction(getSite(), viewer);
        exportOPMLAction = new ExportOPMLAction(getSite(), viewer);
    }

    private void contributeToActionBars() {
        IActionBars bars = getViewSite().getActionBars();
        fillLocalPullDown(bars.getMenuManager());
        fillLocalToolBar(bars.getToolBarManager());
    }
    private void hookContextMenu() {
        MenuManager menuMgr = new MenuManager("#PopupMenu");
        menuMgr.setRemoveAllWhenShown(true);
        menuMgr.addMenuListener(new IMenuListener() {
            public void menuAboutToShow(IMenuManager manager) {
                FavoriteListView.this.fillContextMenu(manager);
            }
        });
        Menu menu = menuMgr.createContextMenu(viewer.getControl());
        viewer.getControl().setMenu(menu);
        getSite().registerContextMenu(menuMgr, viewer);
    }
    private void hookDoubleClickAction() {
        viewer.addDoubleClickListener(new IDoubleClickListener() {
            public void doubleClick(DoubleClickEvent event) {
                IStructuredSelection sel =
                    (IStructuredSelection) event.getSelection();
                for (Iterator ite = sel.iterator(); ite.hasNext(); ) {
                    Object obj = ite.next();
                    if (!(obj instanceof News)) {
                        continue;
                    }
                    News news = (News) obj;
                    try {
                        news.accept(new NewsClippingVisitor());

                        IWorkbenchPage wpage = getSite().getPage();
                        ArticleListView view = (ArticleListView) wpage.showView(
                            ArticleListView.ID,
                            ArticleListView.createSecondaryId(news),
                            IWorkbenchPage.VIEW_ACTIVATE
                        );
                        view.setInput(news);
                    } catch(NewsException e) {
                        MessageDialog.openError(
                            getSite().getShell(),
                            "L̍XVG[",
                            "L̍XVɎs܂:" + e.getMessage()
                        );
                    } catch(Exception e) {
                        throw new IllegalArgumentException(e.getMessage()); // todo:
                    }
                }
            }
        });
    }
    private void fillLocalPullDown(IMenuManager manager) {
        manager.add(addRootGroupAction);
    }
    private void fillLocalToolBar(IToolBarManager manager) {
        manager.add(addRootGroupAction);
    }

    private void fillContextMenu(IMenuManager manager) {
        manager.add(addNewsAction);
        manager.add(addRootGroupAction);
        manager.add(addGroupAction);
        manager.add(new Separator());
        manager.add(editNewsAction);
        manager.add(editGroupAction);
        manager.add(new Separator());
        manager.add(removeFavoriteAction);
        manager.add(new Separator());
        manager.add(markAsReadAllAction);
        manager.add(new Separator());
        manager.add(importOPMLAction);
        manager.add(exportOPMLAction);
        manager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS));
        manager.add(new Separator(IWorkbenchActionConstants.MB_ADDITIONS + "end"));
    }


}
