// MainDialog.h
// 2008/12/04

#pragma once

#include "resource.h"

#include "QAXDecoder.h"

// MainDialog
class MainDialog :
	public ATL::CDialogImpl<MainDialog>,
	public WTL::CMessageFilter {

	enum {
		TID_TICK     =  1,
		TICK_TIMEOUT = 20
	};

	ATL::CComPtr<IDirectSound8> m_ds;

	QAX::DSoundBuffer m_Buffer;

	INT32 m_Channels;
	INT32 m_SamplingRate;

	QAX::QDecoderFactory* m_pDecoderFactory;

	QAX::QFileMapping*   m_pFileMapping;
	QAX::QDecoder*       m_pDecoder;
	QAX::QDecoderOutput* m_pOutput;

	UINT64 m_iPosition;
	bool   m_bEndOfStream;

	bool   m_bLoopMode;
	UINT64 m_iLoopPoint;

public:

	/* */

	enum { IDD = IDD_MAIN };

	/* */

	MainDialog() :
		m_Channels(0),
		m_SamplingRate(0),
		m_pDecoderFactory(0),
		m_pFileMapping(0),
		m_pDecoder(0),
		m_pOutput(0),
		m_iPosition(0),
		m_bEndOfStream(false),
		m_bLoopMode(false),
		m_iLoopPoint(0)
	{
	}

	~MainDialog()
	{
		if (m_pOutput != 0) {
			m_pOutput->Release();
			m_pOutput = 0;
		}

		if (m_pDecoder != 0) {
			m_pDecoder->Release();
			m_pDecoder = 0;
		}

		if (m_pFileMapping != 0) {
			m_pFileMapping->Release();
			m_pFileMapping = 0;
		}

		if (m_pDecoderFactory != 0) {
			m_pDecoderFactory->Release();
			m_pDecoderFactory = 0;
		}
	}

	/* */

	BEGIN_MSG_MAP_EX(MainDialog)
		MESSAGE_HANDLER(WM_INITDIALOG, OnInitDialog)
		MESSAGE_HANDLER(WM_COMMAND,    OnCommand   )

		MSG_WM_DESTROY(OnDestroy)

		MSG_WM_TIMER(OnTimer)
	END_MSG_MAP()

	/* */

	LRESULT OnInitDialog(
		UINT   uMsg,
		WPARAM wParam,
		LPARAM lParam,
		BOOL&  bHandled)
	{
		SetWindowPos(
			0,
			0, 0,
			0, 0,
			SWP_NOZORDER | SWP_NOSIZE);

		SetTimer(TID_TICK, TICK_TIMEOUT);

		SetDlgItemText(IDC_STAT, L"");

		::EnableWindow(GetDlgItem(IDC_OPEN), FALSE);
		::EnableWindow(GetDlgItem(IDC_PLAY), FALSE);
		::EnableWindow(GetDlgItem(IDC_STOP), FALSE);

		try {
			m_pDecoderFactory = QAX::QDecoderFactoryCreate();

		} catch (QAX::Exception&) {
			return TRUE;
		}

		::EnableWindow(GetDlgItem(IDC_OPEN), TRUE);

		return TRUE;
	}

	LRESULT OnCommand(
		UINT   uMsg,
		WPARAM wParam,
		LPARAM lParam,
		BOOL&  bHandled)
	{
		int id   = LOWORD(wParam);
		int code = HIWORD(wParam);

		switch (id) {
		case IDC_ITEMS:
		{
			if (code == LBN_SELCHANGE) {
				::EnableWindow(GetDlgItem(IDC_PLAY), TRUE );
				::EnableWindow(GetDlgItem(IDC_STOP), FALSE);

			} else if (code == LBN_DBLCLK) {
				Play();
			}

			break;
		}

		case IDC_OPEN:
		{
			WTL::CFileDialog dlg(
				TRUE,
				L"qax",
				0,
				OFN_HIDEREADONLY,
				L"QAX Files (*.qax)\0*.qax\0All Files (*.*)\0*.*\0\0");

			if (dlg.DoModal() == IDOK) {
				Open(dlg.m_szFileName);
			}

			break;
		}

		case IDC_PLAY:
		{
			Play();
			break;
		}

		case IDC_STOP:
		{
			Stop();
			break;
		}

		case IDOK:
		case IDCANCEL:
		case IDCLOSE:
			DestroyWindow();
			break;
		}

		return 0;
	}

	/* */

	void OnDestroy()
	{
		m_Buffer.Reset();
		m_Buffer.Release();

		m_ds.Release();

		PostQuitMessage(0);
	}

	/* */

	void OnTimer(UINT_PTR nIDEvent)
	{
		if (nIDEvent == TID_TICK) {
			OnTick();
		}
	}

	/* */

	virtual BOOL PreTranslateMessage(MSG* pMsg)
	{
		HWND hwnd = *this;

		if (hwnd != 0) {
			if ((pMsg->hwnd == hwnd || pMsg->hwnd == GetDlgItem(IDC_ITEMS))
				&& pMsg->message == WM_KEYDOWN) {
				int code = (int)pMsg->wParam;
				if (code == ' ') {
					PlayNext();
					return TRUE;
				}

				if (code == 'N') {
					WTL::CButton check = GetDlgItem(IDC_AUTO);
					int m = check.GetCheck();
					check.SetCheck((m == BST_CHECKED) ? BST_UNCHECKED : BST_CHECKED);
					return TRUE;
				}
			}

			if (::IsDialogMessageW(hwnd, pMsg)) {
				return TRUE;
			}
		}

		return FALSE;
	}

	/* */

	void CreateDSound(IDirectSound8* ds)
	{
		m_ds = ds;
	}

	void CreateBuffer(
		INT32 channels,
		INT32 samplingRate)
	{
		if (m_Channels == channels && m_SamplingRate == samplingRate) {
			return;
		}

		m_Buffer.Release();

		WAVEFORMATEX wfex = { 0 };

		wfex.wFormatTag      = WAVE_FORMAT_PCM;
		wfex.nChannels       = channels;
		wfex.nSamplesPerSec  = samplingRate;
		wfex.nAvgBytesPerSec = samplingRate * channels * 2;
		wfex.nBlockAlign     = channels * 2;
		wfex.wBitsPerSample  = 16;

		HRESULT hRslt = m_Buffer.Create(
			m_ds,
			&wfex,
			samplingRate * channels * 2);
		if (FAILED(hRslt)) {
			DestroyWindow();
			return;
		}

		m_Channels     = channels;
		m_SamplingRate = samplingRate;
	}

	/* */

	void Reset()
	{
		m_Buffer.Reset();

		if (m_pOutput != 0) {
			m_pOutput->Release();
			m_pOutput = 0;
		}

		if (m_pDecoder != 0) {
			m_pDecoder->Release();
			m_pDecoder = 0;
		}

		if (m_pFileMapping != 0) {
			m_pFileMapping->Release();
			m_pFileMapping = 0;
		}

		m_iPosition = 0;
		m_bEndOfStream = false;

		::EnableWindow(GetDlgItem(IDC_PLAY), FALSE);
		::EnableWindow(GetDlgItem(IDC_STOP), FALSE);

		WTL::CListBox box = GetDlgItem(IDC_ITEMS);

		box.ResetContent();
	}

	void Open(LPCWSTR path)
	{
		Reset();

		SetDlgItemText(IDC_STAT, L"Loading...");

		try {
			m_pFileMapping = QAX::QFileMappingCreate(path);

			m_pDecoder = QAX::QDecoderCreate(
				m_pDecoderFactory,
				m_pFileMapping);

			WTL::CListBox box = GetDlgItem(IDC_ITEMS);

			INT32 count = m_pDecoder->GetItemCount();
			for (INT32 i = 0; i < count; i++) {
				QAX::QItem item = m_pDecoder->GetItem(i);
				CStringW name(item.Name.c_str());
				box.AddString(name);
			}

		} catch (QAX::Exception&) {
			return;
		}

		SetDlgItemText(IDC_STAT, L"Ready...");
	}

	/* */

	void Play()
	{
		m_Buffer.Reset();

		if (m_pOutput != 0) {
			m_pOutput->Release();
			m_pOutput = 0;
		}

		SetDlgItemText(IDC_STAT, L"Playing...");

		try {
			WTL::CListBox box = GetDlgItem(IDC_ITEMS);

			INT32 index = box.GetCurSel();
			if (index < 0) {
				return;
			}

			CStringW wname;
			box.GetText(index, wname);

			CStringA name(wname);
			m_pOutput = m_pDecoder->CreateOutput(name);
			if (m_pOutput == 0) {
				return;
			}

			QAX::QDecoderSpec spec = m_pOutput->GetDecoderSpec();

			CreateBuffer(
				spec.Channels,
				spec.SamplingRate);

			m_iPosition = 0;
			m_bEndOfStream = false;

			m_bLoopMode  = false;
			m_iLoopPoint = 0;

			std::string val = m_pOutput->QueryProperty("LOOP_POINT");
			if (!val.empty()) {
				m_bLoopMode  = true;
				m_iLoopPoint = _atoi64(val.c_str());
			}

			UINT32 feedable = m_Buffer.GetFeedable(0);
			Feed(feedable);

			KillTimer(TID_TICK);
			SetTimer(TID_TICK, TICK_TIMEOUT);

			HRESULT hRslt = m_Buffer.Play();
			if (FAILED(hRslt)) {
				return;
			}

		} catch (QAX::Exception&) {
			return;
		}

		::EnableWindow(GetDlgItem(IDC_PLAY), FALSE);
		::EnableWindow(GetDlgItem(IDC_STOP), TRUE );
	}

	void Stop()
	{
		m_Buffer.Reset();

		if (m_pOutput != 0) {
			m_pOutput->Release();
			m_pOutput = 0;
		}

		::EnableWindow(GetDlgItem(IDC_PLAY), TRUE );
		::EnableWindow(GetDlgItem(IDC_STOP), FALSE);

		SetDlgItemText(IDC_STAT, L"Stoped.");
	}

	/* */

	void OnTick()
	{
		if (m_pOutput == 0) {
			return;
		}

		UINT64 current = m_Buffer.Poll();

		if (m_bEndOfStream) {
			UINT64 last = m_iPosition * m_Channels * 2;
			if (current >= last) {
				WTL::CButton check = GetDlgItem(IDC_AUTO);
				if (check.GetCheck() != BST_CHECKED) {
					Stop();
				} else {
					PlayNext();
				}
				return;
			}
		}

		UINT32 size     = m_Buffer.GetBufferSize() / 4;
		UINT32 feedable = m_Buffer.GetFeedable(current);
		if (feedable >= size) {
			Feed(size);
		}

		UINT64 sample = current / (m_Channels * 2);

		INT32 dsecs = INT32(sample / (m_SamplingRate / 10));
		INT32 secs  = dsecs / 10;

		CStringW buf;
		buf.Format(L"[ %d:%02d.%d ]", secs / 60, secs % 60, dsecs % 10);

		SetDlgItemText(IDC_STAT, buf);
	}

	void Feed(UINT32 cbSize)
	{
		VOID*  pv = 0;
		UINT32 cb = 0;

		HRESULT hRslt = m_Buffer.Lock(
			&pv,
			&cb,
			cbSize);
		if (FAILED(hRslt)) {
			Stop();
			return;
		}

		try {
			UINT32 samples = cbSize / (m_Channels * 2);

			INT16* p = static_cast<INT16*>(pv);

			while (samples > 0) {
				UINT32 count = m_pOutput->Decode(
					p,
					samples);
				if (count == 0) {
					if (m_bLoopMode) {
						m_pOutput->Seek(m_iLoopPoint);
						continue;

					} else {
						m_bEndOfStream = true;
						memset(p, 0, samples * m_Channels * 2);
						break;
					}
				}

				m_iPosition += count;

				p += count * m_Channels;

				samples -= count;
			}

		} catch (QAX::Exception&) {

		}

		hRslt = m_Buffer.Unlock(
			pv,
			cb);
		if (FAILED(hRslt)) {
			return;
		}
	}

	/* */

	void PlayNext()
	{
		WTL::CListBox box = GetDlgItem(IDC_ITEMS);

		INT32 index = box.GetCurSel();
		INT32 count = box.GetCount();

		if (index < 0 || index == count - 1) {
			box.SetCurSel(0);
		} else {
			box.SetCurSel(index + 1);
		}

		Play();
	}

	/* */

}; // MainDialog

/* */

