// QAXWriter.cs
// 2008/11/19

using System;
using System.IO;
using System.Text;
using System.Collections.Generic;
using System.Runtime.InteropServices;

namespace QAX {

// QAXHeader
[StructLayout(LayoutKind.Sequential)]
public struct QAXHeader {

	public UInt32 Signature;
	public UInt32 Sign1;
	public UInt32 Sign2;
	public UInt32 Sign3;

	public UInt64 ContentPos;
	public UInt64 ContentSize;

	public UInt64 ItemsPos;
	public UInt32 ItemsSize;
	public UInt32 ItemsIndex;

	public UInt64 StringsPos;
	public UInt32 StringsSize;
	public UInt32 StringsIndex;

	public UInt64 SetupsPos;
	public UInt32 SetupsSize;
	public UInt32 SetupsIndex;

	public UInt64 IndexsPos;
	public UInt32 IndexsSize;
	public UInt32 IndexsIndex;

} // QAXHeader

// QAXWriter
public class QAXWriter {

	private Stream _s;

	private Int64 _position;

	private const Int32 ZERO_PAGE = 0x1000;

	public const Int32 PAGE_UNIT = 0x1000;

	private StringPool _strings;
	private SetupPool  _setups;
	private DataPool   _indexs;

	private ItemPool _items;

	private UInt64 _cposition;

	public const Byte F_START = 0x01;
	public const Byte F_END   = 0x02;

	private PropertyBag _props;

	public QAXWriter(Stream s)
	{
		_s = s;

		_position = 0;

		_strings = new StringPool();
		_setups  = new SetupPool();
		_indexs  = new DataPool();

		_items = new ItemPool();

		_cposition = 0;

		_props = null;
	}

	public void SetupPropertyBag(PropertyBag bag)
	{
		_props = bag;
	}

	public void Prepare()
	{
		_position = _s.Position;

		Byte[] by = new Byte[ZERO_PAGE];
		_s.Write(by, 0, by.Length);
	}

	public void AddVorbisContent(
		String name,
		Stream s)
	{
		var r = new QOgg.Checker(s);

		r.ReadSetup();

		Byte[] id    = r.Id;
		Byte[] setup = r.Setup;

		using (var vsetup = new QVorbis.Setup()) {
			vsetup.Create(id, setup);

			using (var vchecker = new QVorbis.Checker()) {
				vchecker.Setup(vsetup);

				var w = new QAXContentWriter(_s);

				UInt64 last_gpos = 0;

				UInt64 pos = 0;

				while (r.ReadPage()) {
					Byte[][] packets = r.Packets;

					UInt64 gpos  = r.GranulePosition;
					UInt32 flags = r.Flags;

					Int32 duration = 0;
					Int32 offset   = 0;

					for (Int32 i = 0; i < packets.Length; i++) {
						Byte[] packet = packets[i];

						Int32 samples = vchecker.Check(packet);
						if (i == 0) {
							offset = samples;
						}

						duration += samples;
					}

					Byte cflags = 0;
					if (pos == 0) {
						cflags |= F_START;
					}

					if ((flags & QOgg.Header.HT_EOS) != 0) {
						cflags |= F_END;
					}

					w.Write(
						packets,
						cflags,
						duration,
						offset);

					if ((flags & QOgg.Header.HT_EOS) != 0) {
						last_gpos = gpos;
					}

					pos += (UInt64)duration;
				}

				if (last_gpos == 0) {
					last_gpos = pos;
				}

				UInt64 csize = w.Finish();

				Int32 cid = _strings.Add("vorbis");

				Int32 cname  = _strings.Add(name);
				Int32 csetup = _setups.Add(id, setup);
				Int32 cindex = _indexs.Add(w.GetIndex());

				var entry = new ItemEntry();

				entry.Key = name;

				entry.Type  = cid;
				entry.Name  = cname;
				entry.Setup = csetup;
				entry.Index = cindex;

				entry.Position = _cposition;
				entry.Size     = w.Size;

				entry.Samples = last_gpos;

				entry.Props = null;

				if (_props != null) {
					entry.Props = _props.GetProperty(name, _strings);
				}

				_items.Add(entry);

				_cposition += csize;
			}
		}
	}

	/* */

	public void AddTTAContent(
		String name,
		Stream s)
	{
		var tta = new TTAOutputer(s);

		UInt64 csize = tta.OutputContent(_s);

		Int32 cid    = _strings.Add("TTA1");
		Int32 cname  = _strings.Add(name);
		Int32 csetup = _setups.Add(tta.GetSetup());
		Int32 cindex = _indexs.Add(tta.GetIndex());

		var entry = new ItemEntry();

		entry.Key = name;

		entry.Type  = cid;
		entry.Name  = cname;
		entry.Setup = csetup;
		entry.Index = cindex;

		entry.Position = _cposition;
		entry.Size     = tta.Size;

		entry.Samples = tta.Samples;

		entry.Props = null;

		if (_props != null) {
			entry.Props = _props.GetProperty(name, _strings);
		}

		_items.Add(entry);

		_cposition += csize;
	}

	/* */

	public void Finish()
	{
		var c_item = _items.ToChunk();
		var c_strs = _strings.ToChunk();
		var c_sups = _setups.ToChunk();
		var c_idxs = _indexs.ToChunk();

		UInt64 pos_item = _cposition;
		_cposition += (UInt64)WriteChunk(c_item);

		UInt64 pos_strs = _cposition;
		_cposition += (UInt64)WriteChunk(c_strs);

		UInt64 pos_sups = _cposition;
		_cposition += (UInt64)WriteChunk(c_sups);

		UInt64 pos_idxs = _cposition;
		_cposition += (UInt64)WriteChunk(c_idxs);

		/* */

		var header = new QAXHeader();

		header.Signature = ChunkId("QAX1");

		header.ContentPos  = ZERO_PAGE;
		header.ContentSize = pos_item;

		header.ItemsPos   = pos_item + ZERO_PAGE;
		header.ItemsSize  = (UInt32)c_item.Payload.Length;
		header.ItemsIndex = (UInt32)c_item.Index;

		header.StringsPos   = pos_strs + ZERO_PAGE;
		header.StringsSize  = (UInt32)c_strs.Payload.Length;
		header.StringsIndex = (UInt32)c_strs.Index;

		header.SetupsPos   = pos_sups + ZERO_PAGE;
		header.SetupsSize  = (UInt32)c_sups.Payload.Length;
		header.SetupsIndex = (UInt32)c_sups.Index;

		header.IndexsPos   = pos_idxs + ZERO_PAGE;
		header.IndexsSize  = (UInt32)c_idxs.Payload.Length;
		header.IndexsIndex = (UInt32)c_idxs.Index;

		Int32 len = Marshal.SizeOf(typeof(QAXHeader));
		Byte[] by = new Byte[len];

		var gh = GCHandle.Alloc(by, GCHandleType.Pinned);
		try {
			IntPtr p = gh.AddrOfPinnedObject();
			Marshal.StructureToPtr(header, p, false);
		} finally {
			gh.Free();
		}

		_s.Seek(_position, SeekOrigin.Begin);
		_s.Write(by, 0, by.Length);
	}

	private Int32 WriteChunk(Chunk c)
	{
		Byte[] by = c.Payload;

		_s.Write(by, 0, by.Length);

		Int32 sz = ((by.Length + PAGE_UNIT - 1) / PAGE_UNIT) * PAGE_UNIT;

		Int32 len = sz - by.Length;
		if (len > 0) {
			Byte[] pad = new Byte[len];
			for (Int32 i = 0; i < pad.Length; i++) {
				pad[i] = 0xff;
			}
			_s.Write(pad, 0, pad.Length);
		}

		return sz;
	}

	private static UInt32 ChunkId(String id)
	{
		UInt32 v = 0;
		v |=  (UInt32)id[0];
		v |= ((UInt32)id[1] <<  8);
		v |= ((UInt32)id[2] << 16);
		v |= ((UInt32)id[3] << 24);
		return v;
	}

} // QAXWriter

// QAXContentWriter
public class QAXContentWriter {

	private Stream _s;

	private UInt64 _total;

	private Index _index;

	private const UInt64 PAGE_UNIT = 0x1000;

	public QAXContentWriter(Stream s)
	{
		_s = s;

		_total = 0;

		_index = new Index();
	}

	public Index Index {
		get { return _index; }
	}

	public UInt64 Size {
		get { return _total; }
	}

	public void Write(
		Byte[][] packets,
		Byte     flag,
		Int32    duration,
		Int32    offset)
	{
		var ms = new MemoryStream();

		Int32[] lens = new Int32[packets.Length];
		for (Int32 i = 0; i < lens.Length; i++) {
			lens[i] = packets[i].Length;
		}

		using (var w = new BinaryWriter(ms)) {
			w.Write(flag);

			Utils.WriteLengths(w, lens);

			for (Int32 i = 0; i < packets.Length; i++) {
				w.Write(packets[i]);
			}
		}

		Byte[] page = ms.ToArray();

		var crc = new QVorbis.CRC32();
		Byte[] bcrc = BitConverter.GetBytes(
			crc.Generate(page));

		var e = new IndexEntry();

		e.Size     = page.Length + bcrc.Length;
		e.Duration = duration;
		e.Offset   = offset;

		_index.Add(e);

		_total += (UInt64)(page.Length + bcrc.Length);

		_s.Write(page, 0, page.Length);
		_s.Write(bcrc, 0, bcrc.Length);
	}

	public UInt64 Finish()
	{
		return Finish(PAGE_UNIT);
	}

	public UInt64 Finish(UInt64 unit)
	{
		UInt64 sz = ((_total + unit - 1) / unit) * unit;

		Int32 len = (Int32)(sz - _total);
		if (len > 0) {
			Byte[] pad = new Byte[len];
			for (Int32 i = 0; i < pad.Length; i++) {
				pad[i] = 0xff;
			}
			_s.Write(pad, 0, pad.Length);
		}

		return sz;
	}

	public static UInt64 Finish(Stream s, UInt64 total, UInt64 unit)
	{
		UInt64 sz = ((total + unit - 1) / unit) * unit;

		Int32 len = (Int32)(sz - total);
		if (len > 0) {
			Byte[] pad = new Byte[len];
			for (Int32 i = 0; i < pad.Length; i++) {
				pad[i] = 0xff;
			}
			s.Write(pad, 0, pad.Length);
		}

		return sz;
	}

	public Byte[] GetIndex()
	{
		return _index.ToArray();
	}

} // QAXContentWriter

} // namespace QAX

