/* DecoderSetup.c */
/* 2008/11/06     */

#include "StdAfx.h"

#include "DecoderSetup.h"

/* */

BOOL DecoderSetup_Id(
	DecoderContext_t* ctx,
	BitDecoder_t*     d,
	DecoderSetup_t*   setup)
{
	BOOL result = FALSE;

	UINT32 val;

	INT32 i;
	UINT8 sig[6];

	val = BitDecoder_GetBits(d, 8);
	if (val != V_PACKET_ID) {
		THROW0(ctx, "DecoderSetup_Id", "InvalidPacketId")
	}

	for (i = 0; i < 6; i++) {
		sig[i] = BitDecoder_GetBits(d, 8);
	}
	if (memcmp(sig, "vorbis", 6) != 0) {
		THROW0(ctx, "DecoderSetup_Id", "InvalidSignature")
	}

	val = BitDecoder_GetBits(d, 32);
	if (val != 0) {
		THROW0(ctx, "DecoderSetup_Id", "InvalidVersion")
	}

	setup->Channels   = BitDecoder_GetBits(d,  8);
	setup->SampleRate = BitDecoder_GetBits(d, 32);

	val = BitDecoder_GetBits(d, 32); /* max bitrate */
	val = BitDecoder_GetBits(d, 32); /* nom bitrate */
	val = BitDecoder_GetBits(d, 32); /* min bitrate */

	setup->BlockSize0 = 1 << BitDecoder_GetBits(d, 4);
	setup->BlockSize1 = 1 << BitDecoder_GetBits(d, 4);

	if (setup->BlockSize0 < 64 || setup->BlockSize0 > 8192) {
		THROW0(ctx, "DecoderSetup_Id", "InvalidBlockSize0")
	}

	if (setup->BlockSize1 < 64 || setup->BlockSize1 > 8192) {
		THROW0(ctx, "DecoderSetup_Id", "InvalidBlockSize0")
	}

	if (setup->BlockSize1 < setup->BlockSize0) {
		THROW0(ctx, "DecoderSetup_Id", "InvalidBlockSize")
	}

	/* EOP */

	val = BitDecoder_GetBits(d, 1);
	if (val != 1) {
		THROW0(ctx, "DecoderSetup_Id", "EOPMissing")
	}

	/* */

	if (setup->Channels != 1 && setup->Channels != 2) {
		THROW0(ctx, "DecoderSetup_Id", "InvalidChannels")
	}

	/* */

	result = TRUE;

ErrorEnd:

	return result;
}

/* */

static BOOL SkipTimeDomainTransfer(
	DecoderContext_t* ctx,
	BitDecoder_t*     d);

static BOOL MappingSetup_Setup(
	DecoderContext_t* ctx,
	BitDecoder_t*     d,
	INT32             channels,
	MappingSetup_t*   setup);

static BOOL ModeSetup_Setup(
	DecoderContext_t* ctx,
	BitDecoder_t*     d,
	ModeSetup_t*      setup);

/* */

static BOOL CheckIntegrity_Setup(
	DecoderContext_t* ctx,
	DecoderSetup_t*   setup);

/* */

BOOL DecoderSetup_Setup(
	DecoderContext_t* ctx,
	BitDecoder_t*     d,
	DecoderSetup_t*   setup)
{
	BOOL result = FALSE;

	MemoryPool_t* pool = ctx->Pool;

	UINT32 val;

	INT32 i;
	UINT8 sig[6];

	val = BitDecoder_GetBits(d, 8);
	if (val != V_PACKET_SETUP) {
		THROW0(ctx, "DecoderSetup_Setup", "InvalidPacketId")
	}

	for (i = 0; i < 6; i++) {
		sig[i] = BitDecoder_GetBits(d, 8);
	}
	if (memcmp(sig, "vorbis", 6) != 0) {
		THROW0(ctx, "DecoderSetup_Setup", "InvalidSignature")
	}

	/* Huffman CodeBook */

	setup->HuffmanCount   = BitDecoder_GetBits(d, 8) + 1;
	setup->HuffmanDecoder = (HuffmanDecoder_t*)MemoryPool_Allocate(
		pool,
		sizeof(HuffmanDecoder_t) * setup->HuffmanCount);
	if (setup->HuffmanDecoder == NULL) {
		THROW0(ctx, "DecoderSetup_Setup", "OutOfMemory")
	}
	memset(setup->HuffmanDecoder, 0, sizeof(HuffmanDecoder_t) * setup->HuffmanCount);

	for (i = 0; i < setup->HuffmanCount; i++) {
		if (!HuffmanDeocder_Setup(ctx, d, setup->HuffmanDecoder + i)) {
			goto ErrorEnd;
		}
	}

	/* Time Domain Transfer (Skip) */

	if (!SkipTimeDomainTransfer(ctx, d)) {
		goto ErrorEnd;
	}

	/* Floor */

	setup->Floor1Count = BitDecoder_GetBits(d, 6) + 1;
	setup->Floor1Setup = (Floor1Setup_t*)MemoryPool_Allocate(
		pool,
		sizeof(Floor1Setup_t) * setup->Floor1Count);
	if (setup->Floor1Setup == NULL) {
		THROW0(ctx, "DecoderSetup_Setup", "OutOfMemory")
	}
	memset(setup->Floor1Setup, 0, sizeof(Floor1Setup_t) * setup->Floor1Count);

	for (i = 0; i < setup->Floor1Count; i++) {
		if (!Floor1Decoder_Setup(ctx, d, setup->Floor1Setup + i)) {
			goto ErrorEnd;
		}
	}

	/* Residue */

	setup->ResidueCount = BitDecoder_GetBits(d, 6) + 1;
	setup->ResidueSetup = (ResidueSetup_t*)MemoryPool_Allocate(
		pool,
		sizeof(ResidueSetup_t) * setup->ResidueCount);
	if (setup->ResidueSetup == NULL) {
		THROW0(ctx, "DecoderSetup_Setup", "OutOfMemory")
	}
	memset(setup->ResidueSetup, 0, sizeof(ResidueSetup_t) * setup->ResidueCount);

	for (i = 0; i < setup->ResidueCount; i++) {
		if (!ResidueDecoder_Setup(ctx, d, setup->ResidueSetup + i)) {
			goto ErrorEnd;
		}
	}

	/* Mapping */

	setup->MappingCount = BitDecoder_GetBits(d, 6) + 1;
	setup->MappingSetup = (MappingSetup_t*)MemoryPool_Allocate(
		pool,
		sizeof(MappingSetup_t) * setup->MappingCount);
	if (setup->MappingSetup == NULL) {
		THROW0(ctx, "DecoderSetup_Setup", "OutOfMemory")
	}
	memset(setup->MappingSetup, 0, sizeof(MappingSetup_t) * setup->MappingCount);

	for (i = 0; i < setup->MappingCount; i++) {
		if (!MappingSetup_Setup(ctx, d, setup->Channels, setup->MappingSetup + i)) {
			goto ErrorEnd;
		}
	}

	/* Mode */

	setup->ModeCount = BitDecoder_GetBits(d, 6) + 1;
	setup->ModeSetup = (ModeSetup_t*)MemoryPool_Allocate(
		pool,
		sizeof(ModeSetup_t) * setup->ModeCount);
	if (setup->ModeSetup == NULL) {
		THROW0(ctx, "DecoderSetup_Setup", "OutOfMemory")
	}
	memset(setup->ModeSetup, 0, sizeof(ModeSetup_t) * setup->ModeCount);

	for (i = 0; i < setup->ModeCount; i++) {
		if (!ModeSetup_Setup(ctx, d, setup->ModeSetup + i)) {
			goto ErrorEnd;
		}
	}

	setup->ModeBits = ilog(setup->ModeCount - 1);

	/* EOP */

	val = BitDecoder_GetBits(d, 1);
	if (val != 1) {
		THROW0(ctx, "DecoderSetup_Setup", "EOPMissing")
	}

	/* Check Integrity */

	if (!CheckIntegrity_Setup(ctx, setup)) {
		goto ErrorEnd;
	}

	/* Transform */

	setup->Windowing0 = GetWindowingDecoder(setup->BlockSize0);
	setup->Windowing1 = GetWindowingDecoder(setup->BlockSize1);

	if (setup->Windowing0 == NULL || setup->Windowing1 == NULL) {
		THROW0(ctx, "DecoderSetup_Setup", "OutOfMemory")
	}

	result = TRUE;

ErrorEnd:

	return result;
}

/* */

BOOL SkipTimeDomainTransfer(
	DecoderContext_t* ctx,
	BitDecoder_t*     d)
{
	BOOL result = FALSE;

	INT32 i;

	INT32 count = BitDecoder_GetBits(d, 6) + 1;
	for (i = 0; i < count; i++) {
		INT32 x = BitDecoder_GetBits(d, 16);
		if (x != 0) {
			THROW0(ctx, "DecoderSetup_SetupTimeDomainTransfer", "InvalidParam")
		}
	}

	result = TRUE;

ErrorEnd:

	return result;
}

BOOL MappingSetup_Setup(
	DecoderContext_t* ctx,
	BitDecoder_t*     d,
	INT32             channels,
	MappingSetup_t*   setup)
{
	BOOL result = FALSE;

	UINT32 resv = BitDecoder_GetBits(d, 16);
	if (resv != 0) {
		THROW0(ctx, "MappingSetup_Setup", "InvalidType")
	}

	if (BitDecoder_GetBits(d, 1) != 0) {
		INT32 submaps = BitDecoder_GetBits(d, 4);
		if (submaps != 1) {
			THROW0(ctx, "MappingSetup_Setup", "InvalidMappingSubmaps")
		}
	}

	if (BitDecoder_GetBits(d, 1) != 0) {
		setup->CouplingSteps = BitDecoder_GetBits(d, 8) + 1;
		if (setup->CouplingSteps > 1) {
			THROW0(ctx, "MappingSetup_Setup", "InvalidCouplingSteps")
		}
		if (channels == 2) {
			setup->Magnitude = BitDecoder_GetBits(d, 1);
			setup->Angle     = BitDecoder_GetBits(d, 1);
		}
	} else {
		setup->CouplingSteps = 0;
	}

	resv = BitDecoder_GetBits(d, 2);
	if (resv != 0) {
		THROW0(ctx, "MappingSetup_Setup", "InvalidReservedBits")
	}

	/* Skip vorbis_mapping_mux */

	resv = BitDecoder_GetBits(d, 8);

	setup->Floor   = BitDecoder_GetBits(d, 8);
	setup->Residue = BitDecoder_GetBits(d, 8);

	result = TRUE;

ErrorEnd:

	return result;
}

BOOL ModeSetup_Setup(
	DecoderContext_t* ctx,
	BitDecoder_t*     d,
	ModeSetup_t*      setup)
{
	BOOL result = FALSE;

	UINT32 val;

	setup->BlockFlag = BitDecoder_GetBits(d, 1);

	val = BitDecoder_GetBits(d, 16);
	if (val != 0) {
		THROW0(ctx, "ModeSetup_Setup", "InvalidWindowType")
	}

	val = BitDecoder_GetBits(d, 16);
	if (val != 0) {
		THROW0(ctx, "ModeSetup_Setup", "InvalidTransformType")
	}

	setup->Mapping = BitDecoder_GetBits(d, 8);

	result = TRUE;

ErrorEnd:

	return result;
}

/* */

BOOL CheckIntegrity_Setup(
	DecoderContext_t* ctx,
	DecoderSetup_t*   setup)
{
	BOOL result = FALSE;

	INT32 i, j, k;

	/* Floor */

	for (i = 0; i < setup->Floor1Count; i++) {
		Floor1Setup_t* floor = setup->Floor1Setup + i;

		for (j = 0; j < 16; j++) {
			INT32 m = floor->ClassMasterBooks[j];
			if (m >= setup->HuffmanCount) {
				THROW0(ctx, "CheckIntegrity_Setup", "Invalid.Floor1ClassMasterBooks")
			}

			for (k = 0; k < 8; k++) {
				INT32 s = floor->SubclassBooks[j][k];
				if (s >= 0 && s >= setup->HuffmanCount) {
					THROW0(ctx, "CheckIntegrity_Setup", "Invalid.Floor1SubclassBooks")
				}
			}
		}
	}

	/* Residue */

	for (i = 0; i < setup->ResidueCount; i++) {
		ResidueSetup_t* res = setup->ResidueSetup + i;

		if (res->ClassBook >= setup->HuffmanCount) {
			THROW0(ctx, "CheckIntegrity_Setup", "Invalid.ResidueClassBook")
		}

		for (j = 0; j < res->Classifications; j++) {
			for (k = 0; k < 8; k++) {
				INT32 b = res->ResidueBooks[j][k];
				if (b >= 0 && b >= setup->HuffmanCount) {
					THROW0(ctx, "CheckIntegrity_Setup", "Invalid.ResidueBooks")
				}
			}
		}

		{
			const HuffmanDecoder_t* c_dec = setup->HuffmanDecoder + res->ClassBook;

			INT32 vals = 1;
			for (j = 0; j < c_dec->Dimensions; j++) {
				vals *= res->Classifications;
				if (vals > c_dec->Entries) {
					break;
				}
			}

			if (vals != c_dec->Entries) {
				THROW0(ctx, "CheckIntegrity_Setup", "CVE-2008-1420")
			}
		}
	}

	/* Mapping */

	for (i = 0; i < setup->MappingCount; i++) {
		MappingSetup_t* mapping = setup->MappingSetup + i;

		if ((INT32)(mapping->Floor) >= setup->Floor1Count) {
			THROW0(ctx, "CheckIntegrity_Setup", "Invalid.Mapping.Floor")
		}

		if ((INT32)(mapping->Residue) >= setup->ResidueCount) {
			THROW0(ctx, "CheckIntegrity_Setup", "Invalid.Mapping.Residue")
		}
	}

	/* Mode */

	for (i = 0; i < setup->ModeCount; i++) {
		ModeSetup_t* mode = setup->ModeSetup + i;

		if ((INT32)(mode->Mapping) >= setup->MappingCount) {
			THROW0(ctx, "CheckIntegrity_Setup", "Invalid.Mode.Mapping")
		}
	}

	/* */

	result = TRUE;

ErrorEnd:

	return result;
}

/* */

