// Output.h
// 2008/12/01

#pragma once

namespace QAX {

// VorbisOutputSpec
struct VorbisOutputSpec {

	QVorbisDecoder* Decoder;
	UINT64          Samples;
	ItemProperty*   Property;

	QFileMapping* Mapping;
	UINT64        Position;
	UINT64        Size;

	const void* IndexChunk;
	UINT32      IndexSize;

}; // VorbisOutputSpec

// VorbisOutputImpl
class VorbisOutputImpl : public QDecoderOutput {

	QVorbisDecoder* m_Decoder;

	QFileMapping* m_Mapping;

	const BYTE* m_pBase;

	UINT64 m_Samples;

	ItemProperty* m_Property;

	UINT64 m_Current;

	UINT32 m_PendingSamples;

	const INT16* m_SampleBuffer;

	UINT32 m_SampleUnit;

	BYTE m_PageFlags;

	bool m_EndOfStream;

	const BYTE* m_Start;
	const BYTE* m_End;

	const BYTE* m_Position;

	UINT32* m_Lens;
	UINT32  m_LensSize;

	const UINT32* m_PosLens;
	const UINT32* m_EndLens;

	IndexArray m_Index;

	const void* m_pIndexChunk;
	UINT32      m_IndexSize;

	UINT32 m_DecodeOffset;

	enum {
		LEN_SIZE = 32
	};

	enum {
		FLAGS_START = 0x01,
		FLAGS_END   = 0x02
	};

	VorbisOutputImpl() :
		m_Decoder(0),
		m_Mapping(0),
		m_pBase(0),
		m_Samples(0),
		m_Property(0),
		m_Current(0),
		m_PendingSamples(0),
		m_SampleBuffer(0),
		m_SampleUnit(0),
		m_PageFlags(0),
		m_EndOfStream(false),
		m_Start(0),
		m_End(0),
		m_Position(0),
		m_Lens(0),
		m_LensSize(0),
		m_PosLens(0),
		m_EndLens(0),
		m_pIndexChunk(0),
		m_IndexSize(0),
		m_DecodeOffset(0)
	{
	}

	void Setup(
		VorbisOutputSpec* p)
	{
		m_Decoder  = p->Decoder;
		m_Samples  = p->Samples;
		m_Property = p->Property;

		m_Current = 0;

		m_PendingSamples = 0;
		m_SampleBuffer   = 0;

		QVorbisDecoderSetup_t* setup = m_Decoder->GetSetup();

		m_SampleUnit = QV_GetDecoderSetupChannels(setup);

		m_PageFlags = 0;

		m_EndOfStream = false;

		/* */

		UINT64 base = (p->Position / g_OFFSET_ALIGN) * g_OFFSET_ALIGN;
		UINT32 diff = UINT32(p->Position - base);

		m_Mapping = p->Mapping;
		m_pBase = static_cast<const BYTE*>(
			m_Mapping->MapView(
				base,
				SIZE_T(diff + p->Size)));

		SetupData(
			m_pBase + diff,
			SIZE_T(p->Size));

		/* */

		m_pIndexChunk = p->IndexChunk;
		m_IndexSize   = p->IndexSize;

		m_DecodeOffset = 0;
	}

	void SetupData(
		const BYTE* data,
		SIZE_T      size)
	{
		m_Start = data;
		m_End   = data + size;

		m_Position = m_Start;

		m_Lens = static_cast<UINT32*>(malloc(LEN_SIZE * sizeof(UINT32)));
		if (m_Lens == 0) {
			DecoderError::Throw("OutOfMemory");
		}

		m_LensSize = LEN_SIZE;

		m_PosLens = 0;
		m_EndLens = 0;
	}

public:

	static QDecoderOutput* Create(
		VorbisOutputSpec* spec)
	{
		VorbisOutputImpl* p = new VorbisOutputImpl();

		try {
			p->Setup(spec);

		} catch (...) {
			delete p;
			throw;
		}

		return p;
	}

	~VorbisOutputImpl()
	{
		if (m_Mapping != 0 && m_pBase != 0) {
			m_Mapping->UnmapView(m_pBase);
			m_Mapping = 0;
			m_pBase   = 0;
		}

		free(m_Lens);
		m_Lens = 0;

		delete m_Decoder;
		m_Decoder = 0;
	}

public:

	virtual void STDCALL Release()
	{
		delete this;
	}

	virtual QDecoderSpec STDCALL GetDecoderSpec()
	{
		QDecoderSpec spec;

		QVorbisDecoderSetup_t* setup = m_Decoder->GetSetup();

		spec.Channels     = QV_GetDecoderSetupChannels(setup);
		spec.SamplingRate = QV_GetDecoderSetupSamplingRate(setup);

		spec.Samples = m_Samples;

		return spec;
	}

	virtual UINT32 STDCALL Decode(
		VOID*  buffer,
		UINT32 samples)
	{
		UINT32 output = 0;

		while (!m_EndOfStream) {
			if (m_PendingSamples > 0) {
				UINT32 count = samples;
				if (count > m_PendingSamples) {
					count = m_PendingSamples;
				}

				CopyMemory(
					buffer,
					m_SampleBuffer,
					count * m_SampleUnit * sizeof(INT16));

				m_SampleBuffer   += count * m_SampleUnit;
				m_PendingSamples -= count;

				m_Current += count;

				output = count;
				break;
			}

			if (m_PosLens >= m_EndLens) {
				ParsePageHeader();
			}

			if (m_PosLens < m_EndLens) {
				DecodePacket();
			}
		}

		return output;
	}

	virtual void STDCALL Seek(
		UINT64 sample)
	{
		if (m_Index.IsEmpty()) {
			m_Index.ParseIndex(
				m_pIndexChunk,
				m_IndexSize,
				m_Samples);
		}

		const IndexEntry& e = m_Index.SeekIndex(sample);

		m_Position = m_Start + e.Position;

		ParsePageHeader();

		SeekInPage(sample, e);
	}

	virtual std::string STDCALL QueryProperty(
		const char* name)
	{
		ItemProperty::iterator it = m_Property->find(name);
		if (it == m_Property->end()) {
			return std::string();
		}

		return (*it).second;
	}

	/* */

	void ParsePageHeader()
	{
		if (m_Position >= m_End) {
			m_EndOfStream = true;
			return;
		}

		const BYTE* p = m_Position;

		const BYTE* buffer = m_Position;

		m_PageFlags = *(p++);

		UINT32 lens;
		p += DecodeVNumber(p, m_End, &lens);

		if (lens > m_LensSize) {
			UINT32* new_len = static_cast<UINT32*>(realloc(
				m_Lens,
				lens * sizeof(UINT32)));
			if (new_len == 0) {
				DecoderError::Throw("OutOfMemory");
			}

			m_Lens     = new_len;
			m_LensSize = lens;
		}

		for (UINT32 i = 0; i < lens; i++) {
			p += DecodeVNumber(p, m_End, m_Lens + i);
		}

		m_PosLens = m_Lens;
		m_EndLens = m_Lens + lens;

		m_Position = p;

		SIZE_T size = p - buffer;
		for (UINT32 i = 0; i < lens; i++) {
			size += m_Lens[i];
		}

		if (buffer + size + sizeof(UINT32) > m_End) {
			FormatError::Throw("Page.Size.OutOfRange");
		}

		CRC32 crc32;
		UINT32 crc = crc32.Generate(buffer, size);

		UINT32 val;
		CopyMemory(&val, buffer + size, sizeof(UINT32));

		if (val != crc) {
			FormatError::Throw("Page.CRC.Error");
		}
	}

	/* */

	void DecodePacket()
	{
		bool bLastPacket = (m_PosLens == (m_EndLens - 1));

		QVorbisDecoder_t* decoder = m_Decoder->GetDecoder();

		QV_Output_t output = { 0 };
		if (!QV_DecodeFrame(
			decoder,
			m_Position,
			m_PosLens[0],
			&output)) {
			DecoderError::Throw("QV_DecodeFrame");
		}

		UINT32 samples = output.Length;
		if (samples > 0) {
			QV_Convert_t convert = { 0 };
			if (!QV_ConvertFrame(
				decoder,
				&output,
				&convert)) {
				DecoderError::Throw("QV_ConvertFrame");
			}

			if (m_DecodeOffset > samples) {
				FormatError::Throw("DecoderOffset.Invalid");
			}

			m_SampleBuffer   = convert.Sample + m_DecodeOffset * m_SampleUnit;
			m_PendingSamples = samples - m_DecodeOffset;

			m_DecodeOffset = 0;

			if ((m_PageFlags & FLAGS_END) != 0 && bLastPacket) {
				UINT64 rest = m_Samples - m_Current;
				if (rest > samples) {
					DecoderError::Throw("Invalid.Samples");
				}
				m_PendingSamples = UINT32(rest);
			}
		}

		m_Position += m_PosLens[0];
		m_PosLens  += 1;

		if (bLastPacket) {
			m_Position += sizeof(UINT32); /* CRC32 */
		}
	}

	/* */

	void SeekInPage(
		UINT64            pos,
		const IndexEntry& e)
	{
		QVorbisDecoder_t* decoder = m_Decoder->GetDecoder();

		QV_ResetDecoder(decoder);
		QV_ResetDecoderChecker(decoder);

		m_PendingSamples = 0;

		m_EndOfStream = false;

		const UINT32* p   = m_PosLens;
		const UINT32* end = m_EndLens;

		if (p >= end) {
			FormatError::Throw("Page.Invalid");
		}

		const BYTE* frame = m_Position;
		SIZE_T      size  = p[0];

		INT32 samples = 0;
		if (!QV_CheckDecoderChecker(
			decoder,
			frame,
			size,
			&samples)) {
			DecoderError::Throw("QV_CheckDecoderChecker");
		}
		if (samples != 0) {
			FormatError::Throw("Check.Invalid");
		}

		const BYTE* n_frame = frame + size;

		UINT64 c_pos = e.Sample + e.Offset;

		if (pos < c_pos) {
			FormatError::Throw("Page.Position.Invalid");
		}

		while (p < end - 1) {
			SIZE_T n_size = p[1];

			if (!QV_CheckDecoderChecker(
				decoder,
				n_frame,
				n_size,
				&samples)) {
				DecoderError::Throw("QV_CheckDecoderChecker");
			}

			if (pos < c_pos + samples) {
				break;
			}

			c_pos += samples;

			frame = n_frame;

			n_frame += n_size;

			p += 1;
		}

		m_Position = frame;
		m_PosLens  = p;

		m_Current = pos;

		m_DecodeOffset = UINT32(pos - c_pos);
	}

	/* */

}; // VorbisOutputImpl

} // namespace QAX

