// Decoder.h
// 2008/11/27

#pragma once

namespace QAX {

// SetupTable
class SetupTable : public Table {

public:

	// SetupEntry
	struct SetupEntry {
		const void* Hash;
		const void* Id;
		SIZE_T      IdSize;
		const void* Setup;
		SIZE_T      SetupSize;
	}; // SetupEntry

	SetupTable(QFileMapping* m) : Table(m)
	{
	}

	SetupEntry GetSetupEntry(INT32 index)
	{
		UINT32 size;
		const void* pos = GetEntry(index, &size);

		const BYTE* p   = static_cast<const BYTE*>(pos);
		const BYTE* end = p + size;

		return GetSetup(p, end);
	}

	static SetupEntry GetSetup(const BYTE* start, const BYTE* end)
	{
		SetupEntry e;

		const BYTE* p = start;

		UINT32 val;

		if (end - p < 20) {
			FormatError::Throw("SetupTable.InvalidEntry");
		}

		e.Hash = p;
		p += 20;

		p += DecodeVNumber(p, end, &val);
		e.IdSize = val;

		p += DecodeVNumber(p, end, &val);
		e.SetupSize = val;

		if (end < p + e.IdSize) {
			FormatError::Throw("SetupTable.InvalidEntry");
		}

		e.Id = p;
		p += e.IdSize;

		if (end < p + e.SetupSize) {
			FormatError::Throw("SetupTable.InvalidEntry");
		}

		e.Setup = p;

		return e;
	}

}; // SetupTable

// IndexTable
class IndexTable : public Table {

public:

	IndexTable(QFileMapping* m) : Table(m)
	{
	}

}; // IndexTable

// Item
struct Item {

	UINT32 Type;
	UINT32 Name;
	UINT32 Setup;
	UINT32 Index;

	UINT64 Position;
	UINT64 Size;

	UINT64 Samples;

	ItemProperty* Property;

}; // Item

typedef std::map<std::string, Item> ItemMap;

typedef std::vector<QItem> ItemArray;

// DecoderImpl
class DecoderImpl : public QDecoder {

	QDecoderFactory* m_Factory;

	QFileMapping* m_Mapping;

	FileHeader m_Header;

	ItemMap m_Items;

	ItemArray m_ItemEntries;

	StringTable m_Strings;
	SetupTable  m_Setups;
	IndexTable  m_Indexs;

	DecoderImpl(QDecoderFactory* f, QFileMapping* m) :
		m_Factory(f),
		m_Mapping(m),
		m_Strings(m),
		m_Setups(m),
		m_Indexs(m)
	{
	}

public:

	static QDecoder* Create(
		QDecoderFactory* f,
		QFileMapping*    m)
	{
		DecoderImpl* p = new DecoderImpl(f, m);

		try {
			p->Setup();

		} catch (...) {
			delete p;
			throw;
		}

		return p;
	}

	~DecoderImpl()
	{
		for (ItemMap::iterator it = m_Items.begin(); it != m_Items.end(); ++it) {
			const Item& item = (*it).second;
			delete item.Property;
		}
	}

	/* */

private:

	void Setup()
	{
		SetupHeader();

		m_Strings.Setup(
			m_Header.StringsPos,
			m_Header.StringsSize,
			m_Header.StringsIndex);

		SetupItems();

		m_Setups.Setup(
			m_Header.SetupsPos,
			m_Header.SetupsSize,
			m_Header.SetupsIndex);

		m_Indexs.Setup(
			m_Header.IndexsPos,
			m_Header.IndexsSize,
			m_Header.IndexsIndex);
	}

	void SetupHeader()
	{
		const FileHeader* p = static_cast<const FileHeader*>(
			m_Mapping->MapView(
				0,
				sizeof(FileHeader)));

		CopyMemory(&m_Header, p, sizeof(m_Header));

		m_Mapping->UnmapView(p);

		if (memcmp(m_Header.Signature, "QAX1", 4) != 0) {
			FormatError::Throw("FileHeader.Signature");
		}
	}

	void SetupItems()
	{
		Table table(m_Mapping);

		table.Setup(
			m_Header.ItemsPos,
			m_Header.ItemsSize,
			m_Header.ItemsIndex);

		INT32 count = table.GetCount();

		m_ItemEntries.resize(count);

		for (INT32 i = 0; i < count; i++) {
			UINT32 size = 0;
			const BYTE* pos = static_cast<const BYTE*>(
				table.GetEntry(i, &size));
			const BYTE* end = pos + size;

			const BYTE* p = pos;

			Item item;

			p += DecodeVNumber(p, end, &(item.Type ));
			p += DecodeVNumber(p, end, &(item.Name ));
			p += DecodeVNumber(p, end, &(item.Setup));
			p += DecodeVNumber(p, end, &(item.Index));

			p += DecodeVNumber(p, end, &(item.Position));
			p += DecodeVNumber(p, end, &(item.Size    ));

			p += DecodeVNumber(p, end, &(item.Samples));

			AutoPtr<ItemProperty> prop = new ItemProperty();

			p += DecodeProperty(
				&m_Strings,
				p,
				end,
				prop);

			item.Property = prop.Detach();

			std::string key = m_Strings.GetString(item.Name);
			std::string typ = m_Strings.GetString(item.Type);

			m_Items.insert(
				ItemMap::value_type(
					key,
					item));

			m_ItemEntries[i].Type = typ;
			m_ItemEntries[i].Name = key;
		}
	}

public:

	/* */

	virtual void STDCALL Release()
	{
		delete this;
	}

	virtual QDecoderOutput* STDCALL CreateOutput(
		const char* name)
	{
		ItemMap::iterator it = m_Items.find(name);
		if (it == m_Items.end()) {
			return 0;
		}

		const Item& item = (*it).second;

		if (item.Position + item.Size > m_Header.ContentSize) {
			FormatError::Throw("Decoder.OutOfContentRange");
		}

		std::string typ = m_Strings.GetString(item.Type);
		if (typ == "vorbis") {
			return CreateVorbisOutput(item);
		}

		if (typ == "TTA1") {
			return CreateTTAOutput(item);
		}

		return 0;
	}

	virtual INT32 STDCALL GetItemCount()
	{
		return INT32(m_ItemEntries.size());
	}

	virtual QItem STDCALL GetItem(
		INT32 index)
	{
		if (index < 0 || index >= INT32(m_ItemEntries.size())) {
			FormatError::Throw("Decoder.OutOfIndex");
		}

		return m_ItemEntries[index];
	}

	/* */

	QDecoderOutput* CreateVorbisOutput(const Item& item)
	{
		SetupTable::SetupEntry setup = m_Setups.GetSetupEntry(item.Setup);

		BYTE key[20];
		CopyMemory(key, setup.Hash, 20);

		QVorbisDecoder* decoder = m_Factory->CreateVorbisDecoder(
			key,
			setup.Id,
			setup.IdSize,
			setup.Setup,
			setup.SetupSize);

		VorbisOutputSpec spec = { 0 };

		spec.Decoder  = decoder;
		spec.Samples  = item.Samples;
		spec.Property = item.Property;

		spec.Mapping  = m_Mapping;
		spec.Position = m_Header.ContentPos + item.Position;
		spec.Size     = item.Size;

		spec.IndexChunk = m_Indexs.GetEntry(
			item.Index,
			&(spec.IndexSize));

		QDecoderOutput* output = VorbisOutputImpl::Create(&spec);

		return output;
	}

	/* */

	QDecoderOutput* CreateTTAOutput(const Item& item)
	{
		QAXTTAOutputSpec spec = { 0 };

		UINT32 size;
		spec.Setup = m_Setups.GetEntry(
			item.Setup,
			&size);
		if (spec.Setup == 0) {
			return 0;
		}

		spec.SetupSize = size;

		spec.Property = item.Property;

		spec.Samples  = item.Samples;

		spec.Mapping  = m_Mapping;
		spec.Position = m_Header.ContentPos + item.Position;
		spec.Size     = SIZE_T(item.Size);

		spec.Index = m_Indexs.GetEntry(
			item.Index,
			&size);
		if (spec.Index == 0) {
			return 0;
		}

		spec.IndexSize = size;

		QDecoderOutput* output = QAXTTAOutputImpl::Create(&spec);

		return output;
	}

	/* */

}; // DecoderImpl

} // namespace QAX

