/*!
  \file
  \brief 部品生成ツールのウィンドウ管理

  \author Satofumi KAMIMURA

  $Id$

  \todo コンポーネントの変更を、PartsConfig に反映させる
*/

#include <QSettings>
#include <QCloseEvent>
#include <QMessageBox>
#include <QFontDatabase>
#include <QButtonGroup>
#include <QFontComboBox>
#include "PartsGeneratorWindow.h"
#include "PartsDrawWidget.h"
#include "GenerateConfigHandler.h"
#include "PartsConfig.h"


struct PartsGeneratorWindow::pImpl {

  QAction* new_act_;
  QAction* open_act_;
  QAction* save_act_;
  PartsDrawWidget* parts_draw_;
  QButtonGroup* button_group_;
  QComboBox* font_combobox_;
  GenerateConfigHandler parts_configs_;
  size_t parts_size_;


  pImpl(void) : parts_draw_(NULL), parts_size_(0) {
  }


  void initializeForm(PartsGeneratorWindow* parent) {

    // !!! とりあえず、読み込んでしまう
    // !!! いずれ、指定したパスのファイルを読み込むようにすべき
    loadConfigFile("parts_config.txt");

    // アイコンの適用
    parent->setWindowIcon(QIcon(":icons/parts_generator_icon"));

    // フォントツリーの作成
    initializeFontTree(parent);

    // 画像描画ウィジットの配置
    parts_draw_ = new PartsDrawWidget(parent->centralwidget);
    parent->scrollarea_->setWidget(parts_draw_);

    // フォントの選択変更イベント
    // !!! isChanged() で、表示中のフォントが変更されるようにする
    // !!! この処理は、Undo したいかも

    // メニューイベント
    connect(parent->action_quit_, SIGNAL(triggered()), parent, SLOT(close()));

    // 画像ボタンのイベントで設定の表示を変更し、
    // 設定の変更で、描画ウィジットの画像を更新する
    // !!! って感じのイベントを設定

    // ボタンをグループ化
    button_group_ = new QButtonGroup(parent);
    button_group_->addButton(parent->digital_button_, 0);
    button_group_->addButton(parent->number_button_, 1);
    button_group_->addButton(parent->week_button_, 2);
    button_group_->addButton(parent->separator_button_, 3);

    connect(button_group_, SIGNAL(buttonClicked(int)),
            parent, SLOT(update(void)));

    // 最後に押されていたボタンを押下させる
    // !!! とりあえず、最初のにしておく。デフォルト設定で

    // !!! 不要ならば、除去する
    parent->update();

    // 出力イベント処理
    connect(parent->output_button_, SIGNAL(clicked()),
            parent, SLOT(outputPixmap()));
  }


  // 設定をフォームに反映
  void loadPartsConfig(const PartsConfig& parts, PartsGeneratorWindow* parent) {

    parent->output_label_->setText((parts.type + ".png").c_str());

    // !!! フォントを変更する
    parent->font_size_spinbox_->setValue(parts.font_size);
    updateColorButton(parent->font_color_button_, parts.fore_color);
    updateColorButton(parent->background_color_button_, parts.back_color);
    parent->offset_spinbox_->setValue(parts.offset);
  }


  void updateColorButton(QToolButton* button, const std::string& color_str) {

    QColor fore_color = PartsConfig::encodeColor(color_str);
    QPixmap fore_pixmap(32, 32);
    fore_pixmap.fill(fore_color);
    button->setIcon(QIcon(fore_pixmap));
  }


  // プレビュー画像の描画
  void drawPreviewPixmap(const PartsConfig& parts) {

    parts_draw_->updatePreview(parts);
  }


  // フォントツリーの作成
  void initializeFontTree(PartsGeneratorWindow* parent) {

    QFontDatabase database;
    foreach (QString family, database.families()) {
      const QStringList styles = database.styles(family);
      if (styles.isEmpty()) {
        continue;
      }

      QTreeWidgetItem* family_item = new QTreeWidgetItem(parent->font_tree_);
      family_item->setText(0, family);

      foreach (QString style, styles) {
        QTreeWidgetItem* style_item = new QTreeWidgetItem(family_item);
        style_item->setText(0, style);
        style_item->setData(0, Qt::UserRole,
                            QVariant(database.weight(family, style)));
        style_item->setData(0, Qt::UserRole + 1,
                            QVariant(database.italic(family, style)));
      }
    }
  }


  // 設定の読み出し
  void readSettings(PartsGeneratorWindow* parent) {

    QSettings settings("Hyakuren Soft LTD.", "parts_generator");
    parent->restoreGeometry(settings.value("geometry").toByteArray());
  }


  // 設定の保存
  void writeSettings(PartsGeneratorWindow* parent) {

    QSettings settings("Hyakuren Soft LTD.", "parts_generator");
    settings.setValue("geometry", parent->saveGeometry());
  }


  // 設定ファイルの読み込み
  void loadConfigFile(const char* file) {

    parts_configs_.load(file);
    parts_size_ = parts_configs_.size();
  }
};


PartsGeneratorWindow::PartsGeneratorWindow(void)
  : QMainWindow(), pimpl(new pImpl) {
  setupUi(this);
  pimpl->readSettings(this);

  // 以下の初期化は、pImpl 内で行うと tr() が使えないため、ここで行っている

  // ツールバーの初期化
  pimpl->new_act_ = new QAction(QIcon(":icons/new_icon"), tr("New"), this);
  pimpl->new_act_->setStatusTip(tr("Create a new config file."));
  connect(pimpl->new_act_, SIGNAL(triggered()), this, SLOT(newFile()));

  pimpl->open_act_ = new QAction(QIcon(":icons/open_icon"), tr("Open"), this);
  pimpl->open_act_->setStatusTip(tr("Open an exist config file."));
  connect(pimpl->open_act_, SIGNAL(triggered()), this, SLOT(openFile()));

  pimpl->save_act_ = new QAction(QIcon(":icons/save_icon"), tr("Save"), this);
  pimpl->save_act_->setStatusTip(tr("Save the config file to disk."));
  connect(pimpl->save_act_, SIGNAL(triggered()), this, SLOT(saveFile()));

  toolbar_->addAction(pimpl->new_act_);
  toolbar_->addAction(pimpl->open_act_);
  toolbar_->addAction(pimpl->save_act_);

  // フォントツリーのタイトル設定
  font_tree_->setColumnCount(1);
  font_tree_->setHeaderLabels(QStringList() << tr("Font"));

  // フォームの初期化
  pimpl->initializeForm(this);

  // メニューの初期化
  connect(action_about_, SIGNAL(triggered()), this, SLOT(aboutApplication()));
}


PartsGeneratorWindow::~PartsGeneratorWindow(void) {
  pimpl->writeSettings(this);
}


// 終了
void PartsGeneratorWindow::closeEvent(QCloseEvent* event) {

  event->accept();
}


// 新規作成
void PartsGeneratorWindow::newFile(void) {

  // !!!
}


// 読み出し
void PartsGeneratorWindow::openFile(void) {

  // !!! loadConfigFile("parts_config.txt");
  // !!!
  update();
}


// 保存
void PartsGeneratorWindow::saveFile(void) {

  // !!!
}


// このアプリケーションについて
void PartsGeneratorWindow::aboutApplication(void) {

  QMessageBox::about(this, tr("About parts_generator"),
                     tr("<h2>Parts generator</h2>"
                        "<p>Theme making sub tool for Qtmclock.</p>"
                        "<p>Report bugs to "
                        "&lt;satofumi@users.sourceforge.jp&gt;</p>"));
}


// フォーム設定とプレビュー画像の更新
void PartsGeneratorWindow::update(void) {

  int index = pimpl->button_group_->checkedId();
  if (index >= static_cast<int>(pimpl->parts_size_)) {
    return;
  }

  const PartsConfig parts = pimpl->parts_configs_.getPartsConfig(index);
  pimpl->loadPartsConfig(parts, this);
  pimpl->drawPreviewPixmap(parts);
}


// 画像ファイルの出力
void PartsGeneratorWindow::outputPixmap(void) {

  QString output_file = output_label_->text();
  pimpl->parts_draw_->outputPixmap(output_file);
}
