/*!
  \file
  \brief アニメーション管理

  \author Satofumi KAMIMURA

  $Id$

  \todo セルがゼロ枚のパターンは、取り除く
*/

#include <QTimer>
#include <QDateTime>
#include "AnimationManager.h"
#include "AnimationPattern.h"

// !!! アニメーション用の定数定義のみを分離し、処理するべきか？
#include "ClockPartsDraw.h"


struct AnimationManager::pImpl {

  size_t cycle_msec_;
  std::vector<AnimationPattern> patterns_;
  size_t pattern_index_;
  size_t pattern_max_;
  QTimer update_timer_;
  bool is_playing_;
  int current_cell_index_;
  int cell_max_;
  int prev_cell_index_;

  pImpl(size_t cycle_msec, const std::vector<AnimationPattern>& patterns,
        AnimationManager* parent)
    : cycle_msec_(cycle_msec), patterns_(patterns),
      pattern_index_(0), pattern_max_(patterns_.size()), is_playing_(false),
      current_cell_index_(0), cell_max_(0) {

    // 乱数を初期化
    srand(QDateTime::currentDateTime().toTime_t());

    connect(&update_timer_, SIGNAL(timeout()), parent, SLOT(updateAnimation()));
  }


  // 再生の開始評価
  void evaluate(AnimationManager* parent) {

    // 再生中ならば、セルを再生する
    if (is_playing_) {
      return;
    }

    // パターンを再生するかの評価
    for (size_t i = 0; i < pattern_max_; ++i) {
      pattern_index_ = (pattern_index_ + 1) % pattern_max_;

      double random = 1.0 * rand() / (RAND_MAX + 1.0);
      double compare_probability =
        patterns_[pattern_index_].probability * cycle_msec_ / 1000.0;
      if (random < compare_probability) {
        startAnimation(pattern_index_);

        // 最初のセルを更新
        parent->updateAnimation();
        return;
      }
    }
  }


  // アニメーションの開始
  void startAnimation(size_t index) {

    current_cell_index_ = 0;
    cell_max_ = patterns_[index].cells.size();

    // タイマーをスタートさせ、再生中フラグをセットする
    update_timer_.start(patterns_[index].cycle_msec);
    is_playing_ = true;
  }


  // セルの更新
  int updateAnimation(void) {

    if (current_cell_index_ >= cell_max_) {
      update_timer_.stop();
      is_playing_ = false;
      return ClockPartsDraw::NoAnimationCell;
    }

    size_t cell_index = patterns_[pattern_index_].cells[current_cell_index_];
    ++current_cell_index_;

    return cell_index;
  }
};


AnimationManager::AnimationManager(size_t cycle_msec,
                                   const std::vector<AnimationPattern>&
                                   patterns,
                                   QObject* parent)
  : QObject(parent), pimpl(new pImpl(cycle_msec, patterns, this)) {
}


AnimationManager::~AnimationManager(void) {
}


void AnimationManager::evaluate(void) {

  return pimpl->evaluate(this);
}


void AnimationManager::updateAnimation(void) {

  int cell_index = pimpl->updateAnimation();
  if (cell_index != pimpl->prev_cell_index_) {
    pimpl->prev_cell_index_ = cell_index;
    emit updateCell(cell_index);
  }
}
