/*!
  \file
  \brief テーマ時計の描画処理

  \author Satofumi KAMIMURA

  $Id$

  \todo 指定した拡張子のファイルのみ、ドラッグ可能にする
*/

#include <QtGui>
#include <QTimer>
#include <QTime>
#include <QSettings>
#include <QAction>
#include "ThemeClockWidget.h"
#include "ThemeClockDraw.h"
#include "ClockSettingDialog.h"
#include "ConvertStdStringPath.h"

using namespace qrk;


struct ThemeClockWidget::pImpl {

  static const bool ForceCreate = true;

  QTimer clock_timer_;
  ThemeClockDraw* draw_widget_;
  QString pre_theme_file_;
  QString theme_file_;
  QPoint drag_position_;
  size_t scaling_percent_;
  bool no_animation_;
  size_t pre_scaling_percent_;
  bool pre_no_animation_;

  ClockSettingDialog setting_dialog_;


  pImpl(ThemeClockWidget* parent)
    : draw_widget_(new ThemeClockDraw(parent)),
      scaling_percent_(100), pre_scaling_percent_(100),
      pre_no_animation_(false) {
  }

  // フォームの初期化
  void initializeForm(ThemeClockWidget* parent) {

    // アイコンを適用
    parent->setWindowIcon(QIcon(":icons/qtmclock_icon"));

    // 時計の更新設定
    connect(&clock_timer_, SIGNAL(timeout()), parent, SLOT(redraw()));

    // 拡大率の設定
    connect(&setting_dialog_, SIGNAL(scalingValueChanged(int)),
            parent, SLOT(changeScaling(int)));

    // アニメーションの On / Off
    connect(&setting_dialog_, SIGNAL(noAnimationStateChanged(bool)),
            parent, SLOT(changeNoAnimation(bool)));

    // テーマファイルの Drop を許可
    parent->setAcceptDrops(true);

    // 影響が薄そうな色、ということで灰色にしておき、
    // アンチエイリアスの効果を期待して、少しだけ透明にしておく
    QColor color(128, 128, 128, 223);
    parent->setPalette(color);
  }


  // 次に秒針が動くまでのミリ秒だけ待機
  void startClock(ThemeClockWidget* parent) {

    clock_timer_.stop();
    QTime current_time = QTime::currentTime();

    QTimer::singleShot(1000 - current_time.msec(), parent, SLOT(redraw()));
  }


  // 再描画
  void redraw(ThemeClockWidget* parent) {
    static_cast<void>(parent);

    if (! clock_timer_.isActive()) {
      clock_timer_.start(1000);
    }

    // 再描画処理
    draw_widget_->redraw(0, parent);
  }


  // テーマファイルの読み出しを試行
  bool tryLoadTheme(const QString& theme_file, ThemeClockWidget* parent) {

    // テーマの読み出し
    if (loadThemeFile(theme_file, parent)) {
      // 今回指定されたテーマ
      return true;

    } else if (loadThemeFile(pre_theme_file_, parent)) {

      // 前回利用したテーマ。拡大率があれば、設定する
      parent->changeScaling(pre_scaling_percent_);
      setting_dialog_.setScalingPercent(scaling_percent_);

      // アニメーションの指定があれば、設定する
      parent->changeNoAnimation(pre_no_animation_);
      setting_dialog_.setNoAnimationCheckbox(scaling_percent_);

      return true;

    } else {
      // テーマのドロップを促すフォームを作成
      loadThemeFile("", parent);
      return false;
    }
  }


  // テーマファイルの読み出し
  bool loadThemeFile(const QString& theme_file, ThemeClockWidget* parent) {

    if (! draw_widget_->loadThemeFile(theme_file)) {
      return false;
    }

    theme_file_ = theme_file;
    parent->resize(draw_widget_->size());
    return true;
  }


  // テーマファイル名の取得
  QString getThemeFile(void) {
    return draw_widget_->getThemeFile();
  }


  // 設定の読み出し
  void readSettings(bool position_specified, ThemeClockWidget* parent) {

    QSettings settings("Hyakuren Soft LTD.", "qtmclock");
    pre_theme_file_ = settings.value("theme_file", "").toString();
    pre_scaling_percent_ = settings.value("scaling_percent", 0).toInt();
    pre_no_animation_ = settings.value("no_animation", false).toBool();

    if (! position_specified) {
      // 位置が指定されてないときのみ、設定を反映させる
      QRect position = settings.value("geometry", parent->geometry()).toRect();
      parent->setGeometry(position);
    }
  }


  // 設定の書き込み
  void writeSettings(ThemeClockWidget* parent) {

    QSettings settings("Hyakuren Soft LTD.", "qtmclock");
    QFileInfo fi(theme_file_);
    settings.setValue("theme_file", fi.absoluteFilePath());

    settings.setValue("scaling_percent", scaling_percent_);
    settings.setValue("no_animation", no_animation_);

    QRect position = parent->geometry();
    QSize size = parent->size();

    settings.setValue("geometry", QRect(position.x(), position.y(),
                                        size.width(), size.height()));
  }


  // 右クリックメニューの初期化
  void initializeMenu(ThemeClockWidget* parent) {

    // 設定
    QAction* config_action = new QAction(QAction::tr("Config"), parent);
    connect(config_action, SIGNAL(triggered()), parent, SLOT(configHandler()));
    parent->addAction(config_action);

    // 終了
    QAction* quit_action = new QAction(QAction::tr("E&xit"), parent);
    quit_action->setShortcut(tr("Ctrl+Q"));
    connect(quit_action, SIGNAL(triggered()), qApp, SLOT(quit()));
    parent->addAction(quit_action);

    parent->setContextMenuPolicy(Qt::ActionsContextMenu);
  }
};


ThemeClockWidget::ThemeClockWidget(const QString& theme_file,
                                   const bool position_specified,
                                   QWidget* parent)
  : QWidget(parent), pimpl(new pImpl(this)) {

  // 前回動作時の設定を読み出す
  pimpl->readSettings(position_specified, this);

  // テーマの読み出し
  pimpl->tryLoadTheme(theme_file, this);
  resize(pimpl->draw_widget_->size());

  // フォームの初期化
  pimpl->initializeForm(this);

  // 右クリックメニューの初期化
  pimpl->initializeMenu(this);

  // 時計の動作開始
  pimpl->startClock(this);
}


ThemeClockWidget::~ThemeClockWidget(void) {

  // 今回動作時の設定を保存
  pimpl->writeSettings(this);
}


// 再描画
void ThemeClockWidget::redraw(void) {

  pimpl->redraw(this);
}


// リサイズ
void ThemeClockWidget::resizeEvent(QResizeEvent* event) {
  static_cast<void>(event);

  redraw();
}


// 左クリックでのフォーム移動用
void ThemeClockWidget::mousePressEvent(QMouseEvent* event) {

  if (event->button() == Qt::LeftButton) {
    pimpl->drag_position_ = event->globalPos() - frameGeometry().topLeft();
    event->accept();
  }
}


// 左クリックでのフォーム移動
void ThemeClockWidget::mouseMoveEvent(QMouseEvent* event) {

  if (event->buttons() & Qt::LeftButton) {
    move(event->globalPos() - pimpl->drag_position_);
    event->accept();
  }
}


// テーマのリロード用
void ThemeClockWidget::dragEnterEvent(QDragEnterEvent* event) {

  event->acceptProposedAction();
}


// テーマのリロード用
void ThemeClockWidget::dragMoveEvent(QDragEnterEvent* event) {

  event->acceptProposedAction();
}


// テーマのリロード
void ThemeClockWidget::dropEvent(QDropEvent* event) {

  const QMimeData* mime_data = event->mimeData();
  if (! mime_data->hasUrls()) {
    return;
  }

  QList<QUrl> url_list = mime_data->urls();
  for (int i = 0; i < url_list.count(); ++i) {

    QString path = url_list.at(i).path();

    // 先頭の '/' を取り除く
    path.replace(0, 1, "");

    // 日本語のパスを含む場合への対処
    std::string std_path = toStdStringPath(path);
    if (pimpl->loadThemeFile(QString(std_path.c_str()), this)) {
      return;

    } else {
      QString out;
      QMessageBox::warning(this, tr("qtmclock"),
			   tr("%1 is not correct theme file.").arg(path),
			   QMessageBox::Ok);
    }
  }
}


// 設定フォームの表示
void ThemeClockWidget::configHandler(void) {

  pimpl->setting_dialog_.show();
  pimpl->setting_dialog_.raise();
}


void ThemeClockWidget::changeScaling(int percent) {

  pimpl->scaling_percent_ = percent;

  pimpl->draw_widget_->updateScalingPercent(percent);
  resize(pimpl->draw_widget_->size());
}


void ThemeClockWidget::changeNoAnimation(bool checked) {

  pimpl->no_animation_ = checked;
  pimpl->draw_widget_->setAnimationOn(pimpl->no_animation_);
}
