/*!
  \file
  \brief qlm_batmon の描画ウィジット

  \author Satofumi KAMIMURA

  $Id: BatmonWidget.cpp 33 2010-01-06 09:12:34Z satofumi $
*/

#include "BatmonWidget.h"
#include "BatteryState.h"
#include "DrawCharacter.h"
#include <QApplication>
#include <QPainter>
#include <QTime>
#include <QTimer>
#include <QActionGroup>
#include <QAction>
#include <QSettings>
#include <QMouseEvent>
#include <QMenu>
#include <cmath>

using namespace qrk;
using namespace std;


namespace
{
  const char* Organization = "Hyakuren Soft LTD.";
  const char* Application = "qlm_batmon";
};


struct BatmonWidget::pImpl
{
  enum {
    IntervalMsec = 2000,        // [msec]
  };

  BatmonWidget* widget_;
  BatteryState battery_;
  ChargingState battery_state_;
  ChargingState previous_battery_state_;
  size_t percent_;
  size_t previous_percent_;
  QTimer update_timer_;
  vector<character_t> characters_;
  size_t character_index_;
  DrawCharacter draw_character_;
  QPoint drag_position_;

  bool hide_checked_;
  QAction* hide_action_;
  vector<QAction*> skin_actions_;
  bool is_masked_;

  bool is_tray_available_;
  QSystemTrayIcon tray_icon_;


  pImpl(BatmonWidget* widget)
    : widget_(widget),
      battery_state_(Unknown), previous_battery_state_(Unknown),
      percent_(0), previous_percent_(0), character_index_(0),
      hide_checked_(false), hide_action_(NULL), is_masked_(false),
      is_tray_available_(QSystemTrayIcon::isSystemTrayAvailable())
  {
  }


  void initialize(void)
  {
    // アイコンを適用
    widget_->setWindowIcon(QIcon(":icons/qlm_batmon_icon"));

    // 背景を灰色にし、アンチエイリアスの効果を期待して少しだけ透明にする
    QColor color(128, 128, 128, 223);
    widget_->setPalette(color);

    update();

    // 更新用のタイマーを初期化
    update_timer_.setInterval(IntervalMsec);
    connect(&update_timer_, SIGNAL(timeout()), widget_, SLOT(updateTimeout()));
    update_timer_.start();
  }


  void initializeCharacters(void)
  {
    // 女の子
    character_t girl;
    girl.name = tr("girl");
    girl.picture_unknown = ":picture/girl_unknown";
    girl.picture_charging = ":picture/girl_charging";
    girl.picture_discharging = ":picture/girl_discharging";
    girl.percent_bar = ":picture/percent_bar";
    girl.bar_position = QPoint(4, 43);
    characters_.push_back(girl);

    // りもペンギン
    character_t penguin;
    penguin.name = tr("pengui");
    penguin.picture_unknown = ":picture/penguin_unknown";
    penguin.picture_charging = ":picture/penguin_charging";
    penguin.picture_discharging = ":picture/penguin_discharging";
    penguin.percent_bar = ":picture/percent_bar";
    penguin.bar_position = QPoint(5, 44);
    characters_.push_back(penguin);
  }


  void initializeMenu(void)
  {
    // キャラクタ選択メニューの作成
    size_t n = characters_.size();
    for (size_t i = 0; i < n; ++i) {
      QAction* action = new QAction(characters_[i].name, widget_);
      action->setCheckable(true);
      widget_->addAction(action);
      skin_actions_.push_back(action);
    }
    connect(skin_actions_[0], SIGNAL(triggered()),
            widget_, SLOT(girlSkinSelected()));
    connect(skin_actions_[1], SIGNAL(triggered()),
            widget_, SLOT(limoSkinSelected()));
    setSkin(character_index_);

    widget_->addAction(newSeparator());

    // 充電中に非表示にするかのチェックボックス
    hide_action_ = new QAction(tr("Hide while charging"), widget_);
    hide_action_->setCheckable(true);
    hide_action_->setChecked(hide_checked_);
    widget_->addAction(hide_action_);
    connect(hide_action_, SIGNAL(triggered()), widget_, SLOT(hideSpecified()));

    widget_->addAction(newSeparator());

    // 終了
    QAction* quit_action = new QAction(QAction::tr("Quit"), widget_);
    quit_action->setShortcut(tr("Ctrl+Q"));
    connect(quit_action, SIGNAL(triggered()), qApp, SLOT(quit()));
    widget_->addAction(quit_action);

    widget_->setContextMenuPolicy(Qt::ActionsContextMenu);
  }


  void initializeSystemTray(void)
  {
    // システムトレイにアイコンを配置
    tray_icon_.setIcon(widget_->windowIcon());
    tray_icon_.setToolTip(tr("APM Battery Monitor"));

    QMenu* menu = new QMenu(widget_);
    QList<QAction*> actions = widget_->actions();
    for (int i = 0; i < actions.size(); ++i) {
      menu->addAction(actions.at(i));
    }
    tray_icon_.setContextMenu(menu);
    tray_icon_.hide();
  }


  QAction* newSeparator(void)
  {
    QAction* separator = new QAction(widget_);
    separator->setSeparator(true);
    return separator;
  }


  void loadSettings(bool restore_geometry)
  {
    QSettings settings(Organization, Application);

    character_index_ = settings.value("character_index", 0).toInt();
    hide_checked_ = settings.value("hide_checked", false).toBool();

    if (restore_geometry) {
      widget_->restoreGeometry(settings.value("geometry").toByteArray());
    }
  }


  void saveSettings(void)
  {
    QSettings settings(Organization, Application);

    settings.setValue("geometry", widget_->saveGeometry());
    settings.setValue("character_index", character_index_);
    settings.setValue("hide_checked", hide_action_->isChecked());
  }


  ChargingState batteryState(void)
  {
    if (! battery_.isAvailable()) {
      return Unknown;
    }
    return (battery_.isCharging() ? Charging : Discharging);
  }


  void update(void)
  {
    // 描画する状態の取得
    battery_state_ = batteryState();
    percent_ = battery_.remainingPercent();

    // フォーカス時に表示する情報の更新
    QString message = updateMessage();
    widget_->setToolTip(message);

    bool repainted = false;
    if ((battery_state_ != previous_battery_state_) ||
        percent_ != previous_percent_) {
      repainted = true;
      widget_->repaint();
    }
    previous_battery_state_ = battery_state_;
    previous_percent_ = percent_;

    if (hide_checked_ && (battery_state_ == Charging)) {
      // 充電中に描画しないよう指定されているときは隠す
      widget_->hide();
      if (is_tray_available_) {
        tray_icon_.show();
      }
    } else {
      if ((! repainted) && widget_->isHidden()) {
        widget_->show();
        if (is_tray_available_) {
          tray_icon_.hide();
        }
      }
    }
  }


  // マスコットの再描画
  void repaint(void)
  {
    // 再描画
    QPainter painter(widget_);
    QRect base_rect;
    QRegion region;
    if (draw_character_.draw(painter, base_rect, region,
                             characters_[character_index_],
                             battery_state_, percent_)) {
      widget_->resize(base_rect.width(), base_rect.height());
      if (is_masked_) {
        widget_->clearMask();
      }
      widget_->setMask(region);
      is_masked_ = true;
    }
  }


  double remainingTime(size_t total_second)
  {
    return total_second / (60.0 * 60.0);
  }


  QString updateMessage(void)
  {
    double remaining_time = remainingTime(battery_.remainingSecond());
    if (remaining_time > 1.0) {
      remaining_time = floor(remaining_time);
    }

    switch (battery_state_) {
    case Charging:
      return QString(tr("charging: %1 [%]")).arg(percent_);
      break;

    case Discharging:
      if (remaining_time == 0.0) {
	return QString(tr("remaining: %1 [%]")).arg(percent_);
      } else {
	return QString(tr("remaining: %1 [h] (%2 [%])")).arg(remaining_time, 0, 'g', 1).arg(percent_);
      }
      break;

    default:
    case Unknown:
      return QString(tr("unknown system"));
      break;
    }
  }


  void setSkin(size_t index)
  {
    character_index_ = index;

    size_t n = skin_actions_.size();
    for (size_t i = 0; i < n; ++i) {
      bool checked = (i == index) ? true : false;
      skin_actions_[i]->setChecked(checked);
    }
  }
};


BatmonWidget::BatmonWidget(bool geometry_specified, QWidget* widget)
  : QWidget(widget), pimpl(new pImpl(this))
{
  pimpl->initialize();
  pimpl->loadSettings(! geometry_specified);
  pimpl->initializeCharacters();
  pimpl->initializeMenu();
  if (pimpl->is_tray_available_) {
    // システムトレイが利用可能ならば登録
    pimpl->initializeSystemTray();
  }

  if (geometry_specified) {
    // 位置がコマンドラインで指定され直したため、それを保存する
    pimpl->saveSettings();
  }
}


BatmonWidget::~BatmonWidget(void)
{
  pimpl->saveSettings();
}


void BatmonWidget::paintEvent(QPaintEvent* event)
{
  static_cast<void>(event);

  pimpl->repaint();
}


void BatmonWidget::updateTimeout(void)
{
  pimpl->update();
}


// 左クリックでのフォーム移動
void BatmonWidget::mousePressEvent(QMouseEvent* event)
{
  if (event->button() == Qt::LeftButton) {
    pimpl->drag_position_ = event->globalPos() - frameGeometry().topLeft();
    event->accept();
  }
}


// 左クリックでのフォーム移動
void BatmonWidget::mouseMoveEvent(QMouseEvent* event)
{
  if (event->buttons() & Qt::LeftButton) {
    move(event->globalPos() - pimpl->drag_position_);
    pimpl->saveSettings();
    event->accept();
  }
}


void BatmonWidget::hideSpecified(void)
{
  bool check = pimpl->hide_action_->isChecked();
  pimpl->hide_action_->setChecked(check);

  pimpl->hide_checked_ = check;
  pimpl->saveSettings();
}


void BatmonWidget::girlSkinSelected(void)
{
  pimpl->setSkin(0);
  update();
  repaint();
  pimpl->saveSettings();
}


void BatmonWidget::limoSkinSelected(void)
{
  pimpl->setSkin(1);
  update();
  repaint();
  pimpl->saveSettings();
}
