import imp
import os.path
from twisted.internet import reactor
from twisted.internet.defer import Deferred
from twisted.web.client import getPage
from twisted.python import log, logfile
import feedparser
import yaml
import sys

import habu.cron

CURRENT_PATH  = os.path.dirname(__file__)


class ExecuteContext(object):
    def __init__(self, executeManager, name, restInstances):
        self.executeManager = executeManager
        self.name = name
        self.restInstances = restInstances

    def _copy(self, refAdd=True):
        return ExecuteContext(
            self.executeManager, 
            self.name,
            self.restInstances[:])

    def hasNext(self):
        return len(self.restInstances) > 0

    def next(self):
        return self.restInstances.pop(0)
    
        

class ExecuteManager(object):
    HOOK_TYPE_START = 1
    HOOK_TYPE_FOLK = 2
    # We need add end event
    HOOK_TYPE_SUCCESS = 3
    HOOK_TYPE_GOT_ERROR = 4

    
    def __init__(self, name, lineInstances):
        self.lineInstances = lineInstances
        for instance in lineInstances:
            setattr(instance, "executeManager", self)
        self.name = name
        self.executeContexts = []
        self.hooks = []

    def deferredWrapper(self, content):
        deferred = Deferred()
        reactor.callLater(0, deferred.callback, content)
        return deferred

    def addHook(self, hook_type, hook_func):
        self.hooks.append((hook_type, hook_func))
        return self

    def removeHook(self, hook_type, hook_func):
        hook = (hook_type, hook_func)
        if hook in self.hooks:
            self.hooks.remove(hook)

    def _fireHook(self, hook_type):
        myhooks = [ hook for hook in self.hooks
                    if hook[0] == hook_type]
        for hook in myhooks:
            log.msg("Event Fired : " + str(hook_type))
            hook[1](self)
    
    def execute(self):
        log.msg("Start Line : " + self.name)
        if not self.lineInstances:
            log.msg("End Line(no line instance) : " + self.name)
            return
        executeContext = ExecuteContext(
            self,
            self.name,
            self.lineInstances)

        self.executeContexts = [executeContext]
        self._fireHook(self.HOOK_TYPE_START)
        self.runNext(None, executeContext)

    def executeErrback(self, failure, executeContext):
        log.msg("End Line(%s) : %s" % (str(failure), self.name))
        self.executeContexts.remove(executeContext)
        self._fireHook(self.HOOK_TYPE_GOT_ERROR)

    def executeCallback(self, result, executeContext):
        log.msg("Get executeCallback : " + self.name)
        self._fireHook(self.HOOK_TYPE_SUCCESS)
        if result == None:
            log.msg("End Line(no callback data) : " + self.name)
            self.executeContexts.remove(executeContext)
            self._fireHook(self.HOOK_TYPE_SUCCESS)
            return

        if not executeContext or not executeContext.hasNext():
            log.msg("End Line : " + self.name)
            self.executeContexts.remove(executeContext)
            self._fireHook(self.HOOK_TYPE_SUCCESS)
            return

        self.runNext(result, executeContext)

        return result

    def runNext(self, result, executeContext):
        instance = executeContext.next()
        values = instance.execute(result)
        if not values:
            log.msg("End Line(no execute data) : " + self.name)
            self.executeContexts.remove(executeContext)
            self._fireHook(self.HOOK_TYPE_SUCCESS)
            return
        if not executeContext.hasNext():
            log.msg("End Line(no next instance) : " + self.name)
            self.executeContexts.remove(executeContext)
            self._fireHook(self.HOOK_TYPE_SUCCESS)
            return
        
        self.scheduleNext(values, executeContext)

    def scheduleNext(self, values, executeContext):
        if not isinstance(values, list):
            deferredList = [values]
        else:
            deferredList = values
        
        deferred = deferredList.pop(0)
        self.callbackNext(deferred, executeContext)

        for deferred in deferredList:
            log.msg("Folk Line : " + self.name)
            copiedContext = executeContext._copy()
            self.executeContexts.append(copiedContext)
            self._fireHook(self.HOOK_TYPE_FOLK)
            self.callbackNext(deferred, copiedContext)

    def callbackNext(self, deferred, executeContext):
        if not isinstance(deferred, Deferred):
            deferred = self.deferredWrapper(deferred)
        
        deferred.addCallback(self.executeCallback, executeContext)
        deferred.addErrback(self.executeErrback, executeContext)

class Habu(object):
    def __init__(self):
        self.config = None
        self.environ = None
        self.moduleFactories = {}
        self.scheduler = None

    def load(self, yamlStream):
        self.config = yaml.load(yamlStream)
        self.environ = self.config.setdefault("global", None)
        self._loadAfter()
        scheduleConfig = self.config.setdefault("scheduler", None)
        self.setupCron(scheduleConfig)

    def _loadAfter(self):
        # plugin loading path sequence order is
        # 1. current directory
        # 2. plugin path in config file.
        # 3. this module directory path for the standard plugins.
        self.environ.setdefault("plugin_path", ["."])
        if "." not in self.environ["plugin_path"]:
            self.environ["plugin_path"].insert(0, ".")
        if CURRENT_PATH not in self.environ["plugin_path"]:
            self.environ["plugin_path"].append(CURRENT_PATH)


    def startLogging(self):
        output = self.environ.setdefault("log", "null")
        if not output or output == "null":
            return
        if output == "stdout":
            logFile = sys.stdout
        elif output == "stderr":
            logFile = sys.stderr
        else:
            import os.path
            basename = os.path.basename(output)
            dirname = os.path.dirname(output)
            logFile = logfile.DailyLogFile(basename, dirname)
        log.startLogging(logFile)


    def setupCron(self, scheduleConfig):
        if not scheduleConfig:
            return
        if not self.scheduler:
            log.msg("cron created")
            self.scheduler = habu.cron.Cron()
        for crontab in scheduleConfig:
            try:
                name = crontab.get("pipeline")
                expr = crontab.get("expression")
                log.msg("con expression(%s) : %s" % (name, expr))
                expr = habu.cron.parseExpression(expr)
                if name:
                    self.scheduler.add(expr, self.runJob, name)
            except Exception, e:
                log.msg("ERROR: remove cron tab")

    def runJob(self, name):
        log.msg("Run Job : " + name)
        if self.config["pipeline"].has_key(name):
            self.run(name)

    def getPipelines(self):
        return self.config.setdefault("pipeline", None)

    def getPipelineNames(self):
        return config["pipeline"].keys()

    def startScheduler(self):
        if self.scheduler:
            self.scheduler.start()

    def runAll(self):
        return [self.run(key) for key in self.config["pipeline"].keys()]

    def run(self, pipeline):
        line = self.config["pipeline"][pipeline]
        lineInstances = [self.instanciate(lineModule) for lineModule in line]
        executeManager = ExecuteManager(pipeline,
                                        lineInstances)
        reactor.callLater(0, executeManager.execute)

        return executeManager

    def error(self, *args, **kwds):
        log.error("ERROR", *args, **kwds)

    def instanciate(self, lineModule):
        mod = self.loadModule(lineModule["module"])
        config = lineModule.setdefault("config", {})
        obj = mod.create(config, self.environ)
        if hasattr(obj, "setSubModules"):
            subModules = lineModule.get("sub-modules", [])
            subModuleObjs = [self.instanciate(module) for module in subModules]
            obj.setSubModules(subModuleObjs)
        
        return obj        

    def loadModule(self, moduleName):
        if self.moduleFactories.has_key(moduleName):
            return self.moduleFactories[moduleName]
        
        basepath, name = moduleName.rsplit(".", 1)
        basepath = basepath.replace(".", "/")
        
        plugin_path = [os.path.join(dirname, basepath) for dirname 
                       in self.environ.get("plugin_path")]
        plugin_path.append(os.path.join(CURRENT_PATH, basepath))
        modinfo = imp.find_module(name, plugin_path)
        mod = imp.load_module(name, *modinfo)
        self.moduleFactories[moduleName] = mod
        
        return mod

