from twisted.internet.defer import Deferred
from twisted.python import log


class Join(object):
    def __init__(self):
        self.contents = []
        self.deferred = None

    def mergeContents(self):
        """
        merge all contents to one content.
        If you want to merge with your custom way,
        override this method.

        This method provide very simple merge.
        """
        if len(self.contents) == 1:
            return self.contents[0]
        
        content = self.contents.pop()
        for c in self.contents:
            content["entries"].extend(c["entries"])
            
        return content


    def eventFired(self, manager):
        if len(manager.executeContexts) == 1 and self.deferred:
            log.msg("fired next")
            manager.removeHook(manager.HOOK_TYPE_GOT_ERROR, self.eventFired)
            manager.removeHook(manager.HOOK_TYPE_SUCCESS, self.eventFired)
            self.deferred.callback(self.mergeContents())


    def execute(self, content):
        manager = self.executeManager
        self.contents.append(content)
        
        if not self.deferred:
            self.deferred = Deferred()
            manager.addHook(manager.HOOK_TYPE_GOT_ERROR, self.eventFired)
            manager.addHook(manager.HOOK_TYPE_SUCCESS, self.eventFired)
            return self.deferred

        log.msg("Thread Joined")
        return None


#####################
# utility methods
#

def mergeResult(content, subModules, arg = None):
    if not isinstance(content, list):
        content = [content]

    for module in subModules:
        result = module.execute(arg)
        if not isinstance(result, list):
            content.append(result)
        else:
            content.extend(result)

    return content


#####################
# decorators
#


class HabuDecoratorBase(object):
    def __init__(self, *args, **kwds):
        self.args = args
        self.kwds = kwds

    def before(self, content):
        return content

    def after(self, content):
        return content

    def __call__(self, func):
        def modifiedFunc(obj, content):
            content = self.before(content)
            result = func(obj, content)
            return self.after(result)

        return modifiedFunc

class Sort(HabuDecoratorBase):
    def __init__(self, *args, **kwds):
        super(Sort, self).__init__(*args, **kwds)

    def before(self, content):
        key = self.kwds.get("key", "updated_parsed")
        reverse = self.kwds.get("reverse", False)
        entries = content.get("entries", [])
        
        if callable(key):
            getkey = key
        else:
            def getkey(a):
                return a[key]
        
        entries.sort(key=getkey, reverse=reverse)
        return content


def evaluate(item, expr):
    if isinstance(expr, str) or isinstance(expr, unicode):
        return item.find(expr) > -1
    
    if callable(expr):
        return expr(item)
    
    if hasattr(expr, "search") and  expr.search(item):
        return True

    return False

class Grep(HabuDecoratorBase):
    """
    one of conditions match, filter it
    """

    def __init__(self, exclusive = False, *args, **kwds):
        super(Grep, self).__init__(*args, **kwds)
        self.exclusive = exclusive

    def iterEntries(self, content):
        for entry in content.get("entries", []):
            for key, value in self.kwds.iteritems():
                item = entry.get(key, None)
                if item == None:
                    if self.exclusive:
                        yield entry
                elif evaluate(item, value) != self.exclusive:
                    yield entry

    def before(self, content):
        content["entries"] = [ entry for entry in self.iterEntries(content) ]
        
        return content




if __name__ == "__main__":

    class C:
        @Sort(key="updated")
        def myfunc(self, c):
            return c
        

    c = C()
    r = c.myfunc({"entries":[{"updated": "3"},
                             {"updated": "1"},
                             {"updated": "3"},
                             {"updated": "2"},
                             ]})
    print r

    r = c.myfunc({"entries":[{"updated": 3},
                             {"updated": 1},
                             {"updated": 3},
                             {"updated": 2},
                             ]})
    print r

    def myGetKey(a):
        return a.get("updated")

    class C:
        @Grep(exclusive=True, updated="2")
        @Sort(key=myGetKey, reverse=True)
        def myfunc(self, c):
            return c
        

    c = C()
    r = c.myfunc({"entries":[{"updated": "3"},
                             {"updated": "1"},
                             {"updated": "4"},
                             {"updated": "2"},
                             ]})
    
    print "filter sort"
    print r

    import re
    class C:
        @Grep(updated=re.compile("[1|3|4]"))
        @Sort(key=myGetKey, reverse=True)
        def myfunc(self, c):
            return c
        

    c = C()
    r = c.myfunc({"entries":[{"updated": "3"},
                             {"updated": "1"},
                             {"updated": "4"},
                             {"updated": "2"},
                             ]})
    print "gerp sort"

    print r



        
