<?php
/**
 * RdbObject and its Manager
 * class for manupulating data stored in relational database server.
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * @author Haruki Setoyama <haruki@planewave.org> 
 * @copyright Copyright &copy; 2003, Haruki SETOYAMA <haruki@planewave.org>
 * @license http://www.opensource.org/licenses/lgpl-license.php LGPL
 * @package rdbObject
 * @since 2003.06
 * @version 0.2 $Id: rdbobjectmanager.php,v 1.14 2003/11/24 15:18:35 haruki Exp $
 * @link http://www.planewave.org/rdbobject/
 */
/**
 * Factory class for rdbObjectManager
 * 
 * @abstract 
 */
class rdbObjectManagerFactory extends PEAR {
    /**
     * object instanciation.
     * 
     * Use singleton class.
     * singleton::getInstance(NAME_OF_rdbObjectManagerFactory_CLASS)
     * 
     * @access publc
     */
    function &getInstance()
    {
        $rdbm = $this->_rdbm();
        $file = RDBOBJECT_DIR . '/' . strtolower($this->_manager()) . '.' . $rdbm . '.php';
        include_once($file);
        $managerClass = $this->_manager() . '_' . $rdbm;
        if (!class_exists($managerClass)) {
            return PEAR::raiseError("Database server '$rdbm' is not supported." , RDBOBJECT_ERROR_INTERNAL);
        } 
		$rdbObjectClass = $this->_rdbObject();
		if (! class_exists($rdbObjectClass)) {
            return PEAR::raiseError("RdbObject class '$rdbObjectClass' is not defined." , RDBOBJECT_ERROR_CONFIG);
        }
		
        $manager = new $managerClass;
        $manager->_myfactory = get_class($this);
		$manager->_table = $this->_table($rdbm);
        $manager->_linkid = &$this->_linkid($rdbm);
		
        $ret = $manager->_init($rdbObjectClass, $this->_column($rdbm));

        return $manager;
    } 

    /**
     * returns name of database server to use
     * 
     * @return string 
     * @access private 
	 * @abstract
     */
    function _rdbm()
    {
        return 'mysql';
    } 

    /**
     * returns name of rdbObject class
     * 
     * @return string 
     * @access private
	 * @abstract
     */
    function _rdbObject() {}

    /**
     * returns name of rdbObject manager class
     * 
     * @return string 
     * @access private
     */
    function _manager()
    {
        return 'rdbObjectManager';
    } 

    /**
     * returns correspondence of variable name to column name
     * 
     * @return array 
     * @access private
     */
    function _column($rdbm)
    {
        return array();
    } 

    /**
     * returns table name this object use
     * 
     * @return string 
     * @access private
	 * @abstract 
     */
    function _table($rdbm) {} 

    /**
     * returns link id
     * 
     * @return int linkid
     * @access private 
	 * @abstract
     */
    function &_linkid($rdbm) {}
} 

/**
 * rdbObjectManager
 * 
 * Child classes shoulld be implemented for respective database server
 * with the name like 'rdbObjectElement_mysql'
 * 
 * @abstract 
 */
class rdbObjectManager extends PEAR {
	/**
     * logger object or array for log
     * 
     * @access public 
     */
    var $logger = null;
	
    /**
     * the name of manager factory class
     * 
     * @access private 
     * @var string 
     */
    var $_myfactory = '';

    /**
     * table name
     * 
     * @access private 
     * @var string 
     */
    var $_table;

    /**
     * linkid to database server
     * 
     * @access private 
     * @var int 
     */
    var $_linkid;
	
    /**
     * cache of the rdbObjects
     * 
     * @access private 
     * @var array 
     */
    var $_cache = array();

    /**
     * blank rdbObject instance.
     * 
     * @access private 
     * @var rdbObject 
     */
    var $_rdbObject;

    /**
     * reference to $this->_rdbObject->element
     * 
     * @access private 
     * @var array array of rdbObjectElement
     */
    var $_element = array();

    /**
     * column names
     * reference to $this->_rdbObject->_column
     * 
     * @access private 
     */
    var $_column;

    /**
     * quoted column names
     * 
     * @access private 
     * @var array 
     */
    var $_column_quoted = array();

    /**
     * types of variables
     * reference to $this->_rdbObject->$_type
     * 
     * @access private 
     */
	var $_type;

    /**
     * variable name of the primary key
     * 
     * @access private 
     */
    var $_pkey;

    /**
     * column name of the primary key
     * 
     * @access private 
     */
    var $_pkey_column;

    /**
     * initialize
     * 
     * @access private only for rdbObjectManagerHandler
     * @return bool |pearError
     */
    function _init($rdbObjectClass, $columns)
    { 
        $this->_rdbObject = new $rdbObjectClass;

        $this->_rdbObject->_manager = $this->_myfactory;
        $this->_column = &$this->_rdbObject->_column;
        $this->_type = &$this->_rdbObject->_type;

        $names = array_keys($this->_column);
        foreach ($names as $name) {
            if (isset($columns[$name])) {
                $this->_column[$name] = $columns[$name];
            } 
            $this->_column_quoted[$name] = $this->_quote_column($this->_column[$name]);
        } 
        $this->_pkey = &$this->_rdbObject->_pkey;
        $this->_pkey_column = $this->_column[$this->_pkey];
		
		$this->_table = $this->_quote_table($this->_table);

        //return true;
    } 

	/**
     * raiseError() when argument is invalid
     * 
     * @access private
     * @return pearError
     */
	function _argument_error($msg, $func =null)
	{
		$msg = '<b>Invalid Argument</b>: '.$msg;
		if (isset($func)) {
		    $msg .= ' at <b>'.$func.'()</b>';
		}
		if (function_exists('debug_backtrace')) {
			$bt = debug_backtrace();
			$myname = get_class($this);
			$num = count($bt);
			for ($i=0; $i < $num; $i++) {
				if ($bt[$i]['class'] != $myname) {
				    if (!isset($func)) {
					    $msg .= ' at <b>'.$bt[$i-1]['function'].'</b>';
					}
					$msg .= ' in <b>'.$bt[$i-1]['file'].'</b>';
					$msg .= ' on line <b>'.$bt[$i-1]['line'].'</b>'."\n";
					break;
				}
			}
		}
		return $this->raiseError($msg.'<br>', RDBOBJECT_ERROR_ARGUMENT);	
	}

	/**
     * log SQL statement etc...
     * 
     * @access private
     * @return void
     */	
	function _log($msg)
	{
		$t = gettimeofday();
		$time = $t['sec'].'.'.$t['usec'];
		if (is_object($this->logger)) {
		    $this->logger->log($time.': '.$msg, PEAR_LOG_DEBUG);
		}
		else
		{
			$this->logger[] = $time.': '.$msg;
		}
	}

	/**
     * fetch rdbObjects from SQL query result
     * 
     * @access private
     * @return array
     */	
	function &_fetch_objects($sql)
    {
        $result = $this->_query($sql);
        if (! $result) {
            $this->_query_error($sql);
			return array();
        } 
        $ret_objs = array();
        while ($ret = $this->_fetch($result)) {
            $pkey = intval($ret[$this->_pkey_column]);
            if (! isset($this->_cache[$pkey])) {
                $this->_cache[$pkey] = $this->_rdbObject;
                $this->_cache[$pkey]->_dbRawData = $ret;
            } 
            $ret_objs[$pkey] = &$this->_cache[$pkey];
        } 
        return $ret_objs;
    }

	/**
     * fetch one value from SQL query result
     * 
     * @access private
     * @return string
     */	
	function _fetch_one_result($sql)
    {
        $result = $this->_query($sql);
        if (! $result) {
            $this->_query_error($sql);
			return false;
        } 
        $ret = $this->_fetch($result);
        return $ret['vl'];
    }

	/**
     * fetch values from SQL query result
     *
     * @access private
     * @return array
     */
	function _fetch_values($sql, $key = false)
    {
        $result = $this->_query($sql);
        if (! $result) {
            $this->_query_error($sql);
			return false;
        } 
		$vals = array();
		if ($key) {
		    while ($ret = $this->_fetch($result)) {
				$vals[$ret['KY']] = $ret['VL'];
			}
		} else {
        	while ($ret = $this->_fetch($result)) {
				$vals[] = $ret['VL'];
			}
		}
        return $vals;
    }

    /**
     * create new rdbObject instance
     * 
     * @access public 
     * @return rdbObject 
     */
    function &create()
    {
        $obj = $this->_rdbObject;
        //$obj->_blank();
        return $obj;
    } 

    /**
     * clear cache
     * 
     * @access public 
     */
    function clearCache()
    {
        unset($this->_cache);
        $this->_cache = array();
    } 

    /**
     * open one rdbObject instance
     * 
     * @param int $pkey primary key
     * @access public 
     * @return rdbObject|null
     */
    function &open($pkey)
    {
		if (! is_numeric($pkey)) {
			$this->_argument_error('Primary key must be integer', 'open');
			return null;
		}
		$intpkey = intval($pkey);
		if ($intpkey != $pkey) {
		    $this->_argument_error('Primary key must be integer', 'open');
			return null;
		}

        if (isset($this->_cache[$intpkey])) {
            return $this->_cache[$intpkey];
        }
		
        $sql = 'SELECT * FROM '.$this->_table
            . ' WHERE '.$this->_column_quoted[$this->_pkey].' = '. $intpkey;
        $result = $this->_query($sql);
        if (! $result) {
            $this->_query_error($sql);
			return null;
        }
        $ret = $this->_fetch($result);
        if ($ret === false) {
            return null;
        }
        $this->_cache[$intpkey] = $this->_rdbObject;
        $this->_cache[$intpkey]->_dbRawData = $ret;
        return $this->_cache[$intpkey];
    } 

    /**
     * open some rdbObject instances
     * 
     * @param array $pkeys     array of primary keys
     * @access public 
     * @return array array of rdbObject
     */
    function &openObjects($pkeys)
    {
		if (! is_array($pkeys)) {
		    $this->_argument_error('Primary keys must be in array', 'openObjects');		
			return array();
		}
		$intpkeys = array();
		foreach ($pkeys as $pkey) {
			if (! is_numeric($pkey)) {
			    $this->_argument_error('Primary keys must be integer' , 'openObjects');
				return array();
			}
			$intpkey = intval($pkey);
			if ($intpkey != $pkey) {
			    $this->_argument_error('Primary keys must be integer' , 'openObjects');
				return array();
			}
			$intpkeys[] = $intpkey;
		}
		
    	$sql = 'SELECT * FROM ' . $this->_table . '';
        $sql .= ' WHERE ' . $this->_column_quoted[$this->_pkey] . ' = '
         . implode(' OR ' . $this->_column_quoted[$this->_pkey] . ' = ', $intpkeys);
    	return $this->_fetch_objects($sql);
    } 

    /**
     * open some rdbObject instances with $varname be $value
     * 
     * @param string $varname variable name
     * @param mixed $value 
     * @param string $opetator 
     * @return array array of rdbObject
     * @access public  
     */
    function &openWith($varname, $value, $opetator = '=')
    {
		$where = $this->_render_comparison($varname, $value, $opetator);
		if ($where === false) {
		    return array();
		}
        $sql = 'SELECT * FROM ' . $this->_table . ' WHERE '. $where;
        return $this->_fetch_objects($sql);
    } 

    /**
     * open some rdbObject instances with some conditions
     * parameter $condition is rdbObjectCondition object
     * 
     * @param object $condition 
     * @access public 
     * @return array array of rdbObject
     */
    function &openConditional($condition)
    {
		if ($this->_validate_condition($condition, 'openConditional') !== true) {
       		return array();
		}
		$sql = $this->_compose_select_sql('*', $this->_table, $condition);
        return $this->_fetch_objects($sql);
    } 

    /**
     * open rdbObjects instances with other table joined
     * 
     * @access public 
     * @param rdbObjectManamer $manager_related 
     * @param string $fkey_related 
     * @param stirng $varname 
     * @param mixed $value 
     * @param string $opetator 
     * @return array array of rdbObject
     */
    function &openJoin(&$manager_related, $fkey_related, $varname, $value, $opetator = '=')
    {
        if (! isset($manager_related->_column[$varname])) {
            $this->_argument_error("Variable '$varname' is not defined in the related table" , 'openJoin');
        	return array();
		} 
		$awhere = $manager_related->_render_comparison($varname, $value, $opetator);
		if ($awhere === false) {
		    return array();
		}
        $sql = sprintf("SELECT b.* FROM %s as a, %s as b WHERE (a.%s = b.%s) AND (a.%s)" 
						, $manager_related->_table 
						, $this->_table 
						, $manager_related->_column_quoted[$fkey_related] 
						, $this->_column_quoted[$this->_pkey] 
						, $awhere);

        return $this->_fetch_objects($sql);
    } 

    /**
     * get primary keys with some condition
     * 
     * @access public 
     * @return array array of int
     */
    function getPkeys($condition =null)
    {
		return $this->_get_values($this->_pkey, $condition, 'getPkeys');
    } 

    /**
     * get valus of a variable with some condition
     * 
     * @access public 
     * @return array
     */	
	function getValues($variable, $condition =null)
	{
		if ($this->_validate_variable($variable, 'getValues') !== true) {
   			return array();
		}
		return $this->_get_values($variable, $condition, 'getValues');
	}
	
    /**
     * main part of getValues() function
     * 
     * @access private 
     * @return array
     */
	function _get_values($variable, $condition =null, $func)
	{
		if ($this->_validate_condition($condition, $func) !== true) {
      		return array();
		}
		
		$sql = $this->_compose_select_sql($this->_column_quoted[$variable], $this->_table, $condition);
        $result = $this->_query($sql); 
        if (! $result) {
          	$this->_query_error($sql);
			return array();
       	} 
       	$keys = array();
       	while ($ret = $this->_fetch($result)) {
           	$keys[] = $ret[$this->_pkey_column];
       	} 
        return $keys;
	} 

    /**
     * count rows that meets with condition
     * 
     * @access public 
     * @return array 
     */
    function count($condition, $group =null)
    {
		if ($this->_validate_condition($condition, 'count') !== true) {
      		return array();
		}
		if (isset($group)) {
			if ($this->_validate_variable($group, 'count') !== true) {
	   			return array();
			}
			$select = $this->_column_quoted[$group].' as KY, COUNT(*) as VL';
			//$sql = $this->_compose_select_sql($select, $this->_table, $condition, false, 'ky');
			$sql = $this->_compose_select_sql($select, $this->_table, $condition, false, $this->_column_quoted[$group]);
		} else {
			$select = 'COUNT(*) as VL';
			$sql = $this->_compose_select_sql($select, $this->_table, $condition);
		}
        return $this->_fetch_values($sql, $group);
    } 

    /**
     * returns max value within the variables
     * 
     * @access public 
     * @return array  
     */
    function max($variable, $condition = null, $group =null)
    {
	    return $this->_func('MAX', $variable, $condition, $group);
    } 

    /**
     * returns min value within the variables
     * 
     * @access public 
     * @return array 
     */
    function min($variable, $condition = null, $group =null)
    {
	    return $this->_func('MIN', $variable, $condition, $group);
    } 
	
	/**
     * returns sumation within the variables
     * 
     * @access public 
     * @return array 
     */
    function sum($variable, $condition = null, $group =null)
    {
	    return $this->_func('SUM', $variable, $condition, $group);
    } 

	/**
     * main part of min,max etc ...
     *
     * @access private
     * @return array
     */
	function _func($func, $variable, $condition = null, $group =null)
    {
        if ($this->_validate_variable($variable, strtolower($func)) !== true) {
   			return array();
		}
		if ($this->_validate_condition($condition, strtolower($func)) !== true) {
   			return array();
		}
		
		if (isset($group)) {
			if ($this->_validate_variable($group, strtolower($func)) !== true) {
	   			return array();
			}
			$select = $this->_column_quoted[$group].' as KY, '.$func.'('.$this->_column_quoted[$variable].') as VL';
			//$sql = $this->_compose_select_sql($select, $this->_table, $condition, false, 'ky');
			$sql = $this->_compose_select_sql($select, $this->_table, $condition, false, $this->_column_quoted[$group]);
		} else {
			$select = $func.'('.$this->_column_quoted[$variable].') as VL';
			$sql = $this->_compose_select_sql($select, $this->_table, $condition);
		}
		return $this->_fetch_values($sql, $group);
		
    }
	
    /**
     * save rdbObject in databse
     *
     * @param object $object 
     * @access public 
     * @return bool
     * @abstract 
     */
    function save(&$object, $varnames =null)
    {
		if ($this->_valid_rdbobject($object, 'save') !== true) {
            return false;
        } 
        if ($object->pkey() > 0) {
			// Update
			return $this->_update($object, $varnames);
        } else {
        	// Insert new record
			$object->_pre_save(array_keys($this->_column));
        	return $this->_insert($object);
		}
    }

    /**
     * update rdbObject in databse
     *
     * @access private
     * @return bool
     */
	function _update(&$object, $varnames)
    {
        if (!isset($varnames)) {
            $vars = array_keys($this->_column);
        } else {
            $vars = array_intersect(array_keys($this->_column), $varnames);
        } 

        $object->_pre_save($vars);

        $object->change[$this->_pkey] = null;
        $data = array();
        foreach ($vars as $varname) {
            if ($object->change[$varname] == true) {
                if ($object->_dbRawData[$this->_column[$varname]] !== null) {
                    $data[] = '' . $this->_column_quoted[$varname] . ' = '
                     . $this->_quote($object->_dbRawData[$this->_column[$varname]], $varname);
                } else {
                    $data[] = 'NULL';
                } 
            } 
        }
		 
        $pkey = $object->pkey();
		
		$sql = 'SELECT COUNT(*) as VL FROM '.$this->_table;
		$sql .= ' WHERE '.$this->_column_quoted[$this->_pkey].' = '.$pkey;
		$ret = $this->_fetch_one_result($sql);
		if ($ret = 0) {
			$this->raiseError('RdbObject with primary key '.$pkey.'does not exists.' , RDBOBJECT_ERROR_INTERNAL);
			return false;
		}
			
        if (! empty($data)) {
            $sql = sprintf("UPDATE %s SET %s WHERE %s = %u" , $this->_table, implode(', ', $data) , $this->_column_quoted[$this->_pkey], $pkey);
            $result = $this->_query($sql);
            if (! $result) {
                $this->_query_error($sql);
				return false;
            } 
        } 

        foreach ($vars as $varname) {
            $object->change[$varname] = null;
        } 
        $this->_cache[$pkey]->_dbRawData = $object->_dbRawData;
        return $pkey;
    } 

    /**
     * insert rdbObject in databse
     *
     * @access private
     * @return bool
     */
    function _insert(&$object)
    {
        $id = $this->_gen_id();
        if ($id === false) {
            return false;
        }
        $object->_dbRawData[$this->_pkey_column] = $id;

        $fields = array();
        $data = array();
        foreach ($this->_column as $varname => $column) {
            $fields[] = $this->_column_quoted[$varname];
            $data[] = $this->_quote($object->_dbRawData[$column], $varname);
        }
        $sql = sprintf("INSERT INTO %s ( %s ) VALUES ( %s )"
                        , $this->_table
                        , implode(', ', $fields)
                        , implode(', ', $data));
        $result = $this->_query($sql);
        if ($result === false) {
            $this->_query_error($sql);
            return false;
        }

        $object->change = array();
        $this->_cache[$id] = &$object;
        return $id;
    }

    /**
     * ganerate id for insert
     *
     * @access private
     * @return bool
     * @abstract
     */
    function _gen_id() {}

    /**
     * delete rdbObject from databse
     * 
     * @param object $object 
     * @access public 
     * @return bool
     */
    function delete(&$object)
    {
	    if ($this->_valid_rdbobject($object, 'delete') !== true) {
            return false;
        } 

        foreach ($this->_type as $name => $type) {
			if ($type & RDBOBJECT_ELEMENT_VIRTUAL) {
			 	$object->deleteRelated($name);  
			}  
        } 

        $pkey = $object->pkey();
        $sql = sprintf("DELETE FROM %s WHERE %s = %u" , $this->_table, $this->_column_quoted[$this->_pkey], $pkey);
        $result = $this->_query($sql);
        if (! $result) {
            $this->_query_error($sql);
			return false;
        } 
		
		$sql = 'SELECT COUNT(*) as VL FROM '.$this->_table;
		$sql .= ' WHERE '.$this->_column_quoted[$this->_pkey].' = '.$pkey;
		$ret = $this->_fetch_one_result($sql);
		if ($ret > 0) {
			$this->raiseError('Fail to delete the RdbObject with primary key '.$pkey.'.' , RDBOBJECT_ERROR_INTERNAL);
			return false;
		}

        $this->_cache[$pkey] = null;
        $object = null;
        return true;
    }  

    /**
     * check if $object is valid rdbobject
     * 
     * @param object $object 
     * @access private 
     * @return bool 
     */
    function _valid_rdbobject(&$object, $func =null)
    {
        if (!is_a($object, 'rdbObject')) {
			$this->_argument_error('This is not rdbObject instance', $func);
            return false;
        }
		if ($object->_manager != $this->_myfactory) {
			$this->_argument_error('Specified rdbObject is of another manager', $func);
            return false;
        }
        return true;
    } 

    /**
     * validate condition argument
     * 
     * @access private 
     * @return bool 
     */
    function _validate_condition($condition, $func =null)
    {
		if (isset($condition)) {
	        if (!is_a($condition, 'rdbObjectCondition') && !is_a($condition, 'rdbObjectComparison')) {
				$this->_argument_error('Condition is invalid data type', $func);
				return false;
	        } 
			if ($condition->_manager != $this->_myfactory) {
				$this->_argument_error('Condition is for another manager', $func);
				return false;
			}
		}
		return true;
	}
	
	/**
     * validate argument of variable name 
     * 
     * @access private 
     * @return bool 
     */
    function _validate_variable($variable, $func =null)
    {
		if (! @isset($this->_column[$variable])) {
            $this->_argument_error('Variable \''.(string)$variable.'\' is not defined' , $func);
        	return false;
		}
		return true;
	}

    /**
     * render comparison statement of rdbObjectCondition and rdbObjectComparisons
     * 
     * @access private 
     * @return string|PEAR_Error SQL statement
     */
    function _render_comparison($variable, $value, $operator = '=')
    { 
		if (!@isset($this->_column[$variable])) {
		    $this->_argument_error('Variable name \''.(string)$variable.'\' is invalid"');
			return false;
		}
		
	  	$value = $this->_rdbObject->_to_query_string($variable, $value);
		if ($value === false) {
			$this->_argument_error("Value is invalid data type");
			return false;
		}
		
        if (! in_array($operator, array('=', '>', '<', '>=', '<=', '!=', '<>', 'LIKE'))) {
            $this->_argument_error('Operator \''.(string)$operator.'\' is invalid');
        	return false;
		} 
		
		$column = $this->_column_quoted[$variable];
        if ($value === null) {
            switch ($operator) {
            case '=':
            case '<=':
            case '>=':
            case 'LIKE':
                return $column . ' IS NULL';

            default:
                return $column . ' IS NOT NULL';
            } 
        } else {
            return $column . ' ' . $operator . ' ' . $this->_quote($value, $variable) . '';
        } 
    } 

    /**
     * get instance of Comparison 
	 * @see rdbObjectComparison
     * 
     * @access public 
     */	
	function newComparison($variable = '', $value = '', $operator = '=')
	{
		$where = $this->_render_comparison($variable, $value, $operator);
		if ($where === false) {
		    return null;
		}
		$obj = new rdbObjectComparison;
		$obj->_where = $where;
		$obj->_manager = $this->_myfactory;
		return $obj;	
	}

    /**
     * get instance of Codition
	 * @see rdbObjectCondition 
     * 
     * @access public 
     */	
	function newCondition($ope ='AND')
	{
		if ($ope == 'AND' || $ope == 'OR') {
		    $obj = new rdbObjectCondition;
			$obj->_logical_operator = $ope;
			$obj->_manager = $this->_myfactory;
			return $obj; 
		} else {
			$this->_argument_error('Operator \''.(string)$ope.'\' is invalid' , 'newCondition');
			return null;
		}
	}

    /**
     * get instance of Codition
	 * @see rdbObjectOrderBy 
     * 
     * @access public 
     */	
	function newOrderby($variable, $order = 'ASC')
	{
		if (!@isset($this->_column[$variable])) {
		    $this->_argument_error('Variable name \''.(string)$variable.'\' is invalid"' , 'newOrderby');
			return null;
		}
	
		if ($order == 'ASC' || $order == 'DESC') {
		    $obj = new rdbObjectOrderBy;
			$obj->_column = $this->_column_quoted[$variable];
			$obj->_order = $order;
			$obj->_manager = $this->_myfactory;
			return $obj;
		} else {
			$this->_argument_error('Order operater \''.(string)$order.'\' is invalid' , 'newOrderby');
			return null;
		}
	}
	
    /**
     * SQL query
     * 
     * @param $sql 
     * @access private 
     * @return resource  
     * @abstract 
     */	
	function _query($sql) {} 

    /**
     * raiseEror() for query error
     * 
     * @param $sql 
     * @access private 
     * @return PEAR_Error 
     * @abstract 
     */	
    function _query_error($sql = '') {}

    /**
     * fetch_assoc with $result
     * 
     * @param resource $result 
     * @access private 
     * @return array 
     * @abstract 
     */	
	function _fetch($result) {}
	
    /**
     * quote value of mySQL variable for SQL statement
     * 
     * @param string $val 
     * @access private 
     * @return string 
     * @abstract 
     */
    function _quote($val) {}

    /**
     * quote column name
     * 
     * @param string $val 
     * @access private 
     * @return string 
     * @abstract 
     */
    function _quote_column($variable) {}
	
	/**
     * quote column name
     * 
     * @param string $val 
     * @access private 
     * @return string 
     * @abstract 
     */
    function _quote_table($table) {}

    /**
     * compose 'SELECT' SQL statement with condition
     * 
     * @access private 
     * @return string 
     * @abstract 
     */	
	function _compose_select_sql($select, $from, $condition) {}
		
} 

/**
 * abstract rdbObjectCondition
 * Objects for SQL where statement
 * 
 * @abstract 
 */
class rdbObjectCondition_abstract extends PEAR {
    /**
     * name of rdbObhectManagerFactory this belongs to
     * 
     * @access private 
     */	
	var $_manager;
}
	

/**
 * rdbObjectCondition
 * Object for SQL where statement
 * 
 * @access public 
 */
class rdbObjectCondition extends rdbObjectCondition_abstract {
   /**
     * comparisons
     * 
     * @var array array of rdbObjectComparison(s)
     * @access private 
     */
    var $_comparisons = array();
	
    /**
     * logical operator. that is 'AND'/'OR'
     * 
     * @var string 
     * @access private 
     */
    var $_logical_operator = ' AND ';
	
    /**
     * LIMIT number
     * 
     * @var int 
     * @access public 
     */
    var $limit;
	
    /**
     * START number
     * 
     * @var int 
     * @access public 
     */
    var $start;
	
    /**
     * varibale names for order
     * 
     * @var array array of variable name
     * @access private 
     */
    var $_orderby = array();

	/**
     * add condition of comparison
     * 
     * @param rdbObjectComparison $comparison 
     * @access public 
     * @return bool 
     */
    function addCondition($condition)
    {
        if (!is_a($condition, 'rdbObjectCondition') && !is_a($condition, 'rdbObjectComparison')) {
      		return $this->raiseError("Data type of specified condition is invalid. " , RDBOBJECT_ERROR_ARGUMENT);      
        } elseif ($condition->_manager != $this->_manager) {
			return $this->raiseError("Specified condition belogs to another rdbObjectMangerFactory. " , RDBOBJECT_ERROR_ARGUMENT);
		} else {
            $this->_comparisons[] = $condition;
			return true;
        } 
    } 

    /**
     * set LIMIT number of acquired rows
     * 
     * @param int $limit positive
     * @access public 
     * @return void 
     */
    function setLimit($limit = 1)
    {
        $this->limit = intval($limit);
    } 

    /**
     * set START number of acquired rows
     * 
     * @param int $limit positive
     * @access public 
     * @return void 
     */
    function setStart($start = 0)
    {
        $this->start = intval($start);
    } 

    /**
     * add Orderby condition
     * 
     * @param string $field 
     * @param string $order 'ASC' or 'DESC'
     * @access public 
     * @return bool 
     */
    function addOrderby($orderby)
    {
        if (!is_a($orderby, 'rdbObjectOrderBy')) {
      		return $this->raiseError("Data type of specified orderby is invalid. " , RDBOBJECT_ERROR_ARGUMENT);      
        } elseif ($orderby>_manager != $this->_manager) {
			return $this->raiseError("Specified orderby belogs to another rdbObjectMangerFactory. " , RDBOBJECT_ERROR_ARGUMENT);
		} else {
            $this->_orderby[] = $orderby;
			return true;
        }
    } 

    /**
     * render SQL statement
     * 
     * @param bool $where 
     * @access public 
     * @return string 
     */
    function render($where = true)
    {
        $ret = ((! $where) ? '' : ' WHERE ');
        if (! empty($this->_comparisons)) {
			$tmp = array();
            foreach ($this->_comparisons as $comparison) {
                $tmp[] = $comparison->render(false);
            } 
            $ret .= '( ' . implode(' '.$this->_logical_operator.' ', $tmp) . ' )';
        } 
        if (! empty($this->_orderby)) {
			$tmp = array();
			foreach ($this->_orderby as $orderby) {
                $tmp[] = $orderby->render(false);
            }
			$ret .= ' ORDER BY ' . implode(', ', $tmp);
		}
        return $ret;
    } 
} 

/**
 * rdbObjectComparison
 * 
 * @access public 
 */
class rdbObjectComparison extends rdbObjectCondition_abstract {
    /**
     * where clause
     * 
     * @access private 
     */
    var $_where;

    /**
     * render SQL statement
     * 
     * @param rdbObjectManager $mgr 
     * @param bool $where 
     * @access public 
     * @return string 
     */
    function render($where = true)
    {
        return ((! $where) ? '' : ' WHERE ') . $this->_where;
    } 
} 

/**
 * rdbObjectComparison
 * 
 * @access public 
 */
class rdbObjectOrderBy extends rdbObjectCondition_abstract {
    /**
     * where clause
     * 
     * @access private 
     */
    var $_column;
	
	/**
     * order operater 'ASC' 'DESC'
     * 
     * @access private 
     */
	var $_order;

    /**
     * render SQL statement
     * 
     * @param rdbObjectManager $mgr 
     * @param bool $where 
     * @access public 
     * @return string 
     */
    function render($where = true)
    {
        return ((! $where) ? '' : ' WHERE ') .$this->_column.' '.$this->_order;
    } 
} 

?>