////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// asm2obj.cc
// Copyright(C) 2009 OHSAWA Naotaka. All rights reserved.
//
// $Rev$
// $Date$
// $Author$

#include "asm2obj.h"

void obj :: setup_common_functions(void)
{
	// setup builtin function: cons
	m_base = 257;
	char *fun = "((S((S(KS))((S((S(KS))((S(KK))(KS))))((S((S(KS))((S((S(KS))((S(KK))(KS))))((S(KK))(KI)))))((S((S(KS))((S(KK))(KK))))((S(KK))I))))))((S((S(KS))((S(KK))(KK))))(KI)))";

	int idx = m_sym.index_of("cons");
	if(idx < 0) {
		std :: cerr << "error: symbol table full: cons." << std :: endl;
		abort();
	}
	m_sym[idx].term = parse(&fun, 0);

	// setup builtin church number
	m_m[0] = CONS(SYM_K, SYM_I);		// (KI) for cdr and 0 in LazyK
	m_m[1] = CONS(SYM_I, SYM_I);		// I for 1 in LazyK
	for(int i = 2; i < 257; i++) {
		m_m[i] = CONS(CONSA(m_base), CONSA(i-1));
	}

	// setup builtin function: inc
	fun = "(S((S(KS))((S(KK))I)))";
	idx = m_sym.index_of("1+");
	if(idx < 0) {
		std :: cerr << "error: symbol table full: 1+." << std :: endl;
		abort();
	}
	m_sym[idx].term = parse(&fun, 0);

	return ;
}

void obj :: compile_file(char* n)
{
	struct stat st = { 0 };
	if(lstat(n, &st) != 0) {
		std :: cerr << "error: lstat fails for file: " << n << std :: endl;
		abort();
	}

	FILE *fp = fopen(n, "r");
	if(fp) {
		char name[FNAME_MAX];
		char *buf = (char*)malloc(st.st_size);
		if(buf == 0) {
			std :: cerr << "error: out of memory." << std :: endl;
			abort();
		}
		if(fread(buf, sizeof(*buf), st.st_size, fp) != (size_t)st.st_size) {
			std :: cerr << "error: cannot read file: " << n << std :: endl;
			abort();
		}
		fclose (fp);

		char* fun = parse_fun(buf, name, buf + st.st_size);
		while(fun != 0) {
			int idx = m_sym.index_of(name);
			m_sym[idx].term = parse(&fun, buf + st.st_size);
			fun = parse_fun(fun, name, buf + st.st_size);
		}
		link();
		int idx = m_sym.find("main");
		if(idx >= 0) {
			m_entry = m_sym[idx].term;
		} else {
			std :: cerr << "error: cannot find function 'main'." << std :: endl;
			abort();
		}
	} else {
		std :: cerr << "error: cannot open file for read: " << n << std :: endl;
		abort();
	}
}



char* obj :: parse_fun(char* s, char *n, char* eob)
{
	// skip space
	for(;s != eob && isspace(*s); s++) ;
	if(s == eob) return 0;

	// parse function name
	char*	last = n + FNAME_MAX;
	for(; n != last && s != eob && !isspace(*s); *n++ = *s++ ) ;
	if(s == eob || n == last) {
		std :: cerr << "error: cannot find function name." << std :: endl;
		abort();
	}
	*n = '\0';

	return s;
}


uint32_t obj :: parse(char** s, char* eob)
{
	uint32_t	ptr;

	for(;*s != eob && isspace(**s); (*s)++) ;
	if(*s == eob) {
		std :: cerr << "error: no CAF found." << std :: endl;
		abort();
	}

	if(**s == '(' ) {
		if(m_base >= m_size) {
			m_size *= 2;
			m_m = (emd_t*)realloc(m_m, m_size);
			if(m_m == 0) {
				std :: cerr << "error: out of memory." << std :: endl;
				abort();
			}
		}
		ptr = CONSA(m_base);
		m_base++;
		(*s)++;

		emd_t d = CONS((emd_t)parse(s, eob), parse(s, eob));
		m_m[ptr>>2] = d;

		// skip space
		for(;*s != eob && isspace(**s); (*s)++) ;
		if(**s != ')' || *s == eob) {
			fprintf(stderr, "parse error: expected ')'.\n");
			exit(-1);
		}
	} else {
		if(**s == 'S') {
			ptr = SYM_S;
		} else if(**s == 'K') {
			ptr = SYM_K;
		} else if(**s == 'I') {
			ptr = SYM_I;
		} else if(**s == 'L') {
			ptr = SYM_L;
		} else if(**s == 'N') {
			ptr = SYM_N;
		} else if(**s == 'Z') {
			ptr = MINT(0);
		} else {
			ptr = parse_symbol(s, eob);
			(*s)--;
		}
	}

	(*s)++;

	return ptr;
}

uint32_t obj :: parse_symbol(char** s, char* eob)
{
	char	buf[257];
	int 	idx;
	bool	num = true;
	for(idx = 0; idx < 257 && *s != eob && !isspace(**s) && **s != '(' && **s != ')'; (*s)++, idx++) {
		buf[idx] = **s;
		num = num && isdigit(**s) ? true : false;
	}
	if(idx == 257 || idx == 0) {
		std :: cerr << "parse error: cannot find symbol." << std::endl;
		abort();
	}
	buf[idx] = '\0';

	if(num) {	// symbol is number
		int n = atoi(buf);
		if(n > 256) {
			std :: cerr << "parse error: integer must be less than 257." << std :: endl;
			abort();
		}
		return CONSA(n);
	} else {	// symbol is function
		int idx = m_sym.index_of(buf);
		if(idx < 0) {
			std :: cerr << "parse error: symbol table full: " << buf << "." << std :: endl;
			abort();
		}
		return SYMT(idx);
	}
}

void obj :: link(void)
{
	for(unsigned i = 0; i < m_base; i++) {
		emd_t	d = m_m[i];
		uint32_t car = CAR(d);
		if((car & 0x3) == 0x2) {	// symbol reference
			car = m_sym[car >> 2].term;
			if(car == 0) {
				std :: cerr << "error: symbol does not defined: " << m_sym[CAR(d) >> 2].name << std :: endl;
				abort();
			}
		}
		uint32_t cdr = CDR(d);
		if((cdr & 0x3) == 0x2) {	// symbol reference
			cdr = m_sym[cdr >> 2].term;
			if(cdr == 0) {
				std :: cerr << "error: symbol does not defined: " << m_sym[CDR(d) >> 2].name << std :: endl;
				abort();
			}
		}
		m_m[i] = CONS(car, cdr);
	}

	return ;
}

////////////////////////////////////////////////////////////////////////////////
// END OF IMPLEMENTATION
////////////////////////////////////////////////////////////////////////////////
