////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
// PB_SLAVE.v
// Copyright(C) 2009 OHSAWA Naotaka. All rights reserved.
// 
// $Rev$
// $Date$
// $Author$

`include "defs.vh"

////////////////////////////////////////////////////////////////////////////////
// PB_SLAVE
// BUS Slave module for initialize and run from HOST CPU.
module PB_SLAVE
(
        input                                  CLK,
        input                                  RSTn,

	// Slave Interface for init, start and debug
        input                                  SREQ,
        input                                  SWRITE,
        input  [`PB_SI_ABITS-1:0]              SA,
        input  [`PB_SI_DBITS-1:0]              SWD,
        output [`PB_SI_DBITS-1:0]              SRD,
	
	// Interface for init and start
        output                                 INIT,
        output                                 RUN,
        output                                 LAZYK,
        output                                 CONT,
        output                                 SET_CDR,
        output [`PB_DS_BITS-1:0]               SET_CDR_D,

        input                                  BUSY,
        input  [`PB_DS_BITS-1:0]               CAR,
        input  [`PB_DS_BITS-1:0]               CDR,

	// Evaluation Stack status & control
        output                                 ESREQ,
        output [`PB_ES_ABITS-1:0]              ESA,
        input  [`PB_DS_BITS-1:0]               ESRD,
        input  [`PB_ES_ABITS-1:0]              ENUM,
        input                                  EVF,

	// Free Pointer status & control
        output                                 FPSREQ,
        output [`PB_DS_BITS-1:0]               FPSWD,
        input  [`PB_DS_BITS-1:0]               FPA,
        input                                  FPVF
);

	///////////////////////////////////////////////////
	// Control for Reduction core
	assign	CONT	= SREQ && SWRITE && SA == 8'h00 ? SWD[3] : 1'b0;
	assign	LAZYK	= SREQ && SWRITE && SA == 8'h00 ? SWD[2] : 1'b0;
	assign	INIT	= SREQ && SWRITE && SA == 8'h00 ? SWD[1] : 1'b0;
	assign	RUN	= SREQ && SWRITE && SA == 8'h00 ? SWD[0] : 1'b0;
	assign	SET_CDR	= SREQ && SWRITE && SA == 8'h06 ? 1'b1   : 1'b0;
	assign	SET_CDR_D	= SWD[`PB_DS_BITS-1:0];

	///////////////////////////////////////////////////
	// Evaluation Stack Access
	assign	ESREQ	= SREQ && SWRITE && SA == 8'h04 ? 1'b1 : 1'b0;
	assign	ESA	= SWD[`PB_ES_ABITS-1:0];

	reg [`PB_ES_ABITS-1:0]	esa_r;
	always @(posedge CLK `RST_TYPE) begin
		if(!RSTn) begin
			esa_r <= {`PB_ES_ABITS{1'b0}};
		end else if(ESREQ) begin
			esa_r <= `TPD SWD[`PB_ES_ABITS-1:0];
		end
	end

	reg	esrdv;
	always @(posedge CLK `RST_TYPE) begin
		if(!RSTn) begin
			esrdv <= 1'b0;
		end else if(ESREQ) begin
			esrdv <= `TPD ESREQ;
		end
	end

	reg  [`PB_DS_BITS-1:0]	esrd_r;
	always @(posedge CLK `RST_TYPE) begin
		if(!RSTn) begin
			esrd_r <= {`PB_DS_BITS{1'b0}};
		end else if(esrdv) begin
			esrd_r <= `TPD ESRD;
		end
	end


	///////////////////////////////////////////////////
	// Free Pointer Access
	assign	FPSREQ	= SREQ && SWRITE && SA == 8'h02 ? 1'b1 : 1'b0;
	assign	FPSWD	= SWD[`PB_DS_BITS-1:0];


	///////////////////////////////////////////////////
	// Read Status
	reg [`PB_SI_DBITS-1:0]	rd;
	always @* begin
		case(SA)
		8'h00:	rd = {{`PB_SI_DBITS-4{1'b0}}, EVF, FPVF, 1'b0, BUSY};
		8'h01:  rd = `SLV_PAD_DS(CAR);
		8'h02:  rd = `SLV_PAD_DS(FPA);
		8'h03:  rd = {{`PB_SI_DBITS-`PB_ES_ABITS{1'b0}}, ENUM};
		8'h04:  rd = {{`PB_SI_DBITS-`PB_ES_ABITS{1'b0}}, esa_r};
		8'h05:  rd = `SLV_PAD_DS(esrd_r);
		8'h06:  rd = `SLV_PAD_DS(CDR);
		default:rd = {{`PB_SI_DBITS{1'b0}}};
		endcase
	end

	reg [`PB_SI_DBITS-1:0]	rd_r;
	always @(posedge CLK `RST_TYPE) begin
		if(!RSTn) begin
			rd_r <= {`PB_SI_DBITS{1'b0}};
		end else if(SREQ && !SWRITE) begin
			rd_r <= `TPD rd;
		end
	end

	assign	SRD	= rd;

endmodule

////////////////////////////////////////////////////////////////////////////////
// END OF IMPLEMENTATION
////////////////////////////////////////////////////////////////////////////////
