# session.rb
# $Id: session.rb,v 1.4 2004/03/25 17:19:49 komatsu Exp $
#
# Copyright (C) 2001 Satoru Takabayashi <satoru@namazu.org>
# Copyright (C) 2002, 2003, 2004 Hiroyuki Komatsu <komatsu@taiyaki.org>
#     All rights reserved.
#     This is free software with ABSOLUTELY NO WARRANTY.
#
# You can redistribute it and/or modify it under the terms of 
# the GNU General Public License version 2.
#

class Command
  attr_reader :name, :args, :nargs, :min_nargs, :description

  def initialize (name, args, description, min_nargs = nil)
    @name = name
    @args = args
    @nargs = args.length
    @min_nargs = (min_nargs or @nargs)
    @description = description
  end
end

class SessionCore
  def initialize (prime)
    @prime = prime
  end

  def get_line (io_in)
    return io_in.gets()
  end

  def execute (line)
  end
end

class Session < SessionCore
  def initialize (prime, version)
    super(prime)
    @version = version
    @debug   = false
    @command_table = Hash.new
    init_command_table()
  end

  def init_command_table
    add_command(:close,   [], "close the connection")
    add_command(:help,    [], "print the help message")
    add_command(:version, [], "show the version number")
  end

  def add_command (name, args, description, min_nargs = nil)
    command = Command.new(name, args, description, min_nargs)
    @command_table[name] = command
  end

  def reply_successful (result = nil)
    if result then
      result += "\n"
    else
      result = ""
    end
    return ("ok\n" + result + "\n")
  end

  def reply_unsuccessful (result = nil)
    if result then
      result += "\n"
    else
      result = ""
    end
    return ("error\n" + result + "Try `help' for protocol information.\n\n")
  end

  def help
    commands = @command_table.values.sort {|a, b| 
      (a.nargs <=> b.nargs).nonzero? || a.name.to_s <=> b.name.to_s
    }
    help = ""
    commands.each {|c|
      help += format("%-16s %s - %s\n",
                     c.name, c.args.join(" "), c.description)
    }
    help += "Note: Use TAB for delimiters."
    return reply_successful(help)
  end

  def execute (line)
    if line.nil? then
      return false
    end
    chunks = line.gsub(/[\r\n]/, "").split("\t")
    if chunks.length > 0 then
      name = chunks.shift.intern
      args = chunks
      return send_command(name, args)
    end
  end

  def send_command (name, args)
    command = @command_table[name]
    if command.nil?
      return reply_unsuccessful("Unknown command: #{name}")
    elsif command.min_nargs <= args.length and args.length <= command.nargs
      return send(command.name, *args)
    else
      error_msg = "Wrong number of arguments (expecting #{command.nargs})"
      return reply_unsuccessful(error_msg)
    end
  end

  def close
    return false
  end

  def version
    return reply_successful(@version)
  end
end

class SessionPrime < Session
  def initialize (prime, version)
    super

    add_command(:l, [:PATTERN],
		"look up PATTERN", 0)
    add_command(:lookup, [:PATTERN],
		"look up PATTERN", 0)
    add_command(:lookup_all, [:PATTERN],
		"look up PATTERN", 0)
    add_command(:lookup_compact, [:PATTERN],
		"look up PATTERN and return one or two candidate", 0)
    add_command(:lookup_compact_all, [:PATTERN],
		"look up PATTERN and return one or two candidate", 0)
    add_command(:lookup_direct, [:PATTERN],
		"look up PATTERN for direct typing method", 0)
    add_command(:lookup_direct_all, [:PATTERN],
		"look up PATTERN for direct typing method", 0)
    add_command(:lookup_hybrid, [:PATTERN],
		"look up PATTERN with hybrid matching", 0)
    add_command(:lookup_hybrid_all, [:PATTERN],
		"look up PATTERN with hybrid matching", 0)
    add_command(:lookup_prefix, [:PATTERN],
		"look up PATTERN with prefix matching", 0)
    add_command(:lookup_exact, [:PATTERN],
		"look up PATTERN with exact matching", 0)
    add_command(:lookup_expansion, [:PATTERN],
                "look up PATTERN from literal dictionaries", 0)
    add_command(:lookup_mixed, [:PATTERN],
                "look up PATTERN.  PATTERN can be mixed with pron and literal",
                0)
    add_command(:learn_word, [:KEY, :VALUE, :PART, :CONTEXT, :SUFFIX, :REST],
		"learn and record a word to the user dictionary", 2)
    add_command(:reset_context, [], "reset context")
    add_command(:set_context, [:CONTEXT], "set context to CONTEXT")
    add_command(:get_env, [:KEY], "get a variable associated with KEY")
    add_command(:get_label, [:PATTERN],
		"get a label string (e.g. hiragana) from a user input.")
    add_command(:preedit_convert_input, [:PATTERN],
                "convert a PATTERN and return a pair of a converted and a pending string.")
    add_command(:refresh, [],
                "refresh the statuses of the conversion engines.")
  end

  def learn_word (key, value, part = nil,
                  context = nil, suffix = nil, rest = nil)
    @prime.learn_word(key, value, part, context, suffix, rest)
    return reply_successful()
  end

  def get_env (key)
    env = @prime.get_env(key)
    if env.kind_of?(String) then
      output = format("string\t%s", env)
    elsif env.kind_of?(Array) then
      output = format("array\t%s", env.join("\t"))
    elsif env.kind_of?(TrueClass) or env.kind_of?(FalseClass) then
      output = format("boolean\t%s", env ? "true" : "false")
    elsif env == nil then
      output = 'nil'
    else
      output = 'unknown'
    end

    return reply_successful(output)
  end

  def get_label (pattern)
    return reply_successful(@prime.get_label(pattern))
  end

  def preedit_convert_input (pattern)
    return reply_successful(@prime.preedit_convert_input(pattern))
  end

  def set_context (context)
    @prime.set_context(context)
    return reply_successful()
  end
  def reset_context ()
    @prime.set_context(nil)
    return reply_successful()
  end
  
  def l (pattern = "")
    return reply_successful(@prime.lookup(pattern).to_text)
  end
  def lookup (pattern = "")
    return reply_successful(@prime.lookup(pattern).to_text)
  end
  def lookup_all (pattern = "")
    return reply_successful(@prime.lookup_all(pattern).to_text)
  end

  def lookup_compact (pattern = "")
    return reply_successful(@prime.lookup_compact(pattern).to_text)
  end
  def lookup_compact_all (pattern = "")
    return reply_successful(@prime.lookup_compact_all(pattern).to_text)
  end
  def lookup_direct (pattern = "")
    return reply_successful(@prime.lookup_direct(pattern).to_text)
  end
  def lookup_direct_all (pattern = "")
    return reply_successful(@prime.lookup_direct_all(pattern).to_text)
  end
  def lookup_hybrid (pattern = "")
    return reply_successful(@prime.lookup_hybrid(pattern).to_text)
  end
  def lookup_hybrid_all (pattern = "")
    return reply_successful(@prime.lookup_hybrid_all(pattern).to_text)
  end
  def lookup_prefix (pattern = "")
    return reply_successful(@prime.lookup_prefix(pattern).to_text)
  end
  def lookup_exact (pattern = "")
    return reply_successful(@prime.lookup_exact(pattern).to_text)
  end
  def lookup_expansion (pattern = "")
    return reply_successful(@prime.lookup_expansion(pattern).to_text)
  end
  def lookup_mixed (pattern = "")
    return reply_successful(@prime.lookup_mixed(pattern).to_text)
  end

  def refresh ()
    @prime.refresh()
    return reply_successful()
  end
end

# ----

class SessionSKK < SessionCore
  def initialize (prime, portnum)
    super(prime)
    @portnum = portnum
  end

  def get_line (io_in)
    line = ""
    loop {
      char = io_in.getc()
      if char.nil? then
        return
      end
      if char.chr == " " or char.chr == "\n" then
        return line
      else
        line += char.chr
      end
    }
  end

  def lookup (pattern)
    return @prime.lookup_japanese(pattern)
  end

  def execute (line)
    line.chomp!()

    case line[0,1]
    when "0" then
      return false
    when "1" then
      pattern = line.chomp[1..-1]
      results = lookup(pattern)
      if results.empty? then
        return "4" + pattern + "\n"
      else
        result_line = results.map {|result|
          result.to_text_literal()
        }.join('/')
        return "1/" + result_line + "/\n"
      end
    when "2" then
      # FIXME
      return "prime-#{PRIME_VERSION} "
    when "3" then
      # FIXME
      return "localhost:#{@protnum} "
    else
      return ""
    end
  end
end

class SessionPOBox < SessionSKK
  def lookup (pattern)
    return @prime.lookup(pattern)
  end

  def execute (line)
    ## The detail of POBox protocol is here; 
    ## <http://pitecan.com/OpenPOBox/server/protocol.html>

    line.chomp!()

    case line[0,1]
    when "0" then
      return false
    when "1" then
      pattern = line.chomp[1..-1]
      results = lookup(pattern)
      if results.empty? then
        return "4" + pattern + "\n"
      else
        result_line = results.map {|result|
          result.to_text_literal()
        }.join("\t")
        return "1\t" + result_line + "\n"
      end
    when "2" then
      # FIXME
      return "prime-#{PRIME_VERSION} "
    when "3" then
      # FIXME
      return "localhost:#{@protnum} "
    when "4" then
      context = line.chomp[1..-1]
      if context == "" then
        @prime.set_context(nil)
      else
        @prime.set_context(context)
      end
      return "1 "
    when "5" then
      # Not implemented yet.
      return "1 "
    when "6" then
      # Not implemented yet.
      return "1 "
    when "7" then
      # Not implemented yet.
      return "1 "
    when "8" then
      # Not implemented yet.
      return "1 "
    else
      super
    end
  end
end
