# taiyaki.rb: Ruby Libraries by Hiroyuki Komatsu
# $Id: taiyaki.rb,v 1.2.2.4 2004/01/12 23:36:23 komatsu Exp $
#
# Copyright (C) 2002 Hiroyuki Komatsu <komatsu@taiyaki.org>
#     All rights reserved.
#     This is free software with ABSOLUTELY NO WARRANTY.
#
# You can redistribute it and/or modify it under the terms of 
# the GNU General Public License version 2.

def require (name, package = nil, url = nil)
  package_name =
         {'progressbar'                => 'progressbar',
          'sary'                       => 'sary-ruby',
          'prime/prime-dict-config.rb' => 'prime-dict',
          'suikyo/suikyo'              => 'suikyo',
  }
  package_url =
         {'progressbar'   => 'http://www.namazu.org/~satoru/ruby-progressbar/',
          'sary'          => 'http://sary.namazu.org/',
          'prime/prime-dict-config.rb' =>
                             'http://sourceforge.jp/projects/prime/',
          'suikyo/suikyo' => 'http://taiyaki.org/suikyo/',
  }
  begin
    Kernel::require(name)
  rescue LoadError
    package = (package or package_name[name])
    url     = (url     or package_url[name])

    $stderr.puts("ERROR:")
    if ENV['LANG'] == 'ja_JP.eucJP' then
      $stderr.puts("  Ruby 饤֥ '#{name}' ɤǤޤǤ")
      if package then
        $stderr.puts("  餯 #{package} ѥå" +
                     "󥹥ȡ뤹ɬפޤ")
      end
      if url then
        $stderr.puts("    <#{url}>")
      end
      $stderr.puts("  򤪤Ƥߤޤ")
    else
      $stderr.puts("  The required Ruby library '#{name}' is not found.")
      if package then
        $stderr.puts("  You might need to install a #{package} package.")
      end
      if url then
        $stderr.puts("    <#{url}>")
      end
      $stderr.puts("  Sorry for your inconvenience.")
    end
    exit()
  end
end

def with_io(io, option = "w")
  ### with_io('/tmp/logfile') {|io|
  ###   lines.each {|line|
  ###     io.print(line)
  ###   }
  ### }
  ###
  ### with_io($stdout) {|io|
  ###   lines.each {|line|
  ###     io.print(line)
  ###   }
  ### }
  if io.is_a?(String) then
    io = open(io, option)
    yield io
    io.close()
  else
    yield io
  end
end

module Kernel
  def non_nil? ()
    return (not self.nil?)
  end

  def is_in? (*arg)
    return arg.member?(self)
  end
end


class String
  # '-'
  HIRAGANA_SJIS = "\202\237-\202\361"
  HIRAGANA_EUC  = "\244\241-\244\363"
  # '-'
  KATAKANA_SJIS = "\203@-\203\223"
  KATAKANA_EUC  = "\245\241-\245\363"

  def hiragana!
    case $KCODE[0]
    when ?s, ?S
      return self.tr!(KATAKANA_SJIS, HIRAGANA_SJIS)
    when ?e, ?E
      return self.tr!(KATAKANA_EUC,  HIRAGANA_EUC)
    else
      return nil
    end
  end

  def hiragana
    return ((string = self.dup).hiragana! or string)
  end

  def katakana!
    case $KCODE[0]
    when ?s, ?S
      return self.tr!(HIRAGANA_SJIS, KATAKANA_SJIS)
    when ?e, ?E
      return self.tr!(HIRAGANA_EUC,  KATAKANA_EUC)
    else
      return self
    end
  end

  def katakana
    return ((string = self.dup).katakana! or string)
  end

  def increase
    increasing_string = ""
    increasing_list = []
    self.split(//).each {|char|
      increasing_string += char
      increasing_list.push(increasing_string)
    }
    if block_given? then
      increasing_list.each {|string|
	yield string
      }
    else
      increasing_list
    end
  end
end


class File
  def File::join2 (*paths)
    dirs = paths[0..-2].map{|path|
      path ? path.split(File::Separator) : ""
    }
    join(dirs, paths[-1])
  end
  # Check join2(nil, "a")


  def File::delext (path)
    paths = path.split(File::Separator)
    #(dir, file) = [paths[0..-2], paths[-1]]
    ### It's enigma that sub(/(.)\.[^.]*/, $1) doesn't work.
    paths[-1] = paths[-1].sub(/(.)\.[^.]*$/, '') + ($1 or '')
    return paths.join(File::Separator)
  end
  # Check delext("text.txt")

  def File::ensure (file, filemode = nil)
    if File::file?(file) == false then
      File::open(file, "w").close()
    end
    if filemode then
      File::chmod(filemode, file)
    end
  end
end

class Dir
  def Dir::ensure (path)
    if  File::directory?(path) then
      return true
    else
      if File::exist?(path) then
	return false
      else
	upper_dir = File::join(path.split(File::Separator)[0..-2])
	if Dir::ensure(upper_dir) && File::writable_real?(upper_dir) then
	  Dir::mkdir(path)
	  return true
	end
      end
    end
    return false
  end
end


module Debug
  attr_accessor :debug_mode
  def debug_message (message, output = nil)
    if $DEBUG || @debug_mode || output then
      ($DEBUG_IO or $stdout).puts "#{self.class}[#{self.id}]: #{message}"
    end
  end

  def Debug::message (message, output = nil)
    if $DEBUG || output then
      ($DEBUG_IO or $stdout).puts message
    end
  end
end

module Marshal
  def Marshal::init_file (filename, threshold_time = nil)
    if File::exist?(filename) && 
	(threshold_time.nil? || File::mtime(filename) > threshold_time) then
      object = Marshal::load_file(filename)
      Debug::message("Marshal::init_file: Load the ruby object `#{filename}'.")
    else
      object = yield
      Marshal::dump_file(object, filename)
      Debug::message("Marshal::init_file: Initialize renewally `#{filename}'.")
    end
    return object
  end

  def Marshal::load_file (filename)
    object_file = File::open(filename, "r")
    object = Marshal::load(object_file)
    object_file.close
    return object
  end

  def Marshal::dump_file (object, filename)
    object_file = File::open(filename, "w")
    Marshal::dump(object, object_file)
    object_file.close
  end
end


#### ------------------------------------------------------------
#### For ProgressBar.rb
#### ------------------------------------------------------------

module Enumerable
  def each_with_status ()
    length = self.length()
    self.each_with_index {|value, index|
      case index
      when 0 then
	status = :first
      when (length - 1) then
	status = :last
      else
	status = nil
      end
      yield(value, status)
    }
  end

  def each_with_pbar (title = '')
    #| each_with_pbar (title/String)
    #| => self/Enumerable
    #|
    #| Is a each function with a progress bar.
    #| If title is nil, this function is the same function as 'each'. 
    #|
    if title.nil? then
      self.each {|value|
	yield value
      }
    elsif self.kind_of?(IO) then
      size = self.stat.size
      pbar = ProgressBar.new(title, size)
      self.each {|value|
	yield value
	pbar.inc(value.length)
      }
      pbar.finish
    else
      pbar = ProgressBar.new(title, self.size)
      self.each {|value|
	yield value
	pbar.inc
      }
      pbar.finish
    end
    return self
  end

  def sort_with_pbar (title = '')
    #| sort_with_pbar (title/String)
    #| => sorted_value/Enumerable
    #|
    #| Is a sort function with a progress bar.
    #| If title is nil, this function is the same function as 'sort'. 
    #|
    if title.nil? then
      if block_given? then
	sorted = self.sort {|value1, value2|
	  yield value1, value2
	}
      else
	sorted = self.sort
      end
      return sorted
    end

    estimated_length = (1.4 * self.length * Math.log(self.length)).to_i
    pbar = ProgressBar.new(title, estimated_length)
    if block_given? then
      sorted = self.sort {|value1, value2|
	pbar.inc
	yield value1, value2
      }
    else
      sorted = self.sort {|value1, value2|
	pbar.inc
	value1 <=> value2
      }
    end
    pbar.finish
    return sorted
  end
end

class Integer
  def times_with_pbar (title = '')
    #| times_with_pbar (title/String)
    #| => self/Integer
    #|
    #| Is a times function with a progress bar.
    #| If title is nil, this function is the same function as 'times'. 
    #|
    if title.nil? then
      self.times {|value|
	yield value
      }
      return self
    end

    pbar = ProgressBar.new(title, self)
    self.times {|value|
      yield value
      pbar.inc
    }
    pbar.finish
  end
end
