# prime/engine/engine-basic.rb
# $Id: engine-basic.rb,v 1.2 2003/12/31 09:21:24 komatsu Exp $
#
# Copyright (C) 2002 Hiroyuki Komatsu <komatsu@taiyaki.org>
#     All rights reserved.
#     This is free software with ABSOLUTELY NO WARRANTY.
#
# You can redistribute it and/or modify it under the terms of 
# the GNU General Public License version 2.
#

require 'sary'
require 'prime/prime-dict-config.rb'
require 'prime/engine/engine-japanese'

$engine_class_name = 'PrimeEngineBasic'

class PrimeEngineBasic < PrimeEngineJapanese
  def initialize
    super

    @name = "Basic engine"
    @id   = "basic"
    @description = "Basic engine"

    initialize_posdict()
    @dict_file = File::join2(PRIME_DICT_DIR, 'prime-dict')
    @dict = Sary::Searcher.new(@dict_file)

    @max_candidates = 15
  end

  def initialize_posdict ()
    file_posdict  = File::join2(PRIME_DICT_DIR, 'prime-dict-pos')
    file_partdict = File::join2(PRIME_DICT_DIR, 'prime-dict-part')

    if File::exist?(file_posdict) then
      if File::exist?(file_partdict) then
        $stderr.puts("PRIME Warning:")
        $stderr.puts("    Both prime-dict-pos and prime-dict-part exist")
        $stderr.puts("    under '#{PRIME_DICT_DIR}'.")
        $stderr.puts("    The file name 'prime-dict-part' is obsolete.")
      end
      @partdict_file = file_posdict
    elsif File::exist?(file_partdict) then
      @partdict_file = file_partdict
    else
      $stderr.puts("PRIME ERROR:")
      $stderr.puts("    The POS dictionary #{file_posdict} is not found.")
      $stderr.puts("    Please install the latest prime-dict package.")
      Kernel::exit()
    end
    @partdict = Sary::Searcher.new(@partdict_file)
  end

  private
  def lookup (input, method = :prefix)
    results = PrimeResult.new
    queries = make_queries(input.base)
    if method == :prefix then
      input.expands.each {|string|
	queries.add(string) 
      }
      rests = queries.rests
      max   = @max_candidates
    else
      queries.add(input.original, nil, "", "", false)
      rests = [""]
      max   = 100
    end

    rests.each {|rest|
      cur_results = lookup_dict(queries.query_lines(rest), max)
      cur_results.each {|result|
	(pattern, part, word, freq) = result.split(/\t/)
	suffix = get_suffix(input.base, pattern, rest)
	priority2 = freq.to_i + 10000 - rest.length * 1000
	cand = PrimeCandidate.new(pattern, word, priority2,
				  part, suffix, rest)
	results << cand
      }
      max -= cur_results.length
      (max > 0) or break
    }
    return results
  end

  def lookup_dict (query_lines, max = nil)
    results = []
    return results if max == (nil or 0)

    if !(query_lines.empty?) and @dict.multi_search(query_lines) then
      @dict.sort_occurrences
      while line = @dict.get_next_context_line do
	results << line.chomp
	if max and (results.length == max) then
	  break
	end
      end
    end
    return results
  end

  def lookup_part (base)
    if @partdict.search(base + "\t") then
      (yomi, *parts) = @partdict.get_next_context_line.chomp.split(/\t/)
      return parts
    else
      return []
    end
  end
end

