(require 'prime-process)
(require 'mell)

(defcustom prime-engine-prime-command "prime"
  "prime $B$N%3%^%s%I%Q%9(B")
(defcustom prime-engine-prime-args-list
  '() "prime $B$KM?$($k0z?t(B")

(defconst prime-server-buffer " *prime*")
(defconst prime-server-process "prime-server")

(defvar prime-engine-prime-cands        nil)
(defvar prime-engine-prime-cands-alist  nil)
(defvar prime-engine-command-lookup     nil)
(defvar prime-engine-command-lookup-all nil)
(defvar prime-last-results '())

(defvar prime-session-default    nil)
(defvar prime-session-minibuffer nil)

;;;; Mell ------------------------------------------------------------

(defcustom mell-working-buffer-name " *mell-buffer*"
  "Working buffer for mell")
(defvar    mell-working-buffer nil)

(defun mell-string-append-to-file (string filename)
  (save-excursion
    (or (bufferp mell-working-buffer)
	(setq mell-working-buffer
	      (get-buffer-create mell-working-buffer-name)))
    (set-buffer mell-working-buffer)
    (erase-buffer)
    (insert string)
    (append-to-file (point-min) (point-max) (expand-file-name filename))
    ))

;;;; -----------------------------------------------------------------

;;;; ------------------------------------------------------------
;;;; Server $BMQ4X?t(B
;;;; ------------------------------------------------------------

(defun prime-server-init (&optional forcep)
  (if (or forcep
	  (not (member (prime-process-status prime-server-process)
		       '(run error))))
      (if (prime-process-command-start prime-server-process
				       prime-server-buffer 
				       prime-engine-prime-command
				       prime-engine-prime-args-list
				       'prime-server-exit)
	  (progn
	    (prime-server-check-version)
	    (prime-server-send-dummy)
	    (prime-server-init-session)
	    (prime-server-set-lookup-command))
	(prime-process-error prime-server-process 
			     "PRIME $B$N=i4|2=$K<:GT$7$^$7$?(B"))
    ))

(defun prime-server-check-version ()
  "This checks the version of the PRIME server.
If this client, prime-el, does not support the version, 
this aborts the execution."
  (let ((version (prime-server-get-version)))
    (if (or (string< version "0.8.5")
	    (and (string< "0.9.0" version) (string< version "0.9.3")))
	(error "This client does not support the version of your PRIME server.
Please install the latest server.  The your using version is %s" version))))


(defun prime-server-init-session ()
  (setq prime-session-default    (prime-server-session-start))
  (setq prime-session-minibuffer (prime-server-session-start)))


(defun prime-server-select (word)
  (if prime-learn-p
      (prime-server-dict-register-word word
				       (prime-input-get-inserting-label)
				       prime-context)
    ))

(defun prime-server-dict-register-word (word pattern &optional context)
  (let* ((word-data (or (assoc word prime-engine-prime-cands-alist)
			(list  word pattern)))
	 (word-alist (nth 2 word-data))
	 (key     (or (cdr (assoc "basekey" word-alist))  (nth 1 word-data)))
	 (value   (or (cdr (assoc "base"    word-alist))  (nth 0 word-data)))
	 (part    (or (cdr (assoc "part"    word-alist))  ""))
	 (context (or context ""))
	 (suffix  (or (cdr (assoc "conjugation" word-alist)) ""))
	 (rest    (or (cdr (assoc "suffix"      word-alist)) ""))
	 )
    (prime-server-send-command  (format "learn_word\t%s\t%s\t%s\t%s\n"
				       key value part context suffix rest))
    ))


(defun prime-server-dict-delete-word (word pattern &optional context)
  ;; Not implemented yet.
  )

(defun prime-server-get-word-data (literal key)
  "This returns the matched data with literal and key from \
prime-engine-prime-cands-alist"
  (cdr (assoc key (nth 2 (assoc literal prime-engine-prime-cands-alist)))))

(defun prime-server-set-cands (lookup-result)
  (setq prime-engine-prime-cands-alist
	(mapcar '(lambda (word-data)
		   (list (nth 0 word-data)
			 (nth 1 word-data)
			 (prime-server-parse-word (nthcdr 2 word-data))))
		lookup-result))
  (setq prime-engine-prime-cands
	(mapcar '(lambda (x) (car x))
		prime-engine-prime-cands-alist))
  prime-engine-prime-cands)

(defun prime-server-predict (pattern &optional context)
  (prime-server-set-cands
   (prime-server-lookup pattern context prime-conv-exact-p)))

(defun prime-server-convert (pattern &optional context)
  (prime-server-set-cands
   (prime-server-lookup-all pattern context prime-conv-exact-p)))

(defun prime-server-get-env (key)
  (let ((result (prime-server-send-command (format "get_env\t%s\n" key)))
	value)
    (setq value (car (cdr (delete "" (split-string result "\n")))))
    (if (string-match "^\\([^\t]*\\)\t" value)
	(let ((type (match-string 1 value))
	      (data (substring value (match-end 0))))
	  (cond
	   ((string= type "string")
	    data)
	   ((string= type "array")
	    (if (string= data "") nil
	      (mell-string-split data "\t")))
	   ((string= type "boolean")
	    (string= data "true"))
	   (t
	    (list 'unknown data)))))))


(defun prime-server-reconnect ()
  (interactive)
  (prime-server-exit)
  (prime-server-init t))

(defun prime-server-send-dummy ()
  (prime-server-send-command "help\n")
  )

(defun prime-server-send-command (command &optional function)
  (save-excursion
    (or prime-debug-mode
	(prime-server-init))
    (set-buffer prime-server-buffer)
    (erase-buffer)
    (process-send-string prime-server-process command)
    (catch 'process-loop
      (while (process-status prime-server-process)
 	 (accept-process-output (get-process prime-server-process) 1 0)
	 (and (> (buffer-size) 0)
	      (progn (goto-char (1- (point-max))) (looking-at "^$"))
	      (if function (funcall function) t)
	      (throw 'process-loop nil))
	 ))
    (let ((result (buffer-string)))
      (setq prime-last-results (cons (cons command result) prime-last-results))
      (if (>= (length prime-last-results) 10)
	  (setq prime-last-results (mell-sublist prime-last-results 0 10)))
      result)))

(defun prime-server-exit (&optional forcep) ; $B%W%m%;%9$+$i@ZCG(B 
  (condition-case nil
      (progn
	(if (eq (process-status prime-server-process) 'run)
	    (progn
	      (prime-server-session-end prime-session-default)
	      (prime-server-session-end prime-session-minibuffer)
	      (process-send-string prime-server-process "close\n"))))
    (error nil)
    ))

(defun prime-server-parse-word (word-data)
  (mapcar
   '(lambda (annotaion)
      (let ((pair (mell-string-split annotaion "=")))
	(cons (car pair) (nth 1 pair))))
   word-data))

(defun prime-server-parse-cands (cands-string)
  (mapcar
   '(lambda (str-line)
      (let ((tmp-list (mell-string-split str-line "\t")))
	(cons (nth 1 tmp-list) 
	      (cons (car tmp-list)
		    (nthcdr 2 tmp-list)))
	))
   (cdr (delete "" (split-string cands-string "\n")))
   ))

(defun prime-server-get-version ()
  (let* ((result (prime-server-send-command "version\n"))
	 (data (mell-string-split
		(car (cdr (delete "" (split-string result "\n")))) "\t"))
	 (version (if (> (length data) 1) (nth 1 data) (nth 0 data))))
    version))

(defun prime-server-set-lookup-command ()
  ;; FIXME: This version checking routine is not robust.
  ;; FIXME: <komatsu@taiyaki.org> (2004-02-29)
  (cond ((string< (prime-server-get-version) "0.7.9")
	 (setq prime-style-display-candidates 'all
	       prime-engine-command-lookup "lookup"
	       prime-engine-command-lookup-all "lookup"))

	((eq prime-style-display-candidates 'compact)
	 (setq prime-engine-command-lookup "lookup_compact"
	       prime-engine-command-lookup-all "lookup_compact_all"))
	((eq prime-style-display-candidates 'all)
	 (setq prime-engine-command-lookup "lookup"
	       prime-engine-command-lookup-all "lookup_all"))
	(t
	 (setq prime-engine-command-lookup "lookup_compact"
	       prime-engine-command-lookup-all "lookup_compact_all"))
	))

(defun prime-server-set-context (context)
  (prime-server-send-command
   (if context
       (format "set_context\t%s\n" context)
     "reset_context\n"
     )))

(defun prime-server-lookup (pattern &optional context exactp)
  (if (string= pattern "")
      nil
    (prime-server-set-context context)
    (prime-server-parse-cands
     (prime-server-send-command
      ;; FIXME: Consider a lower compatibility of 'lookup'. Check the version.
      ;; FIXME: <komatsu@taiyaki.org> (2004-01-28)
      ;; FIXME: Replase the command 'l' to 'lookup' in the future.
      ;; FIXME: <komatsu@taiyaki.org> (2004-01-28)
      (format (if exactp "lookup_exact\t%s\n" 
		(concat prime-engine-command-lookup "\t%s\n"))
	      pattern)))))

(defun prime-server-lookup-all (pattern &optional context exactp)
  (if (string= pattern "")
      nil
    (prime-server-set-context context)
    (prime-server-parse-cands
     (prime-server-send-command
      ;; FIXME: Consider a lower compatibility of 'lookup'. Check the version.
      ;; FIXME: <komatsu@taiyaki.org> (2004-01-28)
      ;; FIXME: Replase the command 'l' to 'lookup' in the future.
      ;; FIXME: <komatsu@taiyaki.org> (2004-01-28)
      (format (if exactp "lookup_exact\t%s\n" 
		(concat prime-engine-command-lookup-all "\t%s\n"))
	      pattern)))
    ))


;;;
;;; Codes for a preediting string supposed by PRIME 0.9.3 or later.
;;;
(defun prime-server-send-command2 (command)
  "This sends the command and returns a list of items splited by tabs"
  (let ((result (prime-server-send-command command)))
    (mell-string-split (or (car (cdr (delete "" (split-string result "\n"))))
			   "")
		       "\t")
    ))

(defun prime-server-session-start ()
  "This starts a session with PRIME and returns the session id."
  (car (prime-server-send-command2 "session_start\n")))

(defun prime-server-session-end (session-id)
  "This ends the session with PRIME specified with the session-id"
  (prime-server-send-command2 (format "session_close\t%s\n" session-id)))

(defun prime-server-edit-insert (session-id string)
  (prime-server-send-command2
   (format "edit_insert\t%s\t%s\n" session-id string)))

(defun prime-server-edit-delete (session-id)
  (prime-server-send-command2 (format "edit_delete\t%s\n" session-id)))

(defun prime-server-edit-backspace (session-id)
  (prime-server-send-command2 (format "edit_backspace\t%s\n" session-id)))

(defun prime-server-edit-erase (session-id)
  (prime-server-send-command2 (format "edit_erase\t%s\n" session-id)))

(defun prime-server-edit-undo (session-id)
  (prime-server-send-command2 (format "edit_undo\t%s\n" session-id)))

(defun prime-server-edit-cursor-left (session-id)
  (prime-server-send-command2 (format "edit_cursor_left\t%s\n" session-id)))

(defun prime-server-edit-cursor-right (session-id)
  (prime-server-send-command2 (format "edit_cursor_right\t%s\n" session-id)))

(defun prime-server-edit-cursor-left-edge (session-id)
  (prime-server-send-command2
   (format "edit_cursor_left_edge\t%s\n" session-id)))

(defun prime-server-edit-cursor-right-edge (session-id)
  (prime-server-send-command2
   (format "edit_cursor_right_edge\t%s\n" session-id)))

(defun prime-server-edit-get-preedition (session-id)
  (prime-server-send-command2 (format "edit_get_preedition\t%s\n" session-id)))

(defun prime-server-edit-get-query-string (session-id)
  (car (prime-server-send-command2
	(format "edit_get_query_string\t%s\n" session-id))))

(defun prime-server-edit-set-mode (session-id mode)
  (car (prime-server-send-command2
	(format "edit_set_mode\t%s\t%s\n" session-id mode))))

(provide 'prime-server)
