#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# - Messages -
# Copyright (C) 2013-2015 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Simple query by y (yes) or n (no) where yes by default =============
message_query_yn_default_yes ()
{
	local answer
	read answer || :
	case `echo "$answer" | head -c 1` in
	y|Y)	message_echo
		return 0
		;;
	n|N)	message_echo
		return 1
		;;
	'')	return 0
		;;
	*)	message_echo
		message_echo 'Pardon? ([y]/n)'
		message_query_yn_default_yes
		;;
	esac
}

# ============= Simple query by y (yes) or n (no) where no by default =============
message_query_yn_default_no ()
{
	local answer
	read answer || :
	case `echo "$answer" | head -c 1` in
	y|Y)	message_echo
		return 0
		;;
	n|N)	message_echo
		return 1
		;;
	'')	return 1
		;;
	*)	message_echo
		message_echo 'Pardon? (y/[n])'
		message_query_yn_default_no
		;;
	esac
}

# ============= Output the version =============
message_version ()
{
	[ "x$opt_batch_mode" = xno ] && echo -n "${APPNAME} version "
	echo "${MYVERSION}"
}

# ============= Output a message except for the batch mode =============
message_echo ()
{
	[ "x$opt_batch_mode" = xyes ] && return
	echo "$@" | fold -s -w `misc_get_console_column_size`
}

# ============= Output a message from stdin except for the batch mode =============
# Use via pipe or redirection without argument is discouraged because it violates the screen size detection.
message_cat ()
{
	local ncolumns
	[ "x$opt_batch_mode" = xyes ] && return
	ncolumns=`misc_get_console_column_size`
	fold -s -w $ncolumns "$@"
}

# ============= Output a credit =============
message_credit ()
{
	local ncolumns
	[ "x$opt_batch_mode" = xyes ] && return
	ncolumns=`misc_get_console_column_size`
	message_version | fold -s -w $ncolumns
	fold -s -w $ncolumns << eof
 -- Ports upgrading utility for massive forced reinstallation
 -- And for those who are pursuing the perfect packages environment
Copyright (C) 2010 - 2015 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
Email: <sakaue.mamoru@samurai.mwghennn.net>
Homepage: <http://www.mwghennndo.com/software/portsreinstall/>
eof
}

# ============= Output a time stamp =============
message_timestamp ()
{
	env LANG= date
}

# ============= Output an opening notice =============
message_opening_notice ()
{
	local ncolumns
	[ "x$opt_batch_mode" = xyes ] && return
	ncolumns=`misc_get_console_column_size`
	fold -s -w $ncolumns << eof

 Don't hesitate to terminate by CTRL+C anytime you feel the system is heavy to use because you can restart the operation from the terminated point quickly.

 The current time is `message_timestamp`
eof
}

# ============= Report a failure in a process for a port =============
message_report_failure_for_a_port ()
{
	local origin
	origin=$1
	message_echo "*** Giving up for this port $origin and proceeding to next forcibly..."
	message_echo
}

# ============= Output the relation of the current port with specified targets =============
message_target_relations ()
{
	[ "x$opt_batch_mode" = xyes ] && return
	( set -e
		origin=$1
		database_query_get_target_attributes it "$origin"
		[ -z "${it_is_all}" -a -n "${it_is_relevant}" ] || return 0
		[ -n "${it_is_target}" ] \
			&& message_echo "(Target port)"
		[ -n "${it_is_requires_dependents}" ] \
			&& message_echo "(Dependent of the target port(s))"
		[ -n "${it_is_requires_requirements}" ] \
			&& message_echo "(Requirement of the target port(s))"
		[ -n "${it_is_initial_dependents}" -a -z "${it_is_requires_dependents}" ] \
			&& message_echo "(Former dependent of the target port(s))"
		[ -n "${it_is_initial_requirements}" -a -z "${it_is_requires_requirements}" ] \
			&& message_echo "(Former requirement of the target port(s))"
		[ -n "${it_is_requires_requirements_complement}" ] \
			&& message_echo "(Requirement of the target port(s) for installing their requirements)"
		:
	)
}

# ============= Notice that the current run is a restarted one =============
message_restarted_process ()
{
	message_echo "INFO: Restarting from the previously terminated point."
}

# ============= Output of a section title =============
message_section_title ()
{
	message_echo "[`message_timestamp`] $*"
}

# ============= Output of a stage title =============
message_stage_title ()
{
	message_echo "========== $* =========="
	message_echo "[At `message_timestamp`]"
}

# ============= Dummy output for dry run =============
message_dry_run ()
{
	message_echo "@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@"
	message_echo "@@@@@@@@@@@@@@@@@ DRY RUN @@@@@@@@@@@@@@@@@"
	message_echo "@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@"
	message_echo
}

# ============= Output a summary of a resource recorded in a list =============
message_summary_dependents_of_failed_reinstallation ()
{
	local subject comment tmpmsg ncolumns
	subject=$1
	comment=$2
	tmpmsg=${TMPDIR}/message_summary_dependents_of_failed_reinstallation::message
	database_query_show_single_list_exec "$subject" \
		`options_get_dependency_type` `options_get_dependency_level` > $tmpmsg || return 0
	[ $opt_batch_mode = yes ] && return 1
	ncolumns=`misc_get_console_column_size`
	message_echo '****************'
	message_cat "$tmpmsg"
	if [ -n "$comment" -a -e "$comment" ]
	then
		message_echo '****************'
		message_cat "$comment"
	fi
	message_echo '****************'
	message_echo
	return 1
}

# ============= Show a message of a dependency scope =============
message_dependency_scope ()
{
	( set -e
		options_parse_dependency_type "$deptag"
		options_parse_dependency_level "$level"
		message_echo "(Evaluated by `options_get_dependency_msgterm` dependency)"
	)
}

# ============= Output an advice on manual solution to (re)installation failures =============
message_summary_advice_on_manual_solution ()
{
	local tmpcomment command_pkg_updating
	tmpcomment=${TMPDIR}/message_summary_failed_reinstallation::comment
	command_pkg_updating=`pkgsys_show_pkg_updating_commands`
	cat > $tmpcomment << eof
----------------
[Advices on manual solutions to (re)installation failures]
 Simple redo processes may resolve the above problems, but otherwise manual solution is needed.
 In the latter case, you are recommended to read ${PORTSDIR}/UPDATING to resolve the problems.
eof
	[ -n "$command_pkg_updating" ] && cat >> $tmpcomment << eof
 $command_pkg_updating will be useful for this purpose.

eof
	cat >> $tmpcomment << eof
 For unnecessary failed ports (with their unnecessary dependents), you can register it by
        ${APPNAME} noneed add [deleted_port_globs]
and execute
        ${APPNAME}

 For necessary and manually successful ports, after resolving the problems, execute
        ${APPNAME} ok add [resolved_port_globs]
and execute
        ${APPNAME} redo

 Refer to "Workaround for failed ports" sections in the manual page for details.
----------------

eof
	message_cat "$tmpcomment"
}

# ============= Output a warning if no progress was achieved =============
message_warn_no_achieved_progress ()
{
	[ `cat "${DBDIR}/new_success_in_current_run" 2> /dev/null | wc -l` -gt 0 ] && return
	[ $opt_batch_mode = yes ] && return 1
	message_echo '****************'
	message_echo "WARNING: No progress was achieved by this run."
	message_echo "         Manual solution is needed on the failed ports."
	message_echo '****************'
	message_echo
	return 1
}
