#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# - Operations for building the temporary database -
# Copyright (C) 2013-2015 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Variables =============
DATABASE_VERSION=
DATABASE_IS_OBSOLETE=no

# ============= Creation =============
database_build_create ()
{
	[ `id -u` -eq 0 -a ! -d "${DBDIR}" ] && mkdir -p "${DBDIR}"
	misc_lock_duplicated_executions "${DBDIR}/.lock"
	if [ -e "${DBDIR}/MYVERSION" ]
	then
		if ! grep -q -E "$COMPATIBLE_VERSIONS" "${DBDIR}/MYVERSION" 2> /dev/null
		then
			message_echo "ERROR: The current temporary database is incompatible. You must delete it by" >&2
			message_echo "        ${APPNAME} clean force" >&2
			message_echo "       in order to enable the current version." >&2
			exit 1
		fi
	elif misc_is_superuser_privilege
	then
		echo "$MYVERSION" > ${DBDIR}/MYVERSION
	fi
	DATABASE_VERSION=`cat "${DBDIR}"/MYVERSION 2> /dev/null || :`
	misc_is_superuser_privilege || return 0
	for subdir in initial requires replace targets obsolete backup_packages \
		stage.loop_list stage.complete stage.reinit_loop stage.depends
	do
		[ -d "${DBDIR}/$subdir" ] || mkdir -p "${DBDIR}/$subdir"
	done
}

# ============= Refresh the temporary database =============
database_build_refresh ()
{
	misc_is_superuser_privilege || return
	[ $opt_suppress_obsolete_db_clean = no ] || return
	message_echo "INFO: The temporary database is cleaned up."
	message_echo
	[ -d "${DBDIR}" -a ! -d "${DBDIR}.tmp" ] && mv "${DBDIR}" "${DBDIR}.tmp"
	database_build_create
	mv "${DBDIR}.tmp/saved_options.sh" "${DBDIR}" 2> /dev/null || :
	mv "${DBDIR}.tmp/backup_packages" "${DBDIR}" 2> /dev/null || :
	mv "${DBDIR}.tmp/backup_pkgarcs.lst" "${DBDIR}" 2> /dev/null || :
	rm -rf "${DBDIR}.tmp"
}

# ============= Clean up the temporary database for upgrade of this utility =============
database_build_clean_for_self_upgrade ()
{
	misc_is_superuser_privilege || return
	[ $opt_suppress_obsolete_db_clean = no ] || return
	rm -rf "${DBDIR}"
	database_build_create
	[ -e "${DBDIR}/MYVERSION" ] && mv "${DBDIR}/MYVERSION" "${DBDIR}/MYVERSION.prev"
	:
}

# ============= Check whether the temporary database is newer than the ports tree and refresh if so =============
database_build_refresh_if_obsolete ()
{
	if [ "${PORTS_INDEX_DB}" -nt "${DBDIR}"/MYVERSION ] && misc_is_superuser_privilege
	then
		if [ $opt_suppress_obsolete_db_clean = no -a "z${command}" = zclean ]
		then
			DATABASE_IS_OBSOLETE=no
			message_echo "WARNING: The temporary database is older than the ports tree." >&2
			database_build_refresh || DATABASE_IS_OBSOLETE=yes
		else
			DATABASE_IS_OBSOLETE=yes
		fi
	else
		DATABASE_IS_OBSOLETE=no
	fi
}

# ============= Register an obsolete origin =============
database_build_register_obsolete_port ()
{
	local origin dbpath pkgtag
	origin=$1
	dbpath=${DBDIR}/obsolete/$origin
	[ -e "$dbpath/complete_as_node" ] && return
	[ -d "${DBDIR}/obsolete/$origin" ] || mkdir -p "${DBDIR}/obsolete/$origin"
	pkgtag=`pkgsys_pkg_info_qO_init "$origin"`
	[ -n "$pkgtag" ] || pkgtag='[not installed]'
	echo "$pkgtag" > ${DBDIR}/obsolete/$origin/pkgtag
	for table in dependents requirements
	do
		for level in direct full
		do
			for tag in all run build
			do
				srcfile=${DBDIR}/initial/$origin/${table}.${tag}.${level}
				[ -e "$srcfile" ] && ln -f "$srcfile" "$dbpath/${table}.${tag}.${level}.src"
			done
		done
	done
	touch "$dbpath/complete_as_node"
}

# ============= Convert and register if an origin is obsolete =============
database_build_convert_and_register_origin_if_obsolete ()
{
	local origin recursedb_in recursedb output_origin iline_db origin_new date_moved why_moved
	origin=$1
	recursedb_in=$2
	recursedb=${recursedb_in:-${PORTS_MOVED_DB}}
	echo "$origin" > ${TMPDIR}/database_build_convert_and_register_origin_if_obsolete:origin
	[ -d "${PORTSDIR}/$origin" ] && return
	database_build_register_obsolete_port "$origin"
	grep -n -m 1 -E "^`str_escape_regexp \"$origin\"`\|" "$recursedb" 2> /dev/null > ${TMPDIR}/moved.info || :
	if [ `wc -l < ${TMPDIR}/moved.info` -eq 0 ]
	then
		fileedit_add_a_line_if_new "$origin" "${DBDIR}/obsolete_ports"
		if [ -n "$recursedb_in" ]
		then
			message_echo "${DEPTH_INDEX}  ===> Disappeared port (MOVED broken?)"
		else
			message_echo "${DEPTH_INDEX}  ===> Nonexistent port (your original?)"
		fi
		return 1
	else
		iline_db=`cut -d : -f 1 "${TMPDIR}/moved.info"`
		sed 1,${iline_db}d "${PORTS_MOVED_DB}" > ${TMPDIR}/MOVED.DB
		origin_new=`sed -E 's/^[0-9]+://' "${TMPDIR}/moved.info" | cut -d '|' -f 2 || :`
		date_moved=`cut -d '|' -f 3 "${TMPDIR}/moved.info" || :`
		why_moved=`cut -d '|' -f 4 "${TMPDIR}/moved.info" || :`
		if [ -n "$origin_new" ]
		then
			message_echo "${DEPTH_INDEX}  ===> Moved to $origin_new at $date_moved because \"$why_moved\""
			database_build_convert_and_register_origin_if_obsolete "$origin_new" "${TMPDIR}/MOVED.DB" || return 1
		else
			message_echo "${DEPTH_INDEX}  ===> Deleted at $date_moved because \"$why_moved\""
			fileedit_add_a_line_if_new "$origin" "${DBDIR}/obsolete_ports"
			return 1
		fi
	fi
}
# ============= [Sub-function] Get the output value of converted origin =============
_database_build_convert_and_register_origin_if_obsolete__get_origin ()
{
	cat "${TMPDIR}/database_build_convert_and_register_origin_if_obsolete:origin"
}

# ============= Get the make arguments for building the temporary database =============
database_build_setup_make_args ()
{
	local origin
	origin=$1
	{
		for key in LOCALBASE LINUXBASE PORT_DBDIR PORTSDIR DISTDIR PACKAGES PKGREPOSITORY
		do
			eval echo $key=\$$key
		done
		echo 'DISABLE_VULNERABILITIES=yes'
		if [ $opt_apply_default_config = yes ]
		then
			if pkgsys_is_dialog4ports_used
			then
				echo 'NO_DIALOG=yes'
			else
				echo 'BATCH=yes'
			fi
		fi
		dbdir=${DBDIR}/requires/$origin
		[ -d "$dbdir" ] || dbdir=${DBDIR}/conf/each_port/$origin
		cat "$dbdir/MARG.conf" 2> /dev/null || :
	} | tr '\n' ' '
}

# ============= Get the make environment variables for building the temporary database =============
database_build_setup_make_envs ()
{
	local origin dbdir
	origin=$1
	dbdir=${DBDIR}/requires/$origin
	[ -d "$dbdir" ] || dbdir=${DBDIR}/conf/each_port/$origin
	cat "$dbdir/MENV.conf" 2> /dev/null || :
}

# ============= Execute make command for building the temporary database =============
database_build_make ()
{
	local origin MAKE_ARGS MAKE_ENVS
	origin=$1
	shift
	MAKE_ARGS=`database_build_setup_make_args "$origin"`
	MAKE_ENVS=`database_build_setup_make_envs "$origin"`
	env $MAKE_ENVS make -C "${PORTSDIR}/$origin" "$@" $MAKE_ARGS
}

# ============= Set up a temporary database node for the initial state of a port =============
database_build_setup_initial_node ()
{
	local origin pkg dbpath
	origin=$1
	dbpath=${DBDIR}/initial/$origin
	[ -e "$dbpath/complete_as_node" ] && return
	rm -rf "$dbpath"
	mkdir -p "$dbpath"
	pkg=`pkg_info_qO "$origin"`
	if [ -n "$pkg" ]
	then
		echo "$pkg" > $dbpath/installed_version
		ln -f "$dbpath/installed_version" "$dbpath/pkgtag"
		pkg_info_qr "$pkg" | while read requirement
		do
			origin_requirement=`pkgsys_init_pkg_orig_by_ambiguous_matching "$requirement" || :`
			[ -n "$origin_requirement" ] && echo "$origin_requirement"
			:
		done > $dbpath/requirements.all.full
		pkg_info_qR "$pkg" | while read dependent
		do
			origin_dependent=`pkgsys_init_pkg_orig_by_ambiguous_matching "$dependent" || :`
			[ -n "$origin_dependent" ] && echo "$origin_dependent"
			:
		done > $dbpath/dependents.all.full
		for table in dependents requirements
		do
			for level in direct full
			do
				for tag in all run build
				do
					[ "${tag}.${level}" = all.full ] && continue
					ln -f "$dbpath/${table}.all.full" "$dbpath/${table}.${tag}.${level}"
				done
			done
		done
	fi
	if [ `expr "$pkg" : "^${APPNAME}-[0-9].*"` -gt 0 ]
	then
		[ -d "$dbpath" ] && touch "$dbpath/SUPPRESSED_SELF"
	elif [ `expr "$pkg" : "^pkg-[0-9].*"` -gt 0 ]
	then
		[ -d "$dbpath" ] && touch "$dbpath/SUPPRESSED_PKGNG"
	fi
	touch "$dbpath/complete_as_node"
}

# ============= Set up a temporary database node for the replaced/moved information of a port =============
database_build_setup_replace_node ()
{
	local origin_orig origin dbpath
	origin_orig=$1
	dbpath=${DBDIR}/replace/$origin_orig
	if [ ! -e "$dbpath/complete_as_node" ]
	then
		rm -rf "$dbpath"
		mkdir -p "$dbpath"
		origin=$origin_orig
		if echo "$origin_orig" | grep -q -E -f "${DBDIR}/conf/REPLACE.grep_from_pattern"
		then
			origin=`echo "$origin_orig" | sed -E -f "${DBDIR}/conf/REPLACE.sed_pattern"`
			if [ "x$origin_orig" != "x$origin" ]
			then
				if [ -n "$origin" ]
				then
					message_echo "${DEPTH_INDEX}  ===> Replaced with $origin by user configuration"
				else
					database_build_register_obsolete_port "$origin_orig"
					message_echo "${DEPTH_INDEX}  ===> Deleted by user configuration"
				fi > $dbpath/message
			fi
		fi
		if [ -n "$origin" ]
		then
			if database_build_convert_and_register_origin_if_obsolete "$origin" >> $dbpath/message
			then
				origin=`_database_build_convert_and_register_origin_if_obsolete__get_origin`
			else
				if [ "x$origin" != "x$origin_orig" ]
				then
					message_echo "${DEPTH_INDEX}  ===> Going back to the original port $origin_orig"
					if database_build_convert_and_register_origin_if_obsolete "$origin_orig"
					then
						origin=`_database_build_convert_and_register_origin_if_obsolete__get_origin`
					else
						origin=
					fi
				else
					origin=
				fi >> $dbpath/message
			fi
		fi
		[ "x$origin_orig" = "x$origin" ] || echo "$origin" > $dbpath/origin
		touch "$dbpath/complete_as_node"
	fi
	cat "$dbpath/message" 2> /dev/null || :
}

# ============= Get the inspected level for a port with the current option settings =============
database_build_get_inspected_level ()
{
	local origin origin_dependent origin_esc origin_dependent_esc
	origin=$1
	origin_dependent=$2
	origin_esc='^'`str_escape_regexp "$origin"`'$'
	origin_dependent_esc='^'`str_escape_regexp "$origin_dependent"`'$'
	if [ $opt_only_target_scope = no ]
	then
		echo full
	elif ! pkgsys_pkg_info_eO "$origin" \
		|| grep -E "$origin_esc" "${DBDIR}/stage.loop_list/ports_to_inspect" > /dev/null \
		|| [ ! -e "${DBDIR}/requires/$origin_dependent/installed_version" ] \
		|| grep -E "$origin_dependent_esc" "${DBDIR}/stage.loop_list/ports_to_inspect" > /dev/null
	then
		echo direct
	else
		echo node
	fi
}

# ============= Check whether a port has been inspected in a required level =============
database_build_is_port_already_inspected_in_required_level ()
{
	local origin origin_dependent origin_actual origin_esc inspected_level
	origin=$1
	origin_dependent=$2
	origin_actual=`echo "$origin" | sed -E -f "${DBDIR}/REPLACE.complete_sed_pattern" 2> /dev/null || :`
	[ -n "$origin_actual" ] || origin_actual=$origin
	origin_esc='^'`str_escape_regexp "$origin_actual"`'$'
	inspected_level=`database_build_get_inspected_level "$origin_actual" "$origin_dependent"`
	grep -q -E "^$origin_esc$" "${DBDIR}/ports.inspected.${inspected_level}.list" \
		"${DBDIR}/obsolete_ports" 2> /dev/null || return
	fileedit_rm_a_line "$origin" "${DBDIR}/stage.loop_list/ports_to_inspect.remain"
	[ "x$origin" = "$origin_actual" ] || \
		fileedit_rm_a_line "$origin_actual" "${DBDIR}/stage.loop_list/ports_to_inspect.remain"
	:
}

# ============= Update the current package name of a port =============
database_build_update_pkgname ()
{
	local origin pkg savefile origin_orig
	origin=$1
	savefile=${DBDIR}/requires/$origin/current_version
	if [ -e "${DBDIR}/REVERSE_REPLACE.complete_sed_pattern" ]
	then
		origin_orig=`echo "$origin" \
			| sed -E -f "${DBDIR}/REVERSE_REPLACE.complete_sed_pattern"` || :
		[ "x$origin_orig" = "x$origin" ] && origin_orig=
	else
		origin_orig=
	fi
	echo "$origin_orig" "$origin" | tr ' ' '\n' | grep -v '^$' | while read orig
	do
		pkg=`pkg_info_qO "$orig" || :`
		if [ -z "$pkg" -a -e "${DBDIR}/requires/$orig/initial_orig" ]
		then
			orig_init=`cat "${DBDIR}/requires/$orig/initial_orig"`
			pkg=`pkg_info_qO "$orig_init" || :`
		fi
		[ -n "$pkg" ] && echo "$pkg"
		:
	done > $savefile
	cat "$savefile"
}

# ============= Get the current package name of a port =============
database_build_get_pkgname ()
{
	local origin savefile
	origin=$1
	savefile=${DBDIR}/requires/$origin/current_version
	if [ -e "$savefile" ]
	then
		cat "$savefile"
	else
		database_build_update_pkgname "$origin"
	fi
}

# ============= Get the new package name of a port =============
database_build_get_new_pkgname ()
{
	local origin savefile pkg
	origin=$1
	savefile=${DBDIR}/requires/$origin/new_version
	if [ ! -e "$savefile" ]
	then
		database_build_make "$origin" package-name \
			| tr -d '\n' > $savefile
	fi
	cat "$savefile"
}

# ============= Build the original message tag showing the version upgrade =============
database_build_create_pkgtag ()
{
	local origin dbdir pkg_init pkg_new pkgtag
	origin=$1
	dbdir=${DBDIR}/requires/$origin
	pkg_init=`database_build_get_pkgname "$origin"`
	[ -n "$pkg_init" -a ! -e "$dbdir/installed_version" ] && \
		echo -n "$pkg_init" > $dbdir/installed_version
	pkg_new=`database_build_get_new_pkgname "$origin"`
	pkgtag=$pkg_init
	[ -n "$pkgtag" ] || pkgtag=$pkg_new
	if [ -z "$pkgtag" ]
	then
		pkgtag='?'
	
	elif [ "x$pkg_init" != "x$pkg_new" ]
	then
		if [ -n "$pkg_init" ]
		then
			pkgtag="$pkg_init => $pkg_new"
		else
			pkgtag="[new] $pkg_new"
		fi
	fi
	echo "$pkgtag" > $dbdir/pkgtag.orig
	echo "$pkgtag" > $dbdir/pkgtag
}

# ============= Update the message tag showing the version upgrade =============
database_build_update_pkgtag ()
{
	local origin pkg_init pkg_bak pkg_cur detail pkgtag
	origin=$1
	pkg_init=`cat "${DBDIR}/requires/$origin/installed_version" 2> /dev/null | tr '\n' ' ' | sed 's/ *$//'`
	pkg_bak=`cat "${DBDIR}/requires/$origin/backedup_version" 2> /dev/null | tr '\n' ' ' | sed 's/ *$//'`
	pkg_cur=`database_build_get_pkgname "$origin" | tr '\n' ' ' | sed 's/ *$//'`
	detail=
	if [ "x$pkg_init" != "x$pkg_cur" ]
	then
		if [ -n "$pkg_cur" ]
		then
			detail=" [currently installed version: $pkg_cur]"
		elif [ -n "$pkg_bak" ]
		then
			detail=" [currently deinstalled, previously installed version: $pkg_bak]"
		else
			detail=" [currently deinstalled]"
		fi
	fi
	pkgtag=`cat "${DBDIR}/requires/$origin/pkgtag.orig"`
	echo "$pkgtag$detail" > ${DBDIR}/requires/$origin/pkgtag
}

# ============= Check whether the currently installed package version is the latest =============
database_build_is_currentpkg_latest ()
{
	local origin pkg_cur pkg_new
	origin=$1
	pkg_cur=`database_build_get_pkgname "$origin" | tr '\n' ' ' | sed 's/ *$//'`
	pkg_new=`database_build_get_new_pkgname "$origin"`
	[ "x$pkg_cur" = "x$pkg_new" ]
}

# ============= Recursively inspect dependencies of a port and build a node database of the information =============
database_build_inspect_dependencies ()
{
	local origin_orig origin_dependent origin_orig_regexp origin origin_regexp tag stage dbpath origin_id origin_dependency DEPTH_INDEX_orig nlines iline dist_subdir_rpl inspected_level inspected_levels_compatible origin_tmp inspected_level_tmp conf_updated
	origin_orig=$1
	origin_dependent=$2
	[ -z "$origin_orig" ] && return
	database_build_is_port_already_inspected_in_required_level "$origin_orig" "$origin_dependent" && return
	origin_orig_regexp=`str_escape_regexp "$origin_orig"`
	DEPTH_INDEX_orig=${DEPTH_INDEX}
	DEPTH_INDEX="${DEPTH_INDEX}--"
	message_echo "${DEPTH_INDEX} $origin_orig"
	origin_id=`echo "$origin_orig" | tr / :`
	database_build_setup_initial_node "$origin_orig"
	database_build_setup_replace_node "$origin_orig"
	if [ -e "${DBDIR}/replace/$origin_orig/origin" ]
	then
		origin=`cat "${DBDIR}/replace/$origin_orig/origin"`
	else
		origin=$origin_orig
	fi
	origin_regexp=`str_escape_regexp "$origin"`
	inspected_level=
	inspected_levels_compatible=
	if [ -n "$origin" ]
	then
		fileedit_rm_a_line "$origin" "${DBDIR}/obsolete_ports"
		dbpath=${DBDIR}/requires/$origin
		if [ ! -e "$dbpath/complete_as_node" ]
		then
			conf_updated=
			if grep -q -E -e "^$origin_orig_regexp$" -e "^$origin_regexp$" "${DBDIR}/to_be_reconf" 2> /dev/null
			then
				message_echo "${DEPTH_INDEX}  ===> Reconfigured"
				conf_updated=y
			fi
			rm -rf "$dbpath"
			mkdir -p "$dbpath"
			[ -n "$conf_updated" ] && touch "$dbpath/conf_updated"
			[ "x$origin_orig" = "x$origin" ] || echo "$origin_orig" > $dbpath/initial_orig
			[ -e "${DBDIR}/initial/$origin_orig/SUPPRESSED_SELF" ] && touch "$dbpath/SUPPRESSED_SELF"
			[ -e "${DBDIR}/initial/$origin_orig/SUPPRESSED_PKGNG" ] && touch "$dbpath/SUPPRESSED_PKGNG"
			if [ -d "${DBDIR}/conf/each_port/$origin" ]
			then
				cp -R "${DBDIR}/conf/each_port/$origin/"* "$dbpath/" > /dev/null 2>&1 || :
			fi
			tmp_portsdb_work=${TMPDIR}/database_build_inspect_dependencies:portsdb_work
			[ -d "$tmp_portsdb_work" ] || mkdir "$tmp_portsdb_work"
			( set -e
				export PORT_DBDIR=$tmp_portsdb_work
				database_build_make "$origin" showconfig > $dbpath/ports_options.default
			)
			if [ `wc -c < $dbpath/ports_options.default` -gt 0 ]
			then
				if [ $opt_apply_default_config = yes ]
				then
					if ! pkgsys_is_dialog4ports_used
					then
						printf '\t\n' | database_build_make "$origin" config-conditional > /dev/null
					fi
				else
					database_build_make "$origin" config-conditional
				fi
				database_build_make "$origin" showconfig > $dbpath/ports_options.current
			else
				cp /dev/null "$dbpath/ports_options.current"
			fi
			if [ ! -e "${DBDIR}/initial/$origin_orig/installed_version" ]
			then
				cp "${DBDIR}/initial/$origin_orig/installed_version" "$dbpath" 2> /dev/null || :
			fi
			database_build_create_pkgtag "$origin"
			for tag in run build
			do
				database_build_make "$origin" $tag-depends-list \
					| sed -E "s/^`str_escape_regexp "${PORTSDIR}"`\///" \
					> $dbpath/requirements.$tag.orig || :
				sed -E -f "${DBDIR}/conf/REPLACE.sed_pattern" "$dbpath/requirements.$tag.orig" \
					| grep -v '^$' > $dbpath/requirements.$tag.src || :
			done
			for stage in orig src
			do
				sort -u "$dbpath/requirements.run.${stage}" "$dbpath/requirements.build.${stage}" \
					> $dbpath/requirements.all.direct.${stage}
				mv "$dbpath/requirements.build.${stage}" "$dbpath/requirements.build.direct.${stage}"
				mv "$dbpath/requirements.run.${stage}" "$dbpath/requirements.run.direct.${stage}"
			done
			touch "$dbpath/complete_as_node"
			fileedit_rm_a_line "$origin_orig" "${DBDIR}/to_be_reconf"
			fileedit_rm_a_line "$origin" "${DBDIR}/to_be_reconf"
		else
			for level in direct full
			do
				for tag in run build
				do
					rm -f "$dbpath/requirements.${tag}.${level}"
				done
			done
			for origin_tmp in "$origin" "$origin_orig"
			do
				for inspected_level_tmp in full direct node
				do
					fileedit_rm_a_line "$origin_tmp" "${DBDIR}/ports.inspected.${inspected_level_tmp}.list"
				done
			done
		fi
		inspected_level=`database_build_get_inspected_level "$origin" "$origin_dependent"`
		case $inspected_level in
		full)
			grep -v -E -f "${DBDIR}/installed_ports.grep_pattern" \
				"$dbpath/requirements.all.direct.src" > ${TMPDIR}/missing.$origin_id || :
			inspected_levels_compatible='full direct node'
			;;
		direct)
			grep -v -E -f "${DBDIR}/installed_ports.grep_pattern" \
				"$dbpath/requirements.all.direct.src" > ${TMPDIR}/missing.$origin_id || :
			inspected_levels_compatible='direct node'
			;;
		node)
			cp /dev/null "${TMPDIR}/missing.$origin_id"
			inspected_levels_compatible='node'
			;;
		esac
		nlines=`wc -l < ${TMPDIR}/missing.$origin_id`
		iline=1
		while [ $iline -le $nlines ]
		do
			origin_dependency=`sed -n ${iline}p "${TMPDIR}/missing.$origin_id"`
			iline=$(($iline+1))
			grep -q -E '^'`str_escape_regexp "$origin_dependency"`'$' \
				"${DBDIR}/ports.inspected.list" 2> /dev/null && \
					continue
			database_build_inspect_dependencies "$origin_dependency" "$origin"
		done
		rm -f "${TMPDIR}/missing.$origin_id"
		dist_subdir_rpl=`database_query_get_makevar_val "$origin" DIST_SUBDIR | str_escape_replaceval_filter` || :
		[ -n "$dist_subdir_rpl" ] && dist_subdir_rpl="$dist_subdir_rpl\/"
		database_build_make "$origin" fetch-urlall-list \
			| sed -E "s/.*\/([^\/]+)$/$dist_subdir_rpl\1/" \
			| sort -u >> ${DBDIR}/distfiles.inspected
	fi
	if [ "x$origin_orig" != "x$origin" ]
	then
		if [ -n "$origin" ]
		then
			for inspected_level_tmp in $inspected_levels_compatible
			do
				fileedit_add_a_line_if_new "$origin" "${DBDIR}/ports.inspected.${inspected_level_tmp}.list"
			done
			fileedit_add_a_line_if_new "$origin" "${DBDIR}/ports.inspected.list"
			fileedit_add_a_line_if_new "$origin" "${DBDIR}/inspected_ports.update"
			fileedit_rm_a_line "$origin" "${DBDIR}/stage.loop_list/ports_to_inspect.remain"
		fi
	fi
	for inspected_level_tmp in $inspected_levels_compatible
	do
		fileedit_add_a_line_if_new "$origin_orig" "${DBDIR}/ports.inspected.${inspected_level_tmp}.list"
	done
	fileedit_add_a_line_if_new "$origin_orig" "${DBDIR}/ports.inspected.list"
	fileedit_add_a_line_if_new "$origin_orig" "${DBDIR}/inspected_ports.update"
	fileedit_rm_a_line "$origin_orig" "${DBDIR}/stage.loop_list/ports_to_inspect.remain"
	message_echo "${DEPTH_INDEX}  ===> ok"

	DEPTH_INDEX=${DEPTH_INDEX_orig}
}

# ============= Build and get a list of the complete recursive dependencies of a port =============
database_build_get_complete_recursive_dependency ()
{
	local table origin dbpath suffix dstfile
	table=$1
	origin=$2
	suffix=$3
	dbpath=${DBDIR}/requires/$origin
	[ -e "$dbpath/$table.direct$suffix" ] || return 0
	dstfile=$dbpath/$table.full$suffix
	if [ ! -e "$dstfile" ]
	then
		while read origin_requirement
		do
			echo "$origin_requirement"
			database_build_get_complete_recursive_dependency "$table" "$origin_requirement" "$suffix"
		done < $dbpath/$table.direct$suffix | sort -u > $dstfile.tmp
		mv "$dstfile.tmp" "$dstfile"
	fi
	cat "$dstfile"
}

# ============= Inspect the necessity of requirements of a necessary port even in case of skipping unchanged ports =============
# Here the "necessity" means that a port must be kept installed, newly installed or reinstalled for run- or build-time operations.
# In other words, "unnecessary" ports are what are not yet installed and not required by any "necessary" ports neither in run- or build-time.
database_build_inspect_necessity_for_only_new_upgrade ()
{
	local origin level dbpath tmpfile
	origin=$1
	level=$2
	dbpath=${DBDIR}/requires/$origin
	[ ! -d "$dbpath" -o -e "$dbpath/necessary_port.${level}" ] && return
	tmpfile=${TMPDIR}/database_build_inspect_necessity_for_only_new_upgrade:`echo "$origin" | tr / :`
	if database_query_does_a_port_need_update "$origin"
	then
		sort -u "$dbpath/requirements.build.direct" "$dbpath/requirements.run.${level}" || :
	else
		cat "$dbpath/requirements.run.${level}" || :
	fi 2> /dev/null > $tmpfile
	if [ `wc -l < $tmpfile` -gt 0 ]
	then
		while read origin_requirement
		do
			database_build_inspect_necessity_for_only_new_upgrade "$origin_requirement" "$level"
		done < $tmpfile
	fi
	rm "$tmpfile"
	touch "$dbpath/necessary_port.${level}"
}

# ============= Build complement relations to new dependents for targets recursively =============
database_build_complement_to_new_dependents_for_targets ()
{
	local origin dbdir reqdir level
	origin=$1
	dbdir=${DBDIR}/targets/$origin
	reqdir=${DBDIR}/requires/$origin
	[ -d "$dbdir" ] || return 0
	echo "$origin" >> ${DBDIR}/stage.loop_list/parse_target_attr_info
	if [ -e "$dbdir/target_itself" ] && database_query_does_a_port_need_update "$origin"
	then
		for level in direct full
		do
			cat "$reqdir/requirements.all.${level}" 2> /dev/null | while read origin_requirement
			do
				fileedit_add_a_line_if_new "$origin" \
					"${DBDIR}/targets/$origin_requirement/complement_for_new_dependents.${level}"
			done
		done
		cat "$reqdir/requirements.all.direct" "$reqdir/requirements.all.full" 2> /dev/null \
			| sort -u | while read origin_requirement
		do
			database_build_complement_to_new_dependents_for_targets "$origin_requirement"
		done
	fi
}

# ============= Build target attribute files recursively =============
database_build_target_attributes ()
{
	local origin dbdir reqdir _is_target _is_requires_requirements _is_initial_requirements _is_requires_dependents _is_initial_dependents _is_requires_requirements_complement _is_relevant tag level database table
	origin=$1
	dbdir=${DBDIR}/targets/$origin
	reqdir=${DBDIR}/requires/$origin
	[ -d "$dbdir" ] || return 0
	[ -e "$dbdir/COMPLETE_ATTRS" ] && return
	_is_target=
	[ -e "$dbdir/target_itself" ] && _is_target=y
	for tag in all run build none
	do
		for level in direct full
		do
			_is_relevant=${_is_target}
			_is_requires_requirements=
			_is_initial_requirements=
			_is_requires_dependents=
			_is_initial_dependents=
			_is_requires_requirements_complement=
			for database in requires initial
			do
				for table in requirements dependents
				do
					if [ -e "$dbdir/target_${database}_${table}.${tag}.${level}" ]
					then
						eval _is_${database}_${table}=y
						_is_relevant=y
					fi
				done
			done
			if [ -z "${_is_requires_requirements}" -a -e "$dbdir/complement_for_new_dependents.${level}" ]
			then
				_is_requires_requirements_complement=y
				_is_relevant=y
			fi
			[ -n "${_is_relevant}" ] && cat > $dbdir/attrs.${tag}.${level} << eof
_is_target=${_is_target}
_is_requires_requirements=${_is_requires_requirements}
_is_initial_requirements=${_is_initial_requirements}
_is_requires_dependents=${_is_requires_dependents}
_is_initial_dependents=${_is_initial_dependents}
_is_requires_requirements_complement=${_is_requires_requirements_complement}
eof
		done
	done
	touch "$dbdir/COMPLETE_ATTRS"
}

# ============= Order the ports considering dependencies =============
database_build_order_ports_considering_dependencies ()
{
	touch "${DBDIR}/reinst_order.list.tmp"
	cat > ${TMPDIR}/order_dependencies.awk << eof
BEGIN {
	it = 0;
	i = 0;
}
{
	if (NF == 0)
	{
		i = 0;
	}
	else
	{
		if (i == 0)
		{
			target = \$0;
			sub (/\/requirements.all\.full$/, "", target);
			sub (/^\.\//, "", target);
			srcikey[it] = target;
			srckeyi[target] = it;
			it++;
		}
		else
		{
			src[it-1,i-1] = \$0;
			srcimax[it-1] = srcsize[it-1] = i;
		}
		i++;
	}
}
END {
	ntargets = it;
	norder = 0;
	order_str = "";
	icycle = 0;
	lf_order_str = "";
	while (1)
	{
		is_operated = 0;
		for (it = 0; it < ntargets; it++)
		{
			if (!(it in srcikey)) continue;
			if (srcsize[it] > 0) continue;
			is_operated = 1;
			target = srcikey[it];
			delete srcikey[it];
			order[norder++] = target;
			order_str = order_str lf_order_str;
			order_str = sprintf ("%s%s", order_str, target);
			lf_order_str = "\n";
			for (jt = 0; jt < ntargets; jt++)
			{
				for (j = 0; j < srcimax[jt]; j++)
				{
					if ((jt,j) in src && src[jt,j] == target)
					{
						delete src[jt,j];
						srcsize[jt]--;
						break;
					}
				}
			}
		}
		if (is_operated == 0) break;
		icycle++;
	}
	reinst_order_list = sprintf ("%s%s", ENVIRON["DBDIR"], "/reinst_order.list.tmp");
	print order_str > reinst_order_list;
	unsatisfied = "";
	for (it = 0; it < ntargets; it++)
	{
		if (srcsize[it] == 0) continue;
		reqs = "";
		sp_reqs = "";
		for (i = 0; i < srcimax[it]; i++)
		{
			if ((it,i) in src)
			{
				reqs = reqs sp_reqs src[it,i];
				sp_reqs = ", ";
			}
		}
		unsatisfied = sprintf ("%s%s [%d] (%s)\n", unsatisfied, srcikey[it], srcsize[it], reqs);
	}
	if (unsatisfied != "")
	{
		unsatisfied_list = sprintf ("%s%s", ENVIRON["DBDIR"], "/unsatisfied.list");
		print unsatisfied > unsatisfied_list;
		exit 1;
	}
}
eof
	(cd "${DBDIR}/requires" && \
		find . -depth 3 -name requirements.all.full -exec echo {} \; -exec cat {} \; -exec echo \;) | \
		env "DBDIR=${DBDIR}" awk -f "${TMPDIR}"/order_dependencies.awk || return
	grep -v '^$' "${DBDIR}/reinst_order.list.tmp" > "${DBDIR}/reinst_order.list" || :
}

# ============= [Sub-function] Common operations for resetting configurations for a port =============
_database_build_reset_a_port_confdb ()
{
	local origin origin_replace db
	origin=$1
	[ -d "${DBDIR}/initial/$origin" -o -d "${DBDIR}/replace/$origin" \
		-o -d "${DBDIR}/requires/$origin" -o -d "${DBDIR}/targets/$origin" ] || return 0
	touch "${DBDIR}/inspected_ports_only_partially"
	{
		echo "$origin"
		cat "${DBDIR}/replace/$origin/origin" 2> /dev/null || :
		for dbtag in initial requires
		do
			for tabel in requirements dependents
			do
				cat "${DBDIR}/$dbtag/$origin/$tabel.all.full" 2> /dev/null || :
				cat "${DBDIR}/$dbtag/$origin/$tabel.all.full.orig" 2> /dev/null || :
				rm -f "${DBDIR}/$dbtag/$origin/$tabel.run.full" \
					"${DBDIR}/$dbtag/$origin/$tabel.build.full" \
					"${DBDIR}/$dbtag/$origin/is_customized" 2> /dev/null
			done
		done
	} | sort -u >> ${DBDIR}/inspected_ports.update
	for level in full direct node
	do
		fileedit_rm_a_line "$origin" "${DBDIR}/ports.inspected.${level}.list"
	done
	fileedit_rm_a_line "$origin" "${DBDIR}/obsolete_ports"
	fileedit_rm_a_line "$origin" "${DBDIR}/ports.inspected.list"
	cat "${DBDIR}/replace/$origin/origin" 2> /dev/null || :
}

# ============= Clear database directories for an origin =============
database_build_clear_db_dirs ()
{
	local origin db
	origin=$1
	for db in requires replace targets obsolete
	do
		rm -rf "${DBDIR}/$db/$origin"
	done
}

# ============= Remove configurations for a port permanently =============
database_build_forget ()
{
	local origin origin_replace
	origin=$1
	origin_replace=`_database_build_reset_a_port_confdb "$origin"`
	fileedit_rm_a_line "$origin" "${DBDIR}/targets_specified_so_far"
	[ -z "$origin_replace" ] || database_build_forget "$origin_replace"
	database_build_clear_db_dirs "$origin"
}

# ============= Patch to the temporary database so as to re-inspect and reinstall ports whose configurations were changed =============
database_build_patch_reconf ()
{
	local origin origin_replace
	origin=$1
	origin_replace=`_database_build_reset_a_port_confdb "$origin"`
	[ -d "${DBDIR}/initial/$origin" -o -d "${DBDIR}/requires/$origin" ] && \
		fileedit_add_a_line_if_new "$origin" "${DBDIR}/stage.loop_list/ports_to_inspect"
	fileedit_add_a_line_if_new "$origin" "${DBDIR}/to_be_reconf"
	[ -z "$origin_replace" ] || database_build_patch_reconf "$origin_replace"
	database_build_clear_db_dirs "$origin"
}

# ============= Post-processes after finishing to inspect dependencies =============
database_build_post_inspect_dependencies ()
{
	local table
	touch "${DBDIR}/obsolete_ports" "${DBDIR}/inspected_ports.update"
	find "${DBDIR}/requires" -depth 2 -type d \
		| sed -E 's|.*/([^/]+/[^/]+)$|\1|' > ${DBDIR}/inspected_ports
	find "${DBDIR}/initial" -depth 2 -type d \
		| sed -E 's|.*/([^/]+/[^/]+)$|\1|' > ${DBDIR}/inspected_ports.initial
	sort -u "${DBDIR}/inspected_ports" "${DBDIR}/inspected_ports.initial" > ${DBDIR}/inspected_ports.all
	str_escape_regexp_filter < ${DBDIR}/inspected_ports \
		| sed 's/^/^/; s/$/$/' > ${DBDIR}/inspected_ports.grep_pattern
	str_escape_regexp_filter < ${DBDIR}/inspected_ports.initial \
		| sed 's/^/^/; s/$/$/' > ${DBDIR}/inspected_ports.initial.grep_pattern
	str_escape_regexp_filter < ${DBDIR}/inspected_ports.all \
		| sed 's/^/^/; s/$/$/' > ${DBDIR}/inspected_ports.all.grep_pattern
	cat "${DBDIR}/conf/HOLD_PORTS.grep_pattern" "${DBDIR}/need.grep_pattern" \
		2> /dev/null > ${TMPDIR}/INSPECT_ALL_DEPENDENCIES:obsolete_ports.exclude.grep_pattern || :
	grep -v -E -f "${TMPDIR}/INSPECT_ALL_DEPENDENCIES:obsolete_ports.exclude.grep_pattern" \
		"${DBDIR}/obsolete_ports" > ${DBDIR}/obsolete_ports.can_be_deleted || :
	cp /dev/null "${DBDIR}/REPLACE.complete_sed_pattern.tmp"
	cp /dev/null "${DBDIR}/REVERSE_REPLACE.complete_sed_pattern.tmp"
	find "${DBDIR}/replace" -depth 3 -type f -name origin | while read nodepath
	do
		origin_orig=`expr "$nodepath" : '.*/\([^/][^/]*/[^/][^/]*\)/origin$'`
		origin=`cat "$nodepath"`
		origin_orig_regexp=`str_escape_regexp "$origin_orig"`
		origin_orig_esc=`str_escape_replaceval "$origin_orig"`
		origin_regexp=`str_escape_regexp "$origin"`
		origin_esc=`str_escape_replaceval "$origin"`
		echo "s/^$origin_orig_regexp$/$origin_esc/" >> ${DBDIR}/REPLACE.complete_sed_pattern.tmp
		[ -z "$origin_regexp" ] && continue
		echo "s/^$origin_regexp$/$origin_orig_esc/" >> ${DBDIR}/REVERSE_REPLACE.complete_sed_pattern.tmp
	done
	mv "${DBDIR}/REPLACE.complete_sed_pattern.tmp" "${DBDIR}/REPLACE.complete_sed_pattern"
	mv "${DBDIR}/REVERSE_REPLACE.complete_sed_pattern.tmp" "${DBDIR}/REVERSE_REPLACE.complete_sed_pattern"
	if [ `cat "${DBDIR}/inspected_ports.update" 2> /dev/null | wc -l` -gt 0 ]
	then
		grep -E -f "${DBDIR}/inspected_ports.all.grep_pattern" "${DBDIR}/inspected_ports.update" \
			> ${DBDIR}/inspected_ports.update.tmp || :
		mv "${DBDIR}/inspected_ports.update.tmp" "${DBDIR}/inspected_ports.update"
		if [ $opt_only_target_scope = yes ]
		then
			touch "${DBDIR}/inspected_ports_only_partially"
		else
			rm -f "${DBDIR}/inspected_ports_only_partially"
		fi
		if program_chk_stage_loop_complete CONVERT_REQUIREMENTS_LIST
		then
			mv "${DBDIR}/inspected_ports.update" "${DBDIR}/stage.loop_list/convert_dependency_lists"
		else
			cat "${DBDIR}/inspected_ports.update" "${DBDIR}/stage.loop_list/convert_dependency_lists" \
				2> /dev/null | sort -u > ${DBDIR}/stage.loop_list/convert_dependency_lists.tmp
			mv "${DBDIR}/stage.loop_list/convert_dependency_lists.tmp" \
				"${DBDIR}/stage.loop_list/convert_dependency_lists"
			rm -f "${DBDIR}/inspected_ports.update"
		fi
	else
		program_chk_stage_loop_complete CONVERT_REQUIREMENTS_LIST \
			&& rm -f "${DBDIR}/stage.loop_list/convert_dependency_lists"
	fi
	{
		sed -E -f "${DBDIR}/REPLACE.complete_sed_pattern" "${DBDIR}/need.list" || :
		cat "${DBDIR}/need.list" || :
	} 2> /dev/null | sort -u > ${DBDIR}/need.with_replaced.list
	for table in requirements dependents itself
	do
		[ -e "${DBDIR}/stage.loop_list/target_$table.specified" ] || continue
		sed -E -f "${DBDIR}/REPLACE.complete_sed_pattern" \
			"${DBDIR}/stage.loop_list/target_$table.specified" \
			> ${DBDIR}/stage.loop_list/target_$table.replaced.specified
	done	
	cp /dev/null "${DBDIR}/update_dependencies"
}
