#!/bin/sh -e
# ==============================================================================
# portsreinstall library script
# - Operations of commands as well as check of command line arguments -
# Copyright (C) 2013 Mamoru Sakaue, MwGhennndo, All Rights Reserved.
# This software is distributed under the 2-Clause BSD License.
# ==============================================================================

# ============= Variables =============
COMMAND_MODE=do
COMMAND_SHIFT=0
COMMAND_OPERATION=
COMMAND_SAVE_DIR=
COMMAND_LOAD_FILE=
COMMAND_SHOW_SUBJECT=
COMMAND_DO_MODE=all
COMMAND_RESTART=
COMMAND_SHOW_OPTIONS=
COMMAND_SHOW_DEPTAG=
COMMAND_SHOW_LEVEL=

# ============= Check the number of following command line arguments (in case glob arguments are needed) =============
_command_parse_args__chk_glob_args ()
{
	local nargs
	nargs=1
	[ $nargs -gt 0 ] && return
	message_echo "ERROR: No port glob is specified." >&2
	exit 1
}

# ============= Check the number of following command line arguments (in case without arguments) =============
_command_parse_args__chk_no_arg ()
{
	local nargs term_redundant_argument
	nargs=$1
	[ $nargs -eq 0 ] && return
	term_redundant_argument='A redundant argument is'
	[ $nargs -gt 1 ] && term_redundant_argument='Redundant arguments are'
	message_echo "ERROR: $term_redundant_argument specified." >&2
	exit 1
}

# ============= Execute command operations before getting the temporary database ready =============
command_exec_before_db_creation ()
{
	local COMMAND_RESTART COMMAND_MODE COMMAND_OPERATION
	COMMAND_RESTART="$@"
	COMMAND_MODE=${1:-do}
	shift || :
	case $COMMAND_MODE in
	clean)
		COMMAND_OPERATION=${1:-normal}
		shift || :
		case $COMMAND_OPERATION in
		force)
			message_echo "INFO: The temporary database is tried to be cleaned up without checking the privilege."
			rm -rf "${DBDIR}"
			message_echo "Done"
			exit
			;;
		esac
		_command_parse_args__chk_no_arg $#
		;;
	esac
}

# ============= Check and parse command line arguments =============
command_parse_args ()
{
	local num_args_init
	num_args_init=$#
	COMMAND_RESTART="$@"
	COMMAND_MODE=${1:-do}
	shift || :
	case $COMMAND_MODE in
	clean)
		COMMAND_OPERATION=${1:-normal}
		shift || :
		case $COMMAND_OPERATION in
		normal)
			misc_chk_privilege
			;;
		esac
		_command_parse_args__chk_no_arg $#
		;;
	reset)
		misc_chk_privilege
		COMMAND_OPERATION=${1:-all}
		shift || :
		case $COMMAND_OPERATION in
		all|keepopts)
			;;
		*)
			message_echo "ERROR: Invalid operation [$COMMAND_OPERATION]; it must be empty, \"all\" or \"keepopts\"." >&2
			exit 1
			;;
		esac
		_command_parse_args__chk_no_arg $#
		;;
	ok|taboo|need|noneed)
		misc_chk_privilege
		[ $COMMAND_MODE = ok ] && database_query_chk_preparation_completion
		temp_warn_obsolete_temp_db
		COMMAND_OPERATION=$1
		shift || :
		case $COMMAND_OPERATION in
		add|del)
			;;
		'')
			message_echo "ERROR: Missing operation which must be \"add\" or \"del\"." >&2
			exit 1
			;;
		*)
			message_echo "ERROR: Invalid operation [$COMMAND_OPERATION]; it must be \"add\" or \"del\"." >&2
			exit 1
			;;
		esac
		_command_parse_args__chk_glob_args $#
		;;
	reselect)
		misc_chk_privilege
		database_query_chk_preparation_completion
		temp_warn_obsolete_temp_db
		COMMAND_OPERATION=$1
		shift || :
		case $COMMAND_OPERATION in
		leaves)
			if [ -e "${DBDIR}/inspected_ports_only_partially" ]
			then
				message_echo "ERROR: Leaf ports cannot be analyzed because the dependency inspection is partial." >&2
				message_echo "Executing redo command with -N option by disabling -o option fixes this situation." >&2
				exit 1
			fi
			;;
		obsolete)
			;;
		'')
			message_echo "ERROR: Missing operation which must be \"add\" or \"del\"." >&2
			exit 1
			;;
		*)
			message_echo "ERROR: Invalid operation [$COMMAND_OPERATION]; it must be \"add\" or \"del\"." >&2
			exit 1
			;;
		esac
		_command_parse_args__chk_glob_args $#
		;;
	save)
		misc_chk_privilege
		if [ ! -d "${DBDIR}" ]
		then
			message_echo "ERROR: The temporary database has not been created yet." >&2
			exit 1
		fi
		COMMAND_SAVE_DIR=$1
		shift || :
		_command_parse_args__chk_no_arg $#
		if [ -z "$COMMAND_SAVE_DIR" ]
		then
			message_echo "ERROR: Directory to save the temporary database archive is not specified." >&2
			exit 1
		fi
		if [ ! -d "$COMMAND_SAVE_DIR" ]
		then
			message_echo "ERROR: Directory [$COMMAND_SAVE_DIR] is not found." >&2
			exit 1
		fi
		;;
	load)
		misc_chk_privilege
		if [ -e "${DBDIR}/in_use" ]
		then
			message_echo "ERROR: A temporary database exists." >&2
			message_echo "You must execute" >&2
			message_echo "        ${APPNAME} clean" >&2
			message_echo "before executing \"load\" command." >&2
			exit 1
		fi
		COMMAND_LOAD_FILE=$1
		shift || :
		_command_parse_args__chk_no_arg $#
		if [ -z "$COMMAND_LOAD_FILE" ]
		then
			message_echo "ERROR: No temporary database archive is specified." >&2
			exit 1
		fi
		if [ ! -f "$COMMAND_LOAD_FILE" ]
		then
			message_echo "ERROR: The temporary database archive is not found." >&2
			exit 1
		fi
		COMMAND_LOAD_FILE=`realpath "$COMMAND_LOAD_FILE"`
		;;
	glob)
		_command_parse_args__chk_glob_args $#
		;;
	options)
		_command_parse_args__chk_no_arg $#
		;;
	reconf|forget|escape|restore)
		misc_chk_privilege
		temp_warn_obsolete_temp_db
		_command_parse_args__chk_glob_args $#
		;;
	pkgsanity)
		misc_chk_privilege
		temp_warn_obsolete_temp_db
		;;
	show)
		database_query_chk_preparation_completion
		temp_warn_obsolete_temp_db
		if ! expr "$1" : '@.*' > /dev/null
		then
			COMMAND_SHOW_SUBJECT=${1:-todo}
			shift || :
		else
			COMMAND_SHOW_SUBJECT=todo
		fi
		COMMAND_SHOW_OPTIONS=$1
		if expr "$COMMAND_SHOW_OPTIONS" : '@.*' > /dev/null
		then
			COMMAND_SHOW_DEPTAG=`expr "$COMMAND_SHOW_OPTIONS," : '@\(.*\)' | cut -d , -f 1` || :
			COMMAND_SHOW_LEVEL=`expr "$COMMAND_SHOW_OPTIONS," : '@\(.*\)' | cut -d , -f 2` || :
			case $COMMAND_SHOW_DEPTAG in
			all|run|build|'')	;;
			*)
				message_echo "ERROR: Invalid show option [$COMMAND_SHOW_OPTIONS]." >&2
				exit 1
				;;
			esac
			case $COMMAND_SHOW_LEVEL in
			full|direct|'')	;;
			*)
				message_echo "ERROR: Invalid show option [$COMMAND_SHOW_OPTIONS]." >&2
				exit 1
				;;
			esac
			shift || :
		fi
		case $COMMAND_SHOW_SUBJECT in
		todo|done|redo|resolved|failure|taboo|need|noneed|restored|deleted|conflict)
			_command_parse_args__chk_no_arg $#
			;;
		initrequirements|requirements|initdependents|dependents|status)
			_command_parse_args__chk_glob_args $#
			;;
		*)
			message_echo "ERROR: Invalid subject [$COMMAND_SHOW_SUBJECT]." >&2
			exit 1
			;;
		esac
		;;
	all|prepare)
		COMMAND_DO_MODE=$COMMAND_MODE
		COMMAND_MODE=do
		misc_chk_privilege
		temp_warn_obsolete_temp_db
		_command_parse_args__chk_no_arg $#
		;;
	redo|do)
		COMMAND_DO_MODE=${1:-all}
		shift || :
		case $COMMAND_DO_MODE in
		all|prepare);;
		*)
			message_echo "ERROR: Invalid operation mode [$COMMAND_DO_MODE]." >&2
			exit 1
			;;
		esac
		misc_chk_privilege
		temp_warn_obsolete_temp_db
		_command_parse_args__chk_no_arg $#
		if [ "$COMMAND_DO_MODE" = prepare ]
		then
			COMMAND_RESTART=prepare
		else
			COMMAND_RESTART=
		fi
		;;
	*)
		message_echo "ERROR: Invalid command [$COMMAND_MODE]." >&2
		exit 1
		;;
	esac
	COMMAND_SHIFT=$(($num_args_init - $#))
}

# ============= Notify that option settings are reset =============
command_exec_without_pkgtools__notify_reset_options ()
{
	message_echo "NOTE: Option settings are ignored (because of no effect) and reset."
}

# ============= Execute command operations which do not need package tools =============
command_exec_without_pkgtools ()
{
	local dbdir_parent 
	case $COMMAND_MODE in
	clean)
		message_echo "Starting to clean up the temporary database..."
		command_exec_without_pkgtools__notify_reset_options
		rm -rf "${DBDIR}"
		message_echo "Done"
		exit
		;;
	load)
		message_echo "Starting to load the temporary database from the archive..."
		command_exec_without_pkgtools__notify_reset_options
		dbdir_parent=`dirname "${DBDIR}"`
		[ -d "$dbdir_parent" ] || mkdir -p "$dbdir_parent"
		tar xzf "$COMMAND_LOAD_FILE" -C "$dbdir_parent" --exclude "*/.lock"
		message_echo "Done"
		exit
		;;
	esac
}

# ============= Notify that option settings are ignored because of no effect =============
_command_exec_irrespective_of_saved_options__notify_ignored_options ()
{
	message_echo "NOTE: Option settings are ignored because they have no effect on this command."
}

# ============= Operations of forget command =============
command_forget ()
{
	message_echo "The temporary database is trying to forget about the specified ports as much as possible."
	message_echo "Concretely, the data on each of the specified ports and their requirements/dependents is removed unless initially installed or required by other preserved ports."
	message_echo
	
	# Preparation for inspection of the specified ports
	PROGRAM_DEPENDS=''
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Preparation for inspection of the specified ports"
		rm -rf "${DBDIR}/forget"
		mkdir "${DBDIR}/forget"
		for list in masters remove_scope
		do
			rm -f "${DBDIR}/stage.loop_list/forget_$list"*
		done
		message_echo
	}
	program_exec_and_record_completion FORGET::PREPARATION_INSPECT_MASTER
	
	# (Re)initialization of the specified ports to inspect
	pkgsys_eval_ports_glob "$@" > ${DBDIR}/stage.loop_list/forget_masters
	
	# Inspection of the specified ports
	PROGRAM_DEPENDS='FORGET::PREPARATION_INSPECT_MASTER'
	_program_exec_restartable_loop_operation__routine ()
	{
		local origin origin_orig
		origin=$1
		if [ -e "${DBDIR}/REVERSE_REPLACE.complete_sed_pattern" ]
		then
			origin_orig=`echo "$origin" \
				| sed -E -f "${DBDIR}/REVERSE_REPLACE.complete_sed_pattern"`
		else
			origin_orig=$origin
		fi
		if [ ! -e "${DBDIR}/initial/$origin/installed_version" \
			-a ! -e "${DBDIR}/initial/$origin_orig/installed_version" \
			-a `cat "${DBDIR}/requires/$origin/dependents.all.full" 2> /dev/null | wc -l` -eq 0 ]
		then
			message_echo "$origin"
			echo "$origin" >> ${DBDIR}/forget/remove.master
			cat "${DBDIR}/requires/$origin/requirements.all.full" \
				2> /dev/null >> ${DBDIR}/forget/remove.scope || :
			database_build_forget "$origin"
		fi
	}
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Inspection of the specified ports"
		message_echo "----------------"
		program_exec_restartable_loop_operation forget_masters
		message_echo "----------------"
		cat "${DBDIR}/forget/remove.scope" 2> /dev/null \
			| sort -u > ${DBDIR}/forget/remove.scope.tmp
		mv "${DBDIR}/forget/remove.scope.tmp" "${DBDIR}/forget/remove.scope"
		cat "${DBDIR}/forget/remove.master" "${DBDIR}/forget/remove.scope" 2> /dev/null \
			| str_escape_regexp_filter \
			| sed 's/^/^/;s/$/$/' > ${DBDIR}/forget/remove.scope.grep_pattern
		ln -f "${DBDIR}/forget/remove.scope" "${DBDIR}/stage.loop_list/forget_remove_scope"
		message_echo
	}
	program_exec_and_record_completion FORGET::INSPECT_MASTER
	
	# Inspection of the requirements of the specified ports to remove
	PROGRAM_DEPENDS='FORGET::INSPECT_MASTER'
	_program_exec_restartable_loop_operation__routine ()
	{
		local origin origin_orig
		origin=$1
		if [ -e "${DBDIR}/REVERSE_REPLACE.complete_sed_pattern" ]
		then
			origin_orig=`echo "$origin" \
				| sed -E -f "${DBDIR}/REVERSE_REPLACE.complete_sed_pattern"`
		else
			origin_orig=$origin
		fi
		if [ ! -e "${DBDIR}/initial/$origin/installed_version" \
			-a ! -e "${DBDIR}/initial/$origin_orig/installed_version" ] \
			&& ! grep -v -E -f "${DBDIR}/forget/remove.scope.grep_pattern" \
				"${DBDIR}/requires/$origin/dependents.all.full" \
				> /dev/null 2>&1
		then
			message_echo "$origin"
			database_build_forget "$origin"
			echo "$origin" >> ${DBDIR}/forget/remove
		fi
	}
	_program_exec_and_record_completion__operation ()
	{
		message_section_title "Inspection of the requirements of the specified ports to remove"
		message_echo "----------------"
		program_exec_restartable_loop_operation forget_remove_scope
		message_echo "----------------"
		cat "${DBDIR}/forget/remove.master" "${DBDIR}/forget/remove" 2> /dev/null \
			| str_escape_regexp_filter \
			| sed 's/^/^/;s/$/$/' > ${DBDIR}/forget/remove.grep_pattern
		cat "${DBDIR}/inspected_ports.update" 2> /dev/null | sort -u \
			| grep -v -E -f "${DBDIR}/forget/remove.grep_pattern" \
				> ${DBDIR}/inspected_ports.update.tmp || :
		mv "${DBDIR}/inspected_ports.update.tmp" "${DBDIR}/inspected_ports.update"
		message_echo
	}
	program_exec_and_record_completion FORGET::INSPECT_REQUIREMENTS_OF_REMOVED_PORTS
	
	# Set up so that ports are inspected again in the building process of the temporary database
	program_deregister_stage_complete INSPECT_ALL_DEPENDENCIES
	
	# Clean up the database for this command because it is no more effective
	program_deregister_stage_complete FORGET::PREPARATION_INSPECT_MASTER
	message_echo "Done"
}

# ============= Execute command operations which are irrespective of option settings =============
command_exec_irrespective_of_saved_options ()
{
	local dbfile tmp_manually_done_diff evalated_globs dbdir_parent dbdir_node arcfile grandtitle title isfirst origin origin_regexp backup_pkg origin_orig origin_replace pkg_orig tmp_done_orig tmp_pkgsanity nlines iline pkg
	case $COMMAND_MODE in
	ok)
		dbfile=${DBDIR}/manually_done.list
		[ -e "$dbfile" ] || touch "$dbfile"
		cp "$dbfile" "$dbfile.tmp"
		case $COMMAND_OPERATION in
		add)
			pkgsys_register_evaluated_globs add "$dbfile.tmp" "$@"
			message_echo "`str_linearize_list_and \"$*\"` is/are registered to the list of manually resolved ports"
			;;
		del)
			pkgsys_register_evaluated_globs remove "$dbfile.tmp" "$@"
			message_echo "`str_linearize_list_and \"$*\"` is/are deregistered from the list of manually resolved ports"
			;;
		esac
		tmp_manually_done_diff_old=${TMPDIR}/command_exec_irrespective_of_saved_options:manually_done.list.diff_old
		tmp_manually_done_diff_new=${TMPDIR}/command_exec_irrespective_of_saved_options:manually_done.list.diff_new
		if fileedit_manipulate_old_new_lines "$dbfile" "$dbfile.tmp" \
			"$tmp_manually_done_diff_old" "$tmp_manually_done_diff_new"
		then
			while read origin
			do
				database_record_failure "$origin"
			done < $tmp_manually_done_diff_old
			while read origin
			do
				database_record_success "$origin"
			done < $tmp_manually_done_diff_new
		fi
		mv "$dbfile.tmp" "$dbfile"
		_command_exec_irrespective_of_saved_options__notify_ignored_options
		message_echo "Now the following ports have been manually resolved:"
		message_cat "$dbfile"
		exit
		;;
	taboo)
		evalated_globs=`str_linearize_list_and "$@"`
		case $COMMAND_OPERATION in
		add)
			pkgsys_register_evaluated_globs add "${DBDIR}/taboo.list" "$@"
			message_echo "$evalated_globs is/are registered to the list of ports to be ignored."
			;;
		del)
			pkgsys_register_evaluated_globs remove "${DBDIR}/taboo.list" "$@"
			message_echo "$evalated_globs is/are deregistered from the list of ports to be ignored."
			;;
		esac
		fileedit_combine_lists "${DBDIR}/conf/TABOO:PORTS.parsed" "${DBDIR}/taboo.list" > ${DBDIR}/taboo.all.list
		_command_exec_irrespective_of_saved_options__notify_ignored_options
		message_echo "Now the following ports are registered to be ignored:"
		message_cat "${DBDIR}/taboo.all.list"
		exit
		;;
	need)
		evalated_globs=`str_linearize_list_and "$@"`
		case $COMMAND_OPERATION in
		add)
			pkgsys_register_evaluated_globs add "${DBDIR}/need.list" "$@"
			message_echo "$evalated_globs is/are registered to the list of necessary ports."
			;;
		del)
			pkgsys_register_evaluated_globs remove "${DBDIR}/need.list" "$@"
			message_echo "$evalated_globs is/are deregistered from the list of necessary ports."
			;;
		esac
		str_escape_regexp_filter < ${DBDIR}/need.list \
			| sed 's/^/^/; s/$/$/' > ${DBDIR}/need.grep_pattern
		{
			sed -E -f "${DBDIR}/REPLACE.complete_sed_pattern" "${DBDIR}/need.list" || :
			cat "${DBDIR}/need.list" || :
		} 2> /dev/null | sort -u > ${DBDIR}/need.with_replaced.list
		program_deregister_stage_complete DETERMINE_SPECIFIED_TARGETS
		_command_exec_irrespective_of_saved_options__notify_ignored_options
		message_echo "Now the following ports are registered to be necessary:"
		message_cat "${DBDIR}/need.list"
		exit
		;;
	noneed)
		evalated_globs=`str_linearize_list_and "$@"`
		case $COMMAND_OPERATION in
		add)
			pkgsys_register_evaluated_globs add "${DBDIR}/noneed.list" "$@"
			message_echo "$evalated_globs is/are registered to the list of unnecessary ports."
			;;
		del)
			pkgsys_register_evaluated_globs remove "${DBDIR}/noneed.list" "$@"
			message_echo "$evalated_globs is/are deregistered from the list of unnecessary ports."
			;;
		esac
		program_deregister_stage_complete INSPECT_PRIMARY_LEAF_PORTS
		_command_exec_irrespective_of_saved_options__notify_ignored_options
		message_echo "Now the following ports are registered to be unnecessary:"
		message_cat "${DBDIR}/noneed.list"
		exit
		;;
	reselect)
		case $COMMAND_OPERATION in
		leaves)
			if ! deinstall_select_leaf_ports_to_delete force
			then
				case $? in
				2)
					message_echo "INFO: No leaf port is found."
					;;
				3)
					message_echo "INFO: Leaf ports are undefined because requirements of some ports are not fully inspected."
					;;
				esac
			else
				program_deregister_stage_complete COLLECT_LEAF_PORTS_TO_DELETE
			fi
			;;
		obsolete)
			if ! deinstall_select_obsolete_ports_to_delete force
			then
				case $? in
				2)
					message_echo "INFO: No obsolete package is found."
					;;
				esac
			else
				program_deregister_stage_complete COLLECT_OBSOLETE_PORTS_TO_DELETE
			fi
			;;
		esac
		exit
		;;
	save)
		dbdir_parent=`dirname "${DBDIR}"`
		dbdir_node=`basename "${DBDIR}"`
		arcfile=`realpath "$COMMAND_SAVE_DIR"`/${APPNAME}_`date +%Y%m%d_%H%M%S`.tar.gz
		_command_exec_irrespective_of_saved_options__notify_ignored_options
		message_echo "Starting to save the temporary database as [$arcfile]..."
		tar czf "$arcfile" -C "$dbdir_parent" "$dbdir_node"
		message_echo "Done"
		exit
		;;
	glob)
		_command_exec_irrespective_of_saved_options__notify_ignored_options
		message_echo "Evaluated port origins are as follows:"
		pkgsys_eval_ports_glob "$@"
		exit
		;;
	reconf)
		_command_exec_irrespective_of_saved_options__notify_ignored_options
		message_echo "Temporary database sections for the following ports are reset for later rebuild:"
		for origin in `pkgsys_eval_ports_glob "$@"`
		do
			origin_regexp=`str_escape_regexp "$origin"`
			if grep -q -E "^$origin_regexp$" "${DBDIR}/inspected_ports" 2> /dev/null
			then
				database_build_make "$origin" config
				database_build_patch_reconf "$origin"
			else
				message_echo "$origin (not inspected)"
				continue
			fi
		done
		program_deregister_stage_complete INSPECT_ALL_DEPENDENCIES
		message_echo "Done"
		exit
		;;
	forget)
		_command_exec_irrespective_of_saved_options__notify_ignored_options
		command_forget "$@"
		exit
		;;
	escape)
		_command_exec_irrespective_of_saved_options__notify_ignored_options
		message_echo "Backing up and deleting the following packages for a temporary escape:"
		message_echo
		for origin in `pkgsys_eval_ports_glob "$@"`
		do
			pkgsys_register_evaluated_globs add "${DBDIR}/taboo.list" "$origin"
			message_echo "  Registered $origin as taboo."
			pkg=`pkg_info_qO "$origin"` || continue
			[ -n "$pkg" ] || continue
			if backup_pkg=`pkgsys_get_backup_pkg "$origin"`
			then
				message_echo "INFO: A backup package for $pkg ($origin) already exists as $backup_pkg."
			elif backup_pkg=`pkgsys_create_backup_pkg "$pkg" "${DBDIR}/backup_packages"`
			then
				message_echo "  Backed up $pkg ($origin) into $backup_pkg"
			else
				message_echo "ERROR: Failed to back up $pkg ($origin)." >&2
				message_echo >&2
				continue
			fi
			pkg_delete_f "$pkg" || \
			{
				message_echo "ERROR: Failed to deinstall $pkg ($origin)." >&2
				message_echo >&2
			}
			message_echo "  Deinstalled $pkg ($origin)."
			message_echo
		done
		fileedit_combine_lists "${DBDIR}/conf/TABOO:PORTS.parsed" "${DBDIR}/taboo.list" > ${DBDIR}/taboo.all.list
		message_echo "Done"
		exit
		;;
	restore)
		_command_exec_irrespective_of_saved_options__notify_ignored_options
		message_echo "Restoring the following temporary escaped packages:"
		message_echo
		tmp_done_orig=${TMPDIR}/command_exec_irrespective_of_saved_options::restore::done_orig
		cp /dev/null "$tmp_done_orig"
		for origin in `pkgsys_eval_ports_glob "$@"`
		do
			pkgsys_register_evaluated_globs remove "${DBDIR}/taboo.list" "$origin"
			message_echo "  Deregistered $origin from taboo."
			origin_regexp=`str_escape_regexp "$origin"`
			grep -E -q "^$origin_regexp$" "$tmp_done_orig" || :
			if pkg_info_eO "$origin"
			then
				pkg=`pkg_info_qO "$origin"` || :
				message_echo "WARNING: $pkg ($origin) is already installed." >&2
				message_echo >&2
				continue
			fi
			origin_orig=`echo "$origin" \
				| sed -E -f "${DBDIR}/REVERSE_REPLACE.complete_sed_pattern"`
			if [ "x$origin_orig" = "x$origin" ] && pkg_info_eO "$origin_orig"
			then
				pkg_orig=`pkg_info_qO "$origin_orig"` || :
				message_echo "WARNING: An original version of $origin ($pkg_orig, $origin_orig) is already installed." >&2
				message_echo >&2
				continue
			fi
			origin_replace=`echo "$origin" \
				| sed -E -f "${DBDIR}/REPLACE.complete_sed_pattern"`
			if [ "x$origin_replace" != "x$origin" ]
			then
				if pkg_info_eO "$origin_replace"
				then
					pkg_replace=`pkg_info_qO "$origin_replace"` || :
					message_echo "WARNING: A replacement of $origin ($pkg_replace, $origin_replace) is already installed." >&2
					message_echo >&2
					continue
				fi
				if backup_pkg=`pkgsys_get_backup_pkg "$origin_replace" 2> /dev/null`
				then
					message_echo "INFO: $origin is replaced with $origin_replace ($pkg_replace)."
					echo "$origin_replace" >> $tmp_done_orig
					origin=$origin_replace
				fi
			else
				backup_pkg=
			fi
			if [ -z "$backup_pkg" ] && ! backup_pkg=`pkgsys_get_backup_pkg "$origin" 2> /dev/null`
			then
				message_echo "ERROR: Backup for $origin is not found." >&2
				message_echo >&2
				continue
			fi
			pkg=`pkgsys_pkgarc_to_pkgname "$backup_pkg"`
			if reinstall_chk_forbidden_conflicts "$pkg"
			then
				message_echo "WARNING: $pkg ($origin) is skipped because it conflicts with installed packages." >&2
				message_echo >&2
				continue
			fi
			if ! pkg_add_fF "$backup_pkg"
			then
				message_echo "ERROR: Failed to restore $pkg ($origin)." >&2
				message_echo >&2
			fi
			message_echo "  Restored $pkg ($origin)."
			message_echo
		done
		fileedit_combine_lists "${DBDIR}/conf/TABOO:PORTS.parsed" "${DBDIR}/taboo.list" > ${DBDIR}/taboo.all.list
		message_echo "Done"
		exit
		;;
	pkgsanity)
		tmp_pkgsanity=${TMPDIR}/command_exec_irrespective_of_saved_options:pkgsanity
		_command_exec_irrespective_of_saved_options__notify_ignored_options
		if [ $# -eq 0 ]
		then
			message_echo "Sanity check of the installed files for each package:"
			message_echo
			pkg_info_Ea > $tmp_pkgsanity
		else
			message_echo "Examining the installed files for each specified package:"
			message_echo
			pkgsys_eval_ports_glob "$@" | while read origin
			do
				pkg_info_qO "$origin" || :
			done | grep -v '^[[:space:]]*$' > $tmp_pkgsanity
		fi
		nlines=`wc -l < $tmp_pkgsanity`
		iline=1
		while [ $iline -le $nlines ]
		do
			pkg=`sed -n ${iline}p "$tmp_pkgsanity"`
			iline=$((${iline}+1))
			origin=`pkg_info_qo "$pkg"`
			[ -n "$origin" ] || continue
			origin_regexp=`str_escape_regexp "$origin"`
			grep -q -E "^$origin_regexp$" "${DBDIR}/damaged_package" 2>/dev/null && continue
			pkgsys_sanitychk_pkgcontents "$pkg" is_reinstall_encouraged && continue
			if [ ! -d "${PORTSDIR}/$origin" ]
			then
				message_echo "WARNING: $pkg ($origin) is obsolete." >&2
				message_echo >&2
				continue
			fi
			if [ $is_reinstall_encouraged = no ]
			then
				if [ $opt_batch_mode = no ]
				then
					message_echo "Do you want to reinstall it? (y/[n])"
					message_query_yn_default_no || continue
				fi
			else
				if [ $opt_batch_mode = no ]
				then
					message_echo "Do you want to reinstall it? ([y]/n)"
					message_query_yn_default_yes || continue
				fi
				database_record_reconf_recover_sanity "$origin"
			fi
		done
		message_echo "Done"
		exit
		;;
	redo)
		if [ $opt_reload_conf = yes ]
		then
			program_deregister_stage_complete SAVE_PREV_CONF
		fi
		;;
	esac
}

# ============= Output of "options" command =============
_command_exec_before_tools_upgrade__options ()
{
	local opt_control group_reset group_remain reset_opts result_opts str_opt_control str_rstoption notice_reset notice_result
	opt_control=$1
	group_reset=$2
	group_remain=$3
	reset_opts=`options_inverse_parse $group_reset`
	result_opts=`options_inverse_parse $group_remain`
	if [ $is_batch_mode = yes ]
	then
		printf '%s\t%s\t%s\n' "$opt_control" "$reset_opts" "$result_opts"
	else
		str_opt_control=`echo "$opt_control" | sed -E 's/(.)/-\1 /g;s/ *$//'`
		str_rstoption=Option`[ \`echo -n "$opt_control" | wc -c\` -ge 2 ] && echo s` || :
		notice_reset=`[ -z "$reset_opts" ] && echo ' (nothing)'` || :
		notice_result=`[ -z "$result_opts" ] && echo ' (default)'` || :
		echo "$str_rstoption [$str_opt_control] will reset [$reset_opts]$notice_reset so that the settings become [$result_opts]$notice_result."
	fi
}

# ============= Execute command operations which should be done without upgrade of tools =============
command_exec_before_tools_upgrade ()
{
	local flag_filter_skip_unchanged flag_filter_only_target pkgnamedb dbsuffix list origin_target
	case $COMMAND_MODE in
	reset)
		message_echo "Starting to reset the temporary database by preserving the initial snapshot of installed packages..."
		find "${DBDIR}" -depth 1 \
			-not \( -name saved_options.sh \
				-or -name initial -or -name MYVERSION \
				-or -name backup_failure -or -name installed_ports\* \) \
			-exec rm -rf {} \; 2> /dev/null || :
		case $COMMAND_OPERATION in
		all)
			command_exec_without_pkgtools__notify_reset_options
			rm -f "${DBDIR}/saved_options.sh"
			;;
		keepopts)
			message_echo "INFO: option settings are preserved."
			;;
		esac
		find "${DBDIR}/initial" -depth 2 -type d \
			| sed -E 's|.*/([^/]+/[^/]+)$|\1|' > ${DBDIR}/inspected_ports.update
		mkdir -p "${DBDIR}/stage.loop_list"
		message_echo "Done"
		exit
		;;
	options)
		( set -e
			is_batch_mode=$opt_batch_mode
			options_set_default
			[ -e "${DBDIR}/saved_options.sh" ] && . "${DBDIR}/saved_options.sh"
			savedopts=`options_inverse_parse`
			if [ $is_batch_mode = yes ]
			then
				printf '\t%s\n' "$savedopts"
			else
				echo "The saved setting is [$savedopts]."
			fi
			_command_exec_before_tools_upgrade__options M \
				'renewable_anytime' 'non_renewable renewable_in_redo_on_target renewable_in_redo_on_conf'
			_command_exec_before_tools_upgrade__options N \
				'renewable_in_redo_on_target' 'non_renewable renewable_anytime renewable_in_redo_on_conf'
			_command_exec_before_tools_upgrade__options L \
				'renewable_in_redo_on_conf' 'non_renewable renewable_anytime renewable_in_redo_on_target'
			_command_exec_before_tools_upgrade__options MN \
				'renewable_anytime renewable_in_redo_on_target' 'non_renewable renewable_in_redo_on_conf'
			_command_exec_before_tools_upgrade__options ML \
				'renewable_anytime renewable_in_redo_on_conf' 'non_renewable renewable_in_redo_on_target'
			_command_exec_before_tools_upgrade__options NL \
				'renewable_in_redo_on_target renewable_in_redo_on_conf' 'non_renewable renewable_anytime'
			_command_exec_before_tools_upgrade__options MNL \
				'renewable_anytime renewable_in_redo_on_target renewable_in_redo_on_conf' 'non_renewable'
		)
		exit
		;;
	show)
		flag_filter_skip_unchanged=
		flag_filter_only_target=
		pkgnamedb='requires obsolete initial'
		[ -n "$COMMAND_SHOW_DEPTAG" ] || COMMAND_SHOW_DEPTAG=`options_get_dependency_type`
		[ -n "$COMMAND_SHOW_LEVEL" ] || COMMAND_SHOW_LEVEL=`options_get_dependency_level`
		dbsuffix=$COMMAND_SHOW_DEPTAG.$COMMAND_SHOW_LEVEL
		case $COMMAND_SHOW_SUBJECT in
		todo|done|redo|resolved|failure|taboo|need|noneed|deleted|restored|conflict)
			database_query_show_single_list_exec "$COMMAND_SHOW_SUBJECT" \
				"$COMMAND_SHOW_DEPTAG" "$COMMAND_SHOW_LEVEL" || :
			;;
		initrequirements)
			grandtitle="Dependencies based on the initially installed packages"
			title="The following port(s) was/were required by %s:"
			list=requirements.$dbsuffix
			pkgnamedb='initial'
			[ $COMMAND_SHOW_DEPTAG = none ] && \
				message_echo "WARNING: This command has no meaning with the current options setting." >&2
			database_query_for_each_matching_port "$grandtitle" "$title" "$list" "$pkgnamedb" \
				"$COMMAND_SHOW_DEPTAG" "$COMMAND_SHOW_LEVEL" "$@"
			;;
		requirements)
			grandtitle="Dependencies based on the latest ports tree"
			title="The following port(s) is/are required by %s:"
			list=requirements.$dbsuffix
			pkgnamedb='requires obsolete'
			[ $COMMAND_SHOW_DEPTAG = none ] && \
				message_echo "WARNING: This command has no meaning with the current options setting." >&2
			database_query_for_each_matching_port "$grandtitle" "$title" "$list" "$pkgnamedb" \
				"$COMMAND_SHOW_DEPTAG" "$COMMAND_SHOW_LEVEL" "$@"
			;;
		initdependents)
			grandtitle="Dependencies based on the initially installed packages"
			title="The following port(s) depended on %s:"
			list=dependents.$dbsuffix
			pkgnamedb='initial'
			[ $COMMAND_SHOW_DEPTAG = none ] && \
				message_echo "WARNING: This command has no meaning with the current options setting." >&2
			database_query_for_each_matching_port "$grandtitle" "$title" "$list" "$pkgnamedb" \
				"$COMMAND_SHOW_DEPTAG" "$COMMAND_SHOW_LEVEL" "$@"
			;;
		dependents)
			grandtitle="Dependencies based on the latest ports tree"
			title="The following port(s) depend(s) on %s:"
			list=dependents.$dbsuffix
			pkgnamedb='requires obsolete'
			[ $COMMAND_SHOW_DEPTAG = none ] && \
				message_echo "WARNING: This command has no meaning with the current options setting." >&2
			database_query_for_each_matching_port "$grandtitle" "$title" "$list" "$pkgnamedb" \
				"$COMMAND_SHOW_DEPTAG" "$COMMAND_SHOW_LEVEL" "$@"
			;;
		status)
			grandtitle="Success/failure status in (re)installation"
			lists='todo|done|redo|resolved|failure|taboo|need|noneed|deleted|restored|conflict'
			database_query_for_list_inclusion_of_matching_port "$grandtitle" "$lists" "$pkgnamedb" \
				"$COMMAND_SHOW_DEPTAG" "$COMMAND_SHOW_LEVEL" "$@"
			;;
		esac
		exit
		;;
	esac
}

# ============= Execute command operations which must be done before building the temporary database =============
command_exec_before_building_tempdb ()
{
	case $COMMAND_MODE in
	redo)
		program_deregister_stage_complete REDO_INIT
		if [ $opt_reset_targets = yes ]
		then
			program_deregister_stage_complete DETERMINE_SPECIFIED_TARGETS
			program_deregister_stage_complete INSPECT_ALL_DEPENDENCIES
			rm -rf "${DBDIR}/targets"
		fi
		;;
	esac
	program_deregister_stage_complete FORGET::PREPARATION_INSPECT_MASTER
}

# ============= Execute command operations which must be done before actual (re/de)installation processes =============
command_exec_before_actual_re_de_installation ()
{
	case $COMMAND_MODE in
	do|redo)
		case $COMMAND_DO_MODE in
		prepare)
			message_cat 3<< eof
Done (skipped reinstallation) at `message_timestamp`

 You can restart this process from the aborted/terminated point by executing without options or arguments as:
  ${APPNAME}
eof
			temp_terminate_process ()
			{
			}
			exit
			;;
		esac
		;;
	esac
}
